/* eslint-disable */

var noderedRuntime;
function getNoderedRuntime() {
  if (noderedRuntime) return noderedRuntime;
  noderedRuntime = typeof global === 'undefined' ? window.NODERED_RUNTIME : global.NODERED_RUNTIME;
  return noderedRuntime
}

/**
 * © Copyright IBM Corp. 2016, 2017 All Rights Reserved
 *   Project name: JSONata
 *   This project is licensed under the MIT License, see LICENSE
 */

/**
 * @module JSONata
 * @description JSON query and transformation language
 */

/**
 * jsonata
 * @function
 * @param {Object} expr - JSONata expression
 * @returns {{evaluate: evaluate, assign: assign}} Evaluated expression
 */
var jsonata = (function () {
  'use strict';

  var operators = {
    '.': 75,
    '[': 80,
    ']': 0,
    '{': 70,
    '}': 0,
    '(': 80,
    ')': 0,
    ',': 0,
    '@': 75,
    '#': 70,
    ';': 80,
    ':': 80,
    '?': 20,
    '+': 50,
    '-': 50,
    '*': 60,
    '/': 60,
    '%': 60,
    '|': 20,
    '=': 40,
    '<': 40,
    '>': 40,
    '`': 80,
    '**': 60,
    '..': 20,
    ':=': 10,
    '!=': 40,
    '<=': 40,
    '>=': 40,
    '~>': 40,
    'and': 30,
    'or': 25,
    'in': 40,
    '&': 50,
    '!': 0,   // not an operator, but needed as a stop character for name tokens
    '~': 0   // not an operator, but needed as a stop character for name tokens
  };

  var escapes = {  // JSON string escape sequences - see json.org
    '"': '"',
    '\\': '\\',
    '/': '/',
    'b': '\b',
    'f': '\f',
    'n': '\n',
    'r': '\r',
    't': '\t'
  };

  // Tokenizer (lexer) - invoked by the parser to return one token at a time
  var tokenizer = function (path) {
    var position = 0;
    var length = path.length;

    var create = function (type, value) {
      var obj = { type: type, value: value, position: position };
      return obj;
    };

    var scanRegex = function () {
      // the prefix '/' will have been previously scanned. Find the end of the regex.
      // search for closing '/' ignoring any that are escaped, or within brackets
      var start = position;
      var depth = 0;
      var pattern;
      var flags;
      while (position < length) {
        var currentChar = path.charAt(position);
        if (currentChar === '/' && path.charAt(position - 1) !== '\\' && depth === 0) {
          // end of regex found
          pattern = path.substring(start, position);
          if (pattern === '') {
            throw {
              code: "S0301",
              stack: (new Error()).stack,
              position: position
            };
          }
          position++;
          currentChar = path.charAt(position);
          // flags
          start = position;
          while (currentChar === 'i' || currentChar === 'm') {
            position++;
            currentChar = path.charAt(position);
          }
          flags = path.substring(start, position) + 'g';
          return new RegExp(pattern, flags);
        }
        if ((currentChar === '(' || currentChar === '[' || currentChar === '{') && path.charAt(position - 1) !== '\\') {
          depth++;
        }
        if ((currentChar === ')' || currentChar === ']' || currentChar === '}') && path.charAt(position - 1) !== '\\') {
          depth--;
        }

        position++;
      }
      throw {
        code: "S0302",
        stack: (new Error()).stack,
        position: position
      };
    };

    var next = function (prefix) {
      if (position >= length) return null;
      var currentChar = path.charAt(position);
      // skip whitespace
      while (position < length && ' \t\n\r\v'.indexOf(currentChar) > -1) {
        position++;
        currentChar = path.charAt(position);
      }
      // test for regex
      if (prefix !== true && currentChar === '/') {
        position++;
        return create('regex', scanRegex());
      }
      // handle double-char operators
      if (currentChar === '.' && path.charAt(position + 1) === '.') {
        // double-dot .. range operator
        position += 2;
        return create('operator', '..');
      }
      if (currentChar === ':' && path.charAt(position + 1) === '=') {
        // := assignment
        position += 2;
        return create('operator', ':=');
      }
      if (currentChar === '!' && path.charAt(position + 1) === '=') {
        // !=
        position += 2;
        return create('operator', '!=');
      }
      if (currentChar === '>' && path.charAt(position + 1) === '=') {
        // >=
        position += 2;
        return create('operator', '>=');
      }
      if (currentChar === '<' && path.charAt(position + 1) === '=') {
        // <=
        position += 2;
        return create('operator', '<=');
      }
      if (currentChar === '*' && path.charAt(position + 1) === '*') {
        // **  descendant wildcard
        position += 2;
        return create('operator', '**');
      }
      if (currentChar === '~' && path.charAt(position + 1) === '>') {
        // ~>  chain function
        position += 2;
        return create('operator', '~>');
      }
      // test for single char operators
      if (operators.hasOwnProperty(currentChar)) {
        position++;
        return create('operator', currentChar);
      }
      // test for string literals
      if (currentChar === '"' || currentChar === "'") {
        var quoteType = currentChar;
        // double quoted string literal - find end of string
        position++;
        var qstr = "";
        while (position < length) {
          currentChar = path.charAt(position);
          if (currentChar === '\\') { // escape sequence
            position++;
            currentChar = path.charAt(position);
            if (escapes.hasOwnProperty(currentChar)) {
              qstr += escapes[currentChar];
            } else if (currentChar === 'u') {
              // \u should be followed by 4 hex digits
              var octets = path.substr(position + 1, 4);
              if (/^[0-9a-fA-F]+$/.test(octets)) {
                var codepoint = parseInt(octets, 16);
                qstr += String.fromCharCode(codepoint);
                position += 4;
              } else {
                throw {
                  code: "S0104",
                  stack: (new Error()).stack,
                  position: position
                };
              }
            } else {
              // illegal escape sequence
              throw {
                code: "S0103",
                stack: (new Error()).stack,
                position: position,
                token: currentChar
              };

            }
          } else if (currentChar === quoteType) {
            position++;
            return create('string', qstr);
          } else {
            qstr += currentChar;
          }
          position++;
        }
        throw {
          code: "S0101",
          stack: (new Error()).stack,
          position: position
        };
      }
      // test for numbers
      var numregex = /^-?(0|([1-9][0-9]*))(\.[0-9]+)?([Ee][-+]?[0-9]+)?/;
      var match = numregex.exec(path.substring(position));
      if (match !== null) {
        var num = parseFloat(match[0]);
        if (!isNaN(num) && isFinite(num)) {
          position += match[0].length;
          return create('number', num);
        } else {
          throw {
            code: "S0102",
            stack: (new Error()).stack,
            position: position,
            token: match[0]
          };
        }
      }
      // test for names
      var i = position;
      var ch;
      var name;
      for (; ;) {
        ch = path.charAt(i);
        if (i === length || ' \t\n\r\v'.indexOf(ch) > -1 || operators.hasOwnProperty(ch)) {
          if (path.charAt(position) === '$') {
            // variable reference
            name = path.substring(position + 1, i);
            position = i;
            return create('variable', name);
          } else {
            name = path.substring(position, i);
            position = i;
            switch (name) {
              case 'or':
              case 'in':
              case 'and':
                return create('operator', name);
              case 'true':
                return create('value', true);
              case 'false':
                return create('value', false);
              case 'null':
                return create('value', null);
              default:
                if (position === length && name === '') {
                  // whitespace at end of input
                  return null;
                }
                return create('name', name);
            }
          }
        } else {
          i++;
        }
      }
    };

    return next;
  };

  /**
   * Parses a function signature definition and returns a validation function
   * @param {string} signature - the signature between the <angle brackets>
   * @returns {Function} validation function
   */
  function parseSignature(signature) {
    // create a Regex that represents this signature and return a function that when invoked,
    // returns the validated (possibly fixed-up) arguments, or throws a validation error
    // step through the signature, one symbol at a time
    var position = 1;
    var params = [];
    var param = {};
    var prevParam = param;
    while (position < signature.length) {
      var symbol = signature.charAt(position);
      if (symbol === ':') {
        // TODO figure out what to do with the return type
        // ignore it for now
        break;
      }

      var next = function () {
        params.push(param);
        prevParam = param;
        param = {};
      };

      var findClosingBracket = function (str, start, openSymbol, closeSymbol) {
        // returns the position of the closing symbol (e.g. bracket) in a string
        // that balances the opening symbol at position start
        var depth = 1;
        var position = start;
        while (position < str.length) {
          position++;
          symbol = str.charAt(position);
          if (symbol === closeSymbol) {
            depth--;
            if (depth === 0) {
              // we're done
              break; // out of while loop
            }
          } else if (symbol === openSymbol) {
            depth++;
          }
        }
        return position;
      };

      switch (symbol) {
        case 's': // string
        case 'n': // number
        case 'b': // boolean
        case 'l': // not so sure about expecting null?
        case 'o': // object
          param.regex = '[' + symbol + 'm]';
          param.type = symbol;
          next();
          break;
        case 'a': // array
          //  normally treat any value as singleton array
          param.regex = '[asnblfom]';
          param.type = symbol;
          param.array = true;
          next();
          break;
        case 'f': // function
          param.regex = 'f';
          param.type = symbol;
          next();
          break;
        case 'j': // any JSON type
          param.regex = '[asnblom]';
          param.type = symbol;
          next();
          break;
        case 'x': // any type
          param.regex = '[asnblfom]';
          param.type = symbol;
          next();
          break;
        case '-': // use context if param not supplied
          prevParam.context = true;
          prevParam.contextRegex = new RegExp(prevParam.regex); // pre-compiled to test the context type at runtime
          prevParam.regex += '?';
          break;
        case '?': // optional param
        case '+': // one or more
          prevParam.regex += symbol;
          break;
        case '(': // choice of types
          // search forward for matching ')'
          var endParen = findClosingBracket(signature, position, '(', ')');
          var choice = signature.substring(position + 1, endParen);
          if (choice.indexOf('<') === -1) {
            // no parameterized types, simple regex
            param.regex = '[' + choice + 'm]';
          } else {
            // TODO harder
            throw {
              code: "S0402",
              stack: (new Error()).stack,
              value: choice,
              offset: position
            };
          }
          param.type = '(' + choice + ')';
          position = endParen;
          next();
          break;
        case '<': // type parameter - can only be applied to 'a' and 'f'
          if (prevParam.type === 'a' || prevParam.type === 'f') {
            // search forward for matching '>'
            var endPos = findClosingBracket(signature, position, '<', '>');
            prevParam.subtype = signature.substring(position + 1, endPos);
            position = endPos;
          } else {
            throw {
              code: "S0401",
              stack: (new Error()).stack,
              value: prevParam.type,
              offset: position
            };
          }
          break;
      }
      position++;
    }
    var regexStr = '^' +
      params.map(function (param) {
        return '(' + param.regex + ')';
      }).join('') +
      '$';
    var regex = new RegExp(regexStr);
    var getSymbol = function (value) {
      var symbol;
      if (isFunction(value)) {
        symbol = 'f';
      } else {
        var type = typeof value;
        switch (type) {
          case 'string':
            symbol = 's';
            break;
          case 'number':
            symbol = 'n';
            break;
          case 'boolean':
            symbol = 'b';
            break;
          case 'object':
            if (value === null) {
              symbol = 'l';
            } else if (Array.isArray(value)) {
              symbol = 'a';
            } else {
              symbol = 'o';
            }
            break;
          case 'undefined':
            // any value can be undefined, but should be allowed to match
            symbol = 'm'; // m for missing
        }
      }
      return symbol;
    };

    var throwValidationError = function (badArgs, badSig) {
      // to figure out where this went wrong we need apply each component of the
      // regex to each argument until we get to the one that fails to match
      var partialPattern = '^';
      var goodTo = 0;
      for (var index = 0; index < params.length; index++) {
        partialPattern += params[index].regex;
        var match = badSig.match(partialPattern);
        if (match === null) {
          // failed here
          throw {
            code: "T0410",
            stack: (new Error()).stack,
            value: badArgs[goodTo],
            index: goodTo + 1
          };
        }
        goodTo = match[0].length;
      }
      // if it got this far, it's probably because of extraneous arguments (we
      // haven't added the trailing '$' in the regex yet.
      throw {
        code: "T0410",
        stack: (new Error()).stack,
        value: badArgs[goodTo],
        index: goodTo + 1
      };
    };

    return {
      definition: signature,
      validate: function (args, context) {
        var suppliedSig = '';
        args.forEach(function (arg) {
          suppliedSig += getSymbol(arg);
        });
        var isValid = regex.exec(suppliedSig);
        if (isValid) {
          var validatedArgs = [];
          var argIndex = 0;
          params.forEach(function (param, index) {
            var arg = args[argIndex];
            var match = isValid[index + 1];
            if (match === '') {
              if (param.context) {
                // substitute context value for missing arg
                // first check that the context value is the right type
                var contextType = getSymbol(context);
                // test contextType against the regex for this arg (without the trailing ?)
                if (param.contextRegex.test(contextType)) {
                  validatedArgs.push(context);
                } else {
                  // context value not compatible with this argument
                  throw {
                    code: "T0411",
                    stack: (new Error()).stack,
                    value: context,
                    index: argIndex + 1
                  };
                }
              } else {
                validatedArgs.push(arg);
                argIndex++;
              }
            } else {
              // may have matched multiple args (if the regex ends with a '+'
              // split into single tokens
              match.split('').forEach(function (single) {
                if (param.type === 'a') {
                  if (single === 'm') {
                    // missing (undefined)
                    arg = undefined;
                  } else {
                    arg = args[argIndex];
                    var arrayOK = true;
                    // is there type information on the contents of the array?
                    if (typeof param.subtype !== 'undefined') {
                      if (single !== 'a' && match !== param.subtype) {
                        arrayOK = false;
                      } else if (single === 'a') {
                        if (arg.length > 0) {
                          var itemType = getSymbol(arg[0]);
                          if (itemType !== param.subtype.charAt(0)) { // TODO recurse further
                            arrayOK = false;
                          } else {
                            // make sure every item in the array is this type
                            var differentItems = arg.filter(function (val) {
                              return (getSymbol(val) !== itemType);
                            });
                            arrayOK = (differentItems.length === 0);
                          }
                        }
                      }
                    }
                    if (!arrayOK) {
                      throw {
                        code: "T0412",
                        stack: (new Error()).stack,
                        value: arg,
                        index: argIndex + 1,
                        type: param.subtype // TODO translate symbol to type name
                      };
                    }
                    // the function expects an array. If it's not one, make it so
                    if (single !== 'a') {
                      arg = [arg];
                    }
                  }
                  validatedArgs.push(arg);
                  argIndex++;
                } else {
                  validatedArgs.push(arg);
                  argIndex++;
                }
              });
            }
          });
          return validatedArgs;
        }
        throwValidationError(args, suppliedSig);
      }
    };
  }

  // This parser implements the 'Top down operator precedence' algorithm developed by Vaughan R Pratt; http://dl.acm.org/citation.cfm?id=512931.
  // and builds on the Javascript framework described by Douglas Crockford at http://javascript.crockford.com/tdop/tdop.html
  // and in 'Beautiful Code', edited by Andy Oram and Greg Wilson, Copyright 2007 O'Reilly Media, Inc. 798-0-596-51004-6

  var parser = function (source) {
    var node;
    var lexer;

    var symbol_table = {};

    var base_symbol = {
      nud: function () {
        return this;
      }
    };

    var symbol = function (id, bp) {
      var s = symbol_table[id];
      bp = bp || 0;
      if (s) {
        if (bp >= s.lbp) {
          s.lbp = bp;
        }
      } else {
        s = Object.create(base_symbol);
        s.id = s.value = id;
        s.lbp = bp;
        symbol_table[id] = s;
      }
      return s;
    };

    var advance = function (id, infix) {
      if (id && node.id !== id) {
        var code;
        if (node.id === '(end)') {
          // unexpected end of buffer
          code = "S0203";
        } else {
          code = "S0202";
        }
        throw {
          code: code,
          stack: (new Error()).stack,
          position: node.position,
          token: node.id,
          value: id
        };
      }
      var next_token = lexer(infix);
      if (next_token === null) {
        node = symbol_table["(end)"];
        node.position = source.length;
        return node;
      }
      var value = next_token.value;
      var type = next_token.type;
      var symbol;
      switch (type) {
        case 'name':
        case 'variable':
          symbol = symbol_table["(name)"];
          break;
        case 'operator':
          symbol = symbol_table[value];
          if (!symbol) {
            throw {
              code: "S0204",
              stack: (new Error()).stack,
              position: next_token.position,
              token: value
            };
          }
          break;
        case 'string':
        case 'number':
        case 'value':
          type = "literal";
          symbol = symbol_table["(literal)"];
          break;
        case 'regex':
          type = "regex";
          symbol = symbol_table["(regex)"];
          break;
        /* istanbul ignore next */
        default:
          throw {
            code: "S0205",
            stack: (new Error()).stack,
            position: next_token.position,
            token: value
          };
      }

      node = Object.create(symbol);
      node.value = value;
      node.type = type;
      node.position = next_token.position;
      return node;
    };

    // Pratt's algorithm
    var expression = function (rbp) {
      var left;
      var t = node;
      advance(null, true);
      left = t.nud();
      while (rbp < node.lbp) {
        t = node;
        advance();
        left = t.led(left);
      }
      return left;
    };

    // match infix operators
    // <expression> <operator> <expression>
    // left associative
    var infix = function (id, bp, led) {
      var bindingPower = bp || operators[id];
      var s = symbol(id, bindingPower);
      s.led = led || function (left) {
        this.lhs = left;
        this.rhs = expression(bindingPower);
        this.type = "binary";
        return this;
      };
      return s;
    };

    // match infix operators
    // <expression> <operator> <expression>
    // right associative
    var infixr = function (id, bp, led) {
      var bindingPower = bp || operators[id];
      var s = symbol(id, bindingPower);
      s.led = led || function (left) {
        this.lhs = left;
        this.rhs = expression(bindingPower - 1); // subtract 1 from bindingPower for right associative operators
        this.type = "binary";
        return this;
      };
      return s;
    };

    // match prefix operators
    // <operator> <expression>
    var prefix = function (id, nud) {
      var s = symbol(id);
      s.nud = nud || function () {
        this.expression = expression(70);
        this.type = "unary";
        return this;
      };
      return s;
    };

    symbol("(end)");
    symbol("(name)");
    symbol("(literal)");
    symbol("(regex)");
    symbol(":");
    symbol(";");
    symbol(",");
    symbol(")");
    symbol("]");
    symbol("}");
    symbol(".."); // range operator
    infix("."); // field reference
    infix("+"); // numeric addition
    infix("-"); // numeric subtraction
    infix("*"); // numeric multiplication
    infix("/"); // numeric division
    infix("%"); // numeric modulus
    infix("="); // equality
    infix("<"); // less than
    infix(">"); // greater than
    infix("!="); // not equal to
    infix("<="); // less than or equal
    infix(">="); // greater than or equal
    infix("&"); // string concatenation
    infix("and"); // Boolean AND
    infix("or"); // Boolean OR
    infix("in"); // is member of array
    infixr(":="); // bind variable
    prefix("-"); // unary numeric negation
    infix("~>"); // function application

    // field wildcard (single level)
    prefix('*', function () {
      this.type = "wildcard";
      return this;
    });

    // descendant wildcard (multi-level)
    prefix('**', function () {
      this.type = "descendant";
      return this;
    });

    // function invocation
    infix("(", operators['('], function (left) {
      // left is is what we are trying to invoke
      this.procedure = left;
      this.type = 'function';
      this.arguments = [];
      if (node.id !== ')') {
        for (; ;) {
          if (node.type === 'operator' && node.id === '?') {
            // partial function application
            this.type = 'partial';
            this.arguments.push(node);
            advance('?');
          } else {
            this.arguments.push(expression(0));
          }
          if (node.id !== ',') break;
          advance(',');
        }
      }
      advance(")", true);
      // if the name of the function is 'function' or λ, then this is function definition (lambda function)
      if (left.type === 'name' && (left.value === 'function' || left.value === '\u03BB')) {
        // all of the args must be VARIABLE tokens
        this.arguments.forEach(function (arg, index) {
          if (arg.type !== 'variable') {
            throw {
              code: "S0208",
              stack: (new Error()).stack,
              position: arg.position,
              token: arg.value,
              value: index + 1
            };
          }
        });
        this.type = 'lambda';
        // is the next token a '<' - if so, parse the function signature
        if (node.id === '<') {
          var sigPos = node.position;
          var depth = 1;
          var sig = '<';
          while (depth > 0 && node.id !== '{' && node.id !== '(end)') {
            var tok = advance();
            if (tok.id === '>') {
              depth--;
            } else if (tok.id === '<') {
              depth++;
            }
            sig += tok.value;
          }
          advance('>');
          try {
            this.signature = parseSignature(sig);
          } catch (err) {
            // insert the position into this error
            err.position = sigPos + err.offset;
            throw err;
          }
        }
        // parse the function body
        advance('{');
        this.body = expression(0);
        advance('}');
      }
      return this;
    });

    // parenthesis - block expression
    prefix("(", function () {
      var expressions = [];
      while (node.id !== ")") {
        expressions.push(expression(0));
        if (node.id !== ";") {
          break;
        }
        advance(";");
      }
      advance(")", true);
      this.type = 'block';
      this.expressions = expressions;
      return this;
    });

    // array constructor
    prefix("[", function () {
      var a = [];
      if (node.id !== "]") {
        for (; ;) {
          var item = expression(0);
          if (node.id === "..") {
            // range operator
            var range = { type: "binary", value: "..", position: node.position, lhs: item };
            advance("..");
            range.rhs = expression(0);
            item = range;
          }
          a.push(item);
          if (node.id !== ",") {
            break;
          }
          advance(",");
        }
      }
      advance("]", true);
      this.lhs = a;
      this.type = "unary";
      return this;
    });

    // filter - predicate or array index
    infix("[", operators['['], function (left) {
      if (node.id === "]") {
        // empty predicate means maintain singleton arrays in the output
        var step = left;
        while (step && step.type === 'binary' && step.value === '[') {
          step = step.lhs;
        }
        step.keepArray = true;
        advance("]");
        return left;
      } else {
        this.lhs = left;
        this.rhs = expression(operators[']']);
        this.type = 'binary';
        advance("]", true);
        return this;
      }
    });

    var objectParser = function (left) {
      var a = [];
      if (node.id !== "}") {
        for (; ;) {
          var n = expression(0);
          advance(":");
          var v = expression(0);
          a.push([n, v]); // holds an array of name/value expression pairs
          if (node.id !== ",") {
            break;
          }
          advance(",");
        }
      }
      advance("}", true);
      if (typeof left === 'undefined') {
        // NUD - unary prefix form
        this.lhs = a;
        this.type = "unary";
      } else {
        // LED - binary infix form
        this.lhs = left;
        this.rhs = a;
        this.type = 'binary';
      }
      return this;
    };

    // object constructor
    prefix("{", objectParser);

    // object grouping
    infix("{", operators['{'], objectParser);

    // if/then/else ternary operator ?:
    infix("?", operators['?'], function (left) {
      this.type = 'condition';
      this.condition = left;
      this.then = expression(0);
      if (node.id === ':') {
        // else condition
        advance(":");
        this.else = expression(0);
      }
      return this;
    });

    // tail call optimization
    // this is invoked by the post parser to analyse lambda functions to see
    // if they make a tail call.  If so, it is replaced by a thunk which will
    // be invoked by the trampoline loop during function application.
    // This enables tail-recursive functions to be written without growing the stack
    var tail_call_optimize = function (expr) {
      var result;
      if (expr.type === 'function') {
        var thunk = { type: 'lambda', thunk: true, arguments: [], position: expr.position };
        thunk.body = expr;
        result = thunk;
      } else if (expr.type === 'condition') {
        // analyse both branches
        expr.then = tail_call_optimize(expr.then);
        expr.else = tail_call_optimize(expr.else);
        result = expr;
      } else if (expr.type === 'block') {
        // only the last expression in the block
        var length = expr.expressions.length;
        if (length > 0) {
          expr.expressions[length - 1] = tail_call_optimize(expr.expressions[length - 1]);
        }
        result = expr;
      } else {
        result = expr;
      }
      return result;
    };

    // post-parse stage
    // the purpose of this is flatten the parts of the AST representing location paths,
    // converting them to arrays of steps which in turn may contain arrays of predicates.
    // following this, nodes containing '.' and '[' should be eliminated from the AST.
    var post_parse = function (expr) {
      var result = [];
      switch (expr.type) {
        case 'binary':
          switch (expr.value) {
            case '.':
              var lstep = post_parse(expr.lhs);
              if (lstep.type === 'path') {
                Array.prototype.push.apply(result, lstep);
              } else {
                result.push(lstep);
              }
              var rest = post_parse(expr.rhs);
              if (rest.type !== 'path') {
                rest = [rest];
              }
              Array.prototype.push.apply(result, rest);
              result.type = 'path';
              break;
            case '[':
              // predicated step
              // LHS is a step or a predicated step
              // RHS is the predicate expr
              result = post_parse(expr.lhs);
              var step = result;
              if (result.type === 'path') {
                step = result[result.length - 1];
              }
              if (typeof step.group !== 'undefined') {
                throw {
                  code: "S0209",
                  stack: (new Error()).stack,
                  position: expr.position
                };
              }
              if (typeof step.predicate === 'undefined') {
                step.predicate = [];
              }
              step.predicate.push(post_parse(expr.rhs));
              break;
            case '{':
              // group-by
              // LHS is a step or a predicated step
              // RHS is the object constructor expr
              result = post_parse(expr.lhs);
              if (typeof result.group !== 'undefined') {
                throw {
                  code: "S0210",
                  stack: (new Error()).stack,
                  position: expr.position
                };
              }
              // object constructor - process each pair
              result.group = {
                lhs: expr.rhs.map(function (pair) {
                  return [post_parse(pair[0]), post_parse(pair[1])];
                }),
                position: expr.position
              };
              break;
            default:
              result = { type: expr.type, value: expr.value, position: expr.position };
              result.lhs = post_parse(expr.lhs);
              result.rhs = post_parse(expr.rhs);
          }
          break;
        case 'unary':
          result = { type: expr.type, value: expr.value, position: expr.position };
          if (expr.value === '[') {
            // array constructor - process each item
            result.lhs = expr.lhs.map(function (item) {
              return post_parse(item);
            });
          } else if (expr.value === '{') {
            // object constructor - process each pair
            result.lhs = expr.lhs.map(function (pair) {
              return [post_parse(pair[0]), post_parse(pair[1])];
            });
          } else {
            // all other unary expressions - just process the expression
            result.expression = post_parse(expr.expression);
            // if unary minus on a number, then pre-process
            if (expr.value === '-' && result.expression.type === 'literal' && isNumeric(result.expression.value)) {
              result = result.expression;
              result.value = -result.value;
            }
          }
          break;
        case 'function':
        case 'partial':
          result = { type: expr.type, name: expr.name, value: expr.value, position: expr.position };
          result.arguments = expr.arguments.map(function (arg) {
            return post_parse(arg);
          });
          result.procedure = post_parse(expr.procedure);
          break;
        case 'lambda':
          result = { type: expr.type, arguments: expr.arguments, signature: expr.signature, position: expr.position };
          var body = post_parse(expr.body);
          result.body = tail_call_optimize(body);
          break;
        case 'condition':
          result = { type: expr.type, position: expr.position };
          result.condition = post_parse(expr.condition);
          result.then = post_parse(expr.then);
          if (typeof expr.else !== 'undefined') {
            result.else = post_parse(expr.else);
          }
          break;
        case 'block':
          result = { type: expr.type, position: expr.position };
          // array of expressions - process each one
          result.expressions = expr.expressions.map(function (item) {
            return post_parse(item);
          });
          // TODO scan the array of expressions to see if any of them assign variables
          // if so, need to mark the block as one that needs to create a new frame
          break;
        case 'name':
          result = [expr];
          result.type = 'path';
          break;
        case 'literal':
        case 'wildcard':
        case 'descendant':
        case 'variable':
        case 'regex':
          result = expr;
          break;
        case 'operator':
          // the tokens 'and' and 'or' might have been used as a name rather than an operator
          if (expr.value === 'and' || expr.value === 'or' || expr.value === 'in') {
            expr.type = 'name';
            result = post_parse(expr);
          } else if (expr.value === '?') {
            // partial application
            result = expr;
          } else {
            throw {
              code: "S0201",
              stack: (new Error()).stack,
              position: expr.position,
              token: expr.value
            };
          }
          break;
        default:
          var code = "S0206";
          /* istanbul ignore else */
          if (expr.id === '(end)') {
            code = "S0207";
          }
          throw {
            code: code,
            stack: (new Error()).stack,
            position: expr.position,
            token: expr.value
          };
      }
      return result;
    };

    // now invoke the tokenizer and the parser and return the syntax tree

    lexer = tokenizer(source);
    advance();
    // parse the tokens
    var expr = expression(0);
    if (node.id !== '(end)') {
      throw {
        code: "S0201",
        stack: (new Error()).stack,
        position: node.position,
        token: node.value
      };
    }
    expr = post_parse(expr);

    return expr;
  };

  // Start of Evaluator code

  var staticFrame = createFrame(null);

  /**
   * Check if value is a finite number
   * @param {float} n - number to evaluate
   * @returns {boolean} True if n is a finite number
   */
  function isNumeric(n) {
    var isNum = false;
    if (typeof n === 'number') {
      var num = parseFloat(n);
      isNum = !isNaN(num);
      if (isNum && !isFinite(num)) {
        throw {
          code: "D1001",
          value: n,
          stack: (new Error()).stack
        };
      }
    }
    return isNum;
  }

  /**
   * Returns true if the arg is an array of numbers
   * @param {*} arg - the item to test
   * @returns {boolean} True if arg is an array of numbers
   */
  function isArrayOfNumbers(arg) {
    var result = false;
    if (Array.isArray(arg)) {
      result = (arg.filter(function (item) { return !isNumeric(item); }).length === 0);
    }
    return result;
  }

  // Polyfill
  /* istanbul ignore next */
  Number.isInteger = Number.isInteger || function (value) {
    return typeof value === "number" &&
      isFinite(value) &&
      Math.floor(value) === value;
  };

  /**
   * Evaluate expression against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluate(expr, input, environment) {
    var result;

    var entryCallback = environment.lookup('__evaluate_entry');
    if (entryCallback) {
      entryCallback(expr, input, environment);
    }

    switch (expr.type) {
      case 'path':
        result = evaluatePath(expr, input, environment);
        break;
      case 'binary':
        result = evaluateBinary(expr, input, environment);
        break;
      case 'unary':
        result = evaluateUnary(expr, input, environment);
        break;
      case 'name':
        result = evaluateName(expr, input, environment);
        break;
      case 'literal':
        result = evaluateLiteral(expr, input, environment);
        break;
      case 'wildcard':
        result = evaluateWildcard(expr, input, environment);
        break;
      case 'descendant':
        result = evaluateDescendants(expr, input, environment);
        break;
      case 'condition':
        result = evaluateCondition(expr, input, environment);
        break;
      case 'block':
        result = evaluateBlock(expr, input, environment);
        break;
      case 'regex':
        result = evaluateRegex(expr, input, environment);
        break;
      case 'function':
        result = evaluateFunction(expr, input, environment);
        break;
      case 'variable':
        result = evaluateVariable(expr, input, environment);
        break;
      case 'lambda':
        result = evaluateLambda(expr, input, environment);
        break;
      case 'partial':
        result = evaluatePartialApplication(expr, input, environment);
        break;
    }
    if (expr.hasOwnProperty('predicate')) {
      result = applyPredicates(expr.predicate, result, environment);
    }
    if (expr.hasOwnProperty('group')) {
      result = evaluateGroupExpression(expr.group, result, environment);
    }

    var exitCallback = environment.lookup('__evaluate_exit');
    if (exitCallback) {
      exitCallback(expr, input, environment, result);
    }

    return result;
  }

  /**
   * Evaluate path expression against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluatePath(expr, input, environment) {
    var result;
    var inputSequence;
    var keepSingletonArray = false;
    // expr is an array of steps
    // if the first step is a variable reference ($...), including root reference ($$),
    //   then the path is absolute rather than relative
    if (expr[0].type === 'variable') {
      expr[0].absolute = true;
    } else if (expr[0].type === 'unary' && expr[0].value === '[') {
      // array constructor - not relative to the input
      input = [null];// dummy singleton sequence for first step
    }

    // evaluate each step in turn
    for (var ii = 0; ii < expr.length; ii++) {
      var step = expr[ii];
      if (step.keepArray === true) {
        keepSingletonArray = true;
      }
      var resultSequence = [];
      result = undefined;
      // if input is not an array, make it so
      if (step.absolute === true) {
        inputSequence = [input]; // dummy singleton sequence for first (absolute) step
      } else if (Array.isArray(input)) {
        inputSequence = input;
      } else {
        inputSequence = [input];
      }
      // if there is more than one step in the path, handle quoted field names as names not literals
      if (expr.length > 1 && step.type === 'literal') {
        step.type = 'name';
      }
      inputSequence.forEach(function (item) {
        var res = evaluate(step, item, environment);
        if (typeof res !== 'undefined') {
          if (Array.isArray(res) && (step.value !== '[')) {
            // is res an array - if so, flatten it into the parent array
            res.forEach(function (innerRes) {
              if (typeof innerRes !== 'undefined') {
                resultSequence.push(innerRes);
              }
            });
          } else {
            resultSequence.push(res);
          }
        }
      });
      if (resultSequence.length === 1) {
        if (keepSingletonArray) {
          result = resultSequence;
        } else {
          result = resultSequence[0];
        }
      } else if (resultSequence.length > 1) {
        result = resultSequence;
      }

      if (typeof result === 'undefined') {
        break;
      }
      input = result;
    }
    return result;
  }

  /**
   * Apply predicates to input data
   * @param {Object} predicates - Predicates
   * @param {Object} input - Input data to apply predicates against
   * @param {Object} environment - Environment
   * @returns {*} Result after applying predicates
   */
  function applyPredicates(predicates, input, environment) {
    var result;
    var inputSequence = input;
    // lhs potentially holds an array
    // we want to iterate over the array, and only keep the items that are
    // truthy when applied to the predicate.
    // if the predicate evaluates to an integer, then select that index

    var results = [];
    predicates.forEach(function (predicate) {
      // if it's not an array, turn it into one
      // since in XPath >= 2.0 an item is equivalent to a singleton sequence of that item
      // if input is not an array, make it so
      if (!Array.isArray(inputSequence)) {
        inputSequence = [inputSequence];
      }
      results = [];
      result = undefined;
      if (predicate.type === 'literal' && isNumeric(predicate.value)) {
        var index = predicate.value;
        if (!Number.isInteger(index)) {
          // round it down
          index = Math.floor(index);
        }
        if (index < 0) {
          // count in from end of array
          index = inputSequence.length + index;
        }
        result = inputSequence[index];
      } else {
        inputSequence.forEach(function (item, index) {
          var res = evaluate(predicate, item, environment);
          if (isNumeric(res)) {
            res = [res];
          }
          if (isArrayOfNumbers(res)) {
            res.forEach(function (ires) {
              if (!Number.isInteger(ires)) {
                // round it down
                ires = Math.floor(ires);
              }
              if (ires < 0) {
                // count in from end of array
                ires = inputSequence.length + ires;
              }
              if (ires === index) {
                results.push(item);
              }
            });
          } else if (functionBoolean(res)) { // truthy
            results.push(item);
          }
        });
      }
      if (results.length === 1) {
        result = results[0];
      } else if (results.length > 1) {
        result = results;
      }
      inputSequence = result;
    });
    return result;
  }

  /**
   * Evaluate binary expression against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateBinary(expr, input, environment) {
    var result;

    switch (expr.value) {
      case '+':
      case '-':
      case '*':
      case '/':
      case '%':
        result = evaluateNumericExpression(expr, input, environment);
        break;
      case '=':
      case '!=':
      case '<':
      case '<=':
      case '>':
      case '>=':
        result = evaluateComparisonExpression(expr, input, environment);
        break;
      case '&':
        result = evaluateStringConcat(expr, input, environment);
        break;
      case 'and':
      case 'or':
        result = evaluateBooleanExpression(expr, input, environment);
        break;
      case '..':
        result = evaluateRangeExpression(expr, input, environment);
        break;
      case ':=':
        result = evaluateBindExpression(expr, input, environment);
        break;
      case 'in':
        result = evaluateIncludesExpression(expr, input, environment);
        break;
      case '~>':
        result = evaluateApplyExpression(expr, input, environment);
        break;
    }
    return result;
  }

  /**
   * Evaluate unary expression against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateUnary(expr, input, environment) {
    var result;

    switch (expr.value) {
      case '-':
        result = evaluate(expr.expression, input, environment);
        if (isNumeric(result)) {
          result = -result;
        } else {
          throw {
            code: "D1002",
            stack: (new Error()).stack,
            position: expr.position,
            token: expr.value,
            value: result
          };
        }
        break;
      case '[':
        // array constructor - evaluate each item
        result = [];
        expr.lhs.forEach(function (item) {
          var value = evaluate(item, input, environment);
          if (typeof value !== 'undefined') {
            if (item.value === '[') {
              result.push(value);
            } else {
              result = functionAppend(result, value);
            }
          }
        });
        break;
      case '{':
        // object constructor - apply grouping
        result = evaluateGroupExpression(expr, input, environment);
        break;

    }
    return result;
  }

  /**
   * Evaluate name object against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateName(expr, input, environment) {
    // lookup the 'name' item in the input
    var result;
    if (Array.isArray(input)) {
      var results = [];
      input.forEach(function (item) {
        var res = evaluateName(expr, item, environment);
        if (typeof res !== 'undefined') {
          results.push(res);
        }
      });
      if (results.length === 1) {
        result = results[0];
      } else if (results.length > 1) {
        result = results;
      }
    } else if (input !== null && typeof input === 'object') {
      result = input[expr.value];
    }
    return result;
  }

  /**
   * Evaluate literal against input data
   * @param {Object} expr - JSONata expression
   * @returns {*} Evaluated input data
   */
  function evaluateLiteral(expr) {
    return expr.value;
  }

  /**
   * Evaluate wildcard against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @returns {*} Evaluated input data
   */
  function evaluateWildcard(expr, input) {
    var result;
    var results = [];
    if (input !== null && typeof input === 'object') {
      Object.keys(input).forEach(function (key) {
        var value = input[key];
        if (Array.isArray(value)) {
          value = flatten(value);
          results = functionAppend(results, value);
        } else {
          results.push(value);
        }
      });
    }

    if (results.length === 1) {
      result = results[0];
    } else if (results.length > 1) {
      result = results;
    }
    return result;
  }

  /**
   * Returns a flattened array
   * @param {Array} arg - the array to be flatten
   * @param {Array} flattened - carries the flattened array - if not defined, will initialize to []
   * @returns {Array} - the flattened array
   */
  function flatten(arg, flattened) {
    if (typeof flattened === 'undefined') {
      flattened = [];
    }
    if (Array.isArray(arg)) {
      arg.forEach(function (item) {
        flatten(item, flattened);
      });
    } else {
      flattened.push(arg);
    }
    return flattened;
  }

  /**
   * Evaluate descendants against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @returns {*} Evaluated input data
   */
  function evaluateDescendants(expr, input) {
    var result;
    var resultSequence = [];
    if (typeof input !== 'undefined') {
      // traverse all descendants of this object/array
      recurseDescendants(input, resultSequence);
      if (resultSequence.length === 1) {
        result = resultSequence[0];
      } else {
        result = resultSequence;
      }
    }
    return result;
  }

  /**
   * Recurse through descendants
   * @param {Object} input - Input data
   * @param {Object} results - Results
   */
  function recurseDescendants(input, results) {
    // this is the equivalent of //* in XPath
    if (!Array.isArray(input)) {
      results.push(input);
    }
    if (Array.isArray(input)) {
      input.forEach(function (member) {
        recurseDescendants(member, results);
      });
    } else if (input !== null && typeof input === 'object') {
      Object.keys(input).forEach(function (key) {
        recurseDescendants(input[key], results);
      });
    }
  }

  /**
   * Evaluate numeric expression against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateNumericExpression(expr, input, environment) {
    var result;

    var lhs = evaluate(expr.lhs, input, environment);
    var rhs = evaluate(expr.rhs, input, environment);

    if (typeof lhs === 'undefined' || typeof rhs === 'undefined') {
      // if either side is undefined, the result is undefined
      return result;
    }

    if (!isNumeric(lhs)) {
      throw {
        code: "T2001",
        stack: (new Error()).stack,
        position: expr.position,
        token: expr.value,
        value: lhs
      };
    }
    if (!isNumeric(rhs)) {
      throw {
        code: "T2002",
        stack: (new Error()).stack,
        position: expr.position,
        token: expr.value,
        value: rhs
      };
    }

    switch (expr.value) {
      case '+':
        result = lhs + rhs;
        break;
      case '-':
        result = lhs - rhs;
        break;
      case '*':
        result = lhs * rhs;
        break;
      case '/':
        result = lhs / rhs;
        break;
      case '%':
        result = lhs % rhs;
        break;
    }
    return result;
  }

  /**
   * Evaluate comparison expression against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateComparisonExpression(expr, input, environment) {
    var result;

    var lhs = evaluate(expr.lhs, input, environment);
    var rhs = evaluate(expr.rhs, input, environment);

    if (typeof lhs === 'undefined' || typeof rhs === 'undefined') {
      // if either side is undefined, the result is false
      return false;
    }

    switch (expr.value) {
      case '=':
        result = lhs === rhs;
        break;
      case '!=':
        result = (lhs !== rhs);
        break;
      case '<':
        result = lhs < rhs;
        break;
      case '<=':
        result = lhs <= rhs;
        break;
      case '>':
        result = lhs > rhs;
        break;
      case '>=':
        result = lhs >= rhs;
        break;
    }
    return result;
  }

  /**
   * Inclusion operator - in
   *
   * @param {Object} expr - AST
   * @param {*} input - input context
   * @param {Object} environment - frame
   * @returns {boolean} - true if lhs is a member of rhs
   */
  function evaluateIncludesExpression(expr, input, environment) {
    var result = false;

    var lhs = evaluate(expr.lhs, input, environment);
    var rhs = evaluate(expr.rhs, input, environment);

    if (typeof lhs === 'undefined' || typeof rhs === 'undefined') {
      // if either side is undefined, the result is false
      return false;
    }

    if (!Array.isArray(rhs)) {
      rhs = [rhs];
    }

    for (var i = 0; i < rhs.length; i++) {
      if (rhs[i] === lhs) {
        result = true;
        break;
      }
    }

    return result;
  }

  /**
   * Evaluate boolean expression against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateBooleanExpression(expr, input, environment) {
    var result;

    switch (expr.value) {
      case 'and':
        result = functionBoolean(evaluate(expr.lhs, input, environment)) &&
          functionBoolean(evaluate(expr.rhs, input, environment));
        break;
      case 'or':
        result = functionBoolean(evaluate(expr.lhs, input, environment)) ||
          functionBoolean(evaluate(expr.rhs, input, environment));
        break;
    }
    return result;
  }

  /**
   * Evaluate string concatenation against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {string|*} Evaluated input data
   */
  function evaluateStringConcat(expr, input, environment) {
    var result;
    var lhs = evaluate(expr.lhs, input, environment);
    var rhs = evaluate(expr.rhs, input, environment);

    var lstr = '';
    var rstr = '';
    if (typeof lhs !== 'undefined') {
      lstr = functionString(lhs);
    }
    if (typeof rhs !== 'undefined') {
      rstr = functionString(rhs);
    }

    result = lstr.concat(rstr);
    return result;
  }

  /**
   * Evaluate group expression against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {{}} Evaluated input data
   */
  function evaluateGroupExpression(expr, input, environment) {
    var result = {};
    var groups = {};
    // group the input sequence by 'key' expression
    if (!Array.isArray(input)) {
      input = [input];
    }
    input.forEach(function (item) {
      expr.lhs.forEach(function (pair) {
        var key = evaluate(pair[0], item, environment);
        // key has to be a string
        if (typeof key !== 'string') {
          throw {
            code: "T1003",
            stack: (new Error()).stack,
            position: expr.position,
            value: key
          };
        }
        var entry = { data: item, expr: pair[1] };
        if (groups.hasOwnProperty(key)) {
          // a value already exists in this slot
          // append it as an array
          groups[key].data = functionAppend(groups[key].data, item);
        } else {
          groups[key] = entry;
        }
      });
    });
    // iterate over the groups to evaluate the 'value' expression
    for (var key in groups) {
      var entry = groups[key];
      var value = evaluate(entry.expr, entry.data, environment);
      result[key] = value;
    }
    return result;
  }

  /**
   * Evaluate range expression against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateRangeExpression(expr, input, environment) {
    var result;

    var lhs = evaluate(expr.lhs, input, environment);
    var rhs = evaluate(expr.rhs, input, environment);

    if (typeof lhs === 'undefined' || typeof rhs === 'undefined') {
      // if either side is undefined, the result is undefined
      return result;
    }

    if (lhs > rhs) {
      // if the lhs is greater than the rhs, return undefined
      return result;
    }

    if (!Number.isInteger(lhs)) {
      throw {
        code: "T2003",
        stack: (new Error()).stack,
        position: expr.position,
        token: expr.value,
        value: lhs
      };
    }
    if (!Number.isInteger(rhs)) {
      throw {
        code: "T2004",
        stack: (new Error()).stack,
        position: expr.position,
        token: expr.value,
        value: rhs
      };
    }

    result = new Array(rhs - lhs + 1);
    for (var item = lhs, index = 0; item <= rhs; item++, index++) {
      result[index] = item;
    }
    return result;
  }

  /**
   * Evaluate bind expression against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateBindExpression(expr, input, environment) {
    // The RHS is the expression to evaluate
    // The LHS is the name of the variable to bind to - should be a VARIABLE token
    var value = evaluate(expr.rhs, input, environment);
    if (expr.lhs.type !== 'variable') {
      throw {
        code: "D2005",
        stack: (new Error()).stack,
        position: expr.position,
        token: expr.value,
        value: expr.lhs.type === 'path' ? expr.lhs[0].value : expr.lhs.value
      };
    }
    environment.bind(expr.lhs.value, value);
    return value;
  }

  /**
   * Evaluate condition against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateCondition(expr, input, environment) {
    var result;
    var condition = evaluate(expr.condition, input, environment);
    if (functionBoolean(condition)) {
      result = evaluate(expr.then, input, environment);
    } else if (typeof expr.else !== 'undefined') {
      result = evaluate(expr.else, input, environment);
    }
    return result;
  }

  /**
   * Evaluate block against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateBlock(expr, input, environment) {
    var result;
    // create a new frame to limit the scope of variable assignments
    // TODO, only do this if the post-parse stage has flagged this as required
    var frame = createFrame(environment);
    // invoke each expression in turn
    // only return the result of the last one
    expr.expressions.forEach(function (expression) {
      result = evaluate(expression, input, frame);
    });

    return result;
  }

  /**
   * Prepare a regex
   * @param {Object} expr - expression containing regex
   * @returns {Function} Higher order function representing prepared regex
   */
  function evaluateRegex(expr) {
    expr.value.lastIndex = 0;
    var closure = function (str) {
      var re = expr.value;
      //var result = str.match(expr.value);
      var result;
      var match = re.exec(str);
      if (match !== null) {
        result = {
          match: match[0],
          start: match.index,
          end: match.index + match[0].length,
          groups: []
        };
        if (match.length > 1) {
          for (var i = 1; i < match.length; i++) {
            result.groups.push(match[i]);
          }
        }
        result.next = function () {
          if (re.lastIndex >= str.length) {
            return undefined;
          } else {
            var next = closure(str);
            if (next && next.match === '' && re.lastIndex === expr.value.lastIndex) {
              // matches zero length string; this will never progress
              throw {
                code: "D1004",
                stack: (new Error()).stack,
                position: expr.position,
                value: expr.value.source
              };
            }
            return next;
          }
        };
      }

      return result;
    };
    return closure;
  }

  /**
   * Evaluate variable against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateVariable(expr, input, environment) {
    // lookup the variable value in the environment
    var result;
    // if the variable name is empty string, then it refers to context value
    if (expr.value === '') {
      result = input;
    } else {
      result = environment.lookup(expr.value);
    }
    return result;
  }

  var chain = evaluate(parser('function($f, $g) { function($x){ $g($f($x)) } }'), null, staticFrame);

  /**
   * Apply the function on the RHS using the sequence on the LHS as the first argument
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluateApplyExpression(expr, input, environment) {
    var result;

    var arg1 = evaluate(expr.lhs, input, environment);

    if (expr.rhs.type === 'function') {
      // this is a function _invocation_; invoke it with arg1 at the start
      result = evaluateFunction(expr.rhs, input, environment, { context: arg1 });
    } else {
      var func = evaluate(expr.rhs, input, environment);

      if (!isFunction(func)) {
        throw {
          code: "T2006",
          stack: (new Error()).stack,
          position: expr.position,
          value: func
        };
      }

      if (isFunction(arg1)) {
        // this is function chaining (func1 ~> func2)
        // λ($f, $g) { λ($x){ $g($f($x)) } }
        result = apply(chain, [arg1, func], environment, null);
      } else {
        result = apply(func, [arg1], environment, null);
      }

    }

    return result;
  }

  /**
   *
   * @param {Object} arg - expression to test
   * @returns {boolean} - true if it is a function (lambda or built-in)
   */
  function isFunction(arg) {
    return ((arg && (arg._jsonata_function === true || arg._jsonata_lambda === true)) || typeof arg === 'function');
  }

  /**
   * Tests whether arg is a lambda function
   * @param {*} arg - the value to test
   * @returns {boolean} - true if it is a lambda function
   */
  function isLambda(arg) {
    return arg && arg._jsonata_lambda === true;
  }

  /**
   * Evaluate function against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @param {Object} [applyto] - LHS of ~> operator
   * @returns {*} Evaluated input data
   */
  function evaluateFunction(expr, input, environment, applyto) {
    var result;
    // evaluate the arguments
    var evaluatedArgs = [];
    expr.arguments.forEach(function (arg) {
      evaluatedArgs.push(evaluate(arg, input, environment));
    });
    if (applyto) {
      // insert the first argument from the LHS of ~>
      evaluatedArgs.unshift(applyto.context);
    }
    // lambda function on lhs
    // create the procedure
    // can't assume that expr.procedure is a lambda type directly
    // could be an expression that evaluates to a function (e.g. variable reference, parens expr etc.
    // evaluate it generically first, then check that it is a function.  Throw error if not.
    var proc = evaluate(expr.procedure, input, environment);

    if (typeof proc === 'undefined' && expr.procedure.type === 'path' && environment.lookup(expr.procedure[0].value)) {
      // help the user out here if they simply forgot the leading $
      throw {
        code: "T1005",
        stack: (new Error()).stack,
        position: expr.position,
        token: expr.procedure[0].value
      };
    }
    // apply the procedure
    try {
      var validatedArgs = evaluatedArgs;
      if (proc) {
        validatedArgs = validateArguments(proc.signature, evaluatedArgs, input);
      }
      result = apply(proc, validatedArgs, input);
    } catch (err) {
      // add the position field to the error
      err.position = expr.position;
      // and the function identifier
      err.token = expr.procedure.type === 'path' ? expr.procedure[0].value : expr.procedure.value;
      throw err;
    }
    return result;
  }

  /**
   * Apply procedure or function
   * @param {Object} proc - Procedure
   * @param {Array} args - Arguments
   * @param {Object} self - Self
   * @returns {*} Result of procedure
   */
  function apply(proc, args, self) {
    var result;
    result = applyInner(proc, args, self);
    while (isLambda(result) && result.thunk === true) {
      // trampoline loop - this gets invoked as a result of tail-call optimization
      // the function returned a tail-call thunk
      // unpack it, evaluate its arguments, and apply the tail call
      var next = evaluate(result.body.procedure, result.input, result.environment);
      var evaluatedArgs = [];
      result.body.arguments.forEach(function (arg) {
        evaluatedArgs.push(evaluate(arg, result.input, result.environment));
      });

      result = applyInner(next, evaluatedArgs, self);
    }
    return result;
  }

  /**
   * Apply procedure or function
   * @param {Object} proc - Procedure
   * @param {Array} args - Arguments
   * @param {Object} self - Self
   * @returns {*} Result of procedure
   */
  function applyInner(proc, args, self) {
    var result;
    if (isLambda(proc)) {
      result = applyProcedure(proc, args);
    } else if (proc && proc._jsonata_function === true) {
      result = proc.implementation.apply(self, args);
    } else if (typeof proc === 'function') {
      result = proc.apply(self, args);
    } else {
      throw {
        code: "T1006",
        stack: (new Error()).stack
      };
    }
    return result;
  }

  /**
   * Evaluate lambda against input data
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {{lambda: boolean, input: *, environment: *, arguments: *, body: *}} Evaluated input data
   */
  function evaluateLambda(expr, input, environment) {
    // make a function (closure)
    var procedure = {
      _jsonata_lambda: true,
      input: input,
      environment: environment,
      arguments: expr.arguments,
      signature: expr.signature,
      body: expr.body
    };
    if (expr.thunk === true) {
      procedure.thunk = true;
    }
    return procedure;
  }

  /**
   * Evaluate partial application
   * @param {Object} expr - JSONata expression
   * @param {Object} input - Input data to evaluate against
   * @param {Object} environment - Environment
   * @returns {*} Evaluated input data
   */
  function evaluatePartialApplication(expr, input, environment) {
    // partially apply a function
    var result;
    // evaluate the arguments
    var evaluatedArgs = [];
    expr.arguments.forEach(function (arg) {
      if (arg.type === 'operator' && arg.value === '?') {
        evaluatedArgs.push(arg);
      } else {
        evaluatedArgs.push(evaluate(arg, input, environment));
      }
    });
    // lookup the procedure
    var proc = evaluate(expr.procedure, input, environment);
    if (typeof proc === 'undefined' && expr.procedure.type === 'path' && environment.lookup(expr.procedure[0].value)) {
      // help the user out here if they simply forgot the leading $
      throw {
        code: "T1007",
        stack: (new Error()).stack,
        position: expr.position,
        token: expr.procedure[0].value
      };
    }
    if (isLambda(proc)) {
      result = partialApplyProcedure(proc, evaluatedArgs);
    } else if (proc && proc._jsonata_function === true) {
      result = partialApplyNativeFunction(proc.implementation, evaluatedArgs);
    } else if (typeof proc === 'function') {
      result = partialApplyNativeFunction(proc, evaluatedArgs);
    } else {
      throw {
        code: "T1008",
        stack: (new Error()).stack,
        position: expr.position,
        token: expr.procedure.type === 'path' ? expr.procedure[0].value : expr.procedure.value
      };
    }
    return result;
  }

  /**
   * Validate the arguments against the signature validator (if it exists)
   * @param {Function} signature - validator function
   * @param {Array} args - function arguments
   * @param {*} context - context value
   * @returns {Array} - validated arguments
   */
  function validateArguments(signature, args, context) {
    if (typeof signature === 'undefined') {
      // nothing to validate
      return args;
    }
    var validatedArgs = signature.validate(args, context);
    return validatedArgs;
  }

  /**
   * Apply procedure
   * @param {Object} proc - Procedure
   * @param {Array} args - Arguments
   * @returns {*} Result of procedure
   */
  function applyProcedure(proc, args) {
    var result;
    var env = createFrame(proc.environment);
    proc.arguments.forEach(function (param, index) {
      env.bind(param.value, args[index]);
    });
    if (typeof proc.body === 'function') {
      // this is a lambda that wraps a native function - generated by partially evaluating a native
      result = applyNativeFunction(proc.body, env);
    } else {
      result = evaluate(proc.body, proc.input, env);
    }
    return result;
  }

  /**
   * Partially apply procedure
   * @param {Object} proc - Procedure
   * @param {Array} args - Arguments
   * @returns {{lambda: boolean, input: *, environment: {bind, lookup}, arguments: Array, body: *}} Result of partially applied procedure
   */
  function partialApplyProcedure(proc, args) {
    // create a closure, bind the supplied parameters and return a function that takes the remaining (?) parameters
    var env = createFrame(proc.environment);
    var unboundArgs = [];
    proc.arguments.forEach(function (param, index) {
      var arg = args[index];
      if (arg && arg.type === 'operator' && arg.value === '?') {
        unboundArgs.push(param);
      } else {
        env.bind(param.value, arg);
      }
    });
    var procedure = {
      _jsonata_lambda: true,
      input: proc.input,
      environment: env,
      arguments: unboundArgs,
      body: proc.body
    };
    return procedure;
  }

  /**
   * Partially apply native function
   * @param {Function} native - Native function
   * @param {Array} args - Arguments
   * @returns {{lambda: boolean, input: *, environment: {bind, lookup}, arguments: Array, body: *}} Result of partially applying native function
   */
  function partialApplyNativeFunction(native, args) {
    // create a lambda function that wraps and invokes the native function
    // get the list of declared arguments from the native function
    // this has to be picked out from the toString() value
    var sigArgs = getNativeFunctionArguments(native);
    sigArgs = sigArgs.map(function (sigArg) {
      return '$' + sigArg.trim();
    });
    var body = 'function(' + sigArgs.join(', ') + '){ _ }';

    var bodyAST = parser(body);
    bodyAST.body = native;

    var partial = partialApplyProcedure(bodyAST, args);
    return partial;
  }

  /**
   * Apply native function
   * @param {Object} proc - Procedure
   * @param {Object} env - Environment
   * @returns {*} Result of applying native function
   */
  function applyNativeFunction(proc, env) {
    var sigArgs = getNativeFunctionArguments(proc);
    // generate the array of arguments for invoking the function - look them up in the environment
    var args = sigArgs.map(function (sigArg) {
      return env.lookup(sigArg.trim());
    });

    var result = proc.apply(null, args);
    return result;
  }

  /**
   * Get native function arguments
   * @param {Function} func - Function
   * @returns {*|Array} Native function arguments
   */
  function getNativeFunctionArguments(func) {
    var signature = func.toString();
    var sigParens = /\(([^\)]*)\)/.exec(signature)[1]; // the contents of the parens
    var sigArgs = sigParens.split(',');
    return sigArgs;
  }

  /**
   * Creates a function definition
   * @param {Function} func - function implementation in Javascript
   * @param {string} signature - JSONata function signature definition
   * @returns {{implementation: *, signature: *}} function definition
   */
  function defineFunction(func, signature) {
    var definition = {
      _jsonata_function: true,
      implementation: func
    };
    if (typeof signature !== 'undefined') {
      definition.signature = parseSignature(signature);
    }
    return definition;
  }

  /**
   * Sum function
   * @param {Object} args - Arguments
   * @returns {number} Total value of arguments
   */
  function functionSum(args) {
    // undefined inputs always return undefined
    if (typeof args === 'undefined') {
      return undefined;
    }

    var total = 0;
    args.forEach(function (num) { total += num; });
    return total;
  }

  /**
   * Count function
   * @param {Object} args - Arguments
   * @returns {number} Number of elements in the array
   */
  function functionCount(args) {
    // undefined inputs always return undefined
    if (typeof args === 'undefined') {
      return 0;
    }

    return args.length;
  }

  /**
   * Max function
   * @param {Object} args - Arguments
   * @returns {number} Max element in the array
   */
  function functionMax(args) {
    // undefined inputs always return undefined
    if (typeof args === 'undefined' || args.length === 0) {
      return undefined;
    }

    return Math.max.apply(Math, args);
  }

  /**
   * Min function
   * @param {Object} args - Arguments
   * @returns {number} Min element in the array
   */
  function functionMin(args) {
    // undefined inputs always return undefined
    if (typeof args === 'undefined' || args.length === 0) {
      return undefined;
    }

    return Math.min.apply(Math, args);
  }

  /**
   * Average function
   * @param {Object} args - Arguments
   * @returns {number} Average element in the array
   */
  function functionAverage(args) {
    // undefined inputs always return undefined
    if (typeof args === 'undefined' || args.length === 0) {
      return undefined;
    }

    var total = 0;
    args.forEach(function (num) { total += num; });
    return total / args.length;
  }

  /**
   * Stingify arguments
   * @param {Object} arg - Arguments
   * @returns {String} String from arguments
   */
  function functionString(arg) {
    // undefined inputs always return undefined
    if (typeof arg === 'undefined') {
      return undefined;
    }

    var str;

    if (typeof arg === 'string') {
      // already a string
      str = arg;
    } else if (isFunction(arg)) {
      // functions (built-in and lambda convert to empty string
      str = '';
    } else if (typeof arg === 'number' && !isFinite(arg)) {
      throw {
        code: "D3001",
        value: arg,
        stack: (new Error()).stack
      };
    } else
      str = JSON.stringify(arg, function (key, val) {
        return (typeof val !== 'undefined' && val !== null && val.toPrecision && isNumeric(val)) ? Number(val.toPrecision(13)) :
          (val && isFunction(val)) ? '' : val;
      });
    return str;
  }

  /**
   * Create substring based on character number and length
   * @param {String} str - String to evaluate
   * @param {Integer} start - Character number to start substring
   * @param {Integer} [length] - Number of characters in substring
   * @returns {string|*} Substring
   */
  function functionSubstring(str, start, length) {
    // undefined inputs always return undefined
    if (typeof str === 'undefined') {
      return undefined;
    }

    return str.substr(start, length);
  }

  /**
   * Create substring up until a character
   * @param {String} str - String to evaluate
   * @param {String} chars - Character to define substring boundary
   * @returns {*} Substring
   */
  function functionSubstringBefore(str, chars) {
    // undefined inputs always return undefined
    if (typeof str === 'undefined') {
      return undefined;
    }

    var pos = str.indexOf(chars);
    if (pos > -1) {
      return str.substr(0, pos);
    } else {
      return str;
    }
  }

  /**
   * Create substring after a character
   * @param {String} str - String to evaluate
   * @param {String} chars - Character to define substring boundary
   * @returns {*} Substring
   */
  function functionSubstringAfter(str, chars) {
    // undefined inputs always return undefined
    if (typeof str === 'undefined') {
      return undefined;
    }

    var pos = str.indexOf(chars);
    if (pos > -1) {
      return str.substr(pos + chars.length);
    } else {
      return str;
    }
  }

  /**
   * Lowercase a string
   * @param {String} str - String to evaluate
   * @returns {string} Lowercase string
   */
  function functionLowercase(str) {
    // undefined inputs always return undefined
    if (typeof str === 'undefined') {
      return undefined;
    }

    return str.toLowerCase();
  }

  /**
   * Uppercase a string
   * @param {String} str - String to evaluate
   * @returns {string} Uppercase string
   */
  function functionUppercase(str) {
    // undefined inputs always return undefined
    if (typeof str === 'undefined') {
      return undefined;
    }

    return str.toUpperCase();
  }

  /**
   * length of a string
   * @param {String} str - string
   * @returns {Number} The number of characters in the string
   */
  function functionLength(str) {
    // undefined inputs always return undefined
    if (typeof str === 'undefined') {
      return undefined;
    }

    return str.length;
  }

  /**
   * Normalize and trim whitespace within a string
   * @param {string} str - string to be trimmed
   * @returns {string} - trimmed string
   */
  function functionTrim(str) {
    // undefined inputs always return undefined
    if (typeof str === 'undefined') {
      return undefined;
    }

    // normalize whitespace
    var result = str.replace(/[ \t\n\r]+/gm, ' ');
    if (result.charAt(0) === ' ') {
      // strip leading space
      result = result.substring(1);
    }
    if (result.charAt(result.length - 1) === ' ') {
      // strip trailing space
      result = result.substring(0, result.length - 1);
    }
    return result;
  }

  /**
   * Tests if the str contains the token
   * @param {String} str - string to test
   * @param {String} token - substring or regex to find
   * @returns {Boolean} - true if str contains token
   */
  function functionContains(str, token) {
    // undefined inputs always return undefined
    if (typeof str === 'undefined') {
      return undefined;
    }

    var result;

    if (typeof token === 'string') {
      result = (str.indexOf(token) !== -1);
    } else {
      var matches = token(str);
      result = (typeof matches !== 'undefined');
    }

    return result;
  }

  /**
   * Match a string with a regex returning an array of object containing details of each match
   * @param {String} str - string
   * @param {String} regex - the regex applied to the string
   * @param {Integer} [limit] - max number of matches to return
   * @returns {Array} The array of match objects
   */
  function functionMatch(str, regex, limit) {
    // undefined inputs always return undefined
    if (typeof str === 'undefined') {
      return undefined;
    }

    // limit, if specified, must be a non-negative number
    if (limit < 0) {
      throw {
        stack: (new Error()).stack,
        value: limit,
        code: 'D3040',
        index: 3
      };
    }

    var result = [];

    if (typeof limit === 'undefined' || limit > 0) {
      var count = 0;
      var matches = regex(str);
      if (typeof matches !== 'undefined') {
        while (typeof matches !== 'undefined' && (typeof limit === 'undefined' || count < limit)) {
          result.push({
            match: matches.match,
            index: matches.start,
            groups: matches.groups
          });
          matches = matches.next();
          count++;
        }
      }
    }

    return result;
  }

  /**
   * Match a string with a regex returning an array of object containing details of each match
   * @param {String} str - string
   * @param {String} pattern - the substring/regex applied to the string
   * @param {String} replacement - text to replace the matched substrings
   * @param {Integer} [limit] - max number of matches to return
   * @returns {Array} The array of match objects
   */
  function functionReplace(str, pattern, replacement, limit) {
    // undefined inputs always return undefined
    if (typeof str === 'undefined') {
      return undefined;
    }

    // pattern cannot be an empty string
    if (pattern === '') {
      throw {
        code: "D3010",
        stack: (new Error()).stack,
        value: pattern,
        index: 2
      };
    }

    // limit, if specified, must be a non-negative number
    if (limit < 0) {
      throw {
        code: "D3011",
        stack: (new Error()).stack,
        value: limit,
        index: 4
      };
    }

    var replacer;
    if (typeof replacement === 'string') {
      replacer = function (regexMatch) {
        var substitute = '';
        // scan forward, copying the replacement text into the substitute string
        // and replace any occurrence of $n with the values matched by the regex
        var position = 0;
        var index = replacement.indexOf('$', position);
        while (index !== -1 && position < replacement.length) {
          substitute += replacement.substring(position, index);
          position = index + 1;
          var dollarVal = replacement.charAt(position);
          if (dollarVal === '$') {
            // literal $
            substitute += '$';
            position++;
          } else if (dollarVal === '0') {
            substitute += regexMatch.match;
            position++;
          } else {
            var maxDigits;
            if (regexMatch.groups.length === 0) {
              // no sub-matches; any $ followed by a digit will be replaced by an empty string
              maxDigits = 1;
            } else {
              // max number of digits to parse following the $
              maxDigits = Math.floor(Math.log(regexMatch.groups.length) * Math.LOG10E) + 1;
            }
            index = parseInt(replacement.substring(position, position + maxDigits), 10);
            if (maxDigits > 1 && index > regexMatch.groups.length) {
              index = parseInt(replacement.substring(position, position + maxDigits - 1), 10);
            }
            if (!isNaN(index)) {
              if (regexMatch.groups.length > 0) {
                var submatch = regexMatch.groups[index - 1];
                if (typeof submatch !== 'undefined') {
                  substitute += submatch;
                }
              }
              position += index.toString().length;
            } else {
              // not a capture group, treat the $ as literal
              substitute += '$';
            }
          }
          index = replacement.indexOf('$', position);
        }
        substitute += replacement.substring(position);
        return substitute;
      };
    } else {
      replacer = replacement;
    }

    var result = '';
    var position = 0;

    if (typeof limit === 'undefined' || limit > 0) {
      var count = 0;
      if (typeof pattern === 'string') {
        var index = str.indexOf(pattern, position);
        while (index !== -1 && (typeof limit === 'undefined' || count < limit)) {
          result += str.substring(position, index);
          result += replacement;
          position = index + pattern.length;
          count++;
          index = str.indexOf(pattern, position);
        }
        result += str.substring(position);
      } else {
        var matches = pattern(str);
        if (typeof matches !== 'undefined') {
          while (typeof matches !== 'undefined' && (typeof limit === 'undefined' || count < limit)) {
            result += str.substring(position, matches.start);
            var replacedWith = apply(replacer, [matches], null);
            // check replacedWith is a string
            if (typeof replacedWith === 'string') {
              result += replacedWith;
            } else {
              // not a string - throw error
              throw {
                code: "D3012",
                stack: (new Error()).stack,
                value: replacedWith
              };
            }
            position = matches.start + matches.match.length;
            count++;
            matches = matches.next();
          }
          result += str.substring(position);
        } else {
          result = str;
        }
      }
    } else {
      result = str;
    }

    return result;
  }


  /**
   * Split a string into an array of substrings
   * @param {String} str - string
   * @param {String} separator - the token or regex that splits the string
   * @param {Integer} [limit] - max number of substrings
   * @returns {Array} The array of string
   */
  function functionSplit(str, separator, limit) {
    // undefined inputs always return undefined
    if (typeof str === 'undefined') {
      return undefined;
    }

    // limit, if specified, must be a non-negative number
    if (limit < 0) {
      throw {
        code: "D3020",
        stack: (new Error()).stack,
        value: limit,
        index: 3
      };
    }

    var result = [];

    if (typeof limit === 'undefined' || limit > 0) {
      if (typeof separator === 'string') {
        result = str.split(separator, limit);
      } else {
        var count = 0;
        var matches = separator(str);
        if (typeof matches !== 'undefined') {
          var start = 0;
          while (typeof matches !== 'undefined' && (typeof limit === 'undefined' || count < limit)) {
            result.push(str.substring(start, matches.start));
            start = matches.end;
            matches = matches.next();
            count++;
          }
          if (typeof limit === 'undefined' || count < limit) {
            result.push(str.substring(start));
          }
        } else {
          result = [str];
        }
      }
    }

    return result;
  }

  /**
   * Join an array of strings
   * @param {Array} strs - array of string
   * @param {String} [separator] - the token that splits the string
   * @returns {String} The concatenated string
   */
  function functionJoin(strs, separator) {
    // undefined inputs always return undefined
    if (typeof strs === 'undefined') {
      return undefined;
    }

    // if separator is not specified, default to empty string
    if (typeof separator === 'undefined') {
      separator = "";
    }

    return strs.join(separator);
  }

  /**
   * Cast argument to number
   * @param {Object} arg - Argument
   * @returns {Number} numeric value of argument
   */
  function functionNumber(arg) {
    var result;

    // undefined inputs always return undefined
    if (typeof arg === 'undefined') {
      return undefined;
    }

    if (typeof arg === 'number') {
      // already a number
      result = arg;
    } else if (typeof arg === 'string' && /^-?(0|([1-9][0-9]*))(\.[0-9]+)?([Ee][-+]?[0-9]+)?$/.test(arg) && !isNaN(parseFloat(arg)) && isFinite(arg)) {
      result = parseFloat(arg);
    } else {
      throw {
        code: "D3030",
        value: arg,
        stack: (new Error()).stack,
        index: 1
      };
    }
    return result;
  }


  /**
   * Evaluate an input and return a boolean
   * @param {*} arg - Arguments
   * @returns {boolean} Boolean
   */
  function functionBoolean(arg) {
    // cast arg to its effective boolean value
    // boolean: unchanged
    // string: zero-length -> false; otherwise -> true
    // number: 0 -> false; otherwise -> true
    // null -> false
    // array: empty -> false; length > 1 -> true
    // object: empty -> false; non-empty -> true
    // function -> false

    // undefined inputs always return undefined
    if (typeof arg === 'undefined') {
      return undefined;
    }

    var result = false;
    if (Array.isArray(arg)) {
      if (arg.length === 1) {
        result = functionBoolean(arg[0]);
      } else if (arg.length > 1) {
        var trues = arg.filter(function (val) { return functionBoolean(val); });
        result = trues.length > 0;
      }
    } else if (typeof arg === 'string') {
      if (arg.length > 0) {
        result = true;
      }
    } else if (isNumeric(arg)) {
      if (arg !== 0) {
        result = true;
      }
    } else if (arg !== null && typeof arg === 'object') {
      if (Object.keys(arg).length > 0) {
        // make sure it's not a lambda function
        if (!(isLambda(arg) || arg._jsonata_function)) {
          result = true;
        }
      }
    } else if (typeof arg === 'boolean' && arg === true) {
      result = true;
    }
    return result;
  }

  /**
   * returns the Boolean NOT of the arg
   * @param {*} arg - argument
   * @returns {boolean} - NOT arg
   */
  function functionNot(arg) {
    return !functionBoolean(arg);
  }

  /**
   * Create a map from an array of arguments
   * @param {Function} func - function to apply
   * @param {Array} [arr] - array to map over
   * @returns {Array} Map array
   */
  function functionMap(func, arr) {
    // this can take a variable number of arguments - each one should be mapped to the equivalent arg of func
    // assert that func is a function
    var varargs = arguments;
    var result = [];

    // each subsequent arg must be an array - coerce if not
    var args = arr;
    args = [];
    for (var ii = 1; ii < varargs.length; ii++) {
      args.push(varargs[ii]);
    }
    // do the map - iterate over the arrays, and invoke func
    for (var i = 0; i < args[0].length; i++) {
      var func_args = [];
      var length = typeof func === 'function' ? func.length :
        func._jsonata_function === true ? func.implementation.length : func.arguments.length;
      for (var j = 0; j < length; j++) {
        func_args.push(args[j][i]);
      }
      // invoke func
      result.push(apply(func, func_args, null));
    }

    return result;
  }

  /**
   * Fold left function
   * @param {Function} func - Function
   * @param {Array} sequence - Sequence
   * @param {Object} init - Initial value
   * @returns {*} Result
   */
  function functionFoldLeft(func, sequence, init) {
    var result;

    if (!(func.length === 2 || (func._jsonata_function === true && func.implementation.length === 2) || func.arguments.length === 2)) {
      throw {
        stack: (new Error()).stack,
        code: "D3050",
        index: 1
      };
    }

    var index;
    if (typeof init === 'undefined' && sequence.length > 0) {
      result = sequence[0];
      index = 1;
    } else {
      result = init;
      index = 0;
    }

    while (index < sequence.length) {
      result = apply(func, [result, sequence[index]], null);
      index++;
    }

    return result;
  }

  /**
   * Return keys for an object
   * @param {Object} arg - Object
   * @returns {Array} Array of keys
   */
  function functionKeys(arg) {
    var result = [];

    if (Array.isArray(arg)) {
      // merge the keys of all of the items in the array
      var merge = {};
      arg.forEach(function (item) {
        var keys = functionKeys(item);
        if (Array.isArray(keys)) {
          keys.forEach(function (key) {
            merge[key] = true;
          });
        }
      });
      result = functionKeys(merge);
    } else if (arg !== null && typeof arg === 'object' && !(isLambda(arg))) {
      result = Object.keys(arg);
      if (result.length === 0) {
        result = undefined;
      }
    } else {
      result = undefined;
    }
    return result;
  }

  /**
   * Return value from an object for a given key
   * @param {Object} object - Object
   * @param {String} key - Key in object
   * @returns {*} Value of key in object
   */
  function functionLookup(object, key) {
    var result = evaluateName({ value: key }, object);
    return result;
  }

  /**
   * Append second argument to first
   * @param {Array|Object} arg1 - First argument
   * @param {Array|Object} arg2 - Second argument
   * @returns {*} Appended arguments
   */
  function functionAppend(arg1, arg2) {
    // disregard undefined args
    if (typeof arg1 === 'undefined') {
      return arg2;
    }
    if (typeof arg2 === 'undefined') {
      return arg1;
    }
    // if either argument is not an array, make it so
    if (!Array.isArray(arg1)) {
      arg1 = [arg1];
    }
    if (!Array.isArray(arg2)) {
      arg2 = [arg2];
    }
    Array.prototype.push.apply(arg1, arg2);
    return arg1;
  }

  /**
   * Determines if the argument is undefined
   * @param {*} arg - argument
   * @returns {boolean} False if argument undefined, otherwise true
   */
  function functionExists(arg) {
    if (typeof arg === 'undefined') {
      return false;
    } else {
      return true;
    }
  }

  /**
   * Splits an object into an array of object with one property each
   * @param {*} arg - the object to split
   * @returns {*} - the array
   */
  function functionSpread(arg) {
    var result = [];

    if (Array.isArray(arg)) {
      // spread all of the items in the array
      arg.forEach(function (item) {
        result = functionAppend(result, functionSpread(item));
      });
    } else if (arg !== null && typeof arg === 'object' && !isLambda(arg)) {
      for (var key in arg) {
        var obj = {};
        obj[key] = arg[key];
        result.push(obj);
      }
    } else {
      result = arg;
    }
    return result;
  }

  /**
   * Create frame
   * @param {Object} enclosingEnvironment - Enclosing environment
   * @returns {{bind: bind, lookup: lookup}} Created frame
   */
  function createFrame(enclosingEnvironment) {
    var bindings = {};
    return {
      bind: function (name, value) {
        bindings[name] = value;
      },
      lookup: function (name) {
        var value;
        if (bindings.hasOwnProperty(name)) {
          value = bindings[name];
        } else if (enclosingEnvironment) {
          value = enclosingEnvironment.lookup(name);
        }
        return value;
      }
    };
  }

  // Function registration
  staticFrame.bind('sum', defineFunction(functionSum, '<a<n>:n>'));
  staticFrame.bind('count', defineFunction(functionCount, '<a:n>'));
  staticFrame.bind('max', defineFunction(functionMax, '<a<n>:n>'));
  staticFrame.bind('min', defineFunction(functionMin, '<a<n>:n>'));
  staticFrame.bind('average', defineFunction(functionAverage, '<a<n>:n>'));
  staticFrame.bind('string', defineFunction(functionString, '<x-:s>'));
  staticFrame.bind('substring', defineFunction(functionSubstring, '<s-nn?:s>'));
  staticFrame.bind('substringBefore', defineFunction(functionSubstringBefore, '<s-s:s>'));
  staticFrame.bind('substringAfter', defineFunction(functionSubstringAfter, '<s-s:s>'));
  staticFrame.bind('lowercase', defineFunction(functionLowercase, '<s-:s>'));
  staticFrame.bind('uppercase', defineFunction(functionUppercase, '<s-:s>'));
  staticFrame.bind('length', defineFunction(functionLength, '<s-:n>'));
  staticFrame.bind('trim', defineFunction(functionTrim, '<s-:s>'));
  staticFrame.bind('match', defineFunction(functionMatch, '<s-f<s:o>n?:a<o>>'));
  staticFrame.bind('contains', defineFunction(functionContains, '<s-(sf):b>')); // TODO <s-(sf<s:o>):b>
  staticFrame.bind('replace', defineFunction(functionReplace, '<s-(sf)(sf)n?:s>')); // TODO <s-(sf<s:o>)(sf<o:s>)n?:s>
  staticFrame.bind('split', defineFunction(functionSplit, '<s-(sf)n?:a<s>>')); // TODO <s-(sf<s:o>)n?:a<s>>
  staticFrame.bind('join', defineFunction(functionJoin, '<a<s>s?:s>'));
  staticFrame.bind('number', defineFunction(functionNumber, '<(ns)-:n>'));
  staticFrame.bind('boolean', defineFunction(functionBoolean, '<x-:b>'));
  staticFrame.bind('not', defineFunction(functionNot, '<x-:b>'));
  staticFrame.bind('map', defineFunction(functionMap, '<fa+>'));
  staticFrame.bind('reduce', defineFunction(functionFoldLeft, '<faj?:j>')); // TODO <f<jj:j>a<j>j?:j>
  staticFrame.bind('keys', defineFunction(functionKeys, '<x-:a<s>>'));
  staticFrame.bind('lookup', defineFunction(functionLookup, '<x-s:x>'));
  staticFrame.bind('append', defineFunction(functionAppend, '<xx:a>'));
  staticFrame.bind('exists', defineFunction(functionExists, '<x:b>'));
  staticFrame.bind('spread', defineFunction(functionSpread, '<x-:a<o>>'));

  /**
   * Error codes
   *
   */
  var errorCodes = {
    "S0101": "No terminating quote found in string literal",
    "S0102": "Number out of range: {{token}}",
    "S0103": "unsupported escape sequence: \\{{token}}",
    "S0104": "The escape sequence \\u must be followed by 4 hex digits",
    "S0203": "Syntax error: expected '{{value}}' before end of expression",
    "S0202": "Syntax error: expected '{{value}}', got '{{token}}'",
    "S0204": "Unknown operator: {{token}}",
    "S0205": "Unexpected token: {{token}}",
    "S0208": "Parameter {{value}} of function definition must be a variable name (start with $)",
    "S0209": "A predicate cannot follow a grouping expression in a step.",
    "S0210": "Each step can only have one grouping expression.",
    "S0201": "Syntax error: {{token}}",
    "S0206": "Unknown expression type: {{token}}",
    "S0207": "Syntax error: unexpected end of expression",
    "S0301": "Empty regular expressions are not allowed",
    "S0302": "No terminating / in regular expression",
    "S0402": "Choice groups containing parameterized types not supported",
    "S0401": "Type parameters can only be applied to functions and arrays",
    "T0410": "Argument {{index}} of function '{{token}}' does not match function signature",
    "T0411": "Context value is not a compatible type with argument {{index}} of function '{{token}}'",
    "T0412": "Argument {{index}} of function '{{token}}' must be an array of {{type}}",
    "D1001": "Number out of range: {{value}}",
    "D1002": "Cannot negate a non-numeric value: {{value}}",
    "T2001": "LHS of {{token}} operator must evaluate to a number",
    "T2002": "RHS of {{token}} operator must evaluate to a number",
    "T1003": "Key in object structure must evaluate to a string. Got: {{value}}",
    "T2003": "LHS of range operator (..) must evaluate to an integer",
    "T2004": "RHS of range operator (..) must evaluate to an integer",
    "D2005": "Left hand side of := must be a variable name (start with $)",
    "D1004": "Regular expression matches zero length string",
    "T2006": "RHS of function application operator ~> is not a function",
    "T1005": "Attempted to invoke a non-function. Did you mean '${{token}}'?",
    "T1006": "Attempted to invoke a non-function",
    "T1007": "Attempted to partially apply a non-function. Did you mean '${{token}}'?",
    "T1008": "Attempted to partially apply a non-function",
    "D3001": "Attempting to invoke string function on Infinity or NaN",
    "D3010": "Second argument of replace function cannot be an empty string",
    "D3011": "Forth argument of replace function must evaluate to a positive number",
    "D3012": "Attempted to replace a matched string with a non-string value",
    "D3020": "Third argument of split function must evaluate to a positive number",
    "D3030": "Unable to cast value to a number: {{value}}"
  };

  /**
   * lookup a message template from the catalog and substitute the inserts
   * @param {string} err - error code to lookup
   * @returns {string} message
   */
  function lookupMessage(err) {
    var message = 'Unknown error';
    if (typeof err.message !== 'undefined') {
      message = err.message;
    }
    var template = errorCodes[err.code];
    if (typeof template !== 'undefined') {
      // if there are any handlebars, replace them with the field references
      message = template.replace(/\{\{([^}]+)}}/g, function () {
        return err[arguments[1]];
      });
    }
    return message;
  }

  /**
   * JSONata
   * @param {Object} expr - JSONata expression
   * @returns {{evaluate: evaluate, assign: assign}} Evaluated expression
   */
  function jsonata(expr) {
    var ast;
    try {
      ast = parser(expr);
    } catch (err) {
      // insert error message into structure
      err.message = lookupMessage(err);
      throw err;
    }
    var environment = createFrame(staticFrame);
    return {
      evaluate: function (input, bindings) {
        if (typeof bindings !== 'undefined') {
          var exec_env;
          // the variable bindings have been passed in - create a frame to hold these
          exec_env = createFrame(environment);
          for (var v in bindings) {
            exec_env.bind(v, bindings[v]);
          }
        } else {
          exec_env = environment;
        }
        // put the input document into the environment as the root object
        exec_env.bind('$', input);
        var result;
        try {
          result = evaluate(ast, input, exec_env);
        } catch (err) {
          // insert error message into structure
          err.message = lookupMessage(err);
          throw err;
        }
        return result;
      },
      assign: function (name, value) {
        environment.bind(name, value);
      },
      registerFunction: function (name, implementation, signature) {
        var func = defineFunction(implementation, signature);
        environment.bind(name, func);
      }
    };
  }

  jsonata.parser = parser;

  return jsonata;

})();

// node.js only - export the jsonata and parser functions
// istanbul ignore else
if (typeof module !== 'undefined') {
  module.exports = jsonata;
}
; (function () {
  function indentLine(str, length) {
    if (length <= 0) {
      return str;
    }
    var i = (new Array(length)).join(" ");
    str = str.replace(/^\s*/, i);
    return str;
  }
  function formatExpression(str) {
    var length = str.length;
    var start = 0;
    var inString = false;
    var inBox = false;
    var quoteChar;
    var list = [];
    var stack = [];
    var frame;
    var v;
    var matchingBrackets = {
      "(": ")",
      "[": "]",
      "{": "}"
    }
    for (var i = 0; i < length; i++) {
      var c = str[i];
      if (!inString) {
        if (c === "'" || c === '"') {
          inString = true;
          quoteChar = c;
          frame = { type: "string", pos: i };
          list.push(frame);
          stack.push(frame);
        } else if (c === ";") {
          frame = { type: ";", pos: i };
          list.push(frame);
        } else if (c === ",") {
          frame = { type: ",", pos: i };
          list.push(frame);
        } else if (/[\(\[\{]/.test(c)) {
          frame = { type: "open-block", char: c, pos: i };
          list.push(frame);
          stack.push(frame);
        } else if (/[\}\)\]]/.test(c)) {
          var oldFrame = stack.pop();
          if (matchingBrackets[oldFrame.char] !== c) {
            //console.log("Stack frame mismatch",c,"at",i,"expected",matchingBrackets[oldFrame.char],"from",oldFrame.pos);
            return str;
          }
          //console.log("Closing",c,"at",i,"compare",oldFrame.type,oldFrame.pos);
          oldFrame.width = i - oldFrame.pos;
          frame = { type: "close-block", pos: i, char: c, width: oldFrame.width }
          list.push(frame);
        }
      } else {
        if (c === quoteChar) {
          // Next char must be a ]
          inString = false;
          stack.pop();
        }
      }

    }
    // console.log(stack);
    var result = str;
    var indent = 0;
    var offset = 0;
    var pre, post, indented;
    var longStack = [];
    list.forEach(function (f) {
      if (f.type === ";" || f.type === ",") {
        if (longStack[longStack.length - 1]) {
          pre = result.substring(0, offset + f.pos + 1);
          post = result.substring(offset + f.pos + 1);
          indented = indentLine(post, indent);
          result = pre + "\n" + indented;
          offset += indented.length - post.length + 1;
        }
      } else if (f.type === "open-block") {
        if (f.width > 30) {
          longStack.push(true);
          indent += 4;
          pre = result.substring(0, offset + f.pos + 1);
          post = result.substring(offset + f.pos + 1);
          indented = indentLine(post, indent);
          result = pre + "\n" + indented;
          offset += indented.length - post.length + 1;
        } else {
          longStack.push(false);
        }
      } else if (f.type === "close-block") {
        if (f.width > 30) {
          indent -= 4;
          pre = result.substring(0, offset + f.pos);
          post = result.substring(offset + f.pos);
          indented = indentLine(post, indent);
          result = pre + "\n" + indented;
          offset += indented.length - post.length + 1;
        }
        longStack.pop();
      }
    })
    //console.log(result);
    return result;
  }

  jsonata.format = formatExpression;
  jsonata.functions =
  {
    '$append': { args: ['array', 'array'] },
    '$average': { args: ['value'] },
    '$boolean': { args: ['value'] },
    '$contains': { args: ['str', 'pattern'] },
    '$count': { args: ['array'] },
    '$exists': { args: ['value'] },
    '$join': { args: ['array', 'separator'] },
    '$keys': { args: ['object'] },
    '$length': { args: ['string'] },
    '$lookup': { args: ['object', 'key'] },
    '$lowercase': { args: ['string'] },
    '$match': { args: ['str', 'pattern', 'limit'] },
    '$map': { args: [] },
    '$max': { args: ['array'] },
    '$min': { args: ['array'] },
    '$not': { args: ['value'] },
    '$number': { args: ['value'] },
    '$reduce': { args: [] },
    '$replace': { args: ['str', 'pattern', 'replacement', 'limit'] },
    '$split': { args: ['string', 'separator', 'limit'] },
    '$spread': { args: ['object'] },
    '$string': { args: ['value'] },
    '$substring': { args: ['string', 'start', 'length'] },
    '$substringAfter': { args: ['string', 'chars'] },
    '$substringBefore': { args: ['string', 'chars'] },
    '$sum': { args: ['array'] },
    '$trim': { args: ['str'] },
    '$uppercase': { args: ['string'] }
  }
  jsonata.getFunctionSnippet = function (fn) {
    var snippetText = "";
    if (jsonata.functions.hasOwnProperty(fn)) {
      var def = jsonata.functions[fn];
      snippetText = "\\" + fn + "(";
      if (def.args) {
        snippetText += def.args.map(function (a, i) { return "${" + (i + 1) + ":" + a + "}" }).join(", ");
      }
      snippetText += ")\n"
    }
    return snippetText;
  }
})();
; (function () { function r(e, n, t) { function o(i, f) { if (!n[i]) { if (!e[i]) { var c = "function" == typeof require && require; if (!f && c) return c(i, !0); if (u) return u(i, !0); var a = new Error("Cannot find module '" + i + "'"); throw a.code = "MODULE_NOT_FOUND", a } var p = n[i] = { exports: {} }; e[i][0].call(p.exports, function (r) { var n = e[i][1][r]; return o(n || r) }, p, p.exports, r, e, n, t) } return n[i].exports } for (var u = "function" == typeof require && require, i = 0; i < t.length; i++)o(t[i]); return o } return r })()({
  1: [function (require, module, exports) {
    (function (global) {
      // vm-browserify without iframe
      var Object_keys = function (obj) {
        if (Object.keys) return Object.keys(obj)
        else {
          var res = [];
          for (var key in obj) res.push(key)
          return res;
        }
      };

      var forEach = function (xs, fn) {
        if (xs.forEach) return xs.forEach(fn)
        else for (var i = 0; i < xs.length; i++) {
          fn(xs[i], i, xs);
        }
      };

      var defineProp = (function () {
        try {
          Object.defineProperty({}, '_', {});
          return function (obj, name, value) {
            Object.defineProperty(obj, name, {
              writable: true,
              enumerable: false,
              configurable: true,
              value: value
            })
          };
        } catch (e) {
          return function (obj, name, value) {
            obj[name] = value;
          };
        }
      }());

      var globals = ['Array', 'Boolean', 'Date', 'Error', 'EvalError', 'Function',
        'Infinity', 'JSON', 'Math', 'NaN', 'Number', 'Object', 'RangeError',
        'ReferenceError', 'RegExp', 'String', 'SyntaxError', 'TypeError', 'URIError',
        'decodeURI', 'decodeURIComponent', 'encodeURI', 'encodeURIComponent', 'escape',
        'eval', 'isFinite', 'isNaN', 'parseFloat', 'parseInt', 'undefined', 'unescape'];

      function Context() { }
      Context.prototype = {};

      var Script = exports.Script = function NodeScript(code) {
        if (!(this instanceof Script)) return new Script(code);
        this.code = code;
      };

      Script.prototype.runInContext = function (context) {
        if (!global._isNoderedActive) return;
        const contextInfo = context.msg ? context.msg.contextInfo : null;
        if (contextInfo && global.PAGE_CONTEXTS[contextInfo.pageContextId] === undefined) {
          // console.log('AHGJHA', contextInfo, global.PAGE_CONTEXTS, new Error());
          return
        }
        // console.log('CONTEXT2', context, ' : ', context.flow, ' : ');
        if (!(context instanceof Context)) {
          throw new TypeError("needs a 'context' argument.");
        }
        var functionArgs = [];
        forEach(Object_keys(context), function (key) {
          functionArgs.push(key);
        });
        if (context.msg.__currents && context.msg.__currents.length) {
          context.msg.__currents.forEach(function (currentKey) {
            if (functionArgs.indexOf(currentKey) == -1) { functionArgs.push(currentKey) }
            if (!context[currentKey]) {
              Object.defineProperty(context, currentKey, {
                enumerable: true,
                get: function () {
                  if (context["msg"] && context.msg.hasOwnProperty(currentKey)) { return context.msg[currentKey] }
                }
              })
            }
          })

        }
        var res = (Function.apply(null, functionArgs.concat(this.code))).apply(window, functionArgs.map(function (key) { return context[key] }));
        context.results = res;
        return res;
      };

      Script.prototype.runInCleanerContext = function (context) {
        if (!global._isNoderedActive) return;
        const contextInfo = context.msg ? context.msg.contextInfo : null;
        if (contextInfo && global.PAGE_CONTEXTS[contextInfo.pageContextId] === undefined) {
          // console.log('AHGJHA2', contextInfo, global.PAGE_CONTEXTS, new Error());
          return
        }
        if (!(context instanceof Context)) {
          throw new TypeError("needs a 'context' argument.");
        }
        var functionArgs = [];
        forEach(Object_keys(context), function (key) {
          functionArgs.push(key);
        });
        var funcWithArgs = (Function.apply({}, functionArgs.concat(this.code)));
        var contextValues = functionArgs.map(function (key) {
          return context[key]
        });
        var res = funcWithArgs.apply({}, contextValues);
        return res;
      };

      Script.prototype.runInNewContext = function (context) {
        var ctx = Script.createContext(context);
        var res = this.runInContext(ctx);

        forEach(Object_keys(ctx), function (key) {
          context[key] = ctx[key];
        });

        return res;
      };

      forEach(Object_keys(Script.prototype), function (name) {
        exports[name] = Script[name] = function (code) {
          var s = Script(code);
          return s[name].apply(s, [].slice.call(arguments, 1));
        };
      });

      exports.createScript = function (code) {
        return exports.Script(code);
      };

      // var forceUpdate = false;
      //
      // var getDataProxy = function(bcopy) {
      //     var contextInfo = bcopy.msg.contextInfo || console.warn('no contextInfo', bcopy.msg);
      //     var list = supersonic.listPageDataReferences(contextInfo);
      //     var val = {};
      //     list.forEach( function(res) {
      //         var obj = supersonic.context.getDataReference( res, contextInfo ) || {};
      //         if ( Array.isArray(obj) ) {
      //             val[res] = obj; // not proxied
      //             return;
      //         }
      //         var proxy = new Proxy( obj, {
      //             get: function(target, prop, receiver) {
      //                 var objNow = supersonic.context.getDataReference( res, contextInfo );
      //                 if (!objNow) {
      //                     return undefined;
      //                 }
      //                 return objNow[prop];
      //             },
      //             set: function(target, prop, value) {
      //                 var data = {};
      //                 data[prop] = value;
      //                 supersonic.context.setDataReference( res, data, forceUpdate, contextInfo );
      //             }
      //         } );
      //         val[res] = proxy;
      //     } );
      //     return new Proxy( val, {
      //         get: function( target, prop, receiver ) {
      //             return val[prop];
      //         },
      //         set: function( target, prop, value ) {
      //             if ( !Array.isArray(value) ) {
      //                 // Get rid of the proxy
      //                 value = Object.assign( {}, value );
      //             }
      //             supersonic.context.setDataReference( prop, value, forceUpdate, contextInfo );
      //             // Note that further assignments to the insides of value are missed
      //             val[prop] = value;
      //         }
      //     } );
      // }

      // var getComponentProxy = function(bcopy) {
      //     var contextInfo = bcopy.msg.contextInfo || console.warn('no contextInfo', bcopy.msg);
      //     var val = new Proxy( {}, {
      //         get: function( val_target, componentId, val_receiver ) {
      //             return new Proxy( {}, {
      //                 get: function(target, prop, receiver) {
      //                     var component = supersonic.getComponentState( componentId, contextInfo );
      //                     if (!component) return undefined;
      //                     return component[prop];
      //                 },
      //                 set: function(target, prop, value) {
      //                     var data = {};
      //                     data[prop] = value;
      //                     supersonic.setComponentState( componentId, data, contextInfo );
      //                 }
      //             } );
      //         }
      //     } );
      //     return val;
      // }

      // var bindContext = function (bcopy) {
      //   var contextsupersonic = (bcopy && bcopy.msg && bcopy.msg.supersonic) || supersonic;
      //   const ctx = contextsupersonic.context
      //
      //   bcopy.app = {
      //     get: function (name) {
      //       const ctxInfo = bcopy.msg ? bcopy.msg.contextInfo : {};
      //       return ctx.getAppVariable(name, ctxInfo);
      //     },
      //     set: function (name, value) {
      //       const ctxInfo = bcopy.msg ? bcopy.msg.contextInfo : {};
      //       ctx.setAppVariable(name, value, ctxInfo)
      //     }
      //   };
      //   bcopy.page = {
      //     get: function (name) {
      //       const ctxInfo = bcopy.msg ? bcopy.msg.contextInfo : {};
      //       return ctx.getPageVariable(name, ctxInfo)
      //     },
      //     set: function (name, value) {
      //       const ctxInfo = bcopy.msg ? bcopy.msg.contextInfo : {};
      //       ctx.setPageVariable(name, value, ctxInfo)
      //     }
      //   };
      //   bcopy.data = {
      //     get: function (name) {
      //       const ctxInfo = bcopy.msg ? bcopy.msg.contextInfo : {};
      //       return ctx.getDataReference(name, ctxInfo)
      //     },
      //     set: function (name, value) {
      //       const ctxInfo = bcopy.msg ? bcopy.msg.contextInfo : {};
      //       ctx.setDataReference(name, value, ctxInfo)
      //     }
      //   }
      //   bcopy.parameters = {
      //     get: function (name) {
      //       const ctxInfo = bcopy.msg ? bcopy.msg.contextInfo : {};
      //       return ctx.getParameter(name, ctxInfo)
      //     },
      //   }
      //   bcopy.current = {
      //     get: function () { return bcopy.msg.current },
      //   }
      // }

      exports.createContext = Script.createContext = function (context) {
        var bcopy = new Context();
        if (typeof context === 'object') {
          forEach(Object_keys(context), function (key) {
            bcopy[key] = context[key];
          });
        }

        getNoderedRuntime().bindContext(bcopy);

        return bcopy
      };

      exports.createCleanContext = Script.createCleanContext = function (context) {
        var bcopy = new Context();
        if (typeof context === 'object') {
          forEach(Object_keys(context), function (key) {
            bcopy[key] = context[key];
          });
        }
        return bcopy
      };

    }).call(this, typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {})
  }, {}], 2: [function (require, module, exports) {
    var when = require('when');

    var redNodes = require("../../red/runtime/nodes");
    var context = require("../../red/runtime/nodes/context");
    var storage = require("../../red/runtime/storage");
    var log = require("../../red/runtime/log");
    var i18n = require("../../red/runtime/i18n");
    var events = require("../../red/runtime/events");
    var settings = require("../../red/runtime/settings");

    var started = false;

    function init(userSettings) {
      userSettings.version = getVersion();
      userSettings.credentialSecret = 'secret-key-for-node-credentials'; // see appgyver/editor/angular/angular-red.js
      log.init(userSettings);
      settings.init(userSettings);

      redNodes.init(runtime);
    }

    function getVersion() {
      return 'appgyver-custom';
    }

    function start(consoleLog) {
      return i18n.init()
        .then(function () { return storage.init(runtime) })
        .then(function () { return settings.load(storage) })
        .then(function () {
          return when.resolve().then(function () {

            var i;
            var nodeErrors = redNodes.getNodeList(function (n) { return n.err != null; });
            var nodeMissing = redNodes.getNodeList(function (n) { return n.module && n.enabled && !n.loaded && !n.err; });
            if (nodeErrors.length > 0) {
              log.warn("------------------------------------------------------");
              for (i = 0; i < nodeErrors.length; i += 1) {
                log.warn("[" + nodeErrors[i].name + "] " + nodeErrors[i].err);
              }
              log.warn("------------------------------------------------------");
            }
            if (nodeMissing.length > 0) {
              log.warn(log._("server.missing-modules"));
              var missingModules = {};
              for (i = 0; i < nodeMissing.length; i++) {
                var missing = nodeMissing[i];
                missingModules[missing.module] = (missingModules[missing.module] || []).concat(missing.types);
              }
              var promises = [];
              for (i in missingModules) {
                if (missingModules.hasOwnProperty(i)) {
                  log.warn(" - " + i + ": " + missingModules[i].join(", "));
                  if (settings.autoInstallModules && i != "node-red") {
                    redNodes.installModule(i).otherwise(function (err) {
                      // Error already reported. Need the otherwise handler
                      // to stop the error propagating any further
                    });
                  }
                }
              }
              if (!settings.autoInstallModules) {
                log.info(log._("server.removing-modules"));
                redNodes.cleanModuleList();
              }
            }
            redNodes.loadFlows().then(redNodes.startFlows);
            started = true;
          }).otherwise(function (err) {
            logError(consoleLog, err);
          });
        });
    }

    function stop() {
      started = false;
      return redNodes.stopFlows();
    }

    function emit(id, params, consoleLog) {

      var node = redNodes.getNode(id);
      // console.log('SSSSSSLSLSLSLSS', node);
      if (node) {
        node.emit('input', params);
      } else {
        logError(consoleLog, 'Unknown node id: ' + id);
      }
    }

    function logError(consoleLog, error) {
      var log = consoleLog || console.log;
      // var err = new Error('FUCK')
      // console.log('AHAAJHA', err)
      log(`ERROR fron NODERED: ${error}`);
    }

    var runtime = module.exports = {
      init: init,
      start: start,
      stop: stop,

      emit: emit,

      version: getVersion,

      _: function () {
        return arguments;
      },
      log: log,
      i18n: i18n,
      settings: settings,
      storage: storage,
      events: events,
      nodes: redNodes,
      context: context,
      util: require("../../red/runtime/util"),
      httpAdmin: {
        get: function () { },
        post: function () { },
        delete: function () { }
      },
      auth: {
        needsPermission: function () { }
      },
      library: {
        register: function () { }
      },
      isStarted: function () {
        return started;
      }
    };

    if (typeof window !== 'undefined') {
      window.red = runtime;
      window.vm = require('./dolan-vm');
    }

  }, { "../../red/runtime/events": 712, "../../red/runtime/i18n": 713, "../../red/runtime/log": 714, "../../red/runtime/nodes": 721, "../../red/runtime/nodes/context": 716, "../../red/runtime/settings": 728, "../../red/runtime/storage": 729, "../../red/runtime/util": 731, "./dolan-vm": 1, "when": 710 }], 3: [function (require, module, exports) {
    /*!
     * accepts
     * Copyright(c) 2014 Jonathan Ong
     * Copyright(c) 2015 Douglas Christopher Wilson
     * MIT Licensed
     */

    'use strict'

    /**
     * Module dependencies.
     * @private
     */

    var Negotiator = require('negotiator')
    var mime = require('mime-types')

    /**
     * Module exports.
     * @public
     */

    module.exports = Accepts

    /**
     * Create a new Accepts object for the given req.
     *
     * @param {object} req
     * @public
     */

    function Accepts(req) {
      if (!(this instanceof Accepts)) {
        return new Accepts(req)
      }

      this.headers = req.headers
      this.negotiator = new Negotiator(req)
    }

    /**
     * Check if the given `type(s)` is acceptable, returning
     * the best match when true, otherwise `undefined`, in which
     * case you should respond with 406 "Not Acceptable".
     *
     * The `type` value may be a single mime type string
     * such as "application/json", the extension name
     * such as "json" or an array `["json", "html", "text/plain"]`. When a list
     * or array is given the _best_ match, if any is returned.
     *
     * Examples:
     *
     *     // Accept: text/html
     *     this.types('html');
     *     // => "html"
     *
     *     // Accept: text/*, application/json
     *     this.types('html');
     *     // => "html"
     *     this.types('text/html');
     *     // => "text/html"
     *     this.types('json', 'text');
     *     // => "json"
     *     this.types('application/json');
     *     // => "application/json"
     *
     *     // Accept: text/*, application/json
     *     this.types('image/png');
     *     this.types('png');
     *     // => undefined
     *
     *     // Accept: text/*;q=.5, application/json
     *     this.types(['html', 'json']);
     *     this.types('html', 'json');
     *     // => "json"
     *
     * @param {String|Array} types...
     * @return {String|Array|Boolean}
     * @public
     */

    Accepts.prototype.type =
      Accepts.prototype.types = function (types_) {
        var types = types_

        // support flattened arguments
        if (types && !Array.isArray(types)) {
          types = new Array(arguments.length)
          for (var i = 0; i < types.length; i++) {
            types[i] = arguments[i]
          }
        }

        // no types, return all requested types
        if (!types || types.length === 0) {
          return this.negotiator.mediaTypes()
        }

        // no accept header, return first given type
        if (!this.headers.accept) {
          return types[0]
        }

        var mimes = types.map(extToMime)
        var accepts = this.negotiator.mediaTypes(mimes.filter(validMime))
        var first = accepts[0]

        return first
          ? types[mimes.indexOf(first)]
          : false
      }

    /**
     * Return accepted encodings or best fit based on `encodings`.
     *
     * Given `Accept-Encoding: gzip, deflate`
     * an array sorted by quality is returned:
     *
     *     ['gzip', 'deflate']
     *
     * @param {String|Array} encodings...
     * @return {String|Array}
     * @public
     */

    Accepts.prototype.encoding =
      Accepts.prototype.encodings = function (encodings_) {
        var encodings = encodings_

        // support flattened arguments
        if (encodings && !Array.isArray(encodings)) {
          encodings = new Array(arguments.length)
          for (var i = 0; i < encodings.length; i++) {
            encodings[i] = arguments[i]
          }
        }

        // no encodings, return all requested encodings
        if (!encodings || encodings.length === 0) {
          return this.negotiator.encodings()
        }

        return this.negotiator.encodings(encodings)[0] || false
      }

    /**
     * Return accepted charsets or best fit based on `charsets`.
     *
     * Given `Accept-Charset: utf-8, iso-8859-1;q=0.2, utf-7;q=0.5`
     * an array sorted by quality is returned:
     *
     *     ['utf-8', 'utf-7', 'iso-8859-1']
     *
     * @param {String|Array} charsets...
     * @return {String|Array}
     * @public
     */

    Accepts.prototype.charset =
      Accepts.prototype.charsets = function (charsets_) {
        var charsets = charsets_

        // support flattened arguments
        if (charsets && !Array.isArray(charsets)) {
          charsets = new Array(arguments.length)
          for (var i = 0; i < charsets.length; i++) {
            charsets[i] = arguments[i]
          }
        }

        // no charsets, return all requested charsets
        if (!charsets || charsets.length === 0) {
          return this.negotiator.charsets()
        }

        return this.negotiator.charsets(charsets)[0] || false
      }

    /**
     * Return accepted languages or best fit based on `langs`.
     *
     * Given `Accept-Language: en;q=0.8, es, pt`
     * an array sorted by quality is returned:
     *
     *     ['es', 'pt', 'en']
     *
     * @param {String|Array} langs...
     * @return {Array|String}
     * @public
     */

    Accepts.prototype.lang =
      Accepts.prototype.langs =
      Accepts.prototype.language =
      Accepts.prototype.languages = function (languages_) {
        var languages = languages_

        // support flattened arguments
        if (languages && !Array.isArray(languages)) {
          languages = new Array(arguments.length)
          for (var i = 0; i < languages.length; i++) {
            languages[i] = arguments[i]
          }
        }

        // no languages, return all requested languages
        if (!languages || languages.length === 0) {
          return this.negotiator.languages()
        }

        return this.negotiator.languages(languages)[0] || false
      }

    /**
     * Convert extnames to mime.
     *
     * @param {String} type
     * @return {String}
     * @private
     */

    function extToMime(type) {
      return type.indexOf('/') === -1
        ? mime.lookup(type)
        : type
    }

    /**
     * Check if mime is valid.
     *
     * @param {String} type
     * @return {String}
     * @private
     */

    function validMime(type) {
      return typeof type === 'string'
    }

  }, { "mime-types": 587, "negotiator": 593 }], 4: [function (require, module, exports) {
    'use strict'

    /**
     * Expose `arrayFlatten`.
     */
    module.exports = arrayFlatten

    /**
     * Recursive flatten function with depth.
     *
     * @param  {Array}  array
     * @param  {Array}  result
     * @param  {Number} depth
     * @return {Array}
     */
    function flattenWithDepth(array, result, depth) {
      for (var i = 0; i < array.length; i++) {
        var value = array[i]

        if (depth > 0 && Array.isArray(value)) {
          flattenWithDepth(value, result, depth - 1)
        } else {
          result.push(value)
        }
      }

      return result
    }

    /**
     * Recursive flatten function. Omitting depth is slightly faster.
     *
     * @param  {Array} array
     * @param  {Array} result
     * @return {Array}
     */
    function flattenForever(array, result) {
      for (var i = 0; i < array.length; i++) {
        var value = array[i]

        if (Array.isArray(value)) {
          flattenForever(value, result)
        } else {
          result.push(value)
        }
      }

      return result
    }

    /**
     * Flatten an array, with the ability to define a depth.
     *
     * @param  {Array}  array
     * @param  {Number} depth
     * @return {Array}
     */
    function arrayFlatten(array, depth) {
      if (depth == null) {
        return flattenForever(array, [])
      }

      return flattenWithDepth(array, [], depth)
    }

  }, {}], 5: [function (require, module, exports) {
    var asn1 = exports;

    asn1.bignum = require('bn.js');

    asn1.define = require('./asn1/api').define;
    asn1.base = require('./asn1/base');
    asn1.constants = require('./asn1/constants');
    asn1.decoders = require('./asn1/decoders');
    asn1.encoders = require('./asn1/encoders');

  }, { "./asn1/api": 6, "./asn1/base": 8, "./asn1/constants": 12, "./asn1/decoders": 14, "./asn1/encoders": 17, "bn.js": 24 }], 6: [function (require, module, exports) {
    var asn1 = require('../asn1');
    var inherits = require('inherits');

    var api = exports;

    api.define = function define(name, body) {
      return new Entity(name, body);
    };

    function Entity(name, body) {
      this.name = name;
      this.body = body;

      this.decoders = {};
      this.encoders = {};
    };

    Entity.prototype._createNamed = function createNamed(base) {
      var named;
      try {
        named = require('vm').runInThisContext(
          '(function ' + this.name + '(entity) {\n' +
          '  this._initNamed(entity);\n' +
          '})'
        );
      } catch (e) {
        named = function (entity) {
          this._initNamed(entity);
        };
      }
      inherits(named, base);
      named.prototype._initNamed = function initnamed(entity) {
        base.call(this, entity);
      };

      return new named(this);
    };

    Entity.prototype._getDecoder = function _getDecoder(enc) {
      enc = enc || 'der';
      // Lazily create decoder
      if (!this.decoders.hasOwnProperty(enc))
        this.decoders[enc] = this._createNamed(asn1.decoders[enc]);
      return this.decoders[enc];
    };

    Entity.prototype.decode = function decode(data, enc, options) {
      return this._getDecoder(enc).decode(data, options);
    };

    Entity.prototype._getEncoder = function _getEncoder(enc) {
      enc = enc || 'der';
      // Lazily create encoder
      if (!this.encoders.hasOwnProperty(enc))
        this.encoders[enc] = this._createNamed(asn1.encoders[enc]);
      return this.encoders[enc];
    };

    Entity.prototype.encode = function encode(data, enc, /* internal */ reporter) {
      return this._getEncoder(enc).encode(data, reporter);
    };

  }, { "../asn1": 5, "inherits": 569, "vm": 56 }], 7: [function (require, module, exports) {
    var inherits = require('inherits');
    var Reporter = require('../base').Reporter;
    var Buffer = require('buffer').Buffer;

    function DecoderBuffer(base, options) {
      Reporter.call(this, options);
      if (!Buffer.isBuffer(base)) {
        this.error('Input not Buffer');
        return;
      }

      this.base = base;
      this.offset = 0;
      this.length = base.length;
    }
    inherits(DecoderBuffer, Reporter);
    exports.DecoderBuffer = DecoderBuffer;

    DecoderBuffer.prototype.save = function save() {
      return { offset: this.offset, reporter: Reporter.prototype.save.call(this) };
    };

    DecoderBuffer.prototype.restore = function restore(save) {
      // Return skipped data
      var res = new DecoderBuffer(this.base);
      res.offset = save.offset;
      res.length = this.offset;

      this.offset = save.offset;
      Reporter.prototype.restore.call(this, save.reporter);

      return res;
    };

    DecoderBuffer.prototype.isEmpty = function isEmpty() {
      return this.offset === this.length;
    };

    DecoderBuffer.prototype.readUInt8 = function readUInt8(fail) {
      if (this.offset + 1 <= this.length)
        return this.base.readUInt8(this.offset++, true);
      else
        return this.error(fail || 'DecoderBuffer overrun');
    }

    DecoderBuffer.prototype.skip = function skip(bytes, fail) {
      if (!(this.offset + bytes <= this.length))
        return this.error(fail || 'DecoderBuffer overrun');

      var res = new DecoderBuffer(this.base);

      // Share reporter state
      res._reporterState = this._reporterState;

      res.offset = this.offset;
      res.length = this.offset + bytes;
      this.offset += bytes;
      return res;
    }

    DecoderBuffer.prototype.raw = function raw(save) {
      return this.base.slice(save ? save.offset : this.offset, this.length);
    }

    function EncoderBuffer(value, reporter) {
      if (Array.isArray(value)) {
        this.length = 0;
        this.value = value.map(function (item) {
          if (!(item instanceof EncoderBuffer))
            item = new EncoderBuffer(item, reporter);
          this.length += item.length;
          return item;
        }, this);
      } else if (typeof value === 'number') {
        if (!(0 <= value && value <= 0xff))
          return reporter.error('non-byte EncoderBuffer value');
        this.value = value;
        this.length = 1;
      } else if (typeof value === 'string') {
        this.value = value;
        this.length = Buffer.byteLength(value);
      } else if (Buffer.isBuffer(value)) {
        this.value = value;
        this.length = value.length;
      } else {
        return reporter.error('Unsupported type: ' + typeof value);
      }
    }
    exports.EncoderBuffer = EncoderBuffer;

    EncoderBuffer.prototype.join = function join(out, offset) {
      if (!out)
        out = new Buffer(this.length);
      if (!offset)
        offset = 0;

      if (this.length === 0)
        return out;

      if (Array.isArray(this.value)) {
        this.value.forEach(function (item) {
          item.join(out, offset);
          offset += item.length;
        });
      } else {
        if (typeof this.value === 'number')
          out[offset] = this.value;
        else if (typeof this.value === 'string')
          out.write(this.value, offset);
        else if (Buffer.isBuffer(this.value))
          this.value.copy(out, offset);
        offset += this.length;
      }

      return out;
    };

  }, { "../base": 8, "buffer": 58, "inherits": 569 }], 8: [function (require, module, exports) {
    var base = exports;

    base.Reporter = require('./reporter').Reporter;
    base.DecoderBuffer = require('./buffer').DecoderBuffer;
    base.EncoderBuffer = require('./buffer').EncoderBuffer;
    base.Node = require('./node');

  }, { "./buffer": 7, "./node": 9, "./reporter": 10 }], 9: [function (require, module, exports) {
    var Reporter = require('../base').Reporter;
    var EncoderBuffer = require('../base').EncoderBuffer;
    var DecoderBuffer = require('../base').DecoderBuffer;
    var assert = require('minimalistic-assert');

    // Supported tags
    var tags = [
      'seq', 'seqof', 'set', 'setof', 'objid', 'bool',
      'gentime', 'utctime', 'null_', 'enum', 'int', 'objDesc',
      'bitstr', 'bmpstr', 'charstr', 'genstr', 'graphstr', 'ia5str', 'iso646str',
      'numstr', 'octstr', 'printstr', 't61str', 'unistr', 'utf8str', 'videostr'
    ];

    // Public methods list
    var methods = [
      'key', 'obj', 'use', 'optional', 'explicit', 'implicit', 'def', 'choice',
      'any', 'contains'
    ].concat(tags);

    // Overrided methods list
    var overrided = [
      '_peekTag', '_decodeTag', '_use',
      '_decodeStr', '_decodeObjid', '_decodeTime',
      '_decodeNull', '_decodeInt', '_decodeBool', '_decodeList',

      '_encodeComposite', '_encodeStr', '_encodeObjid', '_encodeTime',
      '_encodeNull', '_encodeInt', '_encodeBool'
    ];

    function Node(enc, parent) {
      var state = {};
      this._baseState = state;

      state.enc = enc;

      state.parent = parent || null;
      state.children = null;

      // State
      state.tag = null;
      state.args = null;
      state.reverseArgs = null;
      state.choice = null;
      state.optional = false;
      state.any = false;
      state.obj = false;
      state.use = null;
      state.useDecoder = null;
      state.key = null;
      state['default'] = null;
      state.explicit = null;
      state.implicit = null;
      state.contains = null;

      // Should create new instance on each method
      if (!state.parent) {
        state.children = [];
        this._wrap();
      }
    }
    module.exports = Node;

    var stateProps = [
      'enc', 'parent', 'children', 'tag', 'args', 'reverseArgs', 'choice',
      'optional', 'any', 'obj', 'use', 'alteredUse', 'key', 'default', 'explicit',
      'implicit', 'contains'
    ];

    Node.prototype.clone = function clone() {
      var state = this._baseState;
      var cstate = {};
      stateProps.forEach(function (prop) {
        cstate[prop] = state[prop];
      });
      var res = new this.constructor(cstate.parent);
      res._baseState = cstate;
      return res;
    };

    Node.prototype._wrap = function wrap() {
      var state = this._baseState;
      methods.forEach(function (method) {
        this[method] = function _wrappedMethod() {
          var clone = new this.constructor(this);
          state.children.push(clone);
          return clone[method].apply(clone, arguments);
        };
      }, this);
    };

    Node.prototype._init = function init(body) {
      var state = this._baseState;

      assert(state.parent === null);
      body.call(this);

      // Filter children
      state.children = state.children.filter(function (child) {
        return child._baseState.parent === this;
      }, this);
      assert.equal(state.children.length, 1, 'Root node can have only one child');
    };

    Node.prototype._useArgs = function useArgs(args) {
      var state = this._baseState;

      // Filter children and args
      var children = args.filter(function (arg) {
        return arg instanceof this.constructor;
      }, this);
      args = args.filter(function (arg) {
        return !(arg instanceof this.constructor);
      }, this);

      if (children.length !== 0) {
        assert(state.children === null);
        state.children = children;

        // Replace parent to maintain backward link
        children.forEach(function (child) {
          child._baseState.parent = this;
        }, this);
      }
      if (args.length !== 0) {
        assert(state.args === null);
        state.args = args;
        state.reverseArgs = args.map(function (arg) {
          if (typeof arg !== 'object' || arg.constructor !== Object)
            return arg;

          var res = {};
          Object.keys(arg).forEach(function (key) {
            if (key == (key | 0))
              key |= 0;
            var value = arg[key];
            res[value] = key;
          });
          return res;
        });
      }
    };

    //
    // Overrided methods
    //

    overrided.forEach(function (method) {
      Node.prototype[method] = function _overrided() {
        var state = this._baseState;
        throw new Error(method + ' not implemented for encoding: ' + state.enc);
      };
    });

    //
    // Public methods
    //

    tags.forEach(function (tag) {
      Node.prototype[tag] = function _tagMethod() {
        var state = this._baseState;
        var args = Array.prototype.slice.call(arguments);

        assert(state.tag === null);
        state.tag = tag;

        this._useArgs(args);

        return this;
      };
    });

    Node.prototype.use = function use(item) {
      assert(item);
      var state = this._baseState;

      assert(state.use === null);
      state.use = item;

      return this;
    };

    Node.prototype.optional = function optional() {
      var state = this._baseState;

      state.optional = true;

      return this;
    };

    Node.prototype.def = function def(val) {
      var state = this._baseState;

      assert(state['default'] === null);
      state['default'] = val;
      state.optional = true;

      return this;
    };

    Node.prototype.explicit = function explicit(num) {
      var state = this._baseState;

      assert(state.explicit === null && state.implicit === null);
      state.explicit = num;

      return this;
    };

    Node.prototype.implicit = function implicit(num) {
      var state = this._baseState;

      assert(state.explicit === null && state.implicit === null);
      state.implicit = num;

      return this;
    };

    Node.prototype.obj = function obj() {
      var state = this._baseState;
      var args = Array.prototype.slice.call(arguments);

      state.obj = true;

      if (args.length !== 0)
        this._useArgs(args);

      return this;
    };

    Node.prototype.key = function key(newKey) {
      var state = this._baseState;

      assert(state.key === null);
      state.key = newKey;

      return this;
    };

    Node.prototype.any = function any() {
      var state = this._baseState;

      state.any = true;

      return this;
    };

    Node.prototype.choice = function choice(obj) {
      var state = this._baseState;

      assert(state.choice === null);
      state.choice = obj;
      this._useArgs(Object.keys(obj).map(function (key) {
        return obj[key];
      }));

      return this;
    };

    Node.prototype.contains = function contains(item) {
      var state = this._baseState;

      assert(state.use === null);
      state.contains = item;

      return this;
    };

    //
    // Decoding
    //

    Node.prototype._decode = function decode(input, options) {
      var state = this._baseState;

      // Decode root node
      if (state.parent === null)
        return input.wrapResult(state.children[0]._decode(input, options));

      var result = state['default'];
      var present = true;

      var prevKey = null;
      if (state.key !== null)
        prevKey = input.enterKey(state.key);

      // Check if tag is there
      if (state.optional) {
        var tag = null;
        if (state.explicit !== null)
          tag = state.explicit;
        else if (state.implicit !== null)
          tag = state.implicit;
        else if (state.tag !== null)
          tag = state.tag;

        if (tag === null && !state.any) {
          // Trial and Error
          var save = input.save();
          try {
            if (state.choice === null)
              this._decodeGeneric(state.tag, input, options);
            else
              this._decodeChoice(input, options);
            present = true;
          } catch (e) {
            present = false;
          }
          input.restore(save);
        } else {
          present = this._peekTag(input, tag, state.any);

          if (input.isError(present))
            return present;
        }
      }

      // Push object on stack
      var prevObj;
      if (state.obj && present)
        prevObj = input.enterObject();

      if (present) {
        // Unwrap explicit values
        if (state.explicit !== null) {
          var explicit = this._decodeTag(input, state.explicit);
          if (input.isError(explicit))
            return explicit;
          input = explicit;
        }

        var start = input.offset;

        // Unwrap implicit and normal values
        if (state.use === null && state.choice === null) {
          if (state.any)
            var save = input.save();
          var body = this._decodeTag(
            input,
            state.implicit !== null ? state.implicit : state.tag,
            state.any
          );
          if (input.isError(body))
            return body;

          if (state.any)
            result = input.raw(save);
          else
            input = body;
        }

        if (options && options.track && state.tag !== null)
          options.track(input.path(), start, input.length, 'tagged');

        if (options && options.track && state.tag !== null)
          options.track(input.path(), input.offset, input.length, 'content');

        // Select proper method for tag
        if (state.any)
          result = result;
        else if (state.choice === null)
          result = this._decodeGeneric(state.tag, input, options);
        else
          result = this._decodeChoice(input, options);

        if (input.isError(result))
          return result;

        // Decode children
        if (!state.any && state.choice === null && state.children !== null) {
          state.children.forEach(function decodeChildren(child) {
            // NOTE: We are ignoring errors here, to let parser continue with other
            // parts of encoded data
            child._decode(input, options);
          });
        }

        // Decode contained/encoded by schema, only in bit or octet strings
        if (state.contains && (state.tag === 'octstr' || state.tag === 'bitstr')) {
          var data = new DecoderBuffer(result);
          result = this._getUse(state.contains, input._reporterState.obj)
            ._decode(data, options);
        }
      }

      // Pop object
      if (state.obj && present)
        result = input.leaveObject(prevObj);

      // Set key
      if (state.key !== null && (result !== null || present === true))
        input.leaveKey(prevKey, state.key, result);
      else if (prevKey !== null)
        input.exitKey(prevKey);

      return result;
    };

    Node.prototype._decodeGeneric = function decodeGeneric(tag, input, options) {
      var state = this._baseState;

      if (tag === 'seq' || tag === 'set')
        return null;
      if (tag === 'seqof' || tag === 'setof')
        return this._decodeList(input, tag, state.args[0], options);
      else if (/str$/.test(tag))
        return this._decodeStr(input, tag, options);
      else if (tag === 'objid' && state.args)
        return this._decodeObjid(input, state.args[0], state.args[1], options);
      else if (tag === 'objid')
        return this._decodeObjid(input, null, null, options);
      else if (tag === 'gentime' || tag === 'utctime')
        return this._decodeTime(input, tag, options);
      else if (tag === 'null_')
        return this._decodeNull(input, options);
      else if (tag === 'bool')
        return this._decodeBool(input, options);
      else if (tag === 'objDesc')
        return this._decodeStr(input, tag, options);
      else if (tag === 'int' || tag === 'enum')
        return this._decodeInt(input, state.args && state.args[0], options);

      if (state.use !== null) {
        return this._getUse(state.use, input._reporterState.obj)
          ._decode(input, options);
      } else {
        return input.error('unknown tag: ' + tag);
      }
    };

    Node.prototype._getUse = function _getUse(entity, obj) {

      var state = this._baseState;
      // Create altered use decoder if implicit is set
      state.useDecoder = this._use(entity, obj);
      assert(state.useDecoder._baseState.parent === null);
      state.useDecoder = state.useDecoder._baseState.children[0];
      if (state.implicit !== state.useDecoder._baseState.implicit) {
        state.useDecoder = state.useDecoder.clone();
        state.useDecoder._baseState.implicit = state.implicit;
      }
      return state.useDecoder;
    };

    Node.prototype._decodeChoice = function decodeChoice(input, options) {
      var state = this._baseState;
      var result = null;
      var match = false;

      Object.keys(state.choice).some(function (key) {
        var save = input.save();
        var node = state.choice[key];
        try {
          var value = node._decode(input, options);
          if (input.isError(value))
            return false;

          result = { type: key, value: value };
          match = true;
        } catch (e) {
          input.restore(save);
          return false;
        }
        return true;
      }, this);

      if (!match)
        return input.error('Choice not matched');

      return result;
    };

    //
    // Encoding
    //

    Node.prototype._createEncoderBuffer = function createEncoderBuffer(data) {
      return new EncoderBuffer(data, this.reporter);
    };

    Node.prototype._encode = function encode(data, reporter, parent) {
      var state = this._baseState;
      if (state['default'] !== null && state['default'] === data)
        return;

      var result = this._encodeValue(data, reporter, parent);
      if (result === undefined)
        return;

      if (this._skipDefault(result, reporter, parent))
        return;

      return result;
    };

    Node.prototype._encodeValue = function encode(data, reporter, parent) {
      var state = this._baseState;

      // Decode root node
      if (state.parent === null)
        return state.children[0]._encode(data, reporter || new Reporter());

      var result = null;

      // Set reporter to share it with a child class
      this.reporter = reporter;

      // Check if data is there
      if (state.optional && data === undefined) {
        if (state['default'] !== null)
          data = state['default']
        else
          return;
      }

      // Encode children first
      var content = null;
      var primitive = false;
      if (state.any) {
        // Anything that was given is translated to buffer
        result = this._createEncoderBuffer(data);
      } else if (state.choice) {
        result = this._encodeChoice(data, reporter);
      } else if (state.contains) {
        content = this._getUse(state.contains, parent)._encode(data, reporter);
        primitive = true;
      } else if (state.children) {
        content = state.children.map(function (child) {
          if (child._baseState.tag === 'null_')
            return child._encode(null, reporter, data);

          if (child._baseState.key === null)
            return reporter.error('Child should have a key');
          var prevKey = reporter.enterKey(child._baseState.key);

          if (typeof data !== 'object')
            return reporter.error('Child expected, but input is not object');

          var res = child._encode(data[child._baseState.key], reporter, data);
          reporter.leaveKey(prevKey);

          return res;
        }, this).filter(function (child) {
          return child;
        });
        content = this._createEncoderBuffer(content);
      } else {
        if (state.tag === 'seqof' || state.tag === 'setof') {
          // TODO(indutny): this should be thrown on DSL level
          if (!(state.args && state.args.length === 1))
            return reporter.error('Too many args for : ' + state.tag);

          if (!Array.isArray(data))
            return reporter.error('seqof/setof, but data is not Array');

          var child = this.clone();
          child._baseState.implicit = null;
          content = this._createEncoderBuffer(data.map(function (item) {
            var state = this._baseState;

            return this._getUse(state.args[0], data)._encode(item, reporter);
          }, child));
        } else if (state.use !== null) {
          result = this._getUse(state.use, parent)._encode(data, reporter);
        } else {
          content = this._encodePrimitive(state.tag, data);
          primitive = true;
        }
      }

      // Encode data itself
      var result;
      if (!state.any && state.choice === null) {
        var tag = state.implicit !== null ? state.implicit : state.tag;
        var cls = state.implicit === null ? 'universal' : 'context';

        if (tag === null) {
          if (state.use === null)
            reporter.error('Tag could be omitted only for .use()');
        } else {
          if (state.use === null)
            result = this._encodeComposite(tag, primitive, cls, content);
        }
      }

      // Wrap in explicit
      if (state.explicit !== null)
        result = this._encodeComposite(state.explicit, false, 'context', result);

      return result;
    };

    Node.prototype._encodeChoice = function encodeChoice(data, reporter) {
      var state = this._baseState;

      var node = state.choice[data.type];
      if (!node) {
        assert(
          false,
          data.type + ' not found in ' +
          JSON.stringify(Object.keys(state.choice)));
      }
      return node._encode(data.value, reporter);
    };

    Node.prototype._encodePrimitive = function encodePrimitive(tag, data) {
      var state = this._baseState;

      if (/str$/.test(tag))
        return this._encodeStr(data, tag);
      else if (tag === 'objid' && state.args)
        return this._encodeObjid(data, state.reverseArgs[0], state.args[1]);
      else if (tag === 'objid')
        return this._encodeObjid(data, null, null);
      else if (tag === 'gentime' || tag === 'utctime')
        return this._encodeTime(data, tag);
      else if (tag === 'null_')
        return this._encodeNull();
      else if (tag === 'int' || tag === 'enum')
        return this._encodeInt(data, state.args && state.reverseArgs[0]);
      else if (tag === 'bool')
        return this._encodeBool(data);
      else if (tag === 'objDesc')
        return this._encodeStr(data, tag);
      else
        throw new Error('Unsupported tag: ' + tag);
    };

    Node.prototype._isNumstr = function isNumstr(str) {
      return /^[0-9 ]*$/.test(str);
    };

    Node.prototype._isPrintstr = function isPrintstr(str) {
      return /^[A-Za-z0-9 '\(\)\+,\-\.\/:=\?]*$/.test(str);
    };

  }, { "../base": 8, "minimalistic-assert": 590 }], 10: [function (require, module, exports) {
    var inherits = require('inherits');

    function Reporter(options) {
      this._reporterState = {
        obj: null,
        path: [],
        options: options || {},
        errors: []
      };
    }
    exports.Reporter = Reporter;

    Reporter.prototype.isError = function isError(obj) {
      return obj instanceof ReporterError;
    };

    Reporter.prototype.save = function save() {
      var state = this._reporterState;

      return { obj: state.obj, pathLen: state.path.length };
    };

    Reporter.prototype.restore = function restore(data) {
      var state = this._reporterState;

      state.obj = data.obj;
      state.path = state.path.slice(0, data.pathLen);
    };

    Reporter.prototype.enterKey = function enterKey(key) {
      return this._reporterState.path.push(key);
    };

    Reporter.prototype.exitKey = function exitKey(index) {
      var state = this._reporterState;

      state.path = state.path.slice(0, index - 1);
    };

    Reporter.prototype.leaveKey = function leaveKey(index, key, value) {
      var state = this._reporterState;

      this.exitKey(index);
      if (state.obj !== null)
        state.obj[key] = value;
    };

    Reporter.prototype.path = function path() {
      return this._reporterState.path.join('/');
    };

    Reporter.prototype.enterObject = function enterObject() {
      var state = this._reporterState;

      var prev = state.obj;
      state.obj = {};
      return prev;
    };

    Reporter.prototype.leaveObject = function leaveObject(prev) {
      var state = this._reporterState;

      var now = state.obj;
      state.obj = prev;
      return now;
    };

    Reporter.prototype.error = function error(msg) {
      var err;
      var state = this._reporterState;

      var inherited = msg instanceof ReporterError;
      if (inherited) {
        err = msg;
      } else {
        err = new ReporterError(state.path.map(function (elem) {
          return '[' + JSON.stringify(elem) + ']';
        }).join(''), msg.message || msg, msg.stack);
      }

      if (!state.options.partial)
        throw err;

      if (!inherited)
        state.errors.push(err);

      return err;
    };

    Reporter.prototype.wrapResult = function wrapResult(result) {
      var state = this._reporterState;
      if (!state.options.partial)
        return result;

      return {
        result: this.isError(result) ? null : result,
        errors: state.errors
      };
    };

    function ReporterError(path, msg) {
      this.path = path;
      this.rethrow(msg);
    };
    inherits(ReporterError, Error);

    ReporterError.prototype.rethrow = function rethrow(msg) {
      this.message = msg + ' at: ' + (this.path || '(shallow)');
      if (Error.captureStackTrace)
        Error.captureStackTrace(this, ReporterError);

      if (!this.stack) {
        try {
          // IE only adds stack when thrown
          throw new Error(this.message);
        } catch (e) {
          this.stack = e.stack;
        }
      }
      return this;
    };

  }, { "inherits": 569 }], 11: [function (require, module, exports) {
    var constants = require('../constants');

    exports.tagClass = {
      0: 'universal',
      1: 'application',
      2: 'context',
      3: 'private'
    };
    exports.tagClassByName = constants._reverse(exports.tagClass);

    exports.tag = {
      0x00: 'end',
      0x01: 'bool',
      0x02: 'int',
      0x03: 'bitstr',
      0x04: 'octstr',
      0x05: 'null_',
      0x06: 'objid',
      0x07: 'objDesc',
      0x08: 'external',
      0x09: 'real',
      0x0a: 'enum',
      0x0b: 'embed',
      0x0c: 'utf8str',
      0x0d: 'relativeOid',
      0x10: 'seq',
      0x11: 'set',
      0x12: 'numstr',
      0x13: 'printstr',
      0x14: 't61str',
      0x15: 'videostr',
      0x16: 'ia5str',
      0x17: 'utctime',
      0x18: 'gentime',
      0x19: 'graphstr',
      0x1a: 'iso646str',
      0x1b: 'genstr',
      0x1c: 'unistr',
      0x1d: 'charstr',
      0x1e: 'bmpstr'
    };
    exports.tagByName = constants._reverse(exports.tag);

  }, { "../constants": 12 }], 12: [function (require, module, exports) {
    var constants = exports;

    // Helper
    constants._reverse = function reverse(map) {
      var res = {};

      Object.keys(map).forEach(function (key) {
        // Convert key to integer if it is stringified
        if ((key | 0) == key)
          key = key | 0;

        var value = map[key];
        res[value] = key;
      });

      return res;
    };

    constants.der = require('./der');

  }, { "./der": 11 }], 13: [function (require, module, exports) {
    var inherits = require('inherits');

    var asn1 = require('../../asn1');
    var base = asn1.base;
    var bignum = asn1.bignum;

    // Import DER constants
    var der = asn1.constants.der;

    function DERDecoder(entity) {
      this.enc = 'der';
      this.name = entity.name;
      this.entity = entity;

      // Construct base tree
      this.tree = new DERNode();
      this.tree._init(entity.body);
    };
    module.exports = DERDecoder;

    DERDecoder.prototype.decode = function decode(data, options) {
      if (!(data instanceof base.DecoderBuffer))
        data = new base.DecoderBuffer(data, options);

      return this.tree._decode(data, options);
    };

    // Tree methods

    function DERNode(parent) {
      base.Node.call(this, 'der', parent);
    }
    inherits(DERNode, base.Node);

    DERNode.prototype._peekTag = function peekTag(buffer, tag, any) {
      if (buffer.isEmpty())
        return false;

      var state = buffer.save();
      var decodedTag = derDecodeTag(buffer, 'Failed to peek tag: "' + tag + '"');
      if (buffer.isError(decodedTag))
        return decodedTag;

      buffer.restore(state);

      return decodedTag.tag === tag || decodedTag.tagStr === tag ||
        (decodedTag.tagStr + 'of') === tag || any;
    };

    DERNode.prototype._decodeTag = function decodeTag(buffer, tag, any) {
      var decodedTag = derDecodeTag(buffer,
        'Failed to decode tag of "' + tag + '"');
      if (buffer.isError(decodedTag))
        return decodedTag;

      var len = derDecodeLen(buffer,
        decodedTag.primitive,
        'Failed to get length of "' + tag + '"');

      // Failure
      if (buffer.isError(len))
        return len;

      if (!any &&
        decodedTag.tag !== tag &&
        decodedTag.tagStr !== tag &&
        decodedTag.tagStr + 'of' !== tag) {
        return buffer.error('Failed to match tag: "' + tag + '"');
      }

      if (decodedTag.primitive || len !== null)
        return buffer.skip(len, 'Failed to match body of: "' + tag + '"');

      // Indefinite length... find END tag
      var state = buffer.save();
      var res = this._skipUntilEnd(
        buffer,
        'Failed to skip indefinite length body: "' + this.tag + '"');
      if (buffer.isError(res))
        return res;

      len = buffer.offset - state.offset;
      buffer.restore(state);
      return buffer.skip(len, 'Failed to match body of: "' + tag + '"');
    };

    DERNode.prototype._skipUntilEnd = function skipUntilEnd(buffer, fail) {
      while (true) {
        var tag = derDecodeTag(buffer, fail);
        if (buffer.isError(tag))
          return tag;
        var len = derDecodeLen(buffer, tag.primitive, fail);
        if (buffer.isError(len))
          return len;

        var res;
        if (tag.primitive || len !== null)
          res = buffer.skip(len)
        else
          res = this._skipUntilEnd(buffer, fail);

        // Failure
        if (buffer.isError(res))
          return res;

        if (tag.tagStr === 'end')
          break;
      }
    };

    DERNode.prototype._decodeList = function decodeList(buffer, tag, decoder,
      options) {
      var result = [];
      while (!buffer.isEmpty()) {
        var possibleEnd = this._peekTag(buffer, 'end');
        if (buffer.isError(possibleEnd))
          return possibleEnd;

        var res = decoder.decode(buffer, 'der', options);
        if (buffer.isError(res) && possibleEnd)
          break;
        result.push(res);
      }
      return result;
    };

    DERNode.prototype._decodeStr = function decodeStr(buffer, tag) {
      if (tag === 'bitstr') {
        var unused = buffer.readUInt8();
        if (buffer.isError(unused))
          return unused;
        return { unused: unused, data: buffer.raw() };
      } else if (tag === 'bmpstr') {
        var raw = buffer.raw();
        if (raw.length % 2 === 1)
          return buffer.error('Decoding of string type: bmpstr length mismatch');

        var str = '';
        for (var i = 0; i < raw.length / 2; i++) {
          str += String.fromCharCode(raw.readUInt16BE(i * 2));
        }
        return str;
      } else if (tag === 'numstr') {
        var numstr = buffer.raw().toString('ascii');
        if (!this._isNumstr(numstr)) {
          return buffer.error('Decoding of string type: ' +
            'numstr unsupported characters');
        }
        return numstr;
      } else if (tag === 'octstr') {
        return buffer.raw();
      } else if (tag === 'objDesc') {
        return buffer.raw();
      } else if (tag === 'printstr') {
        var printstr = buffer.raw().toString('ascii');
        if (!this._isPrintstr(printstr)) {
          return buffer.error('Decoding of string type: ' +
            'printstr unsupported characters');
        }
        return printstr;
      } else if (/str$/.test(tag)) {
        return buffer.raw().toString();
      } else {
        return buffer.error('Decoding of string type: ' + tag + ' unsupported');
      }
    };

    DERNode.prototype._decodeObjid = function decodeObjid(buffer, values, relative) {
      var result;
      var identifiers = [];
      var ident = 0;
      while (!buffer.isEmpty()) {
        var subident = buffer.readUInt8();
        ident <<= 7;
        ident |= subident & 0x7f;
        if ((subident & 0x80) === 0) {
          identifiers.push(ident);
          ident = 0;
        }
      }
      if (subident & 0x80)
        identifiers.push(ident);

      var first = (identifiers[0] / 40) | 0;
      var second = identifiers[0] % 40;

      if (relative)
        result = identifiers;
      else
        result = [first, second].concat(identifiers.slice(1));

      if (values) {
        var tmp = values[result.join(' ')];
        if (tmp === undefined)
          tmp = values[result.join('.')];
        if (tmp !== undefined)
          result = tmp;
      }

      return result;
    };

    DERNode.prototype._decodeTime = function decodeTime(buffer, tag) {
      var str = buffer.raw().toString();
      if (tag === 'gentime') {
        var year = str.slice(0, 4) | 0;
        var mon = str.slice(4, 6) | 0;
        var day = str.slice(6, 8) | 0;
        var hour = str.slice(8, 10) | 0;
        var min = str.slice(10, 12) | 0;
        var sec = str.slice(12, 14) | 0;
      } else if (tag === 'utctime') {
        var year = str.slice(0, 2) | 0;
        var mon = str.slice(2, 4) | 0;
        var day = str.slice(4, 6) | 0;
        var hour = str.slice(6, 8) | 0;
        var min = str.slice(8, 10) | 0;
        var sec = str.slice(10, 12) | 0;
        if (year < 70)
          year = 2000 + year;
        else
          year = 1900 + year;
      } else {
        return buffer.error('Decoding ' + tag + ' time is not supported yet');
      }

      return Date.UTC(year, mon - 1, day, hour, min, sec, 0);
    };

    DERNode.prototype._decodeNull = function decodeNull(buffer) {
      return null;
    };

    DERNode.prototype._decodeBool = function decodeBool(buffer) {
      var res = buffer.readUInt8();
      if (buffer.isError(res))
        return res;
      else
        return res !== 0;
    };

    DERNode.prototype._decodeInt = function decodeInt(buffer, values) {
      // Bigint, return as it is (assume big endian)
      var raw = buffer.raw();
      var res = new bignum(raw);

      if (values)
        res = values[res.toString(10)] || res;

      return res;
    };

    DERNode.prototype._use = function use(entity, obj) {
      if (typeof entity === 'function')
        entity = entity(obj);
      return entity._getDecoder('der').tree;
    };

    // Utility methods

    function derDecodeTag(buf, fail) {
      var tag = buf.readUInt8(fail);
      if (buf.isError(tag))
        return tag;

      var cls = der.tagClass[tag >> 6];
      var primitive = (tag & 0x20) === 0;

      // Multi-octet tag - load
      if ((tag & 0x1f) === 0x1f) {
        var oct = tag;
        tag = 0;
        while ((oct & 0x80) === 0x80) {
          oct = buf.readUInt8(fail);
          if (buf.isError(oct))
            return oct;

          tag <<= 7;
          tag |= oct & 0x7f;
        }
      } else {
        tag &= 0x1f;
      }
      var tagStr = der.tag[tag];

      return {
        cls: cls,
        primitive: primitive,
        tag: tag,
        tagStr: tagStr
      };
    }

    function derDecodeLen(buf, primitive, fail) {
      var len = buf.readUInt8(fail);
      if (buf.isError(len))
        return len;

      // Indefinite form
      if (!primitive && len === 0x80)
        return null;

      // Definite form
      if ((len & 0x80) === 0) {
        // Short form
        return len;
      }

      // Long form
      var num = len & 0x7f;
      if (num > 4)
        return buf.error('length octect is too long');

      len = 0;
      for (var i = 0; i < num; i++) {
        len <<= 8;
        var j = buf.readUInt8(fail);
        if (buf.isError(j))
          return j;
        len |= j;
      }

      return len;
    }

  }, { "../../asn1": 5, "inherits": 569 }], 14: [function (require, module, exports) {
    var decoders = exports;

    decoders.der = require('./der');
    decoders.pem = require('./pem');

  }, { "./der": 13, "./pem": 15 }], 15: [function (require, module, exports) {
    var inherits = require('inherits');
    var Buffer = require('buffer').Buffer;

    var DERDecoder = require('./der');

    function PEMDecoder(entity) {
      DERDecoder.call(this, entity);
      this.enc = 'pem';
    };
    inherits(PEMDecoder, DERDecoder);
    module.exports = PEMDecoder;

    PEMDecoder.prototype.decode = function decode(data, options) {
      var lines = data.toString().split(/[\r\n]+/g);

      var label = options.label.toUpperCase();

      var re = /^-----(BEGIN|END) ([^-]+)-----$/;
      var start = -1;
      var end = -1;
      for (var i = 0; i < lines.length; i++) {
        var match = lines[i].match(re);
        if (match === null)
          continue;

        if (match[2] !== label)
          continue;

        if (start === -1) {
          if (match[1] !== 'BEGIN')
            break;
          start = i;
        } else {
          if (match[1] !== 'END')
            break;
          end = i;
          break;
        }
      }
      if (start === -1 || end === -1)
        throw new Error('PEM section not found for: ' + label);

      var base64 = lines.slice(start + 1, end).join('');
      // Remove excessive symbols
      base64.replace(/[^a-z0-9\+\/=]+/gi, '');

      var input = new Buffer(base64, 'base64');
      return DERDecoder.prototype.decode.call(this, input, options);
    };

  }, { "./der": 13, "buffer": 58, "inherits": 569 }], 16: [function (require, module, exports) {
    var inherits = require('inherits');
    var Buffer = require('buffer').Buffer;

    var asn1 = require('../../asn1');
    var base = asn1.base;

    // Import DER constants
    var der = asn1.constants.der;

    function DEREncoder(entity) {
      this.enc = 'der';
      this.name = entity.name;
      this.entity = entity;

      // Construct base tree
      this.tree = new DERNode();
      this.tree._init(entity.body);
    };
    module.exports = DEREncoder;

    DEREncoder.prototype.encode = function encode(data, reporter) {
      return this.tree._encode(data, reporter).join();
    };

    // Tree methods

    function DERNode(parent) {
      base.Node.call(this, 'der', parent);
    }
    inherits(DERNode, base.Node);

    DERNode.prototype._encodeComposite = function encodeComposite(tag,
      primitive,
      cls,
      content) {
      var encodedTag = encodeTag(tag, primitive, cls, this.reporter);

      // Short form
      if (content.length < 0x80) {
        var header = new Buffer(2);
        header[0] = encodedTag;
        header[1] = content.length;
        return this._createEncoderBuffer([header, content]);
      }

      // Long form
      // Count octets required to store length
      var lenOctets = 1;
      for (var i = content.length; i >= 0x100; i >>= 8)
        lenOctets++;

      var header = new Buffer(1 + 1 + lenOctets);
      header[0] = encodedTag;
      header[1] = 0x80 | lenOctets;

      for (var i = 1 + lenOctets, j = content.length; j > 0; i--, j >>= 8)
        header[i] = j & 0xff;

      return this._createEncoderBuffer([header, content]);
    };

    DERNode.prototype._encodeStr = function encodeStr(str, tag) {
      if (tag === 'bitstr') {
        return this._createEncoderBuffer([str.unused | 0, str.data]);
      } else if (tag === 'bmpstr') {
        var buf = new Buffer(str.length * 2);
        for (var i = 0; i < str.length; i++) {
          buf.writeUInt16BE(str.charCodeAt(i), i * 2);
        }
        return this._createEncoderBuffer(buf);
      } else if (tag === 'numstr') {
        if (!this._isNumstr(str)) {
          return this.reporter.error('Encoding of string type: numstr supports ' +
            'only digits and space');
        }
        return this._createEncoderBuffer(str);
      } else if (tag === 'printstr') {
        if (!this._isPrintstr(str)) {
          return this.reporter.error('Encoding of string type: printstr supports ' +
            'only latin upper and lower case letters, ' +
            'digits, space, apostrophe, left and rigth ' +
            'parenthesis, plus sign, comma, hyphen, ' +
            'dot, slash, colon, equal sign, ' +
            'question mark');
        }
        return this._createEncoderBuffer(str);
      } else if (/str$/.test(tag)) {
        return this._createEncoderBuffer(str);
      } else if (tag === 'objDesc') {
        return this._createEncoderBuffer(str);
      } else {
        return this.reporter.error('Encoding of string type: ' + tag +
          ' unsupported');
      }
    };

    DERNode.prototype._encodeObjid = function encodeObjid(id, values, relative) {
      if (typeof id === 'string') {
        if (!values)
          return this.reporter.error('string objid given, but no values map found');
        if (!values.hasOwnProperty(id))
          return this.reporter.error('objid not found in values map');
        id = values[id].split(/[\s\.]+/g);
        for (var i = 0; i < id.length; i++)
          id[i] |= 0;
      } else if (Array.isArray(id)) {
        id = id.slice();
        for (var i = 0; i < id.length; i++)
          id[i] |= 0;
      }

      if (!Array.isArray(id)) {
        return this.reporter.error('objid() should be either array or string, ' +
          'got: ' + JSON.stringify(id));
      }

      if (!relative) {
        if (id[1] >= 40)
          return this.reporter.error('Second objid identifier OOB');
        id.splice(0, 2, id[0] * 40 + id[1]);
      }

      // Count number of octets
      var size = 0;
      for (var i = 0; i < id.length; i++) {
        var ident = id[i];
        for (size++; ident >= 0x80; ident >>= 7)
          size++;
      }

      var objid = new Buffer(size);
      var offset = objid.length - 1;
      for (var i = id.length - 1; i >= 0; i--) {
        var ident = id[i];
        objid[offset--] = ident & 0x7f;
        while ((ident >>= 7) > 0)
          objid[offset--] = 0x80 | (ident & 0x7f);
      }

      return this._createEncoderBuffer(objid);
    };

    function two(num) {
      if (num < 10)
        return '0' + num;
      else
        return num;
    }

    DERNode.prototype._encodeTime = function encodeTime(time, tag) {
      var str;
      var date = new Date(time);

      if (tag === 'gentime') {
        str = [
          two(date.getFullYear()),
          two(date.getUTCMonth() + 1),
          two(date.getUTCDate()),
          two(date.getUTCHours()),
          two(date.getUTCMinutes()),
          two(date.getUTCSeconds()),
          'Z'
        ].join('');
      } else if (tag === 'utctime') {
        str = [
          two(date.getFullYear() % 100),
          two(date.getUTCMonth() + 1),
          two(date.getUTCDate()),
          two(date.getUTCHours()),
          two(date.getUTCMinutes()),
          two(date.getUTCSeconds()),
          'Z'
        ].join('');
      } else {
        this.reporter.error('Encoding ' + tag + ' time is not supported yet');
      }

      return this._encodeStr(str, 'octstr');
    };

    DERNode.prototype._encodeNull = function encodeNull() {
      return this._createEncoderBuffer('');
    };

    DERNode.prototype._encodeInt = function encodeInt(num, values) {
      if (typeof num === 'string') {
        if (!values)
          return this.reporter.error('String int or enum given, but no values map');
        if (!values.hasOwnProperty(num)) {
          return this.reporter.error('Values map doesn\'t contain: ' +
            JSON.stringify(num));
        }
        num = values[num];
      }

      // Bignum, assume big endian
      if (typeof num !== 'number' && !Buffer.isBuffer(num)) {
        var numArray = num.toArray();
        if (!num.sign && numArray[0] & 0x80) {
          numArray.unshift(0);
        }
        num = new Buffer(numArray);
      }

      if (Buffer.isBuffer(num)) {
        var size = num.length;
        if (num.length === 0)
          size++;

        var out = new Buffer(size);
        num.copy(out);
        if (num.length === 0)
          out[0] = 0
        return this._createEncoderBuffer(out);
      }

      if (num < 0x80)
        return this._createEncoderBuffer(num);

      if (num < 0x100)
        return this._createEncoderBuffer([0, num]);

      var size = 1;
      for (var i = num; i >= 0x100; i >>= 8)
        size++;

      var out = new Array(size);
      for (var i = out.length - 1; i >= 0; i--) {
        out[i] = num & 0xff;
        num >>= 8;
      }
      if (out[0] & 0x80) {
        out.unshift(0);
      }

      return this._createEncoderBuffer(new Buffer(out));
    };

    DERNode.prototype._encodeBool = function encodeBool(value) {
      return this._createEncoderBuffer(value ? 0xff : 0);
    };

    DERNode.prototype._use = function use(entity, obj) {
      if (typeof entity === 'function')
        entity = entity(obj);
      return entity._getEncoder('der').tree;
    };

    DERNode.prototype._skipDefault = function skipDefault(dataBuffer, reporter, parent) {
      var state = this._baseState;
      var i;
      if (state['default'] === null)
        return false;

      var data = dataBuffer.join();
      if (state.defaultBuffer === undefined)
        state.defaultBuffer = this._encodeValue(state['default'], reporter, parent).join();

      if (data.length !== state.defaultBuffer.length)
        return false;

      for (i = 0; i < data.length; i++)
        if (data[i] !== state.defaultBuffer[i])
          return false;

      return true;
    };

    // Utility methods

    function encodeTag(tag, primitive, cls, reporter) {
      var res;

      if (tag === 'seqof')
        tag = 'seq';
      else if (tag === 'setof')
        tag = 'set';

      if (der.tagByName.hasOwnProperty(tag))
        res = der.tagByName[tag];
      else if (typeof tag === 'number' && (tag | 0) === tag)
        res = tag;
      else
        return reporter.error('Unknown tag: ' + tag);

      if (res >= 0x1f)
        return reporter.error('Multi-octet tag encoding unsupported');

      if (!primitive)
        res |= 0x20;

      res |= (der.tagClassByName[cls || 'universal'] << 6);

      return res;
    }

  }, { "../../asn1": 5, "buffer": 58, "inherits": 569 }], 17: [function (require, module, exports) {
    var encoders = exports;

    encoders.der = require('./der');
    encoders.pem = require('./pem');

  }, { "./der": 16, "./pem": 18 }], 18: [function (require, module, exports) {
    var inherits = require('inherits');

    var DEREncoder = require('./der');

    function PEMEncoder(entity) {
      DEREncoder.call(this, entity);
      this.enc = 'pem';
    };
    inherits(PEMEncoder, DEREncoder);
    module.exports = PEMEncoder;

    PEMEncoder.prototype.encode = function encode(data, options) {
      var buf = DEREncoder.prototype.encode.call(this, data);

      var p = buf.toString('base64');
      var out = ['-----BEGIN ' + options.label + '-----'];
      for (var i = 0; i < p.length; i += 64)
        out.push(p.slice(i, i + 64));
      out.push('-----END ' + options.label + '-----');
      return out.join('\n');
    };

  }, { "./der": 16, "inherits": 569 }], 19: [function (require, module, exports) {
    (function (global) {
      'use strict';

      // compare and isBuffer taken from https://github.com/feross/buffer/blob/680e9e5e488f22aac27599a57dc844a6315928dd/index.js
      // original notice:

      /*!
       * The buffer module from node.js, for the browser.
       *
       * @author   Feross Aboukhadijeh <feross@feross.org> <http://feross.org>
       * @license  MIT
       */
      function compare(a, b) {
        if (a === b) {
          return 0;
        }

        var x = a.length;
        var y = b.length;

        for (var i = 0, len = Math.min(x, y); i < len; ++i) {
          if (a[i] !== b[i]) {
            x = a[i];
            y = b[i];
            break;
          }
        }

        if (x < y) {
          return -1;
        }
        if (y < x) {
          return 1;
        }
        return 0;
      }
      function isBuffer(b) {
        if (global.Buffer && typeof global.Buffer.isBuffer === 'function') {
          return global.Buffer.isBuffer(b);
        }
        return !!(b != null && b._isBuffer);
      }

      // based on node assert, original notice:

      // http://wiki.commonjs.org/wiki/Unit_Testing/1.0
      //
      // THIS IS NOT TESTED NOR LIKELY TO WORK OUTSIDE V8!
      //
      // Originally from narwhal.js (http://narwhaljs.org)
      // Copyright (c) 2009 Thomas Robinson <280north.com>
      //
      // Permission is hereby granted, free of charge, to any person obtaining a copy
      // of this software and associated documentation files (the 'Software'), to
      // deal in the Software without restriction, including without limitation the
      // rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
      // sell copies of the Software, and to permit persons to whom the Software is
      // furnished to do so, subject to the following conditions:
      //
      // The above copyright notice and this permission notice shall be included in
      // all copies or substantial portions of the Software.
      //
      // THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
      // IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
      // FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
      // AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
      // ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
      // WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

      var util = require('util/');
      var hasOwn = Object.prototype.hasOwnProperty;
      var pSlice = Array.prototype.slice;
      var functionsHaveNames = (function () {
        return function foo() { }.name === 'foo';
      }());
      function pToString(obj) {
        return Object.prototype.toString.call(obj);
      }
      function isView(arrbuf) {
        if (isBuffer(arrbuf)) {
          return false;
        }
        if (typeof global.ArrayBuffer !== 'function') {
          return false;
        }
        if (typeof ArrayBuffer.isView === 'function') {
          return ArrayBuffer.isView(arrbuf);
        }
        if (!arrbuf) {
          return false;
        }
        if (arrbuf instanceof DataView) {
          return true;
        }
        if (arrbuf.buffer && arrbuf.buffer instanceof ArrayBuffer) {
          return true;
        }
        return false;
      }
      // 1. The assert module provides functions that throw
      // AssertionError's when particular conditions are not met. The
      // assert module must conform to the following interface.

      var assert = module.exports = ok;

      // 2. The AssertionError is defined in assert.
      // new assert.AssertionError({ message: message,
      //                             actual: actual,
      //                             expected: expected })

      var regex = /\s*function\s+([^\(\s]*)\s*/;
      // based on https://github.com/ljharb/function.prototype.name/blob/adeeeec8bfcc6068b187d7d9fb3d5bb1d3a30899/implementation.js
      function getName(func) {
        if (!util.isFunction(func)) {
          return;
        }
        if (functionsHaveNames) {
          return func.name;
        }
        var str = func.toString();
        var match = str.match(regex);
        return match && match[1];
      }
      assert.AssertionError = function AssertionError(options) {
        this.name = 'AssertionError';
        this.actual = options.actual;
        this.expected = options.expected;
        this.operator = options.operator;
        if (options.message) {
          this.message = options.message;
          this.generatedMessage = false;
        } else {
          this.message = getMessage(this);
          this.generatedMessage = true;
        }
        var stackStartFunction = options.stackStartFunction || fail;
        if (Error.captureStackTrace) {
          Error.captureStackTrace(this, stackStartFunction);
        } else {
          // non v8 browsers so we can have a stacktrace
          var err = new Error();
          if (err.stack) {
            var out = err.stack;

            // try to strip useless frames
            var fn_name = getName(stackStartFunction);
            var idx = out.indexOf('\n' + fn_name);
            if (idx >= 0) {
              // once we have located the function frame
              // we need to strip out everything before it (and its line)
              var next_line = out.indexOf('\n', idx + 1);
              out = out.substring(next_line + 1);
            }

            this.stack = out;
          }
        }
      };

      // assert.AssertionError instanceof Error
      util.inherits(assert.AssertionError, Error);

      function truncate(s, n) {
        if (typeof s === 'string') {
          return s.length < n ? s : s.slice(0, n);
        } else {
          return s;
        }
      }
      function inspect(something) {
        if (functionsHaveNames || !util.isFunction(something)) {
          return util.inspect(something);
        }
        var rawname = getName(something);
        var name = rawname ? ': ' + rawname : '';
        return '[Function' + name + ']';
      }
      function getMessage(self) {
        return truncate(inspect(self.actual), 128) + ' ' +
          self.operator + ' ' +
          truncate(inspect(self.expected), 128);
      }

      // At present only the three keys mentioned above are used and
      // understood by the spec. Implementations or sub modules can pass
      // other keys to the AssertionError's constructor - they will be
      // ignored.

      // 3. All of the following functions must throw an AssertionError
      // when a corresponding condition is not met, with a message that
      // may be undefined if not provided.  All assertion methods provide
      // both the actual and expected values to the assertion error for
      // display purposes.

      function fail(actual, expected, message, operator, stackStartFunction) {
        throw new assert.AssertionError({
          message: message,
          actual: actual,
          expected: expected,
          operator: operator,
          stackStartFunction: stackStartFunction
        });
      }

      // EXTENSION! allows for well behaved errors defined elsewhere.
      assert.fail = fail;

      // 4. Pure assertion tests whether a value is truthy, as determined
      // by !!guard.
      // assert.ok(guard, message_opt);
      // This statement is equivalent to assert.equal(true, !!guard,
      // message_opt);. To test strictly for the value true, use
      // assert.strictEqual(true, guard, message_opt);.

      function ok(value, message) {
        if (!value) fail(value, true, message, '==', assert.ok);
      }
      assert.ok = ok;

      // 5. The equality assertion tests shallow, coercive equality with
      // ==.
      // assert.equal(actual, expected, message_opt);

      assert.equal = function equal(actual, expected, message) {
        if (actual != expected) fail(actual, expected, message, '==', assert.equal);
      };

      // 6. The non-equality assertion tests for whether two objects are not equal
      // with != assert.notEqual(actual, expected, message_opt);

      assert.notEqual = function notEqual(actual, expected, message) {
        if (actual == expected) {
          fail(actual, expected, message, '!=', assert.notEqual);
        }
      };

      // 7. The equivalence assertion tests a deep equality relation.
      // assert.deepEqual(actual, expected, message_opt);

      assert.deepEqual = function deepEqual(actual, expected, message) {
        if (!_deepEqual(actual, expected, false)) {
          fail(actual, expected, message, 'deepEqual', assert.deepEqual);
        }
      };

      assert.deepStrictEqual = function deepStrictEqual(actual, expected, message) {
        if (!_deepEqual(actual, expected, true)) {
          fail(actual, expected, message, 'deepStrictEqual', assert.deepStrictEqual);
        }
      };

      function _deepEqual(actual, expected, strict, memos) {
        // 7.1. All identical values are equivalent, as determined by ===.
        if (actual === expected) {
          return true;
        } else if (isBuffer(actual) && isBuffer(expected)) {
          return compare(actual, expected) === 0;

          // 7.2. If the expected value is a Date object, the actual value is
          // equivalent if it is also a Date object that refers to the same time.
        } else if (util.isDate(actual) && util.isDate(expected)) {
          return actual.getTime() === expected.getTime();

          // 7.3 If the expected value is a RegExp object, the actual value is
          // equivalent if it is also a RegExp object with the same source and
          // properties (`global`, `multiline`, `lastIndex`, `ignoreCase`).
        } else if (util.isRegExp(actual) && util.isRegExp(expected)) {
          return actual.source === expected.source &&
            actual.global === expected.global &&
            actual.multiline === expected.multiline &&
            actual.lastIndex === expected.lastIndex &&
            actual.ignoreCase === expected.ignoreCase;

          // 7.4. Other pairs that do not both pass typeof value == 'object',
          // equivalence is determined by ==.
        } else if ((actual === null || typeof actual !== 'object') &&
          (expected === null || typeof expected !== 'object')) {
          return strict ? actual === expected : actual == expected;

          // If both values are instances of typed arrays, wrap their underlying
          // ArrayBuffers in a Buffer each to increase performance
          // This optimization requires the arrays to have the same type as checked by
          // Object.prototype.toString (aka pToString). Never perform binary
          // comparisons for Float*Arrays, though, since e.g. +0 === -0 but their
          // bit patterns are not identical.
        } else if (isView(actual) && isView(expected) &&
          pToString(actual) === pToString(expected) &&
          !(actual instanceof Float32Array ||
            actual instanceof Float64Array)) {
          return compare(new Uint8Array(actual.buffer),
            new Uint8Array(expected.buffer)) === 0;

          // 7.5 For all other Object pairs, including Array objects, equivalence is
          // determined by having the same number of owned properties (as verified
          // with Object.prototype.hasOwnProperty.call), the same set of keys
          // (although not necessarily the same order), equivalent values for every
          // corresponding key, and an identical 'prototype' property. Note: this
          // accounts for both named and indexed properties on Arrays.
        } else if (isBuffer(actual) !== isBuffer(expected)) {
          return false;
        } else {
          memos = memos || { actual: [], expected: [] };

          var actualIndex = memos.actual.indexOf(actual);
          if (actualIndex !== -1) {
            if (actualIndex === memos.expected.indexOf(expected)) {
              return true;
            }
          }

          memos.actual.push(actual);
          memos.expected.push(expected);

          return objEquiv(actual, expected, strict, memos);
        }
      }

      function isArguments(object) {
        return Object.prototype.toString.call(object) == '[object Arguments]';
      }

      function objEquiv(a, b, strict, actualVisitedObjects) {
        if (a === null || a === undefined || b === null || b === undefined)
          return false;
        // if one is a primitive, the other must be same
        if (util.isPrimitive(a) || util.isPrimitive(b))
          return a === b;
        if (strict && Object.getPrototypeOf(a) !== Object.getPrototypeOf(b))
          return false;
        var aIsArgs = isArguments(a);
        var bIsArgs = isArguments(b);
        if ((aIsArgs && !bIsArgs) || (!aIsArgs && bIsArgs))
          return false;
        if (aIsArgs) {
          a = pSlice.call(a);
          b = pSlice.call(b);
          return _deepEqual(a, b, strict);
        }
        var ka = objectKeys(a);
        var kb = objectKeys(b);
        var key, i;
        // having the same number of owned properties (keys incorporates
        // hasOwnProperty)
        if (ka.length !== kb.length)
          return false;
        //the same set of keys (although not necessarily the same order),
        ka.sort();
        kb.sort();
        //~~~cheap key test
        for (i = ka.length - 1; i >= 0; i--) {
          if (ka[i] !== kb[i])
            return false;
        }
        //equivalent values for every corresponding key, and
        //~~~possibly expensive deep test
        for (i = ka.length - 1; i >= 0; i--) {
          key = ka[i];
          if (!_deepEqual(a[key], b[key], strict, actualVisitedObjects))
            return false;
        }
        return true;
      }

      // 8. The non-equivalence assertion tests for any deep inequality.
      // assert.notDeepEqual(actual, expected, message_opt);

      assert.notDeepEqual = function notDeepEqual(actual, expected, message) {
        if (_deepEqual(actual, expected, false)) {
          fail(actual, expected, message, 'notDeepEqual', assert.notDeepEqual);
        }
      };

      assert.notDeepStrictEqual = notDeepStrictEqual;
      function notDeepStrictEqual(actual, expected, message) {
        if (_deepEqual(actual, expected, true)) {
          fail(actual, expected, message, 'notDeepStrictEqual', notDeepStrictEqual);
        }
      }


      // 9. The strict equality assertion tests strict equality, as determined by ===.
      // assert.strictEqual(actual, expected, message_opt);

      assert.strictEqual = function strictEqual(actual, expected, message) {
        if (actual !== expected) {
          fail(actual, expected, message, '===', assert.strictEqual);
        }
      };

      // 10. The strict non-equality assertion tests for strict inequality, as
      // determined by !==.  assert.notStrictEqual(actual, expected, message_opt);

      assert.notStrictEqual = function notStrictEqual(actual, expected, message) {
        if (actual === expected) {
          fail(actual, expected, message, '!==', assert.notStrictEqual);
        }
      };

      function expectedException(actual, expected) {
        if (!actual || !expected) {
          return false;
        }

        if (Object.prototype.toString.call(expected) == '[object RegExp]') {
          return expected.test(actual);
        }

        try {
          if (actual instanceof expected) {
            return true;
          }
        } catch (e) {
          // Ignore.  The instanceof check doesn't work for arrow functions.
        }

        if (Error.isPrototypeOf(expected)) {
          return false;
        }

        return expected.call({}, actual) === true;
      }

      function _tryBlock(block) {
        var error;
        try {
          block();
        } catch (e) {
          error = e;
        }
        return error;
      }

      function _throws(shouldThrow, block, expected, message) {
        var actual;

        if (typeof block !== 'function') {
          throw new TypeError('"block" argument must be a function');
        }

        if (typeof expected === 'string') {
          message = expected;
          expected = null;
        }

        actual = _tryBlock(block);

        message = (expected && expected.name ? ' (' + expected.name + ').' : '.') +
          (message ? ' ' + message : '.');

        if (shouldThrow && !actual) {
          fail(actual, expected, 'Missing expected exception' + message);
        }

        var userProvidedMessage = typeof message === 'string';
        var isUnwantedException = !shouldThrow && util.isError(actual);
        var isUnexpectedException = !shouldThrow && actual && !expected;

        if ((isUnwantedException &&
          userProvidedMessage &&
          expectedException(actual, expected)) ||
          isUnexpectedException) {
          fail(actual, expected, 'Got unwanted exception' + message);
        }

        if ((shouldThrow && actual && expected &&
          !expectedException(actual, expected)) || (!shouldThrow && actual)) {
          throw actual;
        }
      }

      // 11. Expected to throw an error:
      // assert.throws(block, Error_opt, message_opt);

      assert.throws = function (block, /*optional*/error, /*optional*/message) {
        _throws(true, block, error, message);
      };

      // EXTENSION! This is annoying to write outside this module.
      assert.doesNotThrow = function (block, /*optional*/error, /*optional*/message) {
        _throws(false, block, error, message);
      };

      assert.ifError = function (err) { if (err) throw err; };

      var objectKeys = Object.keys || function (obj) {
        var keys = [];
        for (var key in obj) {
          if (hasOwn.call(obj, key)) keys.push(key);
        }
        return keys;
      };

    }).call(this, typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {})
  }, { "util/": 22 }], 20: [function (require, module, exports) {
    if (typeof Object.create === 'function') {
      // implementation from standard node.js 'util' module
      module.exports = function inherits(ctor, superCtor) {
        ctor.super_ = superCtor
        ctor.prototype = Object.create(superCtor.prototype, {
          constructor: {
            value: ctor,
            enumerable: false,
            writable: true,
            configurable: true
          }
        });
      };
    } else {
      // old school shim for old browsers
      module.exports = function inherits(ctor, superCtor) {
        ctor.super_ = superCtor
        var TempCtor = function () { }
        TempCtor.prototype = superCtor.prototype
        ctor.prototype = new TempCtor()
        ctor.prototype.constructor = ctor
      }
    }

  }, {}], 21: [function (require, module, exports) {
    module.exports = function isBuffer(arg) {
      return arg && typeof arg === 'object'
        && typeof arg.copy === 'function'
        && typeof arg.fill === 'function'
        && typeof arg.readUInt8 === 'function';
    }
  }, {}], 22: [function (require, module, exports) {
    (function (process, global) {
      // Copyright Joyent, Inc. and other Node contributors.
      //
      // Permission is hereby granted, free of charge, to any person obtaining a
      // copy of this software and associated documentation files (the
      // "Software"), to deal in the Software without restriction, including
      // without limitation the rights to use, copy, modify, merge, publish,
      // distribute, sublicense, and/or sell copies of the Software, and to permit
      // persons to whom the Software is furnished to do so, subject to the
      // following conditions:
      //
      // The above copyright notice and this permission notice shall be included
      // in all copies or substantial portions of the Software.
      //
      // THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
      // OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
      // MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
      // NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
      // DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
      // OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
      // USE OR OTHER DEALINGS IN THE SOFTWARE.

      var formatRegExp = /%[sdj%]/g;
      exports.format = function (f) {
        if (!isString(f)) {
          var objects = [];
          for (var i = 0; i < arguments.length; i++) {
            objects.push(inspect(arguments[i]));
          }
          return objects.join(' ');
        }

        var i = 1;
        var args = arguments;
        var len = args.length;
        var str = String(f).replace(formatRegExp, function (x) {
          if (x === '%%') return '%';
          if (i >= len) return x;
          switch (x) {
            case '%s': return String(args[i++]);
            case '%d': return Number(args[i++]);
            case '%j':
              try {
                return JSON.stringify(args[i++]);
              } catch (_) {
                return '[Circular]';
              }
            default:
              return x;
          }
        });
        for (var x = args[i]; i < len; x = args[++i]) {
          if (isNull(x) || !isObject(x)) {
            str += ' ' + x;
          } else {
            str += ' ' + inspect(x);
          }
        }
        return str;
      };


      // Mark that a method should not be used.
      // Returns a modified function which warns once by default.
      // If --no-deprecation is set, then it is a no-op.
      exports.deprecate = function (fn, msg) {
        // Allow for deprecating things in the process of starting up.
        if (isUndefined(global.process)) {
          return function () {
            return exports.deprecate(fn, msg).apply(this, arguments);
          };
        }

        if (process.noDeprecation === true) {
          return fn;
        }

        var warned = false;
        function deprecated() {
          if (!warned) {
            if (process.throwDeprecation) {
              throw new Error(msg);
            } else if (process.traceDeprecation) {
              console.trace(msg);
            } else {
              console.error(msg);
            }
            warned = true;
          }
          return fn.apply(this, arguments);
        }

        return deprecated;
      };


      var debugs = {};
      var debugEnviron;
      exports.debuglog = function (set) {
        if (isUndefined(debugEnviron))
          debugEnviron = process.env.NODE_DEBUG || '';
        set = set.toUpperCase();
        if (!debugs[set]) {
          if (new RegExp('\\b' + set + '\\b', 'i').test(debugEnviron)) {
            var pid = process.pid;
            debugs[set] = function () {
              var msg = exports.format.apply(exports, arguments);
              console.error('%s %d: %s', set, pid, msg);
            };
          } else {
            debugs[set] = function () { };
          }
        }
        return debugs[set];
      };


      /**
       * Echos the value of a value. Trys to print the value out
       * in the best way possible given the different types.
       *
       * @param {Object} obj The object to print out.
       * @param {Object} opts Optional options object that alters the output.
       */
      /* legacy: obj, showHidden, depth, colors*/
      function inspect(obj, opts) {
        // default options
        var ctx = {
          seen: [],
          stylize: stylizeNoColor
        };
        // legacy...
        if (arguments.length >= 3) ctx.depth = arguments[2];
        if (arguments.length >= 4) ctx.colors = arguments[3];
        if (isBoolean(opts)) {
          // legacy...
          ctx.showHidden = opts;
        } else if (opts) {
          // got an "options" object
          exports._extend(ctx, opts);
        }
        // set default options
        if (isUndefined(ctx.showHidden)) ctx.showHidden = false;
        if (isUndefined(ctx.depth)) ctx.depth = 2;
        if (isUndefined(ctx.colors)) ctx.colors = false;
        if (isUndefined(ctx.customInspect)) ctx.customInspect = true;
        if (ctx.colors) ctx.stylize = stylizeWithColor;
        return formatValue(ctx, obj, ctx.depth);
      }
      exports.inspect = inspect;


      // http://en.wikipedia.org/wiki/ANSI_escape_code#graphics
      inspect.colors = {
        'bold': [1, 22],
        'italic': [3, 23],
        'underline': [4, 24],
        'inverse': [7, 27],
        'white': [37, 39],
        'grey': [90, 39],
        'black': [30, 39],
        'blue': [34, 39],
        'cyan': [36, 39],
        'green': [32, 39],
        'magenta': [35, 39],
        'red': [31, 39],
        'yellow': [33, 39]
      };

      // Don't use 'blue' not visible on cmd.exe
      inspect.styles = {
        'special': 'cyan',
        'number': 'yellow',
        'boolean': 'yellow',
        'undefined': 'grey',
        'null': 'bold',
        'string': 'green',
        'date': 'magenta',
        // "name": intentionally not styling
        'regexp': 'red'
      };


      function stylizeWithColor(str, styleType) {
        var style = inspect.styles[styleType];

        if (style) {
          return '\u001b[' + inspect.colors[style][0] + 'm' + str +
            '\u001b[' + inspect.colors[style][1] + 'm';
        } else {
          return str;
        }
      }


      function stylizeNoColor(str, styleType) {
        return str;
      }


      function arrayToHash(array) {
        var hash = {};

        array.forEach(function (val, idx) {
          hash[val] = true;
        });

        return hash;
      }


      function formatValue(ctx, value, recurseTimes) {
        // Provide a hook for user-specified inspect functions.
        // Check that value is an object with an inspect function on it
        if (ctx.customInspect &&
          value &&
          isFunction(value.inspect) &&
          // Filter out the util module, it's inspect function is special
          value.inspect !== exports.inspect &&
          // Also filter out any prototype objects using the circular check.
          !(value.constructor && value.constructor.prototype === value)) {
          var ret = value.inspect(recurseTimes, ctx);
          if (!isString(ret)) {
            ret = formatValue(ctx, ret, recurseTimes);
          }
          return ret;
        }

        // Primitive types cannot have properties
        var primitive = formatPrimitive(ctx, value);
        if (primitive) {
          return primitive;
        }

        // Look up the keys of the object.
        var keys = Object.keys(value);
        var visibleKeys = arrayToHash(keys);

        if (ctx.showHidden) {
          keys = Object.getOwnPropertyNames(value);
        }

        // IE doesn't make error fields non-enumerable
        // http://msdn.microsoft.com/en-us/library/ie/dww52sbt(v=vs.94).aspx
        if (isError(value)
          && (keys.indexOf('message') >= 0 || keys.indexOf('description') >= 0)) {
          return formatError(value);
        }

        // Some type of object without properties can be shortcutted.
        if (keys.length === 0) {
          if (isFunction(value)) {
            var name = value.name ? ': ' + value.name : '';
            return ctx.stylize('[Function' + name + ']', 'special');
          }
          if (isRegExp(value)) {
            return ctx.stylize(RegExp.prototype.toString.call(value), 'regexp');
          }
          if (isDate(value)) {
            return ctx.stylize(Date.prototype.toString.call(value), 'date');
          }
          if (isError(value)) {
            return formatError(value);
          }
        }

        var base = '', array = false, braces = ['{', '}'];

        // Make Array say that they are Array
        if (isArray(value)) {
          array = true;
          braces = ['[', ']'];
        }

        // Make functions say that they are functions
        if (isFunction(value)) {
          var n = value.name ? ': ' + value.name : '';
          base = ' [Function' + n + ']';
        }

        // Make RegExps say that they are RegExps
        if (isRegExp(value)) {
          base = ' ' + RegExp.prototype.toString.call(value);
        }

        // Make dates with properties first say the date
        if (isDate(value)) {
          base = ' ' + Date.prototype.toUTCString.call(value);
        }

        // Make error with message first say the error
        if (isError(value)) {
          base = ' ' + formatError(value);
        }

        if (keys.length === 0 && (!array || value.length == 0)) {
          return braces[0] + base + braces[1];
        }

        if (recurseTimes < 0) {
          if (isRegExp(value)) {
            return ctx.stylize(RegExp.prototype.toString.call(value), 'regexp');
          } else {
            return ctx.stylize('[Object]', 'special');
          }
        }

        ctx.seen.push(value);

        var output;
        if (array) {
          output = formatArray(ctx, value, recurseTimes, visibleKeys, keys);
        } else {
          output = keys.map(function (key) {
            return formatProperty(ctx, value, recurseTimes, visibleKeys, key, array);
          });
        }

        ctx.seen.pop();

        return reduceToSingleString(output, base, braces);
      }


      function formatPrimitive(ctx, value) {
        if (isUndefined(value))
          return ctx.stylize('undefined', 'undefined');
        if (isString(value)) {
          var simple = '\'' + JSON.stringify(value).replace(/^"|"$/g, '')
            .replace(/'/g, "\\'")
            .replace(/\\"/g, '"') + '\'';
          return ctx.stylize(simple, 'string');
        }
        if (isNumber(value))
          return ctx.stylize('' + value, 'number');
        if (isBoolean(value))
          return ctx.stylize('' + value, 'boolean');
        // For some reason typeof null is "object", so special case here.
        if (isNull(value))
          return ctx.stylize('null', 'null');
      }


      function formatError(value) {
        return '[' + Error.prototype.toString.call(value) + ']';
      }


      function formatArray(ctx, value, recurseTimes, visibleKeys, keys) {
        var output = [];
        for (var i = 0, l = value.length; i < l; ++i) {
          if (hasOwnProperty(value, String(i))) {
            output.push(formatProperty(ctx, value, recurseTimes, visibleKeys,
              String(i), true));
          } else {
            output.push('');
          }
        }
        keys.forEach(function (key) {
          if (!key.match(/^\d+$/)) {
            output.push(formatProperty(ctx, value, recurseTimes, visibleKeys,
              key, true));
          }
        });
        return output;
      }


      function formatProperty(ctx, value, recurseTimes, visibleKeys, key, array) {
        var name, str, desc;
        desc = Object.getOwnPropertyDescriptor(value, key) || { value: value[key] };
        if (desc.get) {
          if (desc.set) {
            str = ctx.stylize('[Getter/Setter]', 'special');
          } else {
            str = ctx.stylize('[Getter]', 'special');
          }
        } else {
          if (desc.set) {
            str = ctx.stylize('[Setter]', 'special');
          }
        }
        if (!hasOwnProperty(visibleKeys, key)) {
          name = '[' + key + ']';
        }
        if (!str) {
          if (ctx.seen.indexOf(desc.value) < 0) {
            if (isNull(recurseTimes)) {
              str = formatValue(ctx, desc.value, null);
            } else {
              str = formatValue(ctx, desc.value, recurseTimes - 1);
            }
            if (str.indexOf('\n') > -1) {
              if (array) {
                str = str.split('\n').map(function (line) {
                  return '  ' + line;
                }).join('\n').substr(2);
              } else {
                str = '\n' + str.split('\n').map(function (line) {
                  return '   ' + line;
                }).join('\n');
              }
            }
          } else {
            str = ctx.stylize('[Circular]', 'special');
          }
        }
        if (isUndefined(name)) {
          if (array && key.match(/^\d+$/)) {
            return str;
          }
          name = JSON.stringify('' + key);
          if (name.match(/^"([a-zA-Z_][a-zA-Z_0-9]*)"$/)) {
            name = name.substr(1, name.length - 2);
            name = ctx.stylize(name, 'name');
          } else {
            name = name.replace(/'/g, "\\'")
              .replace(/\\"/g, '"')
              .replace(/(^"|"$)/g, "'");
            name = ctx.stylize(name, 'string');
          }
        }

        return name + ': ' + str;
      }


      function reduceToSingleString(output, base, braces) {
        var numLinesEst = 0;
        var length = output.reduce(function (prev, cur) {
          numLinesEst++;
          if (cur.indexOf('\n') >= 0) numLinesEst++;
          return prev + cur.replace(/\u001b\[\d\d?m/g, '').length + 1;
        }, 0);

        if (length > 60) {
          return braces[0] +
            (base === '' ? '' : base + '\n ') +
            ' ' +
            output.join(',\n  ') +
            ' ' +
            braces[1];
        }

        return braces[0] + base + ' ' + output.join(', ') + ' ' + braces[1];
      }


      // NOTE: These type checking functions intentionally don't use `instanceof`
      // because it is fragile and can be easily faked with `Object.create()`.
      function isArray(ar) {
        return Array.isArray(ar);
      }
      exports.isArray = isArray;

      function isBoolean(arg) {
        return typeof arg === 'boolean';
      }
      exports.isBoolean = isBoolean;

      function isNull(arg) {
        return arg === null;
      }
      exports.isNull = isNull;

      function isNullOrUndefined(arg) {
        return arg == null;
      }
      exports.isNullOrUndefined = isNullOrUndefined;

      function isNumber(arg) {
        return typeof arg === 'number';
      }
      exports.isNumber = isNumber;

      function isString(arg) {
        return typeof arg === 'string';
      }
      exports.isString = isString;

      function isSymbol(arg) {
        return typeof arg === 'symbol';
      }
      exports.isSymbol = isSymbol;

      function isUndefined(arg) {
        return arg === void 0;
      }
      exports.isUndefined = isUndefined;

      function isRegExp(re) {
        return isObject(re) && objectToString(re) === '[object RegExp]';
      }
      exports.isRegExp = isRegExp;

      function isObject(arg) {
        return typeof arg === 'object' && arg !== null;
      }
      exports.isObject = isObject;

      function isDate(d) {
        return isObject(d) && objectToString(d) === '[object Date]';
      }
      exports.isDate = isDate;

      function isError(e) {
        return isObject(e) &&
          (objectToString(e) === '[object Error]' || e instanceof Error);
      }
      exports.isError = isError;

      function isFunction(arg) {
        return typeof arg === 'function';
      }
      exports.isFunction = isFunction;

      function isPrimitive(arg) {
        return arg === null ||
          typeof arg === 'boolean' ||
          typeof arg === 'number' ||
          typeof arg === 'string' ||
          typeof arg === 'symbol' ||  // ES6 symbol
          typeof arg === 'undefined';
      }
      exports.isPrimitive = isPrimitive;

      exports.isBuffer = require('./support/isBuffer');

      function objectToString(o) {
        return Object.prototype.toString.call(o);
      }


      function pad(n) {
        return n < 10 ? '0' + n.toString(10) : n.toString(10);
      }


      var months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep',
        'Oct', 'Nov', 'Dec'];

      // 26 Feb 16:19:34
      function timestamp() {
        var d = new Date();
        var time = [pad(d.getHours()),
        pad(d.getMinutes()),
        pad(d.getSeconds())].join(':');
        return [d.getDate(), months[d.getMonth()], time].join(' ');
      }


      // log is just a thin wrapper to console.log that prepends a timestamp
      exports.log = function () {
        console.log('%s - %s', timestamp(), exports.format.apply(exports, arguments));
      };


      /**
       * Inherit the prototype methods from one constructor into another.
       *
       * The Function.prototype.inherits from lang.js rewritten as a standalone
       * function (not on Function.prototype). NOTE: If this file is to be loaded
       * during bootstrapping this function needs to be rewritten using some native
       * functions as prototype setup using normal JavaScript does not work as
       * expected during bootstrapping (see mirror.js in r114903).
       *
       * @param {function} ctor Constructor function which needs to inherit the
       *     prototype.
       * @param {function} superCtor Constructor function to inherit prototype from.
       */
      exports.inherits = require('inherits');

      exports._extend = function (origin, add) {
        // Don't do anything if add isn't an object
        if (!add || !isObject(add)) return origin;

        var keys = Object.keys(add);
        var i = keys.length;
        while (i--) {
          origin[keys[i]] = add[keys[i]];
        }
        return origin;
      };

      function hasOwnProperty(obj, prop) {
        return Object.prototype.hasOwnProperty.call(obj, prop);
      }

    }).call(this, require('_process'), typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {})
  }, { "./support/isBuffer": 21, "_process": 614, "inherits": 20 }], 23: [function (require, module, exports) {
    'use strict'

    exports.byteLength = byteLength
    exports.toByteArray = toByteArray
    exports.fromByteArray = fromByteArray

    var lookup = []
    var revLookup = []
    var Arr = typeof Uint8Array !== 'undefined' ? Uint8Array : Array

    var code = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/'
    for (var i = 0, len = code.length; i < len; ++i) {
      lookup[i] = code[i]
      revLookup[code.charCodeAt(i)] = i
    }

    // Support decoding URL-safe base64 strings, as Node.js does.
    // See: https://en.wikipedia.org/wiki/Base64#URL_applications
    revLookup['-'.charCodeAt(0)] = 62
    revLookup['_'.charCodeAt(0)] = 63

    function getLens(b64) {
      var len = b64.length

      if (len % 4 > 0) {
        throw new Error('Invalid string. Length must be a multiple of 4')
      }

      // Trim off extra bytes after placeholder bytes are found
      // See: https://github.com/beatgammit/base64-js/issues/42
      var validLen = b64.indexOf('=')
      if (validLen === -1) validLen = len

      var placeHoldersLen = validLen === len
        ? 0
        : 4 - (validLen % 4)

      return [validLen, placeHoldersLen]
    }

    // base64 is 4/3 + up to two characters of the original data
    function byteLength(b64) {
      var lens = getLens(b64)
      var validLen = lens[0]
      var placeHoldersLen = lens[1]
      return ((validLen + placeHoldersLen) * 3 / 4) - placeHoldersLen
    }

    function _byteLength(b64, validLen, placeHoldersLen) {
      return ((validLen + placeHoldersLen) * 3 / 4) - placeHoldersLen
    }

    function toByteArray(b64) {
      var tmp
      var lens = getLens(b64)
      var validLen = lens[0]
      var placeHoldersLen = lens[1]

      var arr = new Arr(_byteLength(b64, validLen, placeHoldersLen))

      var curByte = 0

      // if there are placeholders, only get up to the last complete 4 chars
      var len = placeHoldersLen > 0
        ? validLen - 4
        : validLen

      for (var i = 0; i < len; i += 4) {
        tmp =
          (revLookup[b64.charCodeAt(i)] << 18) |
          (revLookup[b64.charCodeAt(i + 1)] << 12) |
          (revLookup[b64.charCodeAt(i + 2)] << 6) |
          revLookup[b64.charCodeAt(i + 3)]
        arr[curByte++] = (tmp >> 16) & 0xFF
        arr[curByte++] = (tmp >> 8) & 0xFF
        arr[curByte++] = tmp & 0xFF
      }

      if (placeHoldersLen === 2) {
        tmp =
          (revLookup[b64.charCodeAt(i)] << 2) |
          (revLookup[b64.charCodeAt(i + 1)] >> 4)
        arr[curByte++] = tmp & 0xFF
      }

      if (placeHoldersLen === 1) {
        tmp =
          (revLookup[b64.charCodeAt(i)] << 10) |
          (revLookup[b64.charCodeAt(i + 1)] << 4) |
          (revLookup[b64.charCodeAt(i + 2)] >> 2)
        arr[curByte++] = (tmp >> 8) & 0xFF
        arr[curByte++] = tmp & 0xFF
      }

      return arr
    }

    function tripletToBase64(num) {
      return lookup[num >> 18 & 0x3F] +
        lookup[num >> 12 & 0x3F] +
        lookup[num >> 6 & 0x3F] +
        lookup[num & 0x3F]
    }

    function encodeChunk(uint8, start, end) {
      var tmp
      var output = []
      for (var i = start; i < end; i += 3) {
        tmp =
          ((uint8[i] << 16) & 0xFF0000) +
          ((uint8[i + 1] << 8) & 0xFF00) +
          (uint8[i + 2] & 0xFF)
        output.push(tripletToBase64(tmp))
      }
      return output.join('')
    }

    function fromByteArray(uint8) {
      var tmp
      var len = uint8.length
      var extraBytes = len % 3 // if we have 1 byte left, pad 2 bytes
      var parts = []
      var maxChunkLength = 16383 // must be multiple of 3

      // go through the array every three bytes, we'll deal with trailing stuff later
      for (var i = 0, len2 = len - extraBytes; i < len2; i += maxChunkLength) {
        parts.push(encodeChunk(
          uint8, i, (i + maxChunkLength) > len2 ? len2 : (i + maxChunkLength)
        ))
      }

      // pad the end with zeros, but make sure to not forget the extra bytes
      if (extraBytes === 1) {
        tmp = uint8[len - 1]
        parts.push(
          lookup[tmp >> 2] +
          lookup[(tmp << 4) & 0x3F] +
          '=='
        )
      } else if (extraBytes === 2) {
        tmp = (uint8[len - 2] << 8) + uint8[len - 1]
        parts.push(
          lookup[tmp >> 10] +
          lookup[(tmp >> 4) & 0x3F] +
          lookup[(tmp << 2) & 0x3F] +
          '='
        )
      }

      return parts.join('')
    }

  }, {}], 24: [function (require, module, exports) {
    (function (module, exports) {
      'use strict';

      // Utils
      function assert(val, msg) {
        if (!val) throw new Error(msg || 'Assertion failed');
      }

      // Could use `inherits` module, but don't want to move from single file
      // architecture yet.
      function inherits(ctor, superCtor) {
        ctor.super_ = superCtor;
        var TempCtor = function () { };
        TempCtor.prototype = superCtor.prototype;
        ctor.prototype = new TempCtor();
        ctor.prototype.constructor = ctor;
      }

      // BN

      function BN(number, base, endian) {
        if (BN.isBN(number)) {
          return number;
        }

        this.negative = 0;
        this.words = null;
        this.length = 0;

        // Reduction context
        this.red = null;

        if (number !== null) {
          if (base === 'le' || base === 'be') {
            endian = base;
            base = 10;
          }

          this._init(number || 0, base || 10, endian || 'be');
        }
      }
      if (typeof module === 'object') {
        module.exports = BN;
      } else {
        exports.BN = BN;
      }

      BN.BN = BN;
      BN.wordSize = 26;

      var Buffer;
      try {
        Buffer = require('buffer').Buffer;
      } catch (e) {
      }

      BN.isBN = function isBN(num) {
        if (num instanceof BN) {
          return true;
        }

        return num !== null && typeof num === 'object' &&
          num.constructor.wordSize === BN.wordSize && Array.isArray(num.words);
      };

      BN.max = function max(left, right) {
        if (left.cmp(right) > 0) return left;
        return right;
      };

      BN.min = function min(left, right) {
        if (left.cmp(right) < 0) return left;
        return right;
      };

      BN.prototype._init = function init(number, base, endian) {
        if (typeof number === 'number') {
          return this._initNumber(number, base, endian);
        }

        if (typeof number === 'object') {
          return this._initArray(number, base, endian);
        }

        if (base === 'hex') {
          base = 16;
        }
        assert(base === (base | 0) && base >= 2 && base <= 36);

        number = number.toString().replace(/\s+/g, '');
        var start = 0;
        if (number[0] === '-') {
          start++;
        }

        if (base === 16) {
          this._parseHex(number, start);
        } else {
          this._parseBase(number, base, start);
        }

        if (number[0] === '-') {
          this.negative = 1;
        }

        this.strip();

        if (endian !== 'le') return;

        this._initArray(this.toArray(), base, endian);
      };

      BN.prototype._initNumber = function _initNumber(number, base, endian) {
        if (number < 0) {
          this.negative = 1;
          number = -number;
        }
        if (number < 0x4000000) {
          this.words = [number & 0x3ffffff];
          this.length = 1;
        } else if (number < 0x10000000000000) {
          this.words = [
            number & 0x3ffffff,
            (number / 0x4000000) & 0x3ffffff
          ];
          this.length = 2;
        } else {
          assert(number < 0x20000000000000); // 2 ^ 53 (unsafe)
          this.words = [
            number & 0x3ffffff,
            (number / 0x4000000) & 0x3ffffff,
            1
          ];
          this.length = 3;
        }

        if (endian !== 'le') return;

        // Reverse the bytes
        this._initArray(this.toArray(), base, endian);
      };

      BN.prototype._initArray = function _initArray(number, base, endian) {
        // Perhaps a Uint8Array
        assert(typeof number.length === 'number');
        if (number.length <= 0) {
          this.words = [0];
          this.length = 1;
          return this;
        }

        this.length = Math.ceil(number.length / 3);
        this.words = new Array(this.length);
        for (var i = 0; i < this.length; i++) {
          this.words[i] = 0;
        }

        var j, w;
        var off = 0;
        if (endian === 'be') {
          for (i = number.length - 1, j = 0; i >= 0; i -= 3) {
            w = number[i] | (number[i - 1] << 8) | (number[i - 2] << 16);
            this.words[j] |= (w << off) & 0x3ffffff;
            this.words[j + 1] = (w >>> (26 - off)) & 0x3ffffff;
            off += 24;
            if (off >= 26) {
              off -= 26;
              j++;
            }
          }
        } else if (endian === 'le') {
          for (i = 0, j = 0; i < number.length; i += 3) {
            w = number[i] | (number[i + 1] << 8) | (number[i + 2] << 16);
            this.words[j] |= (w << off) & 0x3ffffff;
            this.words[j + 1] = (w >>> (26 - off)) & 0x3ffffff;
            off += 24;
            if (off >= 26) {
              off -= 26;
              j++;
            }
          }
        }
        return this.strip();
      };

      function parseHex(str, start, end) {
        var r = 0;
        var len = Math.min(str.length, end);
        for (var i = start; i < len; i++) {
          var c = str.charCodeAt(i) - 48;

          r <<= 4;

          // 'a' - 'f'
          if (c >= 49 && c <= 54) {
            r |= c - 49 + 0xa;

            // 'A' - 'F'
          } else if (c >= 17 && c <= 22) {
            r |= c - 17 + 0xa;

            // '0' - '9'
          } else {
            r |= c & 0xf;
          }
        }
        return r;
      }

      BN.prototype._parseHex = function _parseHex(number, start) {
        // Create possibly bigger array to ensure that it fits the number
        this.length = Math.ceil((number.length - start) / 6);
        this.words = new Array(this.length);
        for (var i = 0; i < this.length; i++) {
          this.words[i] = 0;
        }

        var j, w;
        // Scan 24-bit chunks and add them to the number
        var off = 0;
        for (i = number.length - 6, j = 0; i >= start; i -= 6) {
          w = parseHex(number, i, i + 6);
          this.words[j] |= (w << off) & 0x3ffffff;
          // NOTE: `0x3fffff` is intentional here, 26bits max shift + 24bit hex limb
          this.words[j + 1] |= w >>> (26 - off) & 0x3fffff;
          off += 24;
          if (off >= 26) {
            off -= 26;
            j++;
          }
        }
        if (i + 6 !== start) {
          w = parseHex(number, start, i + 6);
          this.words[j] |= (w << off) & 0x3ffffff;
          this.words[j + 1] |= w >>> (26 - off) & 0x3fffff;
        }
        this.strip();
      };

      function parseBase(str, start, end, mul) {
        var r = 0;
        var len = Math.min(str.length, end);
        for (var i = start; i < len; i++) {
          var c = str.charCodeAt(i) - 48;

          r *= mul;

          // 'a'
          if (c >= 49) {
            r += c - 49 + 0xa;

            // 'A'
          } else if (c >= 17) {
            r += c - 17 + 0xa;

            // '0' - '9'
          } else {
            r += c;
          }
        }
        return r;
      }

      BN.prototype._parseBase = function _parseBase(number, base, start) {
        // Initialize as zero
        this.words = [0];
        this.length = 1;

        // Find length of limb in base
        for (var limbLen = 0, limbPow = 1; limbPow <= 0x3ffffff; limbPow *= base) {
          limbLen++;
        }
        limbLen--;
        limbPow = (limbPow / base) | 0;

        var total = number.length - start;
        var mod = total % limbLen;
        var end = Math.min(total, total - mod) + start;

        var word = 0;
        for (var i = start; i < end; i += limbLen) {
          word = parseBase(number, i, i + limbLen, base);

          this.imuln(limbPow);
          if (this.words[0] + word < 0x4000000) {
            this.words[0] += word;
          } else {
            this._iaddn(word);
          }
        }

        if (mod !== 0) {
          var pow = 1;
          word = parseBase(number, i, number.length, base);

          for (i = 0; i < mod; i++) {
            pow *= base;
          }

          this.imuln(pow);
          if (this.words[0] + word < 0x4000000) {
            this.words[0] += word;
          } else {
            this._iaddn(word);
          }
        }
      };

      BN.prototype.copy = function copy(dest) {
        dest.words = new Array(this.length);
        for (var i = 0; i < this.length; i++) {
          dest.words[i] = this.words[i];
        }
        dest.length = this.length;
        dest.negative = this.negative;
        dest.red = this.red;
      };

      BN.prototype.clone = function clone() {
        var r = new BN(null);
        this.copy(r);
        return r;
      };

      BN.prototype._expand = function _expand(size) {
        while (this.length < size) {
          this.words[this.length++] = 0;
        }
        return this;
      };

      // Remove leading `0` from `this`
      BN.prototype.strip = function strip() {
        while (this.length > 1 && this.words[this.length - 1] === 0) {
          this.length--;
        }
        return this._normSign();
      };

      BN.prototype._normSign = function _normSign() {
        // -0 = 0
        if (this.length === 1 && this.words[0] === 0) {
          this.negative = 0;
        }
        return this;
      };

      BN.prototype.inspect = function inspect() {
        return (this.red ? '<BN-R: ' : '<BN: ') + this.toString(16) + '>';
      };

      /*

      var zeros = [];
      var groupSizes = [];
      var groupBases = [];

      var s = '';
      var i = -1;
      while (++i < BN.wordSize) {
        zeros[i] = s;
        s += '0';
      }
      groupSizes[0] = 0;
      groupSizes[1] = 0;
      groupBases[0] = 0;
      groupBases[1] = 0;
      var base = 2 - 1;
      while (++base < 36 + 1) {
        var groupSize = 0;
        var groupBase = 1;
        while (groupBase < (1 << BN.wordSize) / base) {
          groupBase *= base;
          groupSize += 1;
        }
        groupSizes[base] = groupSize;
        groupBases[base] = groupBase;
      }

      */

      var zeros = [
        '',
        '0',
        '00',
        '000',
        '0000',
        '00000',
        '000000',
        '0000000',
        '00000000',
        '000000000',
        '0000000000',
        '00000000000',
        '000000000000',
        '0000000000000',
        '00000000000000',
        '000000000000000',
        '0000000000000000',
        '00000000000000000',
        '000000000000000000',
        '0000000000000000000',
        '00000000000000000000',
        '000000000000000000000',
        '0000000000000000000000',
        '00000000000000000000000',
        '000000000000000000000000',
        '0000000000000000000000000'
      ];

      var groupSizes = [
        0, 0,
        25, 16, 12, 11, 10, 9, 8,
        8, 7, 7, 7, 7, 6, 6,
        6, 6, 6, 6, 6, 5, 5,
        5, 5, 5, 5, 5, 5, 5,
        5, 5, 5, 5, 5, 5, 5
      ];

      var groupBases = [
        0, 0,
        33554432, 43046721, 16777216, 48828125, 60466176, 40353607, 16777216,
        43046721, 10000000, 19487171, 35831808, 62748517, 7529536, 11390625,
        16777216, 24137569, 34012224, 47045881, 64000000, 4084101, 5153632,
        6436343, 7962624, 9765625, 11881376, 14348907, 17210368, 20511149,
        24300000, 28629151, 33554432, 39135393, 45435424, 52521875, 60466176
      ];

      BN.prototype.toString = function toString(base, padding) {
        base = base || 10;
        padding = padding | 0 || 1;

        var out;
        if (base === 16 || base === 'hex') {
          out = '';
          var off = 0;
          var carry = 0;
          for (var i = 0; i < this.length; i++) {
            var w = this.words[i];
            var word = (((w << off) | carry) & 0xffffff).toString(16);
            carry = (w >>> (24 - off)) & 0xffffff;
            if (carry !== 0 || i !== this.length - 1) {
              out = zeros[6 - word.length] + word + out;
            } else {
              out = word + out;
            }
            off += 2;
            if (off >= 26) {
              off -= 26;
              i--;
            }
          }
          if (carry !== 0) {
            out = carry.toString(16) + out;
          }
          while (out.length % padding !== 0) {
            out = '0' + out;
          }
          if (this.negative !== 0) {
            out = '-' + out;
          }
          return out;
        }

        if (base === (base | 0) && base >= 2 && base <= 36) {
          // var groupSize = Math.floor(BN.wordSize * Math.LN2 / Math.log(base));
          var groupSize = groupSizes[base];
          // var groupBase = Math.pow(base, groupSize);
          var groupBase = groupBases[base];
          out = '';
          var c = this.clone();
          c.negative = 0;
          while (!c.isZero()) {
            var r = c.modn(groupBase).toString(base);
            c = c.idivn(groupBase);

            if (!c.isZero()) {
              out = zeros[groupSize - r.length] + r + out;
            } else {
              out = r + out;
            }
          }
          if (this.isZero()) {
            out = '0' + out;
          }
          while (out.length % padding !== 0) {
            out = '0' + out;
          }
          if (this.negative !== 0) {
            out = '-' + out;
          }
          return out;
        }

        assert(false, 'Base should be between 2 and 36');
      };

      BN.prototype.toNumber = function toNumber() {
        var ret = this.words[0];
        if (this.length === 2) {
          ret += this.words[1] * 0x4000000;
        } else if (this.length === 3 && this.words[2] === 0x01) {
          // NOTE: at this stage it is known that the top bit is set
          ret += 0x10000000000000 + (this.words[1] * 0x4000000);
        } else if (this.length > 2) {
          assert(false, 'Number can only safely store up to 53 bits');
        }
        return (this.negative !== 0) ? -ret : ret;
      };

      BN.prototype.toJSON = function toJSON() {
        return this.toString(16);
      };

      BN.prototype.toBuffer = function toBuffer(endian, length) {
        assert(typeof Buffer !== 'undefined');
        return this.toArrayLike(Buffer, endian, length);
      };

      BN.prototype.toArray = function toArray(endian, length) {
        return this.toArrayLike(Array, endian, length);
      };

      BN.prototype.toArrayLike = function toArrayLike(ArrayType, endian, length) {
        var byteLength = this.byteLength();
        var reqLength = length || Math.max(1, byteLength);
        assert(byteLength <= reqLength, 'byte array longer than desired length');
        assert(reqLength > 0, 'Requested array length <= 0');

        this.strip();
        var littleEndian = endian === 'le';
        var res = new ArrayType(reqLength);

        var b, i;
        var q = this.clone();
        if (!littleEndian) {
          // Assume big-endian
          for (i = 0; i < reqLength - byteLength; i++) {
            res[i] = 0;
          }

          for (i = 0; !q.isZero(); i++) {
            b = q.andln(0xff);
            q.iushrn(8);

            res[reqLength - i - 1] = b;
          }
        } else {
          for (i = 0; !q.isZero(); i++) {
            b = q.andln(0xff);
            q.iushrn(8);

            res[i] = b;
          }

          for (; i < reqLength; i++) {
            res[i] = 0;
          }
        }

        return res;
      };

      if (Math.clz32) {
        BN.prototype._countBits = function _countBits(w) {
          return 32 - Math.clz32(w);
        };
      } else {
        BN.prototype._countBits = function _countBits(w) {
          var t = w;
          var r = 0;
          if (t >= 0x1000) {
            r += 13;
            t >>>= 13;
          }
          if (t >= 0x40) {
            r += 7;
            t >>>= 7;
          }
          if (t >= 0x8) {
            r += 4;
            t >>>= 4;
          }
          if (t >= 0x02) {
            r += 2;
            t >>>= 2;
          }
          return r + t;
        };
      }

      BN.prototype._zeroBits = function _zeroBits(w) {
        // Short-cut
        if (w === 0) return 26;

        var t = w;
        var r = 0;
        if ((t & 0x1fff) === 0) {
          r += 13;
          t >>>= 13;
        }
        if ((t & 0x7f) === 0) {
          r += 7;
          t >>>= 7;
        }
        if ((t & 0xf) === 0) {
          r += 4;
          t >>>= 4;
        }
        if ((t & 0x3) === 0) {
          r += 2;
          t >>>= 2;
        }
        if ((t & 0x1) === 0) {
          r++;
        }
        return r;
      };

      // Return number of used bits in a BN
      BN.prototype.bitLength = function bitLength() {
        var w = this.words[this.length - 1];
        var hi = this._countBits(w);
        return (this.length - 1) * 26 + hi;
      };

      function toBitArray(num) {
        var w = new Array(num.bitLength());

        for (var bit = 0; bit < w.length; bit++) {
          var off = (bit / 26) | 0;
          var wbit = bit % 26;

          w[bit] = (num.words[off] & (1 << wbit)) >>> wbit;
        }

        return w;
      }

      // Number of trailing zero bits
      BN.prototype.zeroBits = function zeroBits() {
        if (this.isZero()) return 0;

        var r = 0;
        for (var i = 0; i < this.length; i++) {
          var b = this._zeroBits(this.words[i]);
          r += b;
          if (b !== 26) break;
        }
        return r;
      };

      BN.prototype.byteLength = function byteLength() {
        return Math.ceil(this.bitLength() / 8);
      };

      BN.prototype.toTwos = function toTwos(width) {
        if (this.negative !== 0) {
          return this.abs().inotn(width).iaddn(1);
        }
        return this.clone();
      };

      BN.prototype.fromTwos = function fromTwos(width) {
        if (this.testn(width - 1)) {
          return this.notn(width).iaddn(1).ineg();
        }
        return this.clone();
      };

      BN.prototype.isNeg = function isNeg() {
        return this.negative !== 0;
      };

      // Return negative clone of `this`
      BN.prototype.neg = function neg() {
        return this.clone().ineg();
      };

      BN.prototype.ineg = function ineg() {
        if (!this.isZero()) {
          this.negative ^= 1;
        }

        return this;
      };

      // Or `num` with `this` in-place
      BN.prototype.iuor = function iuor(num) {
        while (this.length < num.length) {
          this.words[this.length++] = 0;
        }

        for (var i = 0; i < num.length; i++) {
          this.words[i] = this.words[i] | num.words[i];
        }

        return this.strip();
      };

      BN.prototype.ior = function ior(num) {
        assert((this.negative | num.negative) === 0);
        return this.iuor(num);
      };

      // Or `num` with `this`
      BN.prototype.or = function or(num) {
        if (this.length > num.length) return this.clone().ior(num);
        return num.clone().ior(this);
      };

      BN.prototype.uor = function uor(num) {
        if (this.length > num.length) return this.clone().iuor(num);
        return num.clone().iuor(this);
      };

      // And `num` with `this` in-place
      BN.prototype.iuand = function iuand(num) {
        // b = min-length(num, this)
        var b;
        if (this.length > num.length) {
          b = num;
        } else {
          b = this;
        }

        for (var i = 0; i < b.length; i++) {
          this.words[i] = this.words[i] & num.words[i];
        }

        this.length = b.length;

        return this.strip();
      };

      BN.prototype.iand = function iand(num) {
        assert((this.negative | num.negative) === 0);
        return this.iuand(num);
      };

      // And `num` with `this`
      BN.prototype.and = function and(num) {
        if (this.length > num.length) return this.clone().iand(num);
        return num.clone().iand(this);
      };

      BN.prototype.uand = function uand(num) {
        if (this.length > num.length) return this.clone().iuand(num);
        return num.clone().iuand(this);
      };

      // Xor `num` with `this` in-place
      BN.prototype.iuxor = function iuxor(num) {
        // a.length > b.length
        var a;
        var b;
        if (this.length > num.length) {
          a = this;
          b = num;
        } else {
          a = num;
          b = this;
        }

        for (var i = 0; i < b.length; i++) {
          this.words[i] = a.words[i] ^ b.words[i];
        }

        if (this !== a) {
          for (; i < a.length; i++) {
            this.words[i] = a.words[i];
          }
        }

        this.length = a.length;

        return this.strip();
      };

      BN.prototype.ixor = function ixor(num) {
        assert((this.negative | num.negative) === 0);
        return this.iuxor(num);
      };

      // Xor `num` with `this`
      BN.prototype.xor = function xor(num) {
        if (this.length > num.length) return this.clone().ixor(num);
        return num.clone().ixor(this);
      };

      BN.prototype.uxor = function uxor(num) {
        if (this.length > num.length) return this.clone().iuxor(num);
        return num.clone().iuxor(this);
      };

      // Not ``this`` with ``width`` bitwidth
      BN.prototype.inotn = function inotn(width) {
        assert(typeof width === 'number' && width >= 0);

        var bytesNeeded = Math.ceil(width / 26) | 0;
        var bitsLeft = width % 26;

        // Extend the buffer with leading zeroes
        this._expand(bytesNeeded);

        if (bitsLeft > 0) {
          bytesNeeded--;
        }

        // Handle complete words
        for (var i = 0; i < bytesNeeded; i++) {
          this.words[i] = ~this.words[i] & 0x3ffffff;
        }

        // Handle the residue
        if (bitsLeft > 0) {
          this.words[i] = ~this.words[i] & (0x3ffffff >> (26 - bitsLeft));
        }

        // And remove leading zeroes
        return this.strip();
      };

      BN.prototype.notn = function notn(width) {
        return this.clone().inotn(width);
      };

      // Set `bit` of `this`
      BN.prototype.setn = function setn(bit, val) {
        assert(typeof bit === 'number' && bit >= 0);

        var off = (bit / 26) | 0;
        var wbit = bit % 26;

        this._expand(off + 1);

        if (val) {
          this.words[off] = this.words[off] | (1 << wbit);
        } else {
          this.words[off] = this.words[off] & ~(1 << wbit);
        }

        return this.strip();
      };

      // Add `num` to `this` in-place
      BN.prototype.iadd = function iadd(num) {
        var r;

        // negative + positive
        if (this.negative !== 0 && num.negative === 0) {
          this.negative = 0;
          r = this.isub(num);
          this.negative ^= 1;
          return this._normSign();

          // positive + negative
        } else if (this.negative === 0 && num.negative !== 0) {
          num.negative = 0;
          r = this.isub(num);
          num.negative = 1;
          return r._normSign();
        }

        // a.length > b.length
        var a, b;
        if (this.length > num.length) {
          a = this;
          b = num;
        } else {
          a = num;
          b = this;
        }

        var carry = 0;
        for (var i = 0; i < b.length; i++) {
          r = (a.words[i] | 0) + (b.words[i] | 0) + carry;
          this.words[i] = r & 0x3ffffff;
          carry = r >>> 26;
        }
        for (; carry !== 0 && i < a.length; i++) {
          r = (a.words[i] | 0) + carry;
          this.words[i] = r & 0x3ffffff;
          carry = r >>> 26;
        }

        this.length = a.length;
        if (carry !== 0) {
          this.words[this.length] = carry;
          this.length++;
          // Copy the rest of the words
        } else if (a !== this) {
          for (; i < a.length; i++) {
            this.words[i] = a.words[i];
          }
        }

        return this;
      };

      // Add `num` to `this`
      BN.prototype.add = function add(num) {
        var res;
        if (num.negative !== 0 && this.negative === 0) {
          num.negative = 0;
          res = this.sub(num);
          num.negative ^= 1;
          return res;
        } else if (num.negative === 0 && this.negative !== 0) {
          this.negative = 0;
          res = num.sub(this);
          this.negative = 1;
          return res;
        }

        if (this.length > num.length) return this.clone().iadd(num);

        return num.clone().iadd(this);
      };

      // Subtract `num` from `this` in-place
      BN.prototype.isub = function isub(num) {
        // this - (-num) = this + num
        if (num.negative !== 0) {
          num.negative = 0;
          var r = this.iadd(num);
          num.negative = 1;
          return r._normSign();

          // -this - num = -(this + num)
        } else if (this.negative !== 0) {
          this.negative = 0;
          this.iadd(num);
          this.negative = 1;
          return this._normSign();
        }

        // At this point both numbers are positive
        var cmp = this.cmp(num);

        // Optimization - zeroify
        if (cmp === 0) {
          this.negative = 0;
          this.length = 1;
          this.words[0] = 0;
          return this;
        }

        // a > b
        var a, b;
        if (cmp > 0) {
          a = this;
          b = num;
        } else {
          a = num;
          b = this;
        }

        var carry = 0;
        for (var i = 0; i < b.length; i++) {
          r = (a.words[i] | 0) - (b.words[i] | 0) + carry;
          carry = r >> 26;
          this.words[i] = r & 0x3ffffff;
        }
        for (; carry !== 0 && i < a.length; i++) {
          r = (a.words[i] | 0) + carry;
          carry = r >> 26;
          this.words[i] = r & 0x3ffffff;
        }

        // Copy rest of the words
        if (carry === 0 && i < a.length && a !== this) {
          for (; i < a.length; i++) {
            this.words[i] = a.words[i];
          }
        }

        this.length = Math.max(this.length, i);

        if (a !== this) {
          this.negative = 1;
        }

        return this.strip();
      };

      // Subtract `num` from `this`
      BN.prototype.sub = function sub(num) {
        return this.clone().isub(num);
      };

      function smallMulTo(self, num, out) {
        out.negative = num.negative ^ self.negative;
        var len = (self.length + num.length) | 0;
        out.length = len;
        len = (len - 1) | 0;

        // Peel one iteration (compiler can't do it, because of code complexity)
        var a = self.words[0] | 0;
        var b = num.words[0] | 0;
        var r = a * b;

        var lo = r & 0x3ffffff;
        var carry = (r / 0x4000000) | 0;
        out.words[0] = lo;

        for (var k = 1; k < len; k++) {
          // Sum all words with the same `i + j = k` and accumulate `ncarry`,
          // note that ncarry could be >= 0x3ffffff
          var ncarry = carry >>> 26;
          var rword = carry & 0x3ffffff;
          var maxJ = Math.min(k, num.length - 1);
          for (var j = Math.max(0, k - self.length + 1); j <= maxJ; j++) {
            var i = (k - j) | 0;
            a = self.words[i] | 0;
            b = num.words[j] | 0;
            r = a * b + rword;
            ncarry += (r / 0x4000000) | 0;
            rword = r & 0x3ffffff;
          }
          out.words[k] = rword | 0;
          carry = ncarry | 0;
        }
        if (carry !== 0) {
          out.words[k] = carry | 0;
        } else {
          out.length--;
        }

        return out.strip();
      }

      // TODO(indutny): it may be reasonable to omit it for users who don't need
      // to work with 256-bit numbers, otherwise it gives 20% improvement for 256-bit
      // multiplication (like elliptic secp256k1).
      var comb10MulTo = function comb10MulTo(self, num, out) {
        var a = self.words;
        var b = num.words;
        var o = out.words;
        var c = 0;
        var lo;
        var mid;
        var hi;
        var a0 = a[0] | 0;
        var al0 = a0 & 0x1fff;
        var ah0 = a0 >>> 13;
        var a1 = a[1] | 0;
        var al1 = a1 & 0x1fff;
        var ah1 = a1 >>> 13;
        var a2 = a[2] | 0;
        var al2 = a2 & 0x1fff;
        var ah2 = a2 >>> 13;
        var a3 = a[3] | 0;
        var al3 = a3 & 0x1fff;
        var ah3 = a3 >>> 13;
        var a4 = a[4] | 0;
        var al4 = a4 & 0x1fff;
        var ah4 = a4 >>> 13;
        var a5 = a[5] | 0;
        var al5 = a5 & 0x1fff;
        var ah5 = a5 >>> 13;
        var a6 = a[6] | 0;
        var al6 = a6 & 0x1fff;
        var ah6 = a6 >>> 13;
        var a7 = a[7] | 0;
        var al7 = a7 & 0x1fff;
        var ah7 = a7 >>> 13;
        var a8 = a[8] | 0;
        var al8 = a8 & 0x1fff;
        var ah8 = a8 >>> 13;
        var a9 = a[9] | 0;
        var al9 = a9 & 0x1fff;
        var ah9 = a9 >>> 13;
        var b0 = b[0] | 0;
        var bl0 = b0 & 0x1fff;
        var bh0 = b0 >>> 13;
        var b1 = b[1] | 0;
        var bl1 = b1 & 0x1fff;
        var bh1 = b1 >>> 13;
        var b2 = b[2] | 0;
        var bl2 = b2 & 0x1fff;
        var bh2 = b2 >>> 13;
        var b3 = b[3] | 0;
        var bl3 = b3 & 0x1fff;
        var bh3 = b3 >>> 13;
        var b4 = b[4] | 0;
        var bl4 = b4 & 0x1fff;
        var bh4 = b4 >>> 13;
        var b5 = b[5] | 0;
        var bl5 = b5 & 0x1fff;
        var bh5 = b5 >>> 13;
        var b6 = b[6] | 0;
        var bl6 = b6 & 0x1fff;
        var bh6 = b6 >>> 13;
        var b7 = b[7] | 0;
        var bl7 = b7 & 0x1fff;
        var bh7 = b7 >>> 13;
        var b8 = b[8] | 0;
        var bl8 = b8 & 0x1fff;
        var bh8 = b8 >>> 13;
        var b9 = b[9] | 0;
        var bl9 = b9 & 0x1fff;
        var bh9 = b9 >>> 13;

        out.negative = self.negative ^ num.negative;
        out.length = 19;
        /* k = 0 */
        lo = Math.imul(al0, bl0);
        mid = Math.imul(al0, bh0);
        mid = (mid + Math.imul(ah0, bl0)) | 0;
        hi = Math.imul(ah0, bh0);
        var w0 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w0 >>> 26)) | 0;
        w0 &= 0x3ffffff;
        /* k = 1 */
        lo = Math.imul(al1, bl0);
        mid = Math.imul(al1, bh0);
        mid = (mid + Math.imul(ah1, bl0)) | 0;
        hi = Math.imul(ah1, bh0);
        lo = (lo + Math.imul(al0, bl1)) | 0;
        mid = (mid + Math.imul(al0, bh1)) | 0;
        mid = (mid + Math.imul(ah0, bl1)) | 0;
        hi = (hi + Math.imul(ah0, bh1)) | 0;
        var w1 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w1 >>> 26)) | 0;
        w1 &= 0x3ffffff;
        /* k = 2 */
        lo = Math.imul(al2, bl0);
        mid = Math.imul(al2, bh0);
        mid = (mid + Math.imul(ah2, bl0)) | 0;
        hi = Math.imul(ah2, bh0);
        lo = (lo + Math.imul(al1, bl1)) | 0;
        mid = (mid + Math.imul(al1, bh1)) | 0;
        mid = (mid + Math.imul(ah1, bl1)) | 0;
        hi = (hi + Math.imul(ah1, bh1)) | 0;
        lo = (lo + Math.imul(al0, bl2)) | 0;
        mid = (mid + Math.imul(al0, bh2)) | 0;
        mid = (mid + Math.imul(ah0, bl2)) | 0;
        hi = (hi + Math.imul(ah0, bh2)) | 0;
        var w2 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w2 >>> 26)) | 0;
        w2 &= 0x3ffffff;
        /* k = 3 */
        lo = Math.imul(al3, bl0);
        mid = Math.imul(al3, bh0);
        mid = (mid + Math.imul(ah3, bl0)) | 0;
        hi = Math.imul(ah3, bh0);
        lo = (lo + Math.imul(al2, bl1)) | 0;
        mid = (mid + Math.imul(al2, bh1)) | 0;
        mid = (mid + Math.imul(ah2, bl1)) | 0;
        hi = (hi + Math.imul(ah2, bh1)) | 0;
        lo = (lo + Math.imul(al1, bl2)) | 0;
        mid = (mid + Math.imul(al1, bh2)) | 0;
        mid = (mid + Math.imul(ah1, bl2)) | 0;
        hi = (hi + Math.imul(ah1, bh2)) | 0;
        lo = (lo + Math.imul(al0, bl3)) | 0;
        mid = (mid + Math.imul(al0, bh3)) | 0;
        mid = (mid + Math.imul(ah0, bl3)) | 0;
        hi = (hi + Math.imul(ah0, bh3)) | 0;
        var w3 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w3 >>> 26)) | 0;
        w3 &= 0x3ffffff;
        /* k = 4 */
        lo = Math.imul(al4, bl0);
        mid = Math.imul(al4, bh0);
        mid = (mid + Math.imul(ah4, bl0)) | 0;
        hi = Math.imul(ah4, bh0);
        lo = (lo + Math.imul(al3, bl1)) | 0;
        mid = (mid + Math.imul(al3, bh1)) | 0;
        mid = (mid + Math.imul(ah3, bl1)) | 0;
        hi = (hi + Math.imul(ah3, bh1)) | 0;
        lo = (lo + Math.imul(al2, bl2)) | 0;
        mid = (mid + Math.imul(al2, bh2)) | 0;
        mid = (mid + Math.imul(ah2, bl2)) | 0;
        hi = (hi + Math.imul(ah2, bh2)) | 0;
        lo = (lo + Math.imul(al1, bl3)) | 0;
        mid = (mid + Math.imul(al1, bh3)) | 0;
        mid = (mid + Math.imul(ah1, bl3)) | 0;
        hi = (hi + Math.imul(ah1, bh3)) | 0;
        lo = (lo + Math.imul(al0, bl4)) | 0;
        mid = (mid + Math.imul(al0, bh4)) | 0;
        mid = (mid + Math.imul(ah0, bl4)) | 0;
        hi = (hi + Math.imul(ah0, bh4)) | 0;
        var w4 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w4 >>> 26)) | 0;
        w4 &= 0x3ffffff;
        /* k = 5 */
        lo = Math.imul(al5, bl0);
        mid = Math.imul(al5, bh0);
        mid = (mid + Math.imul(ah5, bl0)) | 0;
        hi = Math.imul(ah5, bh0);
        lo = (lo + Math.imul(al4, bl1)) | 0;
        mid = (mid + Math.imul(al4, bh1)) | 0;
        mid = (mid + Math.imul(ah4, bl1)) | 0;
        hi = (hi + Math.imul(ah4, bh1)) | 0;
        lo = (lo + Math.imul(al3, bl2)) | 0;
        mid = (mid + Math.imul(al3, bh2)) | 0;
        mid = (mid + Math.imul(ah3, bl2)) | 0;
        hi = (hi + Math.imul(ah3, bh2)) | 0;
        lo = (lo + Math.imul(al2, bl3)) | 0;
        mid = (mid + Math.imul(al2, bh3)) | 0;
        mid = (mid + Math.imul(ah2, bl3)) | 0;
        hi = (hi + Math.imul(ah2, bh3)) | 0;
        lo = (lo + Math.imul(al1, bl4)) | 0;
        mid = (mid + Math.imul(al1, bh4)) | 0;
        mid = (mid + Math.imul(ah1, bl4)) | 0;
        hi = (hi + Math.imul(ah1, bh4)) | 0;
        lo = (lo + Math.imul(al0, bl5)) | 0;
        mid = (mid + Math.imul(al0, bh5)) | 0;
        mid = (mid + Math.imul(ah0, bl5)) | 0;
        hi = (hi + Math.imul(ah0, bh5)) | 0;
        var w5 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w5 >>> 26)) | 0;
        w5 &= 0x3ffffff;
        /* k = 6 */
        lo = Math.imul(al6, bl0);
        mid = Math.imul(al6, bh0);
        mid = (mid + Math.imul(ah6, bl0)) | 0;
        hi = Math.imul(ah6, bh0);
        lo = (lo + Math.imul(al5, bl1)) | 0;
        mid = (mid + Math.imul(al5, bh1)) | 0;
        mid = (mid + Math.imul(ah5, bl1)) | 0;
        hi = (hi + Math.imul(ah5, bh1)) | 0;
        lo = (lo + Math.imul(al4, bl2)) | 0;
        mid = (mid + Math.imul(al4, bh2)) | 0;
        mid = (mid + Math.imul(ah4, bl2)) | 0;
        hi = (hi + Math.imul(ah4, bh2)) | 0;
        lo = (lo + Math.imul(al3, bl3)) | 0;
        mid = (mid + Math.imul(al3, bh3)) | 0;
        mid = (mid + Math.imul(ah3, bl3)) | 0;
        hi = (hi + Math.imul(ah3, bh3)) | 0;
        lo = (lo + Math.imul(al2, bl4)) | 0;
        mid = (mid + Math.imul(al2, bh4)) | 0;
        mid = (mid + Math.imul(ah2, bl4)) | 0;
        hi = (hi + Math.imul(ah2, bh4)) | 0;
        lo = (lo + Math.imul(al1, bl5)) | 0;
        mid = (mid + Math.imul(al1, bh5)) | 0;
        mid = (mid + Math.imul(ah1, bl5)) | 0;
        hi = (hi + Math.imul(ah1, bh5)) | 0;
        lo = (lo + Math.imul(al0, bl6)) | 0;
        mid = (mid + Math.imul(al0, bh6)) | 0;
        mid = (mid + Math.imul(ah0, bl6)) | 0;
        hi = (hi + Math.imul(ah0, bh6)) | 0;
        var w6 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w6 >>> 26)) | 0;
        w6 &= 0x3ffffff;
        /* k = 7 */
        lo = Math.imul(al7, bl0);
        mid = Math.imul(al7, bh0);
        mid = (mid + Math.imul(ah7, bl0)) | 0;
        hi = Math.imul(ah7, bh0);
        lo = (lo + Math.imul(al6, bl1)) | 0;
        mid = (mid + Math.imul(al6, bh1)) | 0;
        mid = (mid + Math.imul(ah6, bl1)) | 0;
        hi = (hi + Math.imul(ah6, bh1)) | 0;
        lo = (lo + Math.imul(al5, bl2)) | 0;
        mid = (mid + Math.imul(al5, bh2)) | 0;
        mid = (mid + Math.imul(ah5, bl2)) | 0;
        hi = (hi + Math.imul(ah5, bh2)) | 0;
        lo = (lo + Math.imul(al4, bl3)) | 0;
        mid = (mid + Math.imul(al4, bh3)) | 0;
        mid = (mid + Math.imul(ah4, bl3)) | 0;
        hi = (hi + Math.imul(ah4, bh3)) | 0;
        lo = (lo + Math.imul(al3, bl4)) | 0;
        mid = (mid + Math.imul(al3, bh4)) | 0;
        mid = (mid + Math.imul(ah3, bl4)) | 0;
        hi = (hi + Math.imul(ah3, bh4)) | 0;
        lo = (lo + Math.imul(al2, bl5)) | 0;
        mid = (mid + Math.imul(al2, bh5)) | 0;
        mid = (mid + Math.imul(ah2, bl5)) | 0;
        hi = (hi + Math.imul(ah2, bh5)) | 0;
        lo = (lo + Math.imul(al1, bl6)) | 0;
        mid = (mid + Math.imul(al1, bh6)) | 0;
        mid = (mid + Math.imul(ah1, bl6)) | 0;
        hi = (hi + Math.imul(ah1, bh6)) | 0;
        lo = (lo + Math.imul(al0, bl7)) | 0;
        mid = (mid + Math.imul(al0, bh7)) | 0;
        mid = (mid + Math.imul(ah0, bl7)) | 0;
        hi = (hi + Math.imul(ah0, bh7)) | 0;
        var w7 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w7 >>> 26)) | 0;
        w7 &= 0x3ffffff;
        /* k = 8 */
        lo = Math.imul(al8, bl0);
        mid = Math.imul(al8, bh0);
        mid = (mid + Math.imul(ah8, bl0)) | 0;
        hi = Math.imul(ah8, bh0);
        lo = (lo + Math.imul(al7, bl1)) | 0;
        mid = (mid + Math.imul(al7, bh1)) | 0;
        mid = (mid + Math.imul(ah7, bl1)) | 0;
        hi = (hi + Math.imul(ah7, bh1)) | 0;
        lo = (lo + Math.imul(al6, bl2)) | 0;
        mid = (mid + Math.imul(al6, bh2)) | 0;
        mid = (mid + Math.imul(ah6, bl2)) | 0;
        hi = (hi + Math.imul(ah6, bh2)) | 0;
        lo = (lo + Math.imul(al5, bl3)) | 0;
        mid = (mid + Math.imul(al5, bh3)) | 0;
        mid = (mid + Math.imul(ah5, bl3)) | 0;
        hi = (hi + Math.imul(ah5, bh3)) | 0;
        lo = (lo + Math.imul(al4, bl4)) | 0;
        mid = (mid + Math.imul(al4, bh4)) | 0;
        mid = (mid + Math.imul(ah4, bl4)) | 0;
        hi = (hi + Math.imul(ah4, bh4)) | 0;
        lo = (lo + Math.imul(al3, bl5)) | 0;
        mid = (mid + Math.imul(al3, bh5)) | 0;
        mid = (mid + Math.imul(ah3, bl5)) | 0;
        hi = (hi + Math.imul(ah3, bh5)) | 0;
        lo = (lo + Math.imul(al2, bl6)) | 0;
        mid = (mid + Math.imul(al2, bh6)) | 0;
        mid = (mid + Math.imul(ah2, bl6)) | 0;
        hi = (hi + Math.imul(ah2, bh6)) | 0;
        lo = (lo + Math.imul(al1, bl7)) | 0;
        mid = (mid + Math.imul(al1, bh7)) | 0;
        mid = (mid + Math.imul(ah1, bl7)) | 0;
        hi = (hi + Math.imul(ah1, bh7)) | 0;
        lo = (lo + Math.imul(al0, bl8)) | 0;
        mid = (mid + Math.imul(al0, bh8)) | 0;
        mid = (mid + Math.imul(ah0, bl8)) | 0;
        hi = (hi + Math.imul(ah0, bh8)) | 0;
        var w8 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w8 >>> 26)) | 0;
        w8 &= 0x3ffffff;
        /* k = 9 */
        lo = Math.imul(al9, bl0);
        mid = Math.imul(al9, bh0);
        mid = (mid + Math.imul(ah9, bl0)) | 0;
        hi = Math.imul(ah9, bh0);
        lo = (lo + Math.imul(al8, bl1)) | 0;
        mid = (mid + Math.imul(al8, bh1)) | 0;
        mid = (mid + Math.imul(ah8, bl1)) | 0;
        hi = (hi + Math.imul(ah8, bh1)) | 0;
        lo = (lo + Math.imul(al7, bl2)) | 0;
        mid = (mid + Math.imul(al7, bh2)) | 0;
        mid = (mid + Math.imul(ah7, bl2)) | 0;
        hi = (hi + Math.imul(ah7, bh2)) | 0;
        lo = (lo + Math.imul(al6, bl3)) | 0;
        mid = (mid + Math.imul(al6, bh3)) | 0;
        mid = (mid + Math.imul(ah6, bl3)) | 0;
        hi = (hi + Math.imul(ah6, bh3)) | 0;
        lo = (lo + Math.imul(al5, bl4)) | 0;
        mid = (mid + Math.imul(al5, bh4)) | 0;
        mid = (mid + Math.imul(ah5, bl4)) | 0;
        hi = (hi + Math.imul(ah5, bh4)) | 0;
        lo = (lo + Math.imul(al4, bl5)) | 0;
        mid = (mid + Math.imul(al4, bh5)) | 0;
        mid = (mid + Math.imul(ah4, bl5)) | 0;
        hi = (hi + Math.imul(ah4, bh5)) | 0;
        lo = (lo + Math.imul(al3, bl6)) | 0;
        mid = (mid + Math.imul(al3, bh6)) | 0;
        mid = (mid + Math.imul(ah3, bl6)) | 0;
        hi = (hi + Math.imul(ah3, bh6)) | 0;
        lo = (lo + Math.imul(al2, bl7)) | 0;
        mid = (mid + Math.imul(al2, bh7)) | 0;
        mid = (mid + Math.imul(ah2, bl7)) | 0;
        hi = (hi + Math.imul(ah2, bh7)) | 0;
        lo = (lo + Math.imul(al1, bl8)) | 0;
        mid = (mid + Math.imul(al1, bh8)) | 0;
        mid = (mid + Math.imul(ah1, bl8)) | 0;
        hi = (hi + Math.imul(ah1, bh8)) | 0;
        lo = (lo + Math.imul(al0, bl9)) | 0;
        mid = (mid + Math.imul(al0, bh9)) | 0;
        mid = (mid + Math.imul(ah0, bl9)) | 0;
        hi = (hi + Math.imul(ah0, bh9)) | 0;
        var w9 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w9 >>> 26)) | 0;
        w9 &= 0x3ffffff;
        /* k = 10 */
        lo = Math.imul(al9, bl1);
        mid = Math.imul(al9, bh1);
        mid = (mid + Math.imul(ah9, bl1)) | 0;
        hi = Math.imul(ah9, bh1);
        lo = (lo + Math.imul(al8, bl2)) | 0;
        mid = (mid + Math.imul(al8, bh2)) | 0;
        mid = (mid + Math.imul(ah8, bl2)) | 0;
        hi = (hi + Math.imul(ah8, bh2)) | 0;
        lo = (lo + Math.imul(al7, bl3)) | 0;
        mid = (mid + Math.imul(al7, bh3)) | 0;
        mid = (mid + Math.imul(ah7, bl3)) | 0;
        hi = (hi + Math.imul(ah7, bh3)) | 0;
        lo = (lo + Math.imul(al6, bl4)) | 0;
        mid = (mid + Math.imul(al6, bh4)) | 0;
        mid = (mid + Math.imul(ah6, bl4)) | 0;
        hi = (hi + Math.imul(ah6, bh4)) | 0;
        lo = (lo + Math.imul(al5, bl5)) | 0;
        mid = (mid + Math.imul(al5, bh5)) | 0;
        mid = (mid + Math.imul(ah5, bl5)) | 0;
        hi = (hi + Math.imul(ah5, bh5)) | 0;
        lo = (lo + Math.imul(al4, bl6)) | 0;
        mid = (mid + Math.imul(al4, bh6)) | 0;
        mid = (mid + Math.imul(ah4, bl6)) | 0;
        hi = (hi + Math.imul(ah4, bh6)) | 0;
        lo = (lo + Math.imul(al3, bl7)) | 0;
        mid = (mid + Math.imul(al3, bh7)) | 0;
        mid = (mid + Math.imul(ah3, bl7)) | 0;
        hi = (hi + Math.imul(ah3, bh7)) | 0;
        lo = (lo + Math.imul(al2, bl8)) | 0;
        mid = (mid + Math.imul(al2, bh8)) | 0;
        mid = (mid + Math.imul(ah2, bl8)) | 0;
        hi = (hi + Math.imul(ah2, bh8)) | 0;
        lo = (lo + Math.imul(al1, bl9)) | 0;
        mid = (mid + Math.imul(al1, bh9)) | 0;
        mid = (mid + Math.imul(ah1, bl9)) | 0;
        hi = (hi + Math.imul(ah1, bh9)) | 0;
        var w10 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w10 >>> 26)) | 0;
        w10 &= 0x3ffffff;
        /* k = 11 */
        lo = Math.imul(al9, bl2);
        mid = Math.imul(al9, bh2);
        mid = (mid + Math.imul(ah9, bl2)) | 0;
        hi = Math.imul(ah9, bh2);
        lo = (lo + Math.imul(al8, bl3)) | 0;
        mid = (mid + Math.imul(al8, bh3)) | 0;
        mid = (mid + Math.imul(ah8, bl3)) | 0;
        hi = (hi + Math.imul(ah8, bh3)) | 0;
        lo = (lo + Math.imul(al7, bl4)) | 0;
        mid = (mid + Math.imul(al7, bh4)) | 0;
        mid = (mid + Math.imul(ah7, bl4)) | 0;
        hi = (hi + Math.imul(ah7, bh4)) | 0;
        lo = (lo + Math.imul(al6, bl5)) | 0;
        mid = (mid + Math.imul(al6, bh5)) | 0;
        mid = (mid + Math.imul(ah6, bl5)) | 0;
        hi = (hi + Math.imul(ah6, bh5)) | 0;
        lo = (lo + Math.imul(al5, bl6)) | 0;
        mid = (mid + Math.imul(al5, bh6)) | 0;
        mid = (mid + Math.imul(ah5, bl6)) | 0;
        hi = (hi + Math.imul(ah5, bh6)) | 0;
        lo = (lo + Math.imul(al4, bl7)) | 0;
        mid = (mid + Math.imul(al4, bh7)) | 0;
        mid = (mid + Math.imul(ah4, bl7)) | 0;
        hi = (hi + Math.imul(ah4, bh7)) | 0;
        lo = (lo + Math.imul(al3, bl8)) | 0;
        mid = (mid + Math.imul(al3, bh8)) | 0;
        mid = (mid + Math.imul(ah3, bl8)) | 0;
        hi = (hi + Math.imul(ah3, bh8)) | 0;
        lo = (lo + Math.imul(al2, bl9)) | 0;
        mid = (mid + Math.imul(al2, bh9)) | 0;
        mid = (mid + Math.imul(ah2, bl9)) | 0;
        hi = (hi + Math.imul(ah2, bh9)) | 0;
        var w11 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w11 >>> 26)) | 0;
        w11 &= 0x3ffffff;
        /* k = 12 */
        lo = Math.imul(al9, bl3);
        mid = Math.imul(al9, bh3);
        mid = (mid + Math.imul(ah9, bl3)) | 0;
        hi = Math.imul(ah9, bh3);
        lo = (lo + Math.imul(al8, bl4)) | 0;
        mid = (mid + Math.imul(al8, bh4)) | 0;
        mid = (mid + Math.imul(ah8, bl4)) | 0;
        hi = (hi + Math.imul(ah8, bh4)) | 0;
        lo = (lo + Math.imul(al7, bl5)) | 0;
        mid = (mid + Math.imul(al7, bh5)) | 0;
        mid = (mid + Math.imul(ah7, bl5)) | 0;
        hi = (hi + Math.imul(ah7, bh5)) | 0;
        lo = (lo + Math.imul(al6, bl6)) | 0;
        mid = (mid + Math.imul(al6, bh6)) | 0;
        mid = (mid + Math.imul(ah6, bl6)) | 0;
        hi = (hi + Math.imul(ah6, bh6)) | 0;
        lo = (lo + Math.imul(al5, bl7)) | 0;
        mid = (mid + Math.imul(al5, bh7)) | 0;
        mid = (mid + Math.imul(ah5, bl7)) | 0;
        hi = (hi + Math.imul(ah5, bh7)) | 0;
        lo = (lo + Math.imul(al4, bl8)) | 0;
        mid = (mid + Math.imul(al4, bh8)) | 0;
        mid = (mid + Math.imul(ah4, bl8)) | 0;
        hi = (hi + Math.imul(ah4, bh8)) | 0;
        lo = (lo + Math.imul(al3, bl9)) | 0;
        mid = (mid + Math.imul(al3, bh9)) | 0;
        mid = (mid + Math.imul(ah3, bl9)) | 0;
        hi = (hi + Math.imul(ah3, bh9)) | 0;
        var w12 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w12 >>> 26)) | 0;
        w12 &= 0x3ffffff;
        /* k = 13 */
        lo = Math.imul(al9, bl4);
        mid = Math.imul(al9, bh4);
        mid = (mid + Math.imul(ah9, bl4)) | 0;
        hi = Math.imul(ah9, bh4);
        lo = (lo + Math.imul(al8, bl5)) | 0;
        mid = (mid + Math.imul(al8, bh5)) | 0;
        mid = (mid + Math.imul(ah8, bl5)) | 0;
        hi = (hi + Math.imul(ah8, bh5)) | 0;
        lo = (lo + Math.imul(al7, bl6)) | 0;
        mid = (mid + Math.imul(al7, bh6)) | 0;
        mid = (mid + Math.imul(ah7, bl6)) | 0;
        hi = (hi + Math.imul(ah7, bh6)) | 0;
        lo = (lo + Math.imul(al6, bl7)) | 0;
        mid = (mid + Math.imul(al6, bh7)) | 0;
        mid = (mid + Math.imul(ah6, bl7)) | 0;
        hi = (hi + Math.imul(ah6, bh7)) | 0;
        lo = (lo + Math.imul(al5, bl8)) | 0;
        mid = (mid + Math.imul(al5, bh8)) | 0;
        mid = (mid + Math.imul(ah5, bl8)) | 0;
        hi = (hi + Math.imul(ah5, bh8)) | 0;
        lo = (lo + Math.imul(al4, bl9)) | 0;
        mid = (mid + Math.imul(al4, bh9)) | 0;
        mid = (mid + Math.imul(ah4, bl9)) | 0;
        hi = (hi + Math.imul(ah4, bh9)) | 0;
        var w13 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w13 >>> 26)) | 0;
        w13 &= 0x3ffffff;
        /* k = 14 */
        lo = Math.imul(al9, bl5);
        mid = Math.imul(al9, bh5);
        mid = (mid + Math.imul(ah9, bl5)) | 0;
        hi = Math.imul(ah9, bh5);
        lo = (lo + Math.imul(al8, bl6)) | 0;
        mid = (mid + Math.imul(al8, bh6)) | 0;
        mid = (mid + Math.imul(ah8, bl6)) | 0;
        hi = (hi + Math.imul(ah8, bh6)) | 0;
        lo = (lo + Math.imul(al7, bl7)) | 0;
        mid = (mid + Math.imul(al7, bh7)) | 0;
        mid = (mid + Math.imul(ah7, bl7)) | 0;
        hi = (hi + Math.imul(ah7, bh7)) | 0;
        lo = (lo + Math.imul(al6, bl8)) | 0;
        mid = (mid + Math.imul(al6, bh8)) | 0;
        mid = (mid + Math.imul(ah6, bl8)) | 0;
        hi = (hi + Math.imul(ah6, bh8)) | 0;
        lo = (lo + Math.imul(al5, bl9)) | 0;
        mid = (mid + Math.imul(al5, bh9)) | 0;
        mid = (mid + Math.imul(ah5, bl9)) | 0;
        hi = (hi + Math.imul(ah5, bh9)) | 0;
        var w14 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w14 >>> 26)) | 0;
        w14 &= 0x3ffffff;
        /* k = 15 */
        lo = Math.imul(al9, bl6);
        mid = Math.imul(al9, bh6);
        mid = (mid + Math.imul(ah9, bl6)) | 0;
        hi = Math.imul(ah9, bh6);
        lo = (lo + Math.imul(al8, bl7)) | 0;
        mid = (mid + Math.imul(al8, bh7)) | 0;
        mid = (mid + Math.imul(ah8, bl7)) | 0;
        hi = (hi + Math.imul(ah8, bh7)) | 0;
        lo = (lo + Math.imul(al7, bl8)) | 0;
        mid = (mid + Math.imul(al7, bh8)) | 0;
        mid = (mid + Math.imul(ah7, bl8)) | 0;
        hi = (hi + Math.imul(ah7, bh8)) | 0;
        lo = (lo + Math.imul(al6, bl9)) | 0;
        mid = (mid + Math.imul(al6, bh9)) | 0;
        mid = (mid + Math.imul(ah6, bl9)) | 0;
        hi = (hi + Math.imul(ah6, bh9)) | 0;
        var w15 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w15 >>> 26)) | 0;
        w15 &= 0x3ffffff;
        /* k = 16 */
        lo = Math.imul(al9, bl7);
        mid = Math.imul(al9, bh7);
        mid = (mid + Math.imul(ah9, bl7)) | 0;
        hi = Math.imul(ah9, bh7);
        lo = (lo + Math.imul(al8, bl8)) | 0;
        mid = (mid + Math.imul(al8, bh8)) | 0;
        mid = (mid + Math.imul(ah8, bl8)) | 0;
        hi = (hi + Math.imul(ah8, bh8)) | 0;
        lo = (lo + Math.imul(al7, bl9)) | 0;
        mid = (mid + Math.imul(al7, bh9)) | 0;
        mid = (mid + Math.imul(ah7, bl9)) | 0;
        hi = (hi + Math.imul(ah7, bh9)) | 0;
        var w16 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w16 >>> 26)) | 0;
        w16 &= 0x3ffffff;
        /* k = 17 */
        lo = Math.imul(al9, bl8);
        mid = Math.imul(al9, bh8);
        mid = (mid + Math.imul(ah9, bl8)) | 0;
        hi = Math.imul(ah9, bh8);
        lo = (lo + Math.imul(al8, bl9)) | 0;
        mid = (mid + Math.imul(al8, bh9)) | 0;
        mid = (mid + Math.imul(ah8, bl9)) | 0;
        hi = (hi + Math.imul(ah8, bh9)) | 0;
        var w17 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w17 >>> 26)) | 0;
        w17 &= 0x3ffffff;
        /* k = 18 */
        lo = Math.imul(al9, bl9);
        mid = Math.imul(al9, bh9);
        mid = (mid + Math.imul(ah9, bl9)) | 0;
        hi = Math.imul(ah9, bh9);
        var w18 = (((c + lo) | 0) + ((mid & 0x1fff) << 13)) | 0;
        c = (((hi + (mid >>> 13)) | 0) + (w18 >>> 26)) | 0;
        w18 &= 0x3ffffff;
        o[0] = w0;
        o[1] = w1;
        o[2] = w2;
        o[3] = w3;
        o[4] = w4;
        o[5] = w5;
        o[6] = w6;
        o[7] = w7;
        o[8] = w8;
        o[9] = w9;
        o[10] = w10;
        o[11] = w11;
        o[12] = w12;
        o[13] = w13;
        o[14] = w14;
        o[15] = w15;
        o[16] = w16;
        o[17] = w17;
        o[18] = w18;
        if (c !== 0) {
          o[19] = c;
          out.length++;
        }
        return out;
      };

      // Polyfill comb
      if (!Math.imul) {
        comb10MulTo = smallMulTo;
      }

      function bigMulTo(self, num, out) {
        out.negative = num.negative ^ self.negative;
        out.length = self.length + num.length;

        var carry = 0;
        var hncarry = 0;
        for (var k = 0; k < out.length - 1; k++) {
          // Sum all words with the same `i + j = k` and accumulate `ncarry`,
          // note that ncarry could be >= 0x3ffffff
          var ncarry = hncarry;
          hncarry = 0;
          var rword = carry & 0x3ffffff;
          var maxJ = Math.min(k, num.length - 1);
          for (var j = Math.max(0, k - self.length + 1); j <= maxJ; j++) {
            var i = k - j;
            var a = self.words[i] | 0;
            var b = num.words[j] | 0;
            var r = a * b;

            var lo = r & 0x3ffffff;
            ncarry = (ncarry + ((r / 0x4000000) | 0)) | 0;
            lo = (lo + rword) | 0;
            rword = lo & 0x3ffffff;
            ncarry = (ncarry + (lo >>> 26)) | 0;

            hncarry += ncarry >>> 26;
            ncarry &= 0x3ffffff;
          }
          out.words[k] = rword;
          carry = ncarry;
          ncarry = hncarry;
        }
        if (carry !== 0) {
          out.words[k] = carry;
        } else {
          out.length--;
        }

        return out.strip();
      }

      function jumboMulTo(self, num, out) {
        var fftm = new FFTM();
        return fftm.mulp(self, num, out);
      }

      BN.prototype.mulTo = function mulTo(num, out) {
        var res;
        var len = this.length + num.length;
        if (this.length === 10 && num.length === 10) {
          res = comb10MulTo(this, num, out);
        } else if (len < 63) {
          res = smallMulTo(this, num, out);
        } else if (len < 1024) {
          res = bigMulTo(this, num, out);
        } else {
          res = jumboMulTo(this, num, out);
        }

        return res;
      };

      // Cooley-Tukey algorithm for FFT
      // slightly revisited to rely on looping instead of recursion

      function FFTM(x, y) {
        this.x = x;
        this.y = y;
      }

      FFTM.prototype.makeRBT = function makeRBT(N) {
        var t = new Array(N);
        var l = BN.prototype._countBits(N) - 1;
        for (var i = 0; i < N; i++) {
          t[i] = this.revBin(i, l, N);
        }

        return t;
      };

      // Returns binary-reversed representation of `x`
      FFTM.prototype.revBin = function revBin(x, l, N) {
        if (x === 0 || x === N - 1) return x;

        var rb = 0;
        for (var i = 0; i < l; i++) {
          rb |= (x & 1) << (l - i - 1);
          x >>= 1;
        }

        return rb;
      };

      // Performs "tweedling" phase, therefore 'emulating'
      // behaviour of the recursive algorithm
      FFTM.prototype.permute = function permute(rbt, rws, iws, rtws, itws, N) {
        for (var i = 0; i < N; i++) {
          rtws[i] = rws[rbt[i]];
          itws[i] = iws[rbt[i]];
        }
      };

      FFTM.prototype.transform = function transform(rws, iws, rtws, itws, N, rbt) {
        this.permute(rbt, rws, iws, rtws, itws, N);

        for (var s = 1; s < N; s <<= 1) {
          var l = s << 1;

          var rtwdf = Math.cos(2 * Math.PI / l);
          var itwdf = Math.sin(2 * Math.PI / l);

          for (var p = 0; p < N; p += l) {
            var rtwdf_ = rtwdf;
            var itwdf_ = itwdf;

            for (var j = 0; j < s; j++) {
              var re = rtws[p + j];
              var ie = itws[p + j];

              var ro = rtws[p + j + s];
              var io = itws[p + j + s];

              var rx = rtwdf_ * ro - itwdf_ * io;

              io = rtwdf_ * io + itwdf_ * ro;
              ro = rx;

              rtws[p + j] = re + ro;
              itws[p + j] = ie + io;

              rtws[p + j + s] = re - ro;
              itws[p + j + s] = ie - io;

              /* jshint maxdepth : false */
              if (j !== l) {
                rx = rtwdf * rtwdf_ - itwdf * itwdf_;

                itwdf_ = rtwdf * itwdf_ + itwdf * rtwdf_;
                rtwdf_ = rx;
              }
            }
          }
        }
      };

      FFTM.prototype.guessLen13b = function guessLen13b(n, m) {
        var N = Math.max(m, n) | 1;
        var odd = N & 1;
        var i = 0;
        for (N = N / 2 | 0; N; N = N >>> 1) {
          i++;
        }

        return 1 << i + 1 + odd;
      };

      FFTM.prototype.conjugate = function conjugate(rws, iws, N) {
        if (N <= 1) return;

        for (var i = 0; i < N / 2; i++) {
          var t = rws[i];

          rws[i] = rws[N - i - 1];
          rws[N - i - 1] = t;

          t = iws[i];

          iws[i] = -iws[N - i - 1];
          iws[N - i - 1] = -t;
        }
      };

      FFTM.prototype.normalize13b = function normalize13b(ws, N) {
        var carry = 0;
        for (var i = 0; i < N / 2; i++) {
          var w = Math.round(ws[2 * i + 1] / N) * 0x2000 +
            Math.round(ws[2 * i] / N) +
            carry;

          ws[i] = w & 0x3ffffff;

          if (w < 0x4000000) {
            carry = 0;
          } else {
            carry = w / 0x4000000 | 0;
          }
        }

        return ws;
      };

      FFTM.prototype.convert13b = function convert13b(ws, len, rws, N) {
        var carry = 0;
        for (var i = 0; i < len; i++) {
          carry = carry + (ws[i] | 0);

          rws[2 * i] = carry & 0x1fff; carry = carry >>> 13;
          rws[2 * i + 1] = carry & 0x1fff; carry = carry >>> 13;
        }

        // Pad with zeroes
        for (i = 2 * len; i < N; ++i) {
          rws[i] = 0;
        }

        assert(carry === 0);
        assert((carry & ~0x1fff) === 0);
      };

      FFTM.prototype.stub = function stub(N) {
        var ph = new Array(N);
        for (var i = 0; i < N; i++) {
          ph[i] = 0;
        }

        return ph;
      };

      FFTM.prototype.mulp = function mulp(x, y, out) {
        var N = 2 * this.guessLen13b(x.length, y.length);

        var rbt = this.makeRBT(N);

        var _ = this.stub(N);

        var rws = new Array(N);
        var rwst = new Array(N);
        var iwst = new Array(N);

        var nrws = new Array(N);
        var nrwst = new Array(N);
        var niwst = new Array(N);

        var rmws = out.words;
        rmws.length = N;

        this.convert13b(x.words, x.length, rws, N);
        this.convert13b(y.words, y.length, nrws, N);

        this.transform(rws, _, rwst, iwst, N, rbt);
        this.transform(nrws, _, nrwst, niwst, N, rbt);

        for (var i = 0; i < N; i++) {
          var rx = rwst[i] * nrwst[i] - iwst[i] * niwst[i];
          iwst[i] = rwst[i] * niwst[i] + iwst[i] * nrwst[i];
          rwst[i] = rx;
        }

        this.conjugate(rwst, iwst, N);
        this.transform(rwst, iwst, rmws, _, N, rbt);
        this.conjugate(rmws, _, N);
        this.normalize13b(rmws, N);

        out.negative = x.negative ^ y.negative;
        out.length = x.length + y.length;
        return out.strip();
      };

      // Multiply `this` by `num`
      BN.prototype.mul = function mul(num) {
        var out = new BN(null);
        out.words = new Array(this.length + num.length);
        return this.mulTo(num, out);
      };

      // Multiply employing FFT
      BN.prototype.mulf = function mulf(num) {
        var out = new BN(null);
        out.words = new Array(this.length + num.length);
        return jumboMulTo(this, num, out);
      };

      // In-place Multiplication
      BN.prototype.imul = function imul(num) {
        return this.clone().mulTo(num, this);
      };

      BN.prototype.imuln = function imuln(num) {
        assert(typeof num === 'number');
        assert(num < 0x4000000);

        // Carry
        var carry = 0;
        for (var i = 0; i < this.length; i++) {
          var w = (this.words[i] | 0) * num;
          var lo = (w & 0x3ffffff) + (carry & 0x3ffffff);
          carry >>= 26;
          carry += (w / 0x4000000) | 0;
          // NOTE: lo is 27bit maximum
          carry += lo >>> 26;
          this.words[i] = lo & 0x3ffffff;
        }

        if (carry !== 0) {
          this.words[i] = carry;
          this.length++;
        }

        return this;
      };

      BN.prototype.muln = function muln(num) {
        return this.clone().imuln(num);
      };

      // `this` * `this`
      BN.prototype.sqr = function sqr() {
        return this.mul(this);
      };

      // `this` * `this` in-place
      BN.prototype.isqr = function isqr() {
        return this.imul(this.clone());
      };

      // Math.pow(`this`, `num`)
      BN.prototype.pow = function pow(num) {
        var w = toBitArray(num);
        if (w.length === 0) return new BN(1);

        // Skip leading zeroes
        var res = this;
        for (var i = 0; i < w.length; i++, res = res.sqr()) {
          if (w[i] !== 0) break;
        }

        if (++i < w.length) {
          for (var q = res.sqr(); i < w.length; i++, q = q.sqr()) {
            if (w[i] === 0) continue;

            res = res.mul(q);
          }
        }

        return res;
      };

      // Shift-left in-place
      BN.prototype.iushln = function iushln(bits) {
        assert(typeof bits === 'number' && bits >= 0);
        var r = bits % 26;
        var s = (bits - r) / 26;
        var carryMask = (0x3ffffff >>> (26 - r)) << (26 - r);
        var i;

        if (r !== 0) {
          var carry = 0;

          for (i = 0; i < this.length; i++) {
            var newCarry = this.words[i] & carryMask;
            var c = ((this.words[i] | 0) - newCarry) << r;
            this.words[i] = c | carry;
            carry = newCarry >>> (26 - r);
          }

          if (carry) {
            this.words[i] = carry;
            this.length++;
          }
        }

        if (s !== 0) {
          for (i = this.length - 1; i >= 0; i--) {
            this.words[i + s] = this.words[i];
          }

          for (i = 0; i < s; i++) {
            this.words[i] = 0;
          }

          this.length += s;
        }

        return this.strip();
      };

      BN.prototype.ishln = function ishln(bits) {
        // TODO(indutny): implement me
        assert(this.negative === 0);
        return this.iushln(bits);
      };

      // Shift-right in-place
      // NOTE: `hint` is a lowest bit before trailing zeroes
      // NOTE: if `extended` is present - it will be filled with destroyed bits
      BN.prototype.iushrn = function iushrn(bits, hint, extended) {
        assert(typeof bits === 'number' && bits >= 0);
        var h;
        if (hint) {
          h = (hint - (hint % 26)) / 26;
        } else {
          h = 0;
        }

        var r = bits % 26;
        var s = Math.min((bits - r) / 26, this.length);
        var mask = 0x3ffffff ^ ((0x3ffffff >>> r) << r);
        var maskedWords = extended;

        h -= s;
        h = Math.max(0, h);

        // Extended mode, copy masked part
        if (maskedWords) {
          for (var i = 0; i < s; i++) {
            maskedWords.words[i] = this.words[i];
          }
          maskedWords.length = s;
        }

        if (s === 0) {
          // No-op, we should not move anything at all
        } else if (this.length > s) {
          this.length -= s;
          for (i = 0; i < this.length; i++) {
            this.words[i] = this.words[i + s];
          }
        } else {
          this.words[0] = 0;
          this.length = 1;
        }

        var carry = 0;
        for (i = this.length - 1; i >= 0 && (carry !== 0 || i >= h); i--) {
          var word = this.words[i] | 0;
          this.words[i] = (carry << (26 - r)) | (word >>> r);
          carry = word & mask;
        }

        // Push carried bits as a mask
        if (maskedWords && carry !== 0) {
          maskedWords.words[maskedWords.length++] = carry;
        }

        if (this.length === 0) {
          this.words[0] = 0;
          this.length = 1;
        }

        return this.strip();
      };

      BN.prototype.ishrn = function ishrn(bits, hint, extended) {
        // TODO(indutny): implement me
        assert(this.negative === 0);
        return this.iushrn(bits, hint, extended);
      };

      // Shift-left
      BN.prototype.shln = function shln(bits) {
        return this.clone().ishln(bits);
      };

      BN.prototype.ushln = function ushln(bits) {
        return this.clone().iushln(bits);
      };

      // Shift-right
      BN.prototype.shrn = function shrn(bits) {
        return this.clone().ishrn(bits);
      };

      BN.prototype.ushrn = function ushrn(bits) {
        return this.clone().iushrn(bits);
      };

      // Test if n bit is set
      BN.prototype.testn = function testn(bit) {
        assert(typeof bit === 'number' && bit >= 0);
        var r = bit % 26;
        var s = (bit - r) / 26;
        var q = 1 << r;

        // Fast case: bit is much higher than all existing words
        if (this.length <= s) return false;

        // Check bit and return
        var w = this.words[s];

        return !!(w & q);
      };

      // Return only lowers bits of number (in-place)
      BN.prototype.imaskn = function imaskn(bits) {
        assert(typeof bits === 'number' && bits >= 0);
        var r = bits % 26;
        var s = (bits - r) / 26;

        assert(this.negative === 0, 'imaskn works only with positive numbers');

        if (this.length <= s) {
          return this;
        }

        if (r !== 0) {
          s++;
        }
        this.length = Math.min(s, this.length);

        if (r !== 0) {
          var mask = 0x3ffffff ^ ((0x3ffffff >>> r) << r);
          this.words[this.length - 1] &= mask;
        }

        return this.strip();
      };

      // Return only lowers bits of number
      BN.prototype.maskn = function maskn(bits) {
        return this.clone().imaskn(bits);
      };

      // Add plain number `num` to `this`
      BN.prototype.iaddn = function iaddn(num) {
        assert(typeof num === 'number');
        assert(num < 0x4000000);
        if (num < 0) return this.isubn(-num);

        // Possible sign change
        if (this.negative !== 0) {
          if (this.length === 1 && (this.words[0] | 0) < num) {
            this.words[0] = num - (this.words[0] | 0);
            this.negative = 0;
            return this;
          }

          this.negative = 0;
          this.isubn(num);
          this.negative = 1;
          return this;
        }

        // Add without checks
        return this._iaddn(num);
      };

      BN.prototype._iaddn = function _iaddn(num) {
        this.words[0] += num;

        // Carry
        for (var i = 0; i < this.length && this.words[i] >= 0x4000000; i++) {
          this.words[i] -= 0x4000000;
          if (i === this.length - 1) {
            this.words[i + 1] = 1;
          } else {
            this.words[i + 1]++;
          }
        }
        this.length = Math.max(this.length, i + 1);

        return this;
      };

      // Subtract plain number `num` from `this`
      BN.prototype.isubn = function isubn(num) {
        assert(typeof num === 'number');
        assert(num < 0x4000000);
        if (num < 0) return this.iaddn(-num);

        if (this.negative !== 0) {
          this.negative = 0;
          this.iaddn(num);
          this.negative = 1;
          return this;
        }

        this.words[0] -= num;

        if (this.length === 1 && this.words[0] < 0) {
          this.words[0] = -this.words[0];
          this.negative = 1;
        } else {
          // Carry
          for (var i = 0; i < this.length && this.words[i] < 0; i++) {
            this.words[i] += 0x4000000;
            this.words[i + 1] -= 1;
          }
        }

        return this.strip();
      };

      BN.prototype.addn = function addn(num) {
        return this.clone().iaddn(num);
      };

      BN.prototype.subn = function subn(num) {
        return this.clone().isubn(num);
      };

      BN.prototype.iabs = function iabs() {
        this.negative = 0;

        return this;
      };

      BN.prototype.abs = function abs() {
        return this.clone().iabs();
      };

      BN.prototype._ishlnsubmul = function _ishlnsubmul(num, mul, shift) {
        var len = num.length + shift;
        var i;

        this._expand(len);

        var w;
        var carry = 0;
        for (i = 0; i < num.length; i++) {
          w = (this.words[i + shift] | 0) + carry;
          var right = (num.words[i] | 0) * mul;
          w -= right & 0x3ffffff;
          carry = (w >> 26) - ((right / 0x4000000) | 0);
          this.words[i + shift] = w & 0x3ffffff;
        }
        for (; i < this.length - shift; i++) {
          w = (this.words[i + shift] | 0) + carry;
          carry = w >> 26;
          this.words[i + shift] = w & 0x3ffffff;
        }

        if (carry === 0) return this.strip();

        // Subtraction overflow
        assert(carry === -1);
        carry = 0;
        for (i = 0; i < this.length; i++) {
          w = -(this.words[i] | 0) + carry;
          carry = w >> 26;
          this.words[i] = w & 0x3ffffff;
        }
        this.negative = 1;

        return this.strip();
      };

      BN.prototype._wordDiv = function _wordDiv(num, mode) {
        var shift = this.length - num.length;

        var a = this.clone();
        var b = num;

        // Normalize
        var bhi = b.words[b.length - 1] | 0;
        var bhiBits = this._countBits(bhi);
        shift = 26 - bhiBits;
        if (shift !== 0) {
          b = b.ushln(shift);
          a.iushln(shift);
          bhi = b.words[b.length - 1] | 0;
        }

        // Initialize quotient
        var m = a.length - b.length;
        var q;

        if (mode !== 'mod') {
          q = new BN(null);
          q.length = m + 1;
          q.words = new Array(q.length);
          for (var i = 0; i < q.length; i++) {
            q.words[i] = 0;
          }
        }

        var diff = a.clone()._ishlnsubmul(b, 1, m);
        if (diff.negative === 0) {
          a = diff;
          if (q) {
            q.words[m] = 1;
          }
        }

        for (var j = m - 1; j >= 0; j--) {
          var qj = (a.words[b.length + j] | 0) * 0x4000000 +
            (a.words[b.length + j - 1] | 0);

          // NOTE: (qj / bhi) is (0x3ffffff * 0x4000000 + 0x3ffffff) / 0x2000000 max
          // (0x7ffffff)
          qj = Math.min((qj / bhi) | 0, 0x3ffffff);

          a._ishlnsubmul(b, qj, j);
          while (a.negative !== 0) {
            qj--;
            a.negative = 0;
            a._ishlnsubmul(b, 1, j);
            if (!a.isZero()) {
              a.negative ^= 1;
            }
          }
          if (q) {
            q.words[j] = qj;
          }
        }
        if (q) {
          q.strip();
        }
        a.strip();

        // Denormalize
        if (mode !== 'div' && shift !== 0) {
          a.iushrn(shift);
        }

        return {
          div: q || null,
          mod: a
        };
      };

      // NOTE: 1) `mode` can be set to `mod` to request mod only,
      //       to `div` to request div only, or be absent to
      //       request both div & mod
      //       2) `positive` is true if unsigned mod is requested
      BN.prototype.divmod = function divmod(num, mode, positive) {
        assert(!num.isZero());

        if (this.isZero()) {
          return {
            div: new BN(0),
            mod: new BN(0)
          };
        }

        var div, mod, res;
        if (this.negative !== 0 && num.negative === 0) {
          res = this.neg().divmod(num, mode);

          if (mode !== 'mod') {
            div = res.div.neg();
          }

          if (mode !== 'div') {
            mod = res.mod.neg();
            if (positive && mod.negative !== 0) {
              mod.iadd(num);
            }
          }

          return {
            div: div,
            mod: mod
          };
        }

        if (this.negative === 0 && num.negative !== 0) {
          res = this.divmod(num.neg(), mode);

          if (mode !== 'mod') {
            div = res.div.neg();
          }

          return {
            div: div,
            mod: res.mod
          };
        }

        if ((this.negative & num.negative) !== 0) {
          res = this.neg().divmod(num.neg(), mode);

          if (mode !== 'div') {
            mod = res.mod.neg();
            if (positive && mod.negative !== 0) {
              mod.isub(num);
            }
          }

          return {
            div: res.div,
            mod: mod
          };
        }

        // Both numbers are positive at this point

        // Strip both numbers to approximate shift value
        if (num.length > this.length || this.cmp(num) < 0) {
          return {
            div: new BN(0),
            mod: this
          };
        }

        // Very short reduction
        if (num.length === 1) {
          if (mode === 'div') {
            return {
              div: this.divn(num.words[0]),
              mod: null
            };
          }

          if (mode === 'mod') {
            return {
              div: null,
              mod: new BN(this.modn(num.words[0]))
            };
          }

          return {
            div: this.divn(num.words[0]),
            mod: new BN(this.modn(num.words[0]))
          };
        }

        return this._wordDiv(num, mode);
      };

      // Find `this` / `num`
      BN.prototype.div = function div(num) {
        return this.divmod(num, 'div', false).div;
      };

      // Find `this` % `num`
      BN.prototype.mod = function mod(num) {
        return this.divmod(num, 'mod', false).mod;
      };

      BN.prototype.umod = function umod(num) {
        return this.divmod(num, 'mod', true).mod;
      };

      // Find Round(`this` / `num`)
      BN.prototype.divRound = function divRound(num) {
        var dm = this.divmod(num);

        // Fast case - exact division
        if (dm.mod.isZero()) return dm.div;

        var mod = dm.div.negative !== 0 ? dm.mod.isub(num) : dm.mod;

        var half = num.ushrn(1);
        var r2 = num.andln(1);
        var cmp = mod.cmp(half);

        // Round down
        if (cmp < 0 || r2 === 1 && cmp === 0) return dm.div;

        // Round up
        return dm.div.negative !== 0 ? dm.div.isubn(1) : dm.div.iaddn(1);
      };

      BN.prototype.modn = function modn(num) {
        assert(num <= 0x3ffffff);
        var p = (1 << 26) % num;

        var acc = 0;
        for (var i = this.length - 1; i >= 0; i--) {
          acc = (p * acc + (this.words[i] | 0)) % num;
        }

        return acc;
      };

      // In-place division by number
      BN.prototype.idivn = function idivn(num) {
        assert(num <= 0x3ffffff);

        var carry = 0;
        for (var i = this.length - 1; i >= 0; i--) {
          var w = (this.words[i] | 0) + carry * 0x4000000;
          this.words[i] = (w / num) | 0;
          carry = w % num;
        }

        return this.strip();
      };

      BN.prototype.divn = function divn(num) {
        return this.clone().idivn(num);
      };

      BN.prototype.egcd = function egcd(p) {
        assert(p.negative === 0);
        assert(!p.isZero());

        var x = this;
        var y = p.clone();

        if (x.negative !== 0) {
          x = x.umod(p);
        } else {
          x = x.clone();
        }

        // A * x + B * y = x
        var A = new BN(1);
        var B = new BN(0);

        // C * x + D * y = y
        var C = new BN(0);
        var D = new BN(1);

        var g = 0;

        while (x.isEven() && y.isEven()) {
          x.iushrn(1);
          y.iushrn(1);
          ++g;
        }

        var yp = y.clone();
        var xp = x.clone();

        while (!x.isZero()) {
          for (var i = 0, im = 1; (x.words[0] & im) === 0 && i < 26; ++i, im <<= 1);
          if (i > 0) {
            x.iushrn(i);
            while (i-- > 0) {
              if (A.isOdd() || B.isOdd()) {
                A.iadd(yp);
                B.isub(xp);
              }

              A.iushrn(1);
              B.iushrn(1);
            }
          }

          for (var j = 0, jm = 1; (y.words[0] & jm) === 0 && j < 26; ++j, jm <<= 1);
          if (j > 0) {
            y.iushrn(j);
            while (j-- > 0) {
              if (C.isOdd() || D.isOdd()) {
                C.iadd(yp);
                D.isub(xp);
              }

              C.iushrn(1);
              D.iushrn(1);
            }
          }

          if (x.cmp(y) >= 0) {
            x.isub(y);
            A.isub(C);
            B.isub(D);
          } else {
            y.isub(x);
            C.isub(A);
            D.isub(B);
          }
        }

        return {
          a: C,
          b: D,
          gcd: y.iushln(g)
        };
      };

      // This is reduced incarnation of the binary EEA
      // above, designated to invert members of the
      // _prime_ fields F(p) at a maximal speed
      BN.prototype._invmp = function _invmp(p) {
        assert(p.negative === 0);
        assert(!p.isZero());

        var a = this;
        var b = p.clone();

        if (a.negative !== 0) {
          a = a.umod(p);
        } else {
          a = a.clone();
        }

        var x1 = new BN(1);
        var x2 = new BN(0);

        var delta = b.clone();

        while (a.cmpn(1) > 0 && b.cmpn(1) > 0) {
          for (var i = 0, im = 1; (a.words[0] & im) === 0 && i < 26; ++i, im <<= 1);
          if (i > 0) {
            a.iushrn(i);
            while (i-- > 0) {
              if (x1.isOdd()) {
                x1.iadd(delta);
              }

              x1.iushrn(1);
            }
          }

          for (var j = 0, jm = 1; (b.words[0] & jm) === 0 && j < 26; ++j, jm <<= 1);
          if (j > 0) {
            b.iushrn(j);
            while (j-- > 0) {
              if (x2.isOdd()) {
                x2.iadd(delta);
              }

              x2.iushrn(1);
            }
          }

          if (a.cmp(b) >= 0) {
            a.isub(b);
            x1.isub(x2);
          } else {
            b.isub(a);
            x2.isub(x1);
          }
        }

        var res;
        if (a.cmpn(1) === 0) {
          res = x1;
        } else {
          res = x2;
        }

        if (res.cmpn(0) < 0) {
          res.iadd(p);
        }

        return res;
      };

      BN.prototype.gcd = function gcd(num) {
        if (this.isZero()) return num.abs();
        if (num.isZero()) return this.abs();

        var a = this.clone();
        var b = num.clone();
        a.negative = 0;
        b.negative = 0;

        // Remove common factor of two
        for (var shift = 0; a.isEven() && b.isEven(); shift++) {
          a.iushrn(1);
          b.iushrn(1);
        }

        do {
          while (a.isEven()) {
            a.iushrn(1);
          }
          while (b.isEven()) {
            b.iushrn(1);
          }

          var r = a.cmp(b);
          if (r < 0) {
            // Swap `a` and `b` to make `a` always bigger than `b`
            var t = a;
            a = b;
            b = t;
          } else if (r === 0 || b.cmpn(1) === 0) {
            break;
          }

          a.isub(b);
        } while (true);

        return b.iushln(shift);
      };

      // Invert number in the field F(num)
      BN.prototype.invm = function invm(num) {
        return this.egcd(num).a.umod(num);
      };

      BN.prototype.isEven = function isEven() {
        return (this.words[0] & 1) === 0;
      };

      BN.prototype.isOdd = function isOdd() {
        return (this.words[0] & 1) === 1;
      };

      // And first word and num
      BN.prototype.andln = function andln(num) {
        return this.words[0] & num;
      };

      // Increment at the bit position in-line
      BN.prototype.bincn = function bincn(bit) {
        assert(typeof bit === 'number');
        var r = bit % 26;
        var s = (bit - r) / 26;
        var q = 1 << r;

        // Fast case: bit is much higher than all existing words
        if (this.length <= s) {
          this._expand(s + 1);
          this.words[s] |= q;
          return this;
        }

        // Add bit and propagate, if needed
        var carry = q;
        for (var i = s; carry !== 0 && i < this.length; i++) {
          var w = this.words[i] | 0;
          w += carry;
          carry = w >>> 26;
          w &= 0x3ffffff;
          this.words[i] = w;
        }
        if (carry !== 0) {
          this.words[i] = carry;
          this.length++;
        }
        return this;
      };

      BN.prototype.isZero = function isZero() {
        return this.length === 1 && this.words[0] === 0;
      };

      BN.prototype.cmpn = function cmpn(num) {
        var negative = num < 0;

        if (this.negative !== 0 && !negative) return -1;
        if (this.negative === 0 && negative) return 1;

        this.strip();

        var res;
        if (this.length > 1) {
          res = 1;
        } else {
          if (negative) {
            num = -num;
          }

          assert(num <= 0x3ffffff, 'Number is too big');

          var w = this.words[0] | 0;
          res = w === num ? 0 : w < num ? -1 : 1;
        }
        if (this.negative !== 0) return -res | 0;
        return res;
      };

      // Compare two numbers and return:
      // 1 - if `this` > `num`
      // 0 - if `this` == `num`
      // -1 - if `this` < `num`
      BN.prototype.cmp = function cmp(num) {
        if (this.negative !== 0 && num.negative === 0) return -1;
        if (this.negative === 0 && num.negative !== 0) return 1;

        var res = this.ucmp(num);
        if (this.negative !== 0) return -res | 0;
        return res;
      };

      // Unsigned comparison
      BN.prototype.ucmp = function ucmp(num) {
        // At this point both numbers have the same sign
        if (this.length > num.length) return 1;
        if (this.length < num.length) return -1;

        var res = 0;
        for (var i = this.length - 1; i >= 0; i--) {
          var a = this.words[i] | 0;
          var b = num.words[i] | 0;

          if (a === b) continue;
          if (a < b) {
            res = -1;
          } else if (a > b) {
            res = 1;
          }
          break;
        }
        return res;
      };

      BN.prototype.gtn = function gtn(num) {
        return this.cmpn(num) === 1;
      };

      BN.prototype.gt = function gt(num) {
        return this.cmp(num) === 1;
      };

      BN.prototype.gten = function gten(num) {
        return this.cmpn(num) >= 0;
      };

      BN.prototype.gte = function gte(num) {
        return this.cmp(num) >= 0;
      };

      BN.prototype.ltn = function ltn(num) {
        return this.cmpn(num) === -1;
      };

      BN.prototype.lt = function lt(num) {
        return this.cmp(num) === -1;
      };

      BN.prototype.lten = function lten(num) {
        return this.cmpn(num) <= 0;
      };

      BN.prototype.lte = function lte(num) {
        return this.cmp(num) <= 0;
      };

      BN.prototype.eqn = function eqn(num) {
        return this.cmpn(num) === 0;
      };

      BN.prototype.eq = function eq(num) {
        return this.cmp(num) === 0;
      };

      //
      // A reduce context, could be using montgomery or something better, depending
      // on the `m` itself.
      //
      BN.red = function red(num) {
        return new Red(num);
      };

      BN.prototype.toRed = function toRed(ctx) {
        assert(!this.red, 'Already a number in reduction context');
        assert(this.negative === 0, 'red works only with positives');
        return ctx.convertTo(this)._forceRed(ctx);
      };

      BN.prototype.fromRed = function fromRed() {
        assert(this.red, 'fromRed works only with numbers in reduction context');
        return this.red.convertFrom(this);
      };

      BN.prototype._forceRed = function _forceRed(ctx) {
        this.red = ctx;
        return this;
      };

      BN.prototype.forceRed = function forceRed(ctx) {
        assert(!this.red, 'Already a number in reduction context');
        return this._forceRed(ctx);
      };

      BN.prototype.redAdd = function redAdd(num) {
        assert(this.red, 'redAdd works only with red numbers');
        return this.red.add(this, num);
      };

      BN.prototype.redIAdd = function redIAdd(num) {
        assert(this.red, 'redIAdd works only with red numbers');
        return this.red.iadd(this, num);
      };

      BN.prototype.redSub = function redSub(num) {
        assert(this.red, 'redSub works only with red numbers');
        return this.red.sub(this, num);
      };

      BN.prototype.redISub = function redISub(num) {
        assert(this.red, 'redISub works only with red numbers');
        return this.red.isub(this, num);
      };

      BN.prototype.redShl = function redShl(num) {
        assert(this.red, 'redShl works only with red numbers');
        return this.red.shl(this, num);
      };

      BN.prototype.redMul = function redMul(num) {
        assert(this.red, 'redMul works only with red numbers');
        this.red._verify2(this, num);
        return this.red.mul(this, num);
      };

      BN.prototype.redIMul = function redIMul(num) {
        assert(this.red, 'redMul works only with red numbers');
        this.red._verify2(this, num);
        return this.red.imul(this, num);
      };

      BN.prototype.redSqr = function redSqr() {
        assert(this.red, 'redSqr works only with red numbers');
        this.red._verify1(this);
        return this.red.sqr(this);
      };

      BN.prototype.redISqr = function redISqr() {
        assert(this.red, 'redISqr works only with red numbers');
        this.red._verify1(this);
        return this.red.isqr(this);
      };

      // Square root over p
      BN.prototype.redSqrt = function redSqrt() {
        assert(this.red, 'redSqrt works only with red numbers');
        this.red._verify1(this);
        return this.red.sqrt(this);
      };

      BN.prototype.redInvm = function redInvm() {
        assert(this.red, 'redInvm works only with red numbers');
        this.red._verify1(this);
        return this.red.invm(this);
      };

      // Return negative clone of `this` % `red modulo`
      BN.prototype.redNeg = function redNeg() {
        assert(this.red, 'redNeg works only with red numbers');
        this.red._verify1(this);
        return this.red.neg(this);
      };

      BN.prototype.redPow = function redPow(num) {
        assert(this.red && !num.red, 'redPow(normalNum)');
        this.red._verify1(this);
        return this.red.pow(this, num);
      };

      // Prime numbers with efficient reduction
      var primes = {
        k256: null,
        p224: null,
        p192: null,
        p25519: null
      };

      // Pseudo-Mersenne prime
      function MPrime(name, p) {
        // P = 2 ^ N - K
        this.name = name;
        this.p = new BN(p, 16);
        this.n = this.p.bitLength();
        this.k = new BN(1).iushln(this.n).isub(this.p);

        this.tmp = this._tmp();
      }

      MPrime.prototype._tmp = function _tmp() {
        var tmp = new BN(null);
        tmp.words = new Array(Math.ceil(this.n / 13));
        return tmp;
      };

      MPrime.prototype.ireduce = function ireduce(num) {
        // Assumes that `num` is less than `P^2`
        // num = HI * (2 ^ N - K) + HI * K + LO = HI * K + LO (mod P)
        var r = num;
        var rlen;

        do {
          this.split(r, this.tmp);
          r = this.imulK(r);
          r = r.iadd(this.tmp);
          rlen = r.bitLength();
        } while (rlen > this.n);

        var cmp = rlen < this.n ? -1 : r.ucmp(this.p);
        if (cmp === 0) {
          r.words[0] = 0;
          r.length = 1;
        } else if (cmp > 0) {
          r.isub(this.p);
        } else {
          r.strip();
        }

        return r;
      };

      MPrime.prototype.split = function split(input, out) {
        input.iushrn(this.n, 0, out);
      };

      MPrime.prototype.imulK = function imulK(num) {
        return num.imul(this.k);
      };

      function K256() {
        MPrime.call(
          this,
          'k256',
          'ffffffff ffffffff ffffffff ffffffff ffffffff ffffffff fffffffe fffffc2f');
      }
      inherits(K256, MPrime);

      K256.prototype.split = function split(input, output) {
        // 256 = 9 * 26 + 22
        var mask = 0x3fffff;

        var outLen = Math.min(input.length, 9);
        for (var i = 0; i < outLen; i++) {
          output.words[i] = input.words[i];
        }
        output.length = outLen;

        if (input.length <= 9) {
          input.words[0] = 0;
          input.length = 1;
          return;
        }

        // Shift by 9 limbs
        var prev = input.words[9];
        output.words[output.length++] = prev & mask;

        for (i = 10; i < input.length; i++) {
          var next = input.words[i] | 0;
          input.words[i - 10] = ((next & mask) << 4) | (prev >>> 22);
          prev = next;
        }
        prev >>>= 22;
        input.words[i - 10] = prev;
        if (prev === 0 && input.length > 10) {
          input.length -= 10;
        } else {
          input.length -= 9;
        }
      };

      K256.prototype.imulK = function imulK(num) {
        // K = 0x1000003d1 = [ 0x40, 0x3d1 ]
        num.words[num.length] = 0;
        num.words[num.length + 1] = 0;
        num.length += 2;

        // bounded at: 0x40 * 0x3ffffff + 0x3d0 = 0x100000390
        var lo = 0;
        for (var i = 0; i < num.length; i++) {
          var w = num.words[i] | 0;
          lo += w * 0x3d1;
          num.words[i] = lo & 0x3ffffff;
          lo = w * 0x40 + ((lo / 0x4000000) | 0);
        }

        // Fast length reduction
        if (num.words[num.length - 1] === 0) {
          num.length--;
          if (num.words[num.length - 1] === 0) {
            num.length--;
          }
        }
        return num;
      };

      function P224() {
        MPrime.call(
          this,
          'p224',
          'ffffffff ffffffff ffffffff ffffffff 00000000 00000000 00000001');
      }
      inherits(P224, MPrime);

      function P192() {
        MPrime.call(
          this,
          'p192',
          'ffffffff ffffffff ffffffff fffffffe ffffffff ffffffff');
      }
      inherits(P192, MPrime);

      function P25519() {
        // 2 ^ 255 - 19
        MPrime.call(
          this,
          '25519',
          '7fffffffffffffff ffffffffffffffff ffffffffffffffff ffffffffffffffed');
      }
      inherits(P25519, MPrime);

      P25519.prototype.imulK = function imulK(num) {
        // K = 0x13
        var carry = 0;
        for (var i = 0; i < num.length; i++) {
          var hi = (num.words[i] | 0) * 0x13 + carry;
          var lo = hi & 0x3ffffff;
          hi >>>= 26;

          num.words[i] = lo;
          carry = hi;
        }
        if (carry !== 0) {
          num.words[num.length++] = carry;
        }
        return num;
      };

      // Exported mostly for testing purposes, use plain name instead
      BN._prime = function prime(name) {
        // Cached version of prime
        if (primes[name]) return primes[name];

        var prime;
        if (name === 'k256') {
          prime = new K256();
        } else if (name === 'p224') {
          prime = new P224();
        } else if (name === 'p192') {
          prime = new P192();
        } else if (name === 'p25519') {
          prime = new P25519();
        } else {
          throw new Error('Unknown prime ' + name);
        }
        primes[name] = prime;

        return prime;
      };

      //
      // Base reduction engine
      //
      function Red(m) {
        if (typeof m === 'string') {
          var prime = BN._prime(m);
          this.m = prime.p;
          this.prime = prime;
        } else {
          assert(m.gtn(1), 'modulus must be greater than 1');
          this.m = m;
          this.prime = null;
        }
      }

      Red.prototype._verify1 = function _verify1(a) {
        assert(a.negative === 0, 'red works only with positives');
        assert(a.red, 'red works only with red numbers');
      };

      Red.prototype._verify2 = function _verify2(a, b) {
        assert((a.negative | b.negative) === 0, 'red works only with positives');
        assert(a.red && a.red === b.red,
          'red works only with red numbers');
      };

      Red.prototype.imod = function imod(a) {
        if (this.prime) return this.prime.ireduce(a)._forceRed(this);
        return a.umod(this.m)._forceRed(this);
      };

      Red.prototype.neg = function neg(a) {
        if (a.isZero()) {
          return a.clone();
        }

        return this.m.sub(a)._forceRed(this);
      };

      Red.prototype.add = function add(a, b) {
        this._verify2(a, b);

        var res = a.add(b);
        if (res.cmp(this.m) >= 0) {
          res.isub(this.m);
        }
        return res._forceRed(this);
      };

      Red.prototype.iadd = function iadd(a, b) {
        this._verify2(a, b);

        var res = a.iadd(b);
        if (res.cmp(this.m) >= 0) {
          res.isub(this.m);
        }
        return res;
      };

      Red.prototype.sub = function sub(a, b) {
        this._verify2(a, b);

        var res = a.sub(b);
        if (res.cmpn(0) < 0) {
          res.iadd(this.m);
        }
        return res._forceRed(this);
      };

      Red.prototype.isub = function isub(a, b) {
        this._verify2(a, b);

        var res = a.isub(b);
        if (res.cmpn(0) < 0) {
          res.iadd(this.m);
        }
        return res;
      };

      Red.prototype.shl = function shl(a, num) {
        this._verify1(a);
        return this.imod(a.ushln(num));
      };

      Red.prototype.imul = function imul(a, b) {
        this._verify2(a, b);
        return this.imod(a.imul(b));
      };

      Red.prototype.mul = function mul(a, b) {
        this._verify2(a, b);
        return this.imod(a.mul(b));
      };

      Red.prototype.isqr = function isqr(a) {
        return this.imul(a, a.clone());
      };

      Red.prototype.sqr = function sqr(a) {
        return this.mul(a, a);
      };

      Red.prototype.sqrt = function sqrt(a) {
        if (a.isZero()) return a.clone();

        var mod3 = this.m.andln(3);
        assert(mod3 % 2 === 1);

        // Fast case
        if (mod3 === 3) {
          var pow = this.m.add(new BN(1)).iushrn(2);
          return this.pow(a, pow);
        }

        // Tonelli-Shanks algorithm (Totally unoptimized and slow)
        //
        // Find Q and S, that Q * 2 ^ S = (P - 1)
        var q = this.m.subn(1);
        var s = 0;
        while (!q.isZero() && q.andln(1) === 0) {
          s++;
          q.iushrn(1);
        }
        assert(!q.isZero());

        var one = new BN(1).toRed(this);
        var nOne = one.redNeg();

        // Find quadratic non-residue
        // NOTE: Max is such because of generalized Riemann hypothesis.
        var lpow = this.m.subn(1).iushrn(1);
        var z = this.m.bitLength();
        z = new BN(2 * z * z).toRed(this);

        while (this.pow(z, lpow).cmp(nOne) !== 0) {
          z.redIAdd(nOne);
        }

        var c = this.pow(z, q);
        var r = this.pow(a, q.addn(1).iushrn(1));
        var t = this.pow(a, q);
        var m = s;
        while (t.cmp(one) !== 0) {
          var tmp = t;
          for (var i = 0; tmp.cmp(one) !== 0; i++) {
            tmp = tmp.redSqr();
          }
          assert(i < m);
          var b = this.pow(c, new BN(1).iushln(m - i - 1));

          r = r.redMul(b);
          c = b.redSqr();
          t = t.redMul(c);
          m = i;
        }

        return r;
      };

      Red.prototype.invm = function invm(a) {
        var inv = a._invmp(this.m);
        if (inv.negative !== 0) {
          inv.negative = 0;
          return this.imod(inv).redNeg();
        } else {
          return this.imod(inv);
        }
      };

      Red.prototype.pow = function pow(a, num) {
        if (num.isZero()) return new BN(1).toRed(this);
        if (num.cmpn(1) === 0) return a.clone();

        var windowSize = 4;
        var wnd = new Array(1 << windowSize);
        wnd[0] = new BN(1).toRed(this);
        wnd[1] = a;
        for (var i = 2; i < wnd.length; i++) {
          wnd[i] = this.mul(wnd[i - 1], a);
        }

        var res = wnd[0];
        var current = 0;
        var currentLen = 0;
        var start = num.bitLength() % 26;
        if (start === 0) {
          start = 26;
        }

        for (i = num.length - 1; i >= 0; i--) {
          var word = num.words[i];
          for (var j = start - 1; j >= 0; j--) {
            var bit = (word >> j) & 1;
            if (res !== wnd[0]) {
              res = this.sqr(res);
            }

            if (bit === 0 && current === 0) {
              currentLen = 0;
              continue;
            }

            current <<= 1;
            current |= bit;
            currentLen++;
            if (currentLen !== windowSize && (i !== 0 || j !== 0)) continue;

            res = this.mul(res, wnd[current]);
            currentLen = 0;
            current = 0;
          }
          start = 26;
        }

        return res;
      };

      Red.prototype.convertTo = function convertTo(num) {
        var r = num.umod(this.m);

        return r === num ? r.clone() : r;
      };

      Red.prototype.convertFrom = function convertFrom(num) {
        var res = num.clone();
        res.red = null;
        return res;
      };

      //
      // Montgomery method engine
      //

      BN.mont = function mont(num) {
        return new Mont(num);
      };

      function Mont(m) {
        Red.call(this, m);

        this.shift = this.m.bitLength();
        if (this.shift % 26 !== 0) {
          this.shift += 26 - (this.shift % 26);
        }

        this.r = new BN(1).iushln(this.shift);
        this.r2 = this.imod(this.r.sqr());
        this.rinv = this.r._invmp(this.m);

        this.minv = this.rinv.mul(this.r).isubn(1).div(this.m);
        this.minv = this.minv.umod(this.r);
        this.minv = this.r.sub(this.minv);
      }
      inherits(Mont, Red);

      Mont.prototype.convertTo = function convertTo(num) {
        return this.imod(num.ushln(this.shift));
      };

      Mont.prototype.convertFrom = function convertFrom(num) {
        var r = this.imod(num.mul(this.rinv));
        r.red = null;
        return r;
      };

      Mont.prototype.imul = function imul(a, b) {
        if (a.isZero() || b.isZero()) {
          a.words[0] = 0;
          a.length = 1;
          return a;
        }

        var t = a.imul(b);
        var c = t.maskn(this.shift).mul(this.minv).imaskn(this.shift).mul(this.m);
        var u = t.isub(c).iushrn(this.shift);
        var res = u;

        if (u.cmp(this.m) >= 0) {
          res = u.isub(this.m);
        } else if (u.cmpn(0) < 0) {
          res = u.iadd(this.m);
        }

        return res._forceRed(this);
      };

      Mont.prototype.mul = function mul(a, b) {
        if (a.isZero() || b.isZero()) return new BN(0)._forceRed(this);

        var t = a.mul(b);
        var c = t.maskn(this.shift).mul(this.minv).imaskn(this.shift).mul(this.m);
        var u = t.isub(c).iushrn(this.shift);
        var res = u;
        if (u.cmp(this.m) >= 0) {
          res = u.isub(this.m);
        } else if (u.cmpn(0) < 0) {
          res = u.iadd(this.m);
        }

        return res._forceRed(this);
      };

      Mont.prototype.invm = function invm(a) {
        // (AR)^-1 * R^2 = (A^-1 * R^-1) * R^2 = A^-1 * R
        var res = this.imod(a._invmp(this.m).mul(this.r2));
        return res._forceRed(this);
      };
    })(typeof module === 'undefined' || module, this);

  }, { "buffer": 26 }], 25: [function (require, module, exports) {
    var r;

    module.exports = function rand(len) {
      if (!r)
        r = new Rand(null);

      return r.generate(len);
    };

    function Rand(rand) {
      this.rand = rand;
    }
    module.exports.Rand = Rand;

    Rand.prototype.generate = function generate(len) {
      return this._rand(len);
    };

    // Emulate crypto API using randy
    Rand.prototype._rand = function _rand(n) {
      if (this.rand.getBytes)
        return this.rand.getBytes(n);

      var res = new Uint8Array(n);
      for (var i = 0; i < res.length; i++)
        res[i] = this.rand.getByte();
      return res;
    };

    if (typeof self === 'object') {
      if (self.crypto && self.crypto.getRandomValues) {
        // Modern browsers
        Rand.prototype._rand = function _rand(n) {
          var arr = new Uint8Array(n);
          self.crypto.getRandomValues(arr);
          return arr;
        };
      } else if (self.msCrypto && self.msCrypto.getRandomValues) {
        // IE
        Rand.prototype._rand = function _rand(n) {
          var arr = new Uint8Array(n);
          self.msCrypto.getRandomValues(arr);
          return arr;
        };

        // Safari's WebWorkers do not have `crypto`
      } else if (typeof window === 'object') {
        // Old junk
        Rand.prototype._rand = function () {
          throw new Error('Not implemented yet');
        };
      }
    } else {
      // Node.js or Web worker with no crypto support
      try {
        var crypto = require('crypto');
        if (typeof crypto.randomBytes !== 'function')
          throw new Error('Not supported');

        Rand.prototype._rand = function _rand(n) {
          return crypto.randomBytes(n);
        };
      } catch (e) {
      }
    }

  }, { "crypto": 26 }], 26: [function (require, module, exports) {

  }, {}], 27: [function (require, module, exports) {
    // based on the aes implimentation in triple sec
    // https://github.com/keybase/triplesec
    // which is in turn based on the one from crypto-js
    // https://code.google.com/p/crypto-js/

    var Buffer = require('safe-buffer').Buffer

    function asUInt32Array(buf) {
      if (!Buffer.isBuffer(buf)) buf = Buffer.from(buf)

      var len = (buf.length / 4) | 0
      var out = new Array(len)

      for (var i = 0; i < len; i++) {
        out[i] = buf.readUInt32BE(i * 4)
      }

      return out
    }

    function scrubVec(v) {
      for (var i = 0; i < v.length; v++) {
        v[i] = 0
      }
    }

    function cryptBlock(M, keySchedule, SUB_MIX, SBOX, nRounds) {
      var SUB_MIX0 = SUB_MIX[0]
      var SUB_MIX1 = SUB_MIX[1]
      var SUB_MIX2 = SUB_MIX[2]
      var SUB_MIX3 = SUB_MIX[3]

      var s0 = M[0] ^ keySchedule[0]
      var s1 = M[1] ^ keySchedule[1]
      var s2 = M[2] ^ keySchedule[2]
      var s3 = M[3] ^ keySchedule[3]
      var t0, t1, t2, t3
      var ksRow = 4

      for (var round = 1; round < nRounds; round++) {
        t0 = SUB_MIX0[s0 >>> 24] ^ SUB_MIX1[(s1 >>> 16) & 0xff] ^ SUB_MIX2[(s2 >>> 8) & 0xff] ^ SUB_MIX3[s3 & 0xff] ^ keySchedule[ksRow++]
        t1 = SUB_MIX0[s1 >>> 24] ^ SUB_MIX1[(s2 >>> 16) & 0xff] ^ SUB_MIX2[(s3 >>> 8) & 0xff] ^ SUB_MIX3[s0 & 0xff] ^ keySchedule[ksRow++]
        t2 = SUB_MIX0[s2 >>> 24] ^ SUB_MIX1[(s3 >>> 16) & 0xff] ^ SUB_MIX2[(s0 >>> 8) & 0xff] ^ SUB_MIX3[s1 & 0xff] ^ keySchedule[ksRow++]
        t3 = SUB_MIX0[s3 >>> 24] ^ SUB_MIX1[(s0 >>> 16) & 0xff] ^ SUB_MIX2[(s1 >>> 8) & 0xff] ^ SUB_MIX3[s2 & 0xff] ^ keySchedule[ksRow++]
        s0 = t0
        s1 = t1
        s2 = t2
        s3 = t3
      }

      t0 = ((SBOX[s0 >>> 24] << 24) | (SBOX[(s1 >>> 16) & 0xff] << 16) | (SBOX[(s2 >>> 8) & 0xff] << 8) | SBOX[s3 & 0xff]) ^ keySchedule[ksRow++]
      t1 = ((SBOX[s1 >>> 24] << 24) | (SBOX[(s2 >>> 16) & 0xff] << 16) | (SBOX[(s3 >>> 8) & 0xff] << 8) | SBOX[s0 & 0xff]) ^ keySchedule[ksRow++]
      t2 = ((SBOX[s2 >>> 24] << 24) | (SBOX[(s3 >>> 16) & 0xff] << 16) | (SBOX[(s0 >>> 8) & 0xff] << 8) | SBOX[s1 & 0xff]) ^ keySchedule[ksRow++]
      t3 = ((SBOX[s3 >>> 24] << 24) | (SBOX[(s0 >>> 16) & 0xff] << 16) | (SBOX[(s1 >>> 8) & 0xff] << 8) | SBOX[s2 & 0xff]) ^ keySchedule[ksRow++]
      t0 = t0 >>> 0
      t1 = t1 >>> 0
      t2 = t2 >>> 0
      t3 = t3 >>> 0

      return [t0, t1, t2, t3]
    }

    // AES constants
    var RCON = [0x00, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36]
    var G = (function () {
      // Compute double table
      var d = new Array(256)
      for (var j = 0; j < 256; j++) {
        if (j < 128) {
          d[j] = j << 1
        } else {
          d[j] = (j << 1) ^ 0x11b
        }
      }

      var SBOX = []
      var INV_SBOX = []
      var SUB_MIX = [[], [], [], []]
      var INV_SUB_MIX = [[], [], [], []]

      // Walk GF(2^8)
      var x = 0
      var xi = 0
      for (var i = 0; i < 256; ++i) {
        // Compute sbox
        var sx = xi ^ (xi << 1) ^ (xi << 2) ^ (xi << 3) ^ (xi << 4)
        sx = (sx >>> 8) ^ (sx & 0xff) ^ 0x63
        SBOX[x] = sx
        INV_SBOX[sx] = x

        // Compute multiplication
        var x2 = d[x]
        var x4 = d[x2]
        var x8 = d[x4]

        // Compute sub bytes, mix columns tables
        var t = (d[sx] * 0x101) ^ (sx * 0x1010100)
        SUB_MIX[0][x] = (t << 24) | (t >>> 8)
        SUB_MIX[1][x] = (t << 16) | (t >>> 16)
        SUB_MIX[2][x] = (t << 8) | (t >>> 24)
        SUB_MIX[3][x] = t

        // Compute inv sub bytes, inv mix columns tables
        t = (x8 * 0x1010101) ^ (x4 * 0x10001) ^ (x2 * 0x101) ^ (x * 0x1010100)
        INV_SUB_MIX[0][sx] = (t << 24) | (t >>> 8)
        INV_SUB_MIX[1][sx] = (t << 16) | (t >>> 16)
        INV_SUB_MIX[2][sx] = (t << 8) | (t >>> 24)
        INV_SUB_MIX[3][sx] = t

        if (x === 0) {
          x = xi = 1
        } else {
          x = x2 ^ d[d[d[x8 ^ x2]]]
          xi ^= d[d[xi]]
        }
      }

      return {
        SBOX: SBOX,
        INV_SBOX: INV_SBOX,
        SUB_MIX: SUB_MIX,
        INV_SUB_MIX: INV_SUB_MIX
      }
    })()

    function AES(key) {
      this._key = asUInt32Array(key)
      this._reset()
    }

    AES.blockSize = 4 * 4
    AES.keySize = 256 / 8
    AES.prototype.blockSize = AES.blockSize
    AES.prototype.keySize = AES.keySize
    AES.prototype._reset = function () {
      var keyWords = this._key
      var keySize = keyWords.length
      var nRounds = keySize + 6
      var ksRows = (nRounds + 1) * 4

      var keySchedule = []
      for (var k = 0; k < keySize; k++) {
        keySchedule[k] = keyWords[k]
      }

      for (k = keySize; k < ksRows; k++) {
        var t = keySchedule[k - 1]

        if (k % keySize === 0) {
          t = (t << 8) | (t >>> 24)
          t =
            (G.SBOX[t >>> 24] << 24) |
            (G.SBOX[(t >>> 16) & 0xff] << 16) |
            (G.SBOX[(t >>> 8) & 0xff] << 8) |
            (G.SBOX[t & 0xff])

          t ^= RCON[(k / keySize) | 0] << 24
        } else if (keySize > 6 && k % keySize === 4) {
          t =
            (G.SBOX[t >>> 24] << 24) |
            (G.SBOX[(t >>> 16) & 0xff] << 16) |
            (G.SBOX[(t >>> 8) & 0xff] << 8) |
            (G.SBOX[t & 0xff])
        }

        keySchedule[k] = keySchedule[k - keySize] ^ t
      }

      var invKeySchedule = []
      for (var ik = 0; ik < ksRows; ik++) {
        var ksR = ksRows - ik
        var tt = keySchedule[ksR - (ik % 4 ? 0 : 4)]

        if (ik < 4 || ksR <= 4) {
          invKeySchedule[ik] = tt
        } else {
          invKeySchedule[ik] =
            G.INV_SUB_MIX[0][G.SBOX[tt >>> 24]] ^
            G.INV_SUB_MIX[1][G.SBOX[(tt >>> 16) & 0xff]] ^
            G.INV_SUB_MIX[2][G.SBOX[(tt >>> 8) & 0xff]] ^
            G.INV_SUB_MIX[3][G.SBOX[tt & 0xff]]
        }
      }

      this._nRounds = nRounds
      this._keySchedule = keySchedule
      this._invKeySchedule = invKeySchedule
    }

    AES.prototype.encryptBlockRaw = function (M) {
      M = asUInt32Array(M)
      return cryptBlock(M, this._keySchedule, G.SUB_MIX, G.SBOX, this._nRounds)
    }

    AES.prototype.encryptBlock = function (M) {
      var out = this.encryptBlockRaw(M)
      var buf = Buffer.allocUnsafe(16)
      buf.writeUInt32BE(out[0], 0)
      buf.writeUInt32BE(out[1], 4)
      buf.writeUInt32BE(out[2], 8)
      buf.writeUInt32BE(out[3], 12)
      return buf
    }

    AES.prototype.decryptBlock = function (M) {
      M = asUInt32Array(M)

      // swap
      var m1 = M[1]
      M[1] = M[3]
      M[3] = m1

      var out = cryptBlock(M, this._invKeySchedule, G.INV_SUB_MIX, G.INV_SBOX, this._nRounds)
      var buf = Buffer.allocUnsafe(16)
      buf.writeUInt32BE(out[0], 0)
      buf.writeUInt32BE(out[3], 4)
      buf.writeUInt32BE(out[2], 8)
      buf.writeUInt32BE(out[1], 12)
      return buf
    }

    AES.prototype.scrub = function () {
      scrubVec(this._keySchedule)
      scrubVec(this._invKeySchedule)
      scrubVec(this._key)
    }

    module.exports.AES = AES

  }, { "safe-buffer": 643 }], 28: [function (require, module, exports) {
    var aes = require('./aes')
    var Buffer = require('safe-buffer').Buffer
    var Transform = require('cipher-base')
    var inherits = require('inherits')
    var GHASH = require('./ghash')
    var xor = require('buffer-xor')
    var incr32 = require('./incr32')

    function xorTest(a, b) {
      var out = 0
      if (a.length !== b.length) out++

      var len = Math.min(a.length, b.length)
      for (var i = 0; i < len; ++i) {
        out += (a[i] ^ b[i])
      }

      return out
    }

    function calcIv(self, iv, ck) {
      if (iv.length === 12) {
        self._finID = Buffer.concat([iv, Buffer.from([0, 0, 0, 1])])
        return Buffer.concat([iv, Buffer.from([0, 0, 0, 2])])
      }
      var ghash = new GHASH(ck)
      var len = iv.length
      var toPad = len % 16
      ghash.update(iv)
      if (toPad) {
        toPad = 16 - toPad
        ghash.update(Buffer.alloc(toPad, 0))
      }
      ghash.update(Buffer.alloc(8, 0))
      var ivBits = len * 8
      var tail = Buffer.alloc(8)
      tail.writeUIntBE(ivBits, 0, 8)
      ghash.update(tail)
      self._finID = ghash.state
      var out = Buffer.from(self._finID)
      incr32(out)
      return out
    }
    function StreamCipher(mode, key, iv, decrypt) {
      Transform.call(this)

      var h = Buffer.alloc(4, 0)

      this._cipher = new aes.AES(key)
      var ck = this._cipher.encryptBlock(h)
      this._ghash = new GHASH(ck)
      iv = calcIv(this, iv, ck)

      this._prev = Buffer.from(iv)
      this._cache = Buffer.allocUnsafe(0)
      this._secCache = Buffer.allocUnsafe(0)
      this._decrypt = decrypt
      this._alen = 0
      this._len = 0
      this._mode = mode

      this._authTag = null
      this._called = false
    }

    inherits(StreamCipher, Transform)

    StreamCipher.prototype._update = function (chunk) {
      if (!this._called && this._alen) {
        var rump = 16 - (this._alen % 16)
        if (rump < 16) {
          rump = Buffer.alloc(rump, 0)
          this._ghash.update(rump)
        }
      }

      this._called = true
      var out = this._mode.encrypt(this, chunk)
      if (this._decrypt) {
        this._ghash.update(chunk)
      } else {
        this._ghash.update(out)
      }
      this._len += chunk.length
      return out
    }

    StreamCipher.prototype._final = function () {
      if (this._decrypt && !this._authTag) throw new Error('Unsupported state or unable to authenticate data')

      var tag = xor(this._ghash.final(this._alen * 8, this._len * 8), this._cipher.encryptBlock(this._finID))
      if (this._decrypt && xorTest(tag, this._authTag)) throw new Error('Unsupported state or unable to authenticate data')

      this._authTag = tag
      this._cipher.scrub()
    }

    StreamCipher.prototype.getAuthTag = function getAuthTag() {
      if (this._decrypt || !Buffer.isBuffer(this._authTag)) throw new Error('Attempting to get auth tag in unsupported state')

      return this._authTag
    }

    StreamCipher.prototype.setAuthTag = function setAuthTag(tag) {
      if (!this._decrypt) throw new Error('Attempting to set auth tag in unsupported state')

      this._authTag = tag
    }

    StreamCipher.prototype.setAAD = function setAAD(buf) {
      if (this._called) throw new Error('Attempting to set AAD in unsupported state')

      this._ghash.update(buf)
      this._alen += buf.length
    }

    module.exports = StreamCipher

  }, { "./aes": 27, "./ghash": 32, "./incr32": 33, "buffer-xor": 57, "cipher-base": 61, "inherits": 569, "safe-buffer": 643 }], 29: [function (require, module, exports) {
    var ciphers = require('./encrypter')
    var deciphers = require('./decrypter')
    var modes = require('./modes/list.json')

    function getCiphers() {
      return Object.keys(modes)
    }

    exports.createCipher = exports.Cipher = ciphers.createCipher
    exports.createCipheriv = exports.Cipheriv = ciphers.createCipheriv
    exports.createDecipher = exports.Decipher = deciphers.createDecipher
    exports.createDecipheriv = exports.Decipheriv = deciphers.createDecipheriv
    exports.listCiphers = exports.getCiphers = getCiphers

  }, { "./decrypter": 30, "./encrypter": 31, "./modes/list.json": 41 }], 30: [function (require, module, exports) {
    var AuthCipher = require('./authCipher')
    var Buffer = require('safe-buffer').Buffer
    var MODES = require('./modes')
    var StreamCipher = require('./streamCipher')
    var Transform = require('cipher-base')
    var aes = require('./aes')
    var ebtk = require('evp_bytestokey')
    var inherits = require('inherits')

    function Decipher(mode, key, iv) {
      Transform.call(this)

      this._cache = new Splitter()
      this._last = void 0
      this._cipher = new aes.AES(key)
      this._prev = Buffer.from(iv)
      this._mode = mode
      this._autopadding = true
    }

    inherits(Decipher, Transform)

    Decipher.prototype._update = function (data) {
      this._cache.add(data)
      var chunk
      var thing
      var out = []
      while ((chunk = this._cache.get(this._autopadding))) {
        thing = this._mode.decrypt(this, chunk)
        out.push(thing)
      }
      return Buffer.concat(out)
    }

    Decipher.prototype._final = function () {
      var chunk = this._cache.flush()
      if (this._autopadding) {
        return unpad(this._mode.decrypt(this, chunk))
      } else if (chunk) {
        throw new Error('data not multiple of block length')
      }
    }

    Decipher.prototype.setAutoPadding = function (setTo) {
      this._autopadding = !!setTo
      return this
    }

    function Splitter() {
      this.cache = Buffer.allocUnsafe(0)
    }

    Splitter.prototype.add = function (data) {
      this.cache = Buffer.concat([this.cache, data])
    }

    Splitter.prototype.get = function (autoPadding) {
      var out
      if (autoPadding) {
        if (this.cache.length > 16) {
          out = this.cache.slice(0, 16)
          this.cache = this.cache.slice(16)
          return out
        }
      } else {
        if (this.cache.length >= 16) {
          out = this.cache.slice(0, 16)
          this.cache = this.cache.slice(16)
          return out
        }
      }

      return null
    }

    Splitter.prototype.flush = function () {
      if (this.cache.length) return this.cache
    }

    function unpad(last) {
      var padded = last[15]
      if (padded < 1 || padded > 16) {
        throw new Error('unable to decrypt data')
      }
      var i = -1
      while (++i < padded) {
        if (last[(i + (16 - padded))] !== padded) {
          throw new Error('unable to decrypt data')
        }
      }
      if (padded === 16) return

      return last.slice(0, 16 - padded)
    }

    function createDecipheriv(suite, password, iv) {
      var config = MODES[suite.toLowerCase()]
      if (!config) throw new TypeError('invalid suite type')

      if (typeof iv === 'string') iv = Buffer.from(iv)
      if (config.mode !== 'GCM' && iv.length !== config.iv) throw new TypeError('invalid iv length ' + iv.length)

      if (typeof password === 'string') password = Buffer.from(password)
      if (password.length !== config.key / 8) throw new TypeError('invalid key length ' + password.length)

      if (config.type === 'stream') {
        return new StreamCipher(config.module, password, iv, true)
      } else if (config.type === 'auth') {
        return new AuthCipher(config.module, password, iv, true)
      }

      return new Decipher(config.module, password, iv)
    }

    function createDecipher(suite, password) {
      var config = MODES[suite.toLowerCase()]
      if (!config) throw new TypeError('invalid suite type')

      var keys = ebtk(password, false, config.key, config.iv)
      return createDecipheriv(suite, keys.key, keys.iv)
    }

    exports.createDecipher = createDecipher
    exports.createDecipheriv = createDecipheriv

  }, { "./aes": 27, "./authCipher": 28, "./modes": 40, "./streamCipher": 43, "cipher-base": 61, "evp_bytestokey": 483, "inherits": 569, "safe-buffer": 643 }], 31: [function (require, module, exports) {
    var MODES = require('./modes')
    var AuthCipher = require('./authCipher')
    var Buffer = require('safe-buffer').Buffer
    var StreamCipher = require('./streamCipher')
    var Transform = require('cipher-base')
    var aes = require('./aes')
    var ebtk = require('evp_bytestokey')
    var inherits = require('inherits')

    function Cipher(mode, key, iv) {
      Transform.call(this)

      this._cache = new Splitter()
      this._cipher = new aes.AES(key)
      this._prev = Buffer.from(iv)
      this._mode = mode
      this._autopadding = true
    }

    inherits(Cipher, Transform)

    Cipher.prototype._update = function (data) {
      this._cache.add(data)
      var chunk
      var thing
      var out = []

      while ((chunk = this._cache.get())) {
        thing = this._mode.encrypt(this, chunk)
        out.push(thing)
      }

      return Buffer.concat(out)
    }

    var PADDING = Buffer.alloc(16, 0x10)

    Cipher.prototype._final = function () {
      var chunk = this._cache.flush()
      if (this._autopadding) {
        chunk = this._mode.encrypt(this, chunk)
        this._cipher.scrub()
        return chunk
      }

      if (!chunk.equals(PADDING)) {
        this._cipher.scrub()
        throw new Error('data not multiple of block length')
      }
    }

    Cipher.prototype.setAutoPadding = function (setTo) {
      this._autopadding = !!setTo
      return this
    }

    function Splitter() {
      this.cache = Buffer.allocUnsafe(0)
    }

    Splitter.prototype.add = function (data) {
      this.cache = Buffer.concat([this.cache, data])
    }

    Splitter.prototype.get = function () {
      if (this.cache.length > 15) {
        var out = this.cache.slice(0, 16)
        this.cache = this.cache.slice(16)
        return out
      }
      return null
    }

    Splitter.prototype.flush = function () {
      var len = 16 - this.cache.length
      var padBuff = Buffer.allocUnsafe(len)

      var i = -1
      while (++i < len) {
        padBuff.writeUInt8(len, i)
      }

      return Buffer.concat([this.cache, padBuff])
    }

    function createCipheriv(suite, password, iv) {
      var config = MODES[suite.toLowerCase()]
      if (!config) throw new TypeError('invalid suite type')

      if (typeof password === 'string') password = Buffer.from(password)
      if (password.length !== config.key / 8) throw new TypeError('invalid key length ' + password.length)

      if (typeof iv === 'string') iv = Buffer.from(iv)
      if (config.mode !== 'GCM' && iv.length !== config.iv) throw new TypeError('invalid iv length ' + iv.length)

      if (config.type === 'stream') {
        return new StreamCipher(config.module, password, iv)
      } else if (config.type === 'auth') {
        return new AuthCipher(config.module, password, iv)
      }

      return new Cipher(config.module, password, iv)
    }

    function createCipher(suite, password) {
      var config = MODES[suite.toLowerCase()]
      if (!config) throw new TypeError('invalid suite type')

      var keys = ebtk(password, false, config.key, config.iv)
      return createCipheriv(suite, keys.key, keys.iv)
    }

    exports.createCipheriv = createCipheriv
    exports.createCipher = createCipher

  }, { "./aes": 27, "./authCipher": 28, "./modes": 40, "./streamCipher": 43, "cipher-base": 61, "evp_bytestokey": 483, "inherits": 569, "safe-buffer": 643 }], 32: [function (require, module, exports) {
    var Buffer = require('safe-buffer').Buffer
    var ZEROES = Buffer.alloc(16, 0)

    function toArray(buf) {
      return [
        buf.readUInt32BE(0),
        buf.readUInt32BE(4),
        buf.readUInt32BE(8),
        buf.readUInt32BE(12)
      ]
    }

    function fromArray(out) {
      var buf = Buffer.allocUnsafe(16)
      buf.writeUInt32BE(out[0] >>> 0, 0)
      buf.writeUInt32BE(out[1] >>> 0, 4)
      buf.writeUInt32BE(out[2] >>> 0, 8)
      buf.writeUInt32BE(out[3] >>> 0, 12)
      return buf
    }

    function GHASH(key) {
      this.h = key
      this.state = Buffer.alloc(16, 0)
      this.cache = Buffer.allocUnsafe(0)
    }

    // from http://bitwiseshiftleft.github.io/sjcl/doc/symbols/src/core_gcm.js.html
    // by Juho Vähä-Herttua
    GHASH.prototype.ghash = function (block) {
      var i = -1
      while (++i < block.length) {
        this.state[i] ^= block[i]
      }
      this._multiply()
    }

    GHASH.prototype._multiply = function () {
      var Vi = toArray(this.h)
      var Zi = [0, 0, 0, 0]
      var j, xi, lsbVi
      var i = -1
      while (++i < 128) {
        xi = (this.state[~~(i / 8)] & (1 << (7 - (i % 8)))) !== 0
        if (xi) {
          // Z_i+1 = Z_i ^ V_i
          Zi[0] ^= Vi[0]
          Zi[1] ^= Vi[1]
          Zi[2] ^= Vi[2]
          Zi[3] ^= Vi[3]
        }

        // Store the value of LSB(V_i)
        lsbVi = (Vi[3] & 1) !== 0

        // V_i+1 = V_i >> 1
        for (j = 3; j > 0; j--) {
          Vi[j] = (Vi[j] >>> 1) | ((Vi[j - 1] & 1) << 31)
        }
        Vi[0] = Vi[0] >>> 1

        // If LSB(V_i) is 1, V_i+1 = (V_i >> 1) ^ R
        if (lsbVi) {
          Vi[0] = Vi[0] ^ (0xe1 << 24)
        }
      }
      this.state = fromArray(Zi)
    }

    GHASH.prototype.update = function (buf) {
      this.cache = Buffer.concat([this.cache, buf])
      var chunk
      while (this.cache.length >= 16) {
        chunk = this.cache.slice(0, 16)
        this.cache = this.cache.slice(16)
        this.ghash(chunk)
      }
    }

    GHASH.prototype.final = function (abl, bl) {
      if (this.cache.length) {
        this.ghash(Buffer.concat([this.cache, ZEROES], 16))
      }

      this.ghash(fromArray([0, abl, 0, bl]))
      return this.state
    }

    module.exports = GHASH

  }, { "safe-buffer": 643 }], 33: [function (require, module, exports) {
    function incr32(iv) {
      var len = iv.length
      var item
      while (len--) {
        item = iv.readUInt8(len)
        if (item === 255) {
          iv.writeUInt8(0, len)
        } else {
          item++
          iv.writeUInt8(item, len)
          break
        }
      }
    }
    module.exports = incr32

  }, {}], 34: [function (require, module, exports) {
    var xor = require('buffer-xor')

    exports.encrypt = function (self, block) {
      var data = xor(block, self._prev)

      self._prev = self._cipher.encryptBlock(data)
      return self._prev
    }

    exports.decrypt = function (self, block) {
      var pad = self._prev

      self._prev = block
      var out = self._cipher.decryptBlock(block)

      return xor(out, pad)
    }

  }, { "buffer-xor": 57 }], 35: [function (require, module, exports) {
    var Buffer = require('safe-buffer').Buffer
    var xor = require('buffer-xor')

    function encryptStart(self, data, decrypt) {
      var len = data.length
      var out = xor(data, self._cache)
      self._cache = self._cache.slice(len)
      self._prev = Buffer.concat([self._prev, decrypt ? data : out])
      return out
    }

    exports.encrypt = function (self, data, decrypt) {
      var out = Buffer.allocUnsafe(0)
      var len

      while (data.length) {
        if (self._cache.length === 0) {
          self._cache = self._cipher.encryptBlock(self._prev)
          self._prev = Buffer.allocUnsafe(0)
        }

        if (self._cache.length <= data.length) {
          len = self._cache.length
          out = Buffer.concat([out, encryptStart(self, data.slice(0, len), decrypt)])
          data = data.slice(len)
        } else {
          out = Buffer.concat([out, encryptStart(self, data, decrypt)])
          break
        }
      }

      return out
    }

  }, { "buffer-xor": 57, "safe-buffer": 643 }], 36: [function (require, module, exports) {
    var Buffer = require('safe-buffer').Buffer

    function encryptByte(self, byteParam, decrypt) {
      var pad
      var i = -1
      var len = 8
      var out = 0
      var bit, value
      while (++i < len) {
        pad = self._cipher.encryptBlock(self._prev)
        bit = (byteParam & (1 << (7 - i))) ? 0x80 : 0
        value = pad[0] ^ bit
        out += ((value & 0x80) >> (i % 8))
        self._prev = shiftIn(self._prev, decrypt ? bit : value)
      }
      return out
    }

    function shiftIn(buffer, value) {
      var len = buffer.length
      var i = -1
      var out = Buffer.allocUnsafe(buffer.length)
      buffer = Buffer.concat([buffer, Buffer.from([value])])

      while (++i < len) {
        out[i] = buffer[i] << 1 | buffer[i + 1] >> (7)
      }

      return out
    }

    exports.encrypt = function (self, chunk, decrypt) {
      var len = chunk.length
      var out = Buffer.allocUnsafe(len)
      var i = -1

      while (++i < len) {
        out[i] = encryptByte(self, chunk[i], decrypt)
      }

      return out
    }

  }, { "safe-buffer": 643 }], 37: [function (require, module, exports) {
    var Buffer = require('safe-buffer').Buffer

    function encryptByte(self, byteParam, decrypt) {
      var pad = self._cipher.encryptBlock(self._prev)
      var out = pad[0] ^ byteParam

      self._prev = Buffer.concat([
        self._prev.slice(1),
        Buffer.from([decrypt ? byteParam : out])
      ])

      return out
    }

    exports.encrypt = function (self, chunk, decrypt) {
      var len = chunk.length
      var out = Buffer.allocUnsafe(len)
      var i = -1

      while (++i < len) {
        out[i] = encryptByte(self, chunk[i], decrypt)
      }

      return out
    }

  }, { "safe-buffer": 643 }], 38: [function (require, module, exports) {
    var xor = require('buffer-xor')
    var Buffer = require('safe-buffer').Buffer
    var incr32 = require('../incr32')

    function getBlock(self) {
      var out = self._cipher.encryptBlockRaw(self._prev)
      incr32(self._prev)
      return out
    }

    var blockSize = 16
    exports.encrypt = function (self, chunk) {
      var chunkNum = Math.ceil(chunk.length / blockSize)
      var start = self._cache.length
      self._cache = Buffer.concat([
        self._cache,
        Buffer.allocUnsafe(chunkNum * blockSize)
      ])
      for (var i = 0; i < chunkNum; i++) {
        var out = getBlock(self)
        var offset = start + i * blockSize
        self._cache.writeUInt32BE(out[0], offset + 0)
        self._cache.writeUInt32BE(out[1], offset + 4)
        self._cache.writeUInt32BE(out[2], offset + 8)
        self._cache.writeUInt32BE(out[3], offset + 12)
      }
      var pad = self._cache.slice(0, chunk.length)
      self._cache = self._cache.slice(chunk.length)
      return xor(chunk, pad)
    }

  }, { "../incr32": 33, "buffer-xor": 57, "safe-buffer": 643 }], 39: [function (require, module, exports) {
    exports.encrypt = function (self, block) {
      return self._cipher.encryptBlock(block)
    }

    exports.decrypt = function (self, block) {
      return self._cipher.decryptBlock(block)
    }

  }, {}], 40: [function (require, module, exports) {
    var modeModules = {
      ECB: require('./ecb'),
      CBC: require('./cbc'),
      CFB: require('./cfb'),
      CFB8: require('./cfb8'),
      CFB1: require('./cfb1'),
      OFB: require('./ofb'),
      CTR: require('./ctr'),
      GCM: require('./ctr')
    }

    var modes = require('./list.json')

    for (var key in modes) {
      modes[key].module = modeModules[modes[key].mode]
    }

    module.exports = modes

  }, { "./cbc": 34, "./cfb": 35, "./cfb1": 36, "./cfb8": 37, "./ctr": 38, "./ecb": 39, "./list.json": 41, "./ofb": 42 }], 41: [function (require, module, exports) {
    module.exports = {
      "aes-128-ecb": {
        "cipher": "AES",
        "key": 128,
        "iv": 0,
        "mode": "ECB",
        "type": "block"
      },
      "aes-192-ecb": {
        "cipher": "AES",
        "key": 192,
        "iv": 0,
        "mode": "ECB",
        "type": "block"
      },
      "aes-256-ecb": {
        "cipher": "AES",
        "key": 256,
        "iv": 0,
        "mode": "ECB",
        "type": "block"
      },
      "aes-128-cbc": {
        "cipher": "AES",
        "key": 128,
        "iv": 16,
        "mode": "CBC",
        "type": "block"
      },
      "aes-192-cbc": {
        "cipher": "AES",
        "key": 192,
        "iv": 16,
        "mode": "CBC",
        "type": "block"
      },
      "aes-256-cbc": {
        "cipher": "AES",
        "key": 256,
        "iv": 16,
        "mode": "CBC",
        "type": "block"
      },
      "aes128": {
        "cipher": "AES",
        "key": 128,
        "iv": 16,
        "mode": "CBC",
        "type": "block"
      },
      "aes192": {
        "cipher": "AES",
        "key": 192,
        "iv": 16,
        "mode": "CBC",
        "type": "block"
      },
      "aes256": {
        "cipher": "AES",
        "key": 256,
        "iv": 16,
        "mode": "CBC",
        "type": "block"
      },
      "aes-128-cfb": {
        "cipher": "AES",
        "key": 128,
        "iv": 16,
        "mode": "CFB",
        "type": "stream"
      },
      "aes-192-cfb": {
        "cipher": "AES",
        "key": 192,
        "iv": 16,
        "mode": "CFB",
        "type": "stream"
      },
      "aes-256-cfb": {
        "cipher": "AES",
        "key": 256,
        "iv": 16,
        "mode": "CFB",
        "type": "stream"
      },
      "aes-128-cfb8": {
        "cipher": "AES",
        "key": 128,
        "iv": 16,
        "mode": "CFB8",
        "type": "stream"
      },
      "aes-192-cfb8": {
        "cipher": "AES",
        "key": 192,
        "iv": 16,
        "mode": "CFB8",
        "type": "stream"
      },
      "aes-256-cfb8": {
        "cipher": "AES",
        "key": 256,
        "iv": 16,
        "mode": "CFB8",
        "type": "stream"
      },
      "aes-128-cfb1": {
        "cipher": "AES",
        "key": 128,
        "iv": 16,
        "mode": "CFB1",
        "type": "stream"
      },
      "aes-192-cfb1": {
        "cipher": "AES",
        "key": 192,
        "iv": 16,
        "mode": "CFB1",
        "type": "stream"
      },
      "aes-256-cfb1": {
        "cipher": "AES",
        "key": 256,
        "iv": 16,
        "mode": "CFB1",
        "type": "stream"
      },
      "aes-128-ofb": {
        "cipher": "AES",
        "key": 128,
        "iv": 16,
        "mode": "OFB",
        "type": "stream"
      },
      "aes-192-ofb": {
        "cipher": "AES",
        "key": 192,
        "iv": 16,
        "mode": "OFB",
        "type": "stream"
      },
      "aes-256-ofb": {
        "cipher": "AES",
        "key": 256,
        "iv": 16,
        "mode": "OFB",
        "type": "stream"
      },
      "aes-128-ctr": {
        "cipher": "AES",
        "key": 128,
        "iv": 16,
        "mode": "CTR",
        "type": "stream"
      },
      "aes-192-ctr": {
        "cipher": "AES",
        "key": 192,
        "iv": 16,
        "mode": "CTR",
        "type": "stream"
      },
      "aes-256-ctr": {
        "cipher": "AES",
        "key": 256,
        "iv": 16,
        "mode": "CTR",
        "type": "stream"
      },
      "aes-128-gcm": {
        "cipher": "AES",
        "key": 128,
        "iv": 12,
        "mode": "GCM",
        "type": "auth"
      },
      "aes-192-gcm": {
        "cipher": "AES",
        "key": 192,
        "iv": 12,
        "mode": "GCM",
        "type": "auth"
      },
      "aes-256-gcm": {
        "cipher": "AES",
        "key": 256,
        "iv": 12,
        "mode": "GCM",
        "type": "auth"
      }
    }

  }, {}], 42: [function (require, module, exports) {
    (function (Buffer) {
      var xor = require('buffer-xor')

      function getBlock(self) {
        self._prev = self._cipher.encryptBlock(self._prev)
        return self._prev
      }

      exports.encrypt = function (self, chunk) {
        while (self._cache.length < chunk.length) {
          self._cache = Buffer.concat([self._cache, getBlock(self)])
        }

        var pad = self._cache.slice(0, chunk.length)
        self._cache = self._cache.slice(chunk.length)
        return xor(chunk, pad)
      }

    }).call(this, require("buffer").Buffer)
  }, { "buffer": 58, "buffer-xor": 57 }], 43: [function (require, module, exports) {
    var aes = require('./aes')
    var Buffer = require('safe-buffer').Buffer
    var Transform = require('cipher-base')
    var inherits = require('inherits')

    function StreamCipher(mode, key, iv, decrypt) {
      Transform.call(this)

      this._cipher = new aes.AES(key)
      this._prev = Buffer.from(iv)
      this._cache = Buffer.allocUnsafe(0)
      this._secCache = Buffer.allocUnsafe(0)
      this._decrypt = decrypt
      this._mode = mode
    }

    inherits(StreamCipher, Transform)

    StreamCipher.prototype._update = function (chunk) {
      return this._mode.encrypt(this, chunk, this._decrypt)
    }

    StreamCipher.prototype._final = function () {
      this._cipher.scrub()
    }

    module.exports = StreamCipher

  }, { "./aes": 27, "cipher-base": 61, "inherits": 569, "safe-buffer": 643 }], 44: [function (require, module, exports) {
    var DES = require('browserify-des')
    var aes = require('browserify-aes/browser')
    var aesModes = require('browserify-aes/modes')
    var desModes = require('browserify-des/modes')
    var ebtk = require('evp_bytestokey')

    function createCipher(suite, password) {
      suite = suite.toLowerCase()

      var keyLen, ivLen
      if (aesModes[suite]) {
        keyLen = aesModes[suite].key
        ivLen = aesModes[suite].iv
      } else if (desModes[suite]) {
        keyLen = desModes[suite].key * 8
        ivLen = desModes[suite].iv
      } else {
        throw new TypeError('invalid suite type')
      }

      var keys = ebtk(password, false, keyLen, ivLen)
      return createCipheriv(suite, keys.key, keys.iv)
    }

    function createDecipher(suite, password) {
      suite = suite.toLowerCase()

      var keyLen, ivLen
      if (aesModes[suite]) {
        keyLen = aesModes[suite].key
        ivLen = aesModes[suite].iv
      } else if (desModes[suite]) {
        keyLen = desModes[suite].key * 8
        ivLen = desModes[suite].iv
      } else {
        throw new TypeError('invalid suite type')
      }

      var keys = ebtk(password, false, keyLen, ivLen)
      return createDecipheriv(suite, keys.key, keys.iv)
    }

    function createCipheriv(suite, key, iv) {
      suite = suite.toLowerCase()
      if (aesModes[suite]) return aes.createCipheriv(suite, key, iv)
      if (desModes[suite]) return new DES({ key: key, iv: iv, mode: suite })

      throw new TypeError('invalid suite type')
    }

    function createDecipheriv(suite, key, iv) {
      suite = suite.toLowerCase()
      if (aesModes[suite]) return aes.createDecipheriv(suite, key, iv)
      if (desModes[suite]) return new DES({ key: key, iv: iv, mode: suite, decrypt: true })

      throw new TypeError('invalid suite type')
    }

    function getCiphers() {
      return Object.keys(desModes).concat(aes.getCiphers())
    }

    exports.createCipher = exports.Cipher = createCipher
    exports.createCipheriv = exports.Cipheriv = createCipheriv
    exports.createDecipher = exports.Decipher = createDecipher
    exports.createDecipheriv = exports.Decipheriv = createDecipheriv
    exports.listCiphers = exports.getCiphers = getCiphers

  }, { "browserify-aes/browser": 29, "browserify-aes/modes": 40, "browserify-des": 45, "browserify-des/modes": 46, "evp_bytestokey": 483 }], 45: [function (require, module, exports) {
    var CipherBase = require('cipher-base')
    var des = require('des.js')
    var inherits = require('inherits')
    var Buffer = require('safe-buffer').Buffer

    var modes = {
      'des-ede3-cbc': des.CBC.instantiate(des.EDE),
      'des-ede3': des.EDE,
      'des-ede-cbc': des.CBC.instantiate(des.EDE),
      'des-ede': des.EDE,
      'des-cbc': des.CBC.instantiate(des.DES),
      'des-ecb': des.DES
    }
    modes.des = modes['des-cbc']
    modes.des3 = modes['des-ede3-cbc']
    module.exports = DES
    inherits(DES, CipherBase)
    function DES(opts) {
      CipherBase.call(this)
      var modeName = opts.mode.toLowerCase()
      var mode = modes[modeName]
      var type
      if (opts.decrypt) {
        type = 'decrypt'
      } else {
        type = 'encrypt'
      }
      var key = opts.key
      if (!Buffer.isBuffer(key)) {
        key = Buffer.from(key)
      }
      if (modeName === 'des-ede' || modeName === 'des-ede-cbc') {
        key = Buffer.concat([key, key.slice(0, 8)])
      }
      var iv = opts.iv
      if (!Buffer.isBuffer(iv)) {
        iv = Buffer.from(iv)
      }
      this._des = mode.create({
        key: key,
        iv: iv,
        type: type
      })
    }
    DES.prototype._update = function (data) {
      return Buffer.from(this._des.update(data))
    }
    DES.prototype._final = function () {
      return Buffer.from(this._des.final())
    }

  }, { "cipher-base": 61, "des.js": 78, "inherits": 569, "safe-buffer": 643 }], 46: [function (require, module, exports) {
    exports['des-ecb'] = {
      key: 8,
      iv: 0
    }
    exports['des-cbc'] = exports.des = {
      key: 8,
      iv: 8
    }
    exports['des-ede3-cbc'] = exports.des3 = {
      key: 24,
      iv: 8
    }
    exports['des-ede3'] = {
      key: 24,
      iv: 0
    }
    exports['des-ede-cbc'] = {
      key: 16,
      iv: 8
    }
    exports['des-ede'] = {
      key: 16,
      iv: 0
    }

  }, {}], 47: [function (require, module, exports) {
    (function (Buffer) {
      var bn = require('bn.js');
      var randomBytes = require('randombytes');
      module.exports = crt;
      function blind(priv) {
        var r = getr(priv);
        var blinder = r.toRed(bn.mont(priv.modulus))
          .redPow(new bn(priv.publicExponent)).fromRed();
        return {
          blinder: blinder,
          unblinder: r.invm(priv.modulus)
        };
      }
      function crt(msg, priv) {
        var blinds = blind(priv);
        var len = priv.modulus.byteLength();
        var mod = bn.mont(priv.modulus);
        var blinded = new bn(msg).mul(blinds.blinder).umod(priv.modulus);
        var c1 = blinded.toRed(bn.mont(priv.prime1));
        var c2 = blinded.toRed(bn.mont(priv.prime2));
        var qinv = priv.coefficient;
        var p = priv.prime1;
        var q = priv.prime2;
        var m1 = c1.redPow(priv.exponent1);
        var m2 = c2.redPow(priv.exponent2);
        m1 = m1.fromRed();
        m2 = m2.fromRed();
        var h = m1.isub(m2).imul(qinv).umod(p);
        h.imul(q);
        m2.iadd(h);
        return new Buffer(m2.imul(blinds.unblinder).umod(priv.modulus).toArray(false, len));
      }
      crt.getr = getr;
      function getr(priv) {
        var len = priv.modulus.byteLength();
        var r = new bn(randomBytes(len));
        while (r.cmp(priv.modulus) >= 0 || !r.umod(priv.prime1) || !r.umod(priv.prime2)) {
          r = new bn(randomBytes(len));
        }
        return r;
      }

    }).call(this, require("buffer").Buffer)
  }, { "bn.js": 24, "buffer": 58, "randombytes": 626 }], 48: [function (require, module, exports) {
    module.exports = require('./browser/algorithms.json')

  }, { "./browser/algorithms.json": 49 }], 49: [function (require, module, exports) {
    module.exports = {
      "sha224WithRSAEncryption": {
        "sign": "rsa",
        "hash": "sha224",
        "id": "302d300d06096086480165030402040500041c"
      },
      "RSA-SHA224": {
        "sign": "ecdsa/rsa",
        "hash": "sha224",
        "id": "302d300d06096086480165030402040500041c"
      },
      "sha256WithRSAEncryption": {
        "sign": "rsa",
        "hash": "sha256",
        "id": "3031300d060960864801650304020105000420"
      },
      "RSA-SHA256": {
        "sign": "ecdsa/rsa",
        "hash": "sha256",
        "id": "3031300d060960864801650304020105000420"
      },
      "sha384WithRSAEncryption": {
        "sign": "rsa",
        "hash": "sha384",
        "id": "3041300d060960864801650304020205000430"
      },
      "RSA-SHA384": {
        "sign": "ecdsa/rsa",
        "hash": "sha384",
        "id": "3041300d060960864801650304020205000430"
      },
      "sha512WithRSAEncryption": {
        "sign": "rsa",
        "hash": "sha512",
        "id": "3051300d060960864801650304020305000440"
      },
      "RSA-SHA512": {
        "sign": "ecdsa/rsa",
        "hash": "sha512",
        "id": "3051300d060960864801650304020305000440"
      },
      "RSA-SHA1": {
        "sign": "rsa",
        "hash": "sha1",
        "id": "3021300906052b0e03021a05000414"
      },
      "ecdsa-with-SHA1": {
        "sign": "ecdsa",
        "hash": "sha1",
        "id": ""
      },
      "sha256": {
        "sign": "ecdsa",
        "hash": "sha256",
        "id": ""
      },
      "sha224": {
        "sign": "ecdsa",
        "hash": "sha224",
        "id": ""
      },
      "sha384": {
        "sign": "ecdsa",
        "hash": "sha384",
        "id": ""
      },
      "sha512": {
        "sign": "ecdsa",
        "hash": "sha512",
        "id": ""
      },
      "DSA-SHA": {
        "sign": "dsa",
        "hash": "sha1",
        "id": ""
      },
      "DSA-SHA1": {
        "sign": "dsa",
        "hash": "sha1",
        "id": ""
      },
      "DSA": {
        "sign": "dsa",
        "hash": "sha1",
        "id": ""
      },
      "DSA-WITH-SHA224": {
        "sign": "dsa",
        "hash": "sha224",
        "id": ""
      },
      "DSA-SHA224": {
        "sign": "dsa",
        "hash": "sha224",
        "id": ""
      },
      "DSA-WITH-SHA256": {
        "sign": "dsa",
        "hash": "sha256",
        "id": ""
      },
      "DSA-SHA256": {
        "sign": "dsa",
        "hash": "sha256",
        "id": ""
      },
      "DSA-WITH-SHA384": {
        "sign": "dsa",
        "hash": "sha384",
        "id": ""
      },
      "DSA-SHA384": {
        "sign": "dsa",
        "hash": "sha384",
        "id": ""
      },
      "DSA-WITH-SHA512": {
        "sign": "dsa",
        "hash": "sha512",
        "id": ""
      },
      "DSA-SHA512": {
        "sign": "dsa",
        "hash": "sha512",
        "id": ""
      },
      "DSA-RIPEMD160": {
        "sign": "dsa",
        "hash": "rmd160",
        "id": ""
      },
      "ripemd160WithRSA": {
        "sign": "rsa",
        "hash": "rmd160",
        "id": "3021300906052b2403020105000414"
      },
      "RSA-RIPEMD160": {
        "sign": "rsa",
        "hash": "rmd160",
        "id": "3021300906052b2403020105000414"
      },
      "md5WithRSAEncryption": {
        "sign": "rsa",
        "hash": "md5",
        "id": "3020300c06082a864886f70d020505000410"
      },
      "RSA-MD5": {
        "sign": "rsa",
        "hash": "md5",
        "id": "3020300c06082a864886f70d020505000410"
      }
    }

  }, {}], 50: [function (require, module, exports) {
    module.exports = {
      "1.3.132.0.10": "secp256k1",
      "1.3.132.0.33": "p224",
      "1.2.840.10045.3.1.1": "p192",
      "1.2.840.10045.3.1.7": "p256",
      "1.3.132.0.34": "p384",
      "1.3.132.0.35": "p521"
    }

  }, {}], 51: [function (require, module, exports) {
    (function (Buffer) {
      var createHash = require('create-hash')
      var stream = require('stream')
      var inherits = require('inherits')
      var sign = require('./sign')
      var verify = require('./verify')

      var algorithms = require('./algorithms.json')
      Object.keys(algorithms).forEach(function (key) {
        algorithms[key].id = new Buffer(algorithms[key].id, 'hex')
        algorithms[key.toLowerCase()] = algorithms[key]
      })

      function Sign(algorithm) {
        stream.Writable.call(this)

        var data = algorithms[algorithm]
        if (!data) throw new Error('Unknown message digest')

        this._hashType = data.hash
        this._hash = createHash(data.hash)
        this._tag = data.id
        this._signType = data.sign
      }
      inherits(Sign, stream.Writable)

      Sign.prototype._write = function _write(data, _, done) {
        this._hash.update(data)
        done()
      }

      Sign.prototype.update = function update(data, enc) {
        if (typeof data === 'string') data = new Buffer(data, enc)

        this._hash.update(data)
        return this
      }

      Sign.prototype.sign = function signMethod(key, enc) {
        this.end()
        var hash = this._hash.digest()
        var sig = sign(hash, key, this._hashType, this._signType, this._tag)

        return enc ? sig.toString(enc) : sig
      }

      function Verify(algorithm) {
        stream.Writable.call(this)

        var data = algorithms[algorithm]
        if (!data) throw new Error('Unknown message digest')

        this._hash = createHash(data.hash)
        this._tag = data.id
        this._signType = data.sign
      }
      inherits(Verify, stream.Writable)

      Verify.prototype._write = function _write(data, _, done) {
        this._hash.update(data)
        done()
      }

      Verify.prototype.update = function update(data, enc) {
        if (typeof data === 'string') data = new Buffer(data, enc)

        this._hash.update(data)
        return this
      }

      Verify.prototype.verify = function verifyMethod(key, sig, enc) {
        if (typeof sig === 'string') sig = new Buffer(sig, enc)

        this.end()
        var hash = this._hash.digest()
        return verify(sig, hash, key, this._signType, this._tag)
      }

      function createSign(algorithm) {
        return new Sign(algorithm)
      }

      function createVerify(algorithm) {
        return new Verify(algorithm)
      }

      module.exports = {
        Sign: createSign,
        Verify: createVerify,
        createSign: createSign,
        createVerify: createVerify
      }

    }).call(this, require("buffer").Buffer)
  }, { "./algorithms.json": 49, "./sign": 52, "./verify": 53, "buffer": 58, "create-hash": 71, "inherits": 569, "stream": 671 }], 52: [function (require, module, exports) {
    (function (Buffer) {
      // much of this based on https://github.com/indutny/self-signed/blob/gh-pages/lib/rsa.js
      var createHmac = require('create-hmac')
      var crt = require('browserify-rsa')
      var EC = require('elliptic').ec
      var BN = require('bn.js')
      var parseKeys = require('parse-asn1')
      var curves = require('./curves.json')

      function sign(hash, key, hashType, signType, tag) {
        var priv = parseKeys(key)
        if (priv.curve) {
          // rsa keys can be interpreted as ecdsa ones in openssl
          if (signType !== 'ecdsa' && signType !== 'ecdsa/rsa') throw new Error('wrong private key type')
          return ecSign(hash, priv)
        } else if (priv.type === 'dsa') {
          if (signType !== 'dsa') throw new Error('wrong private key type')
          return dsaSign(hash, priv, hashType)
        } else {
          if (signType !== 'rsa' && signType !== 'ecdsa/rsa') throw new Error('wrong private key type')
        }
        hash = Buffer.concat([tag, hash])
        var len = priv.modulus.byteLength()
        var pad = [0, 1]
        while (hash.length + pad.length + 1 < len) pad.push(0xff)
        pad.push(0x00)
        var i = -1
        while (++i < hash.length) pad.push(hash[i])

        var out = crt(pad, priv)
        return out
      }

      function ecSign(hash, priv) {
        var curveId = curves[priv.curve.join('.')]
        if (!curveId) throw new Error('unknown curve ' + priv.curve.join('.'))

        var curve = new EC(curveId)
        var key = curve.keyFromPrivate(priv.privateKey)
        var out = key.sign(hash)

        return new Buffer(out.toDER())
      }

      function dsaSign(hash, priv, algo) {
        var x = priv.params.priv_key
        var p = priv.params.p
        var q = priv.params.q
        var g = priv.params.g
        var r = new BN(0)
        var k
        var H = bits2int(hash, q).mod(q)
        var s = false
        var kv = getKey(x, q, hash, algo)
        while (s === false) {
          k = makeKey(q, kv, algo)
          r = makeR(g, k, p, q)
          s = k.invm(q).imul(H.add(x.mul(r))).mod(q)
          if (s.cmpn(0) === 0) {
            s = false
            r = new BN(0)
          }
        }
        return toDER(r, s)
      }

      function toDER(r, s) {
        r = r.toArray()
        s = s.toArray()

        // Pad values
        if (r[0] & 0x80) r = [0].concat(r)
        if (s[0] & 0x80) s = [0].concat(s)

        var total = r.length + s.length + 4
        var res = [0x30, total, 0x02, r.length]
        res = res.concat(r, [0x02, s.length], s)
        return new Buffer(res)
      }

      function getKey(x, q, hash, algo) {
        x = new Buffer(x.toArray())
        if (x.length < q.byteLength()) {
          var zeros = new Buffer(q.byteLength() - x.length)
          zeros.fill(0)
          x = Buffer.concat([zeros, x])
        }
        var hlen = hash.length
        var hbits = bits2octets(hash, q)
        var v = new Buffer(hlen)
        v.fill(1)
        var k = new Buffer(hlen)
        k.fill(0)
        k = createHmac(algo, k).update(v).update(new Buffer([0])).update(x).update(hbits).digest()
        v = createHmac(algo, k).update(v).digest()
        k = createHmac(algo, k).update(v).update(new Buffer([1])).update(x).update(hbits).digest()
        v = createHmac(algo, k).update(v).digest()
        return { k: k, v: v }
      }

      function bits2int(obits, q) {
        var bits = new BN(obits)
        var shift = (obits.length << 3) - q.bitLength()
        if (shift > 0) bits.ishrn(shift)
        return bits
      }

      function bits2octets(bits, q) {
        bits = bits2int(bits, q)
        bits = bits.mod(q)
        var out = new Buffer(bits.toArray())
        if (out.length < q.byteLength()) {
          var zeros = new Buffer(q.byteLength() - out.length)
          zeros.fill(0)
          out = Buffer.concat([zeros, out])
        }
        return out
      }

      function makeKey(q, kv, algo) {
        var t
        var k

        do {
          t = new Buffer(0)

          while (t.length * 8 < q.bitLength()) {
            kv.v = createHmac(algo, kv.k).update(kv.v).digest()
            t = Buffer.concat([t, kv.v])
          }

          k = bits2int(t, q)
          kv.k = createHmac(algo, kv.k).update(kv.v).update(new Buffer([0])).digest()
          kv.v = createHmac(algo, kv.k).update(kv.v).digest()
        } while (k.cmp(q) !== -1)

        return k
      }

      function makeR(g, k, p, q) {
        return g.toRed(BN.mont(p)).redPow(k).fromRed().mod(q)
      }

      module.exports = sign
      module.exports.getKey = getKey
      module.exports.makeKey = makeKey

    }).call(this, require("buffer").Buffer)
  }, { "./curves.json": 50, "bn.js": 24, "browserify-rsa": 47, "buffer": 58, "create-hmac": 73, "elliptic": 463, "parse-asn1": 604 }], 53: [function (require, module, exports) {
    (function (Buffer) {
      // much of this based on https://github.com/indutny/self-signed/blob/gh-pages/lib/rsa.js
      var BN = require('bn.js')
      var EC = require('elliptic').ec
      var parseKeys = require('parse-asn1')
      var curves = require('./curves.json')

      function verify(sig, hash, key, signType, tag) {
        var pub = parseKeys(key)
        if (pub.type === 'ec') {
          // rsa keys can be interpreted as ecdsa ones in openssl
          if (signType !== 'ecdsa' && signType !== 'ecdsa/rsa') throw new Error('wrong public key type')
          return ecVerify(sig, hash, pub)
        } else if (pub.type === 'dsa') {
          if (signType !== 'dsa') throw new Error('wrong public key type')
          return dsaVerify(sig, hash, pub)
        } else {
          if (signType !== 'rsa' && signType !== 'ecdsa/rsa') throw new Error('wrong public key type')
        }
        hash = Buffer.concat([tag, hash])
        var len = pub.modulus.byteLength()
        var pad = [1]
        var padNum = 0
        while (hash.length + pad.length + 2 < len) {
          pad.push(0xff)
          padNum++
        }
        pad.push(0x00)
        var i = -1
        while (++i < hash.length) {
          pad.push(hash[i])
        }
        pad = new Buffer(pad)
        var red = BN.mont(pub.modulus)
        sig = new BN(sig).toRed(red)

        sig = sig.redPow(new BN(pub.publicExponent))
        sig = new Buffer(sig.fromRed().toArray())
        var out = padNum < 8 ? 1 : 0
        len = Math.min(sig.length, pad.length)
        if (sig.length !== pad.length) out = 1

        i = -1
        while (++i < len) out |= sig[i] ^ pad[i]
        return out === 0
      }

      function ecVerify(sig, hash, pub) {
        var curveId = curves[pub.data.algorithm.curve.join('.')]
        if (!curveId) throw new Error('unknown curve ' + pub.data.algorithm.curve.join('.'))

        var curve = new EC(curveId)
        var pubkey = pub.data.subjectPrivateKey.data

        return curve.verify(hash, sig, pubkey)
      }

      function dsaVerify(sig, hash, pub) {
        var p = pub.data.p
        var q = pub.data.q
        var g = pub.data.g
        var y = pub.data.pub_key
        var unpacked = parseKeys.signature.decode(sig, 'der')
        var s = unpacked.s
        var r = unpacked.r
        checkValue(s, q)
        checkValue(r, q)
        var montp = BN.mont(p)
        var w = s.invm(q)
        var v = g.toRed(montp)
          .redPow(new BN(hash).mul(w).mod(q))
          .fromRed()
          .mul(y.toRed(montp).redPow(r.mul(w).mod(q)).fromRed())
          .mod(p)
          .mod(q)
        return v.cmp(r) === 0
      }

      function checkValue(b, q) {
        if (b.cmpn(0) <= 0) throw new Error('invalid sig')
        if (b.cmp(q) >= q) throw new Error('invalid sig')
      }

      module.exports = verify

    }).call(this, require("buffer").Buffer)
  }, { "./curves.json": 50, "bn.js": 24, "buffer": 58, "elliptic": 463, "parse-asn1": 604 }], 54: [function (require, module, exports) {
    arguments[4][26][0].apply(exports, arguments)
  }, { "dup": 26 }], 55: [function (require, module, exports) {
    (function (global) {
      /*! https://mths.be/punycode v1.4.1 by @mathias */
      ; (function (root) {

        /** Detect free variables */
        var freeExports = typeof exports == 'object' && exports &&
          !exports.nodeType && exports;
        var freeModule = typeof module == 'object' && module &&
          !module.nodeType && module;
        var freeGlobal = typeof global == 'object' && global;
        if (
          freeGlobal.global === freeGlobal ||
          freeGlobal.window === freeGlobal ||
          freeGlobal.self === freeGlobal
        ) {
          root = freeGlobal;
        }

        /**
         * The `punycode` object.
         * @name punycode
         * @type Object
         */
        var punycode,

          /** Highest positive signed 32-bit float value */
          maxInt = 2147483647, // aka. 0x7FFFFFFF or 2^31-1

          /** Bootstring parameters */
          base = 36,
          tMin = 1,
          tMax = 26,
          skew = 38,
          damp = 700,
          initialBias = 72,
          initialN = 128, // 0x80
          delimiter = '-', // '\x2D'

          /** Regular expressions */
          regexPunycode = /^xn--/,
          regexNonASCII = /[^\x20-\x7E]/, // unprintable ASCII chars + non-ASCII chars
          regexSeparators = /[\x2E\u3002\uFF0E\uFF61]/g, // RFC 3490 separators

          /** Error messages */
          errors = {
            'overflow': 'Overflow: input needs wider integers to process',
            'not-basic': 'Illegal input >= 0x80 (not a basic code point)',
            'invalid-input': 'Invalid input'
          },

          /** Convenience shortcuts */
          baseMinusTMin = base - tMin,
          floor = Math.floor,
          stringFromCharCode = String.fromCharCode,

          /** Temporary variable */
          key;

        /*--------------------------------------------------------------------------*/

        /**
         * A generic error utility function.
         * @private
         * @param {String} type The error type.
         * @returns {Error} Throws a `RangeError` with the applicable error message.
         */
        function error(type) {
          throw new RangeError(errors[type]);
        }

        /**
         * A generic `Array#map` utility function.
         * @private
         * @param {Array} array The array to iterate over.
         * @param {Function} callback The function that gets called for every array
         * item.
         * @returns {Array} A new array of values returned by the callback function.
         */
        function map(array, fn) {
          var length = array.length;
          var result = [];
          while (length--) {
            result[length] = fn(array[length]);
          }
          return result;
        }

        /**
         * A simple `Array#map`-like wrapper to work with domain name strings or email
         * addresses.
         * @private
         * @param {String} domain The domain name or email address.
         * @param {Function} callback The function that gets called for every
         * character.
         * @returns {Array} A new string of characters returned by the callback
         * function.
         */
        function mapDomain(string, fn) {
          var parts = string.split('@');
          var result = '';
          if (parts.length > 1) {
            // In email addresses, only the domain name should be punycoded. Leave
            // the local part (i.e. everything up to `@`) intact.
            result = parts[0] + '@';
            string = parts[1];
          }
          // Avoid `split(regex)` for IE8 compatibility. See #17.
          string = string.replace(regexSeparators, '\x2E');
          var labels = string.split('.');
          var encoded = map(labels, fn).join('.');
          return result + encoded;
        }

        /**
         * Creates an array containing the numeric code points of each Unicode
         * character in the string. While JavaScript uses UCS-2 internally,
         * this function will convert a pair of surrogate halves (each of which
         * UCS-2 exposes as separate characters) into a single code point,
         * matching UTF-16.
         * @see `punycode.ucs2.encode`
         * @see <https://mathiasbynens.be/notes/javascript-encoding>
         * @memberOf punycode.ucs2
         * @name decode
         * @param {String} string The Unicode input string (UCS-2).
         * @returns {Array} The new array of code points.
         */
        function ucs2decode(string) {
          var output = [],
            counter = 0,
            length = string.length,
            value,
            extra;
          while (counter < length) {
            value = string.charCodeAt(counter++);
            if (value >= 0xD800 && value <= 0xDBFF && counter < length) {
              // high surrogate, and there is a next character
              extra = string.charCodeAt(counter++);
              if ((extra & 0xFC00) == 0xDC00) { // low surrogate
                output.push(((value & 0x3FF) << 10) + (extra & 0x3FF) + 0x10000);
              } else {
                // unmatched surrogate; only append this code unit, in case the next
                // code unit is the high surrogate of a surrogate pair
                output.push(value);
                counter--;
              }
            } else {
              output.push(value);
            }
          }
          return output;
        }

        /**
         * Creates a string based on an array of numeric code points.
         * @see `punycode.ucs2.decode`
         * @memberOf punycode.ucs2
         * @name encode
         * @param {Array} codePoints The array of numeric code points.
         * @returns {String} The new Unicode string (UCS-2).
         */
        function ucs2encode(array) {
          return map(array, function (value) {
            var output = '';
            if (value > 0xFFFF) {
              value -= 0x10000;
              output += stringFromCharCode(value >>> 10 & 0x3FF | 0xD800);
              value = 0xDC00 | value & 0x3FF;
            }
            output += stringFromCharCode(value);
            return output;
          }).join('');
        }

        /**
         * Converts a basic code point into a digit/integer.
         * @see `digitToBasic()`
         * @private
         * @param {Number} codePoint The basic numeric code point value.
         * @returns {Number} The numeric value of a basic code point (for use in
         * representing integers) in the range `0` to `base - 1`, or `base` if
         * the code point does not represent a value.
         */
        function basicToDigit(codePoint) {
          if (codePoint - 48 < 10) {
            return codePoint - 22;
          }
          if (codePoint - 65 < 26) {
            return codePoint - 65;
          }
          if (codePoint - 97 < 26) {
            return codePoint - 97;
          }
          return base;
        }

        /**
         * Converts a digit/integer into a basic code point.
         * @see `basicToDigit()`
         * @private
         * @param {Number} digit The numeric value of a basic code point.
         * @returns {Number} The basic code point whose value (when used for
         * representing integers) is `digit`, which needs to be in the range
         * `0` to `base - 1`. If `flag` is non-zero, the uppercase form is
         * used; else, the lowercase form is used. The behavior is undefined
         * if `flag` is non-zero and `digit` has no uppercase form.
         */
        function digitToBasic(digit, flag) {
          //  0..25 map to ASCII a..z or A..Z
          // 26..35 map to ASCII 0..9
          return digit + 22 + 75 * (digit < 26) - ((flag != 0) << 5);
        }

        /**
         * Bias adaptation function as per section 3.4 of RFC 3492.
         * https://tools.ietf.org/html/rfc3492#section-3.4
         * @private
         */
        function adapt(delta, numPoints, firstTime) {
          var k = 0;
          delta = firstTime ? floor(delta / damp) : delta >> 1;
          delta += floor(delta / numPoints);
          for (/* no initialization */; delta > baseMinusTMin * tMax >> 1; k += base) {
            delta = floor(delta / baseMinusTMin);
          }
          return floor(k + (baseMinusTMin + 1) * delta / (delta + skew));
        }

        /**
         * Converts a Punycode string of ASCII-only symbols to a string of Unicode
         * symbols.
         * @memberOf punycode
         * @param {String} input The Punycode string of ASCII-only symbols.
         * @returns {String} The resulting string of Unicode symbols.
         */
        function decode(input) {
          // Don't use UCS-2
          var output = [],
            inputLength = input.length,
            out,
            i = 0,
            n = initialN,
            bias = initialBias,
            basic,
            j,
            index,
            oldi,
            w,
            k,
            digit,
            t,
            /** Cached calculation results */
            baseMinusT;

          // Handle the basic code points: let `basic` be the number of input code
          // points before the last delimiter, or `0` if there is none, then copy
          // the first basic code points to the output.

          basic = input.lastIndexOf(delimiter);
          if (basic < 0) {
            basic = 0;
          }

          for (j = 0; j < basic; ++j) {
            // if it's not a basic code point
            if (input.charCodeAt(j) >= 0x80) {
              error('not-basic');
            }
            output.push(input.charCodeAt(j));
          }

          // Main decoding loop: start just after the last delimiter if any basic code
          // points were copied; start at the beginning otherwise.

          for (index = basic > 0 ? basic + 1 : 0; index < inputLength; /* no final expression */) {

            // `index` is the index of the next character to be consumed.
            // Decode a generalized variable-length integer into `delta`,
            // which gets added to `i`. The overflow checking is easier
            // if we increase `i` as we go, then subtract off its starting
            // value at the end to obtain `delta`.
            for (oldi = i, w = 1, k = base; /* no condition */; k += base) {

              if (index >= inputLength) {
                error('invalid-input');
              }

              digit = basicToDigit(input.charCodeAt(index++));

              if (digit >= base || digit > floor((maxInt - i) / w)) {
                error('overflow');
              }

              i += digit * w;
              t = k <= bias ? tMin : (k >= bias + tMax ? tMax : k - bias);

              if (digit < t) {
                break;
              }

              baseMinusT = base - t;
              if (w > floor(maxInt / baseMinusT)) {
                error('overflow');
              }

              w *= baseMinusT;

            }

            out = output.length + 1;
            bias = adapt(i - oldi, out, oldi == 0);

            // `i` was supposed to wrap around from `out` to `0`,
            // incrementing `n` each time, so we'll fix that now:
            if (floor(i / out) > maxInt - n) {
              error('overflow');
            }

            n += floor(i / out);
            i %= out;

            // Insert `n` at position `i` of the output
            output.splice(i++, 0, n);

          }

          return ucs2encode(output);
        }

        /**
         * Converts a string of Unicode symbols (e.g. a domain name label) to a
         * Punycode string of ASCII-only symbols.
         * @memberOf punycode
         * @param {String} input The string of Unicode symbols.
         * @returns {String} The resulting Punycode string of ASCII-only symbols.
         */
        function encode(input) {
          var n,
            delta,
            handledCPCount,
            basicLength,
            bias,
            j,
            m,
            q,
            k,
            t,
            currentValue,
            output = [],
            /** `inputLength` will hold the number of code points in `input`. */
            inputLength,
            /** Cached calculation results */
            handledCPCountPlusOne,
            baseMinusT,
            qMinusT;

          // Convert the input in UCS-2 to Unicode
          input = ucs2decode(input);

          // Cache the length
          inputLength = input.length;

          // Initialize the state
          n = initialN;
          delta = 0;
          bias = initialBias;

          // Handle the basic code points
          for (j = 0; j < inputLength; ++j) {
            currentValue = input[j];
            if (currentValue < 0x80) {
              output.push(stringFromCharCode(currentValue));
            }
          }

          handledCPCount = basicLength = output.length;

          // `handledCPCount` is the number of code points that have been handled;
          // `basicLength` is the number of basic code points.

          // Finish the basic string - if it is not empty - with a delimiter
          if (basicLength) {
            output.push(delimiter);
          }

          // Main encoding loop:
          while (handledCPCount < inputLength) {

            // All non-basic code points < n have been handled already. Find the next
            // larger one:
            for (m = maxInt, j = 0; j < inputLength; ++j) {
              currentValue = input[j];
              if (currentValue >= n && currentValue < m) {
                m = currentValue;
              }
            }

            // Increase `delta` enough to advance the decoder's <n,i> state to <m,0>,
            // but guard against overflow
            handledCPCountPlusOne = handledCPCount + 1;
            if (m - n > floor((maxInt - delta) / handledCPCountPlusOne)) {
              error('overflow');
            }

            delta += (m - n) * handledCPCountPlusOne;
            n = m;

            for (j = 0; j < inputLength; ++j) {
              currentValue = input[j];

              if (currentValue < n && ++delta > maxInt) {
                error('overflow');
              }

              if (currentValue == n) {
                // Represent delta as a generalized variable-length integer
                for (q = delta, k = base; /* no condition */; k += base) {
                  t = k <= bias ? tMin : (k >= bias + tMax ? tMax : k - bias);
                  if (q < t) {
                    break;
                  }
                  qMinusT = q - t;
                  baseMinusT = base - t;
                  output.push(
                    stringFromCharCode(digitToBasic(t + qMinusT % baseMinusT, 0))
                  );
                  q = floor(qMinusT / baseMinusT);
                }

                output.push(stringFromCharCode(digitToBasic(q, 0)));
                bias = adapt(delta, handledCPCountPlusOne, handledCPCount == basicLength);
                delta = 0;
                ++handledCPCount;
              }
            }

            ++delta;
            ++n;

          }
          return output.join('');
        }

        /**
         * Converts a Punycode string representing a domain name or an email address
         * to Unicode. Only the Punycoded parts of the input will be converted, i.e.
         * it doesn't matter if you call it on a string that has already been
         * converted to Unicode.
         * @memberOf punycode
         * @param {String} input The Punycoded domain name or email address to
         * convert to Unicode.
         * @returns {String} The Unicode representation of the given Punycode
         * string.
         */
        function toUnicode(input) {
          return mapDomain(input, function (string) {
            return regexPunycode.test(string)
              ? decode(string.slice(4).toLowerCase())
              : string;
          });
        }

        /**
         * Converts a Unicode string representing a domain name or an email address to
         * Punycode. Only the non-ASCII parts of the domain name will be converted,
         * i.e. it doesn't matter if you call it with a domain that's already in
         * ASCII.
         * @memberOf punycode
         * @param {String} input The domain name or email address to convert, as a
         * Unicode string.
         * @returns {String} The Punycode representation of the given domain name or
         * email address.
         */
        function toASCII(input) {
          return mapDomain(input, function (string) {
            return regexNonASCII.test(string)
              ? 'xn--' + encode(string)
              : string;
          });
        }

        /*--------------------------------------------------------------------------*/

        /** Define the public API */
        punycode = {
          /**
           * A string representing the current Punycode.js version number.
           * @memberOf punycode
           * @type String
           */
          'version': '1.4.1',
          /**
           * An object of methods to convert from JavaScript's internal character
           * representation (UCS-2) to Unicode code points, and back.
           * @see <https://mathiasbynens.be/notes/javascript-encoding>
           * @memberOf punycode
           * @type Object
           */
          'ucs2': {
            'decode': ucs2decode,
            'encode': ucs2encode
          },
          'decode': decode,
          'encode': encode,
          'toASCII': toASCII,
          'toUnicode': toUnicode
        };

        /** Expose `punycode` */
        // Some AMD build optimizers, like r.js, check for specific condition patterns
        // like the following:
        if (
          typeof define == 'function' &&
          typeof define.amd == 'object' &&
          define.amd
        ) {
          define('punycode', function () {
            return punycode;
          });
        } else if (freeExports && freeModule) {
          if (module.exports == freeExports) {
            // in Node.js, io.js, or RingoJS v0.8.0+
            freeModule.exports = punycode;
          } else {
            // in Narwhal or RingoJS v0.7.0-
            for (key in punycode) {
              punycode.hasOwnProperty(key) && (freeExports[key] = punycode[key]);
            }
          }
        } else {
          // in Rhino or a web browser
          root.punycode = punycode;
        }

      }(this));

    }).call(this, typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {})
  }, {}], 56: [function (require, module, exports) {
    var indexOf = function (xs, item) {
      if (xs.indexOf) return xs.indexOf(item);
      else for (var i = 0; i < xs.length; i++) {
        if (xs[i] === item) return i;
      }
      return -1;
    };
    var Object_keys = function (obj) {
      if (Object.keys) return Object.keys(obj)
      else {
        var res = [];
        for (var key in obj) res.push(key)
        return res;
      }
    };

    var forEach = function (xs, fn) {
      if (xs.forEach) return xs.forEach(fn)
      else for (var i = 0; i < xs.length; i++) {
        fn(xs[i], i, xs);
      }
    };

    var defineProp = (function () {
      try {
        Object.defineProperty({}, '_', {});
        return function (obj, name, value) {
          Object.defineProperty(obj, name, {
            writable: true,
            enumerable: false,
            configurable: true,
            value: value
          })
        };
      } catch (e) {
        return function (obj, name, value) {
          obj[name] = value;
        };
      }
    }());

    var globals = ['Array', 'Boolean', 'Date', 'Error', 'EvalError', 'Function',
      'Infinity', 'JSON', 'Math', 'NaN', 'Number', 'Object', 'RangeError',
      'ReferenceError', 'RegExp', 'String', 'SyntaxError', 'TypeError', 'URIError',
      'decodeURI', 'decodeURIComponent', 'encodeURI', 'encodeURIComponent', 'escape',
      'eval', 'isFinite', 'isNaN', 'parseFloat', 'parseInt', 'undefined', 'unescape'];

    function Context() { }
    Context.prototype = {};

    var Script = exports.Script = function NodeScript(code) {
      if (!(this instanceof Script)) return new Script(code);
      this.code = code;
    };

    Script.prototype.runInContext = function (context) {
      if (!(context instanceof Context)) {
        throw new TypeError("needs a 'context' argument.");
      }

      var iframe = document.createElement('iframe');
      if (!iframe.style) iframe.style = {};
      iframe.style.display = 'none';

      document.body.appendChild(iframe);

      var win = iframe.contentWindow;
      var wEval = win.eval, wExecScript = win.execScript;

      if (!wEval && wExecScript) {
        // win.eval() magically appears when this is called in IE:
        wExecScript.call(win, 'null');
        wEval = win.eval;
      }

      forEach(Object_keys(context), function (key) {
        win[key] = context[key];
      });
      forEach(globals, function (key) {
        if (context[key]) {
          win[key] = context[key];
        }
      });

      var winKeys = Object_keys(win);

      var res = wEval.call(win, this.code);

      forEach(Object_keys(win), function (key) {
        // Avoid copying circular objects like `top` and `window` by only
        // updating existing context properties or new properties in the `win`
        // that was only introduced after the eval.
        if (key in context || indexOf(winKeys, key) === -1) {
          context[key] = win[key];
        }
      });

      forEach(globals, function (key) {
        if (!(key in context)) {
          defineProp(context, key, win[key]);
        }
      });

      document.body.removeChild(iframe);

      return res;
    };

    Script.prototype.runInThisContext = function () {
      return eval(this.code); // maybe...
    };

    Script.prototype.runInNewContext = function (context) {
      var ctx = Script.createContext(context);
      var res = this.runInContext(ctx);
      if (context) {
        forEach(Object_keys(ctx), function (key) {
          context[key] = ctx[key];
        });
      }

      return res;
    };

    forEach(Object_keys(Script.prototype), function (name) {
      exports[name] = Script[name] = function (code) {
        var s = Script(code);
        return s[name].apply(s, [].slice.call(arguments, 1));
      };
    });

    exports.isContext = function (context) {
      return context instanceof Context;
    };

    exports.createScript = function (code) {
      return exports.Script(code);
    };

    exports.createContext = Script.createContext = function (context) {
      var copy = new Context();
      if (typeof context === 'object') {
        forEach(Object_keys(context), function (key) {
          copy[key] = context[key];
        });
      }
      return copy;
    };

  }, {}], 57: [function (require, module, exports) {
    (function (Buffer) {
      module.exports = function xor(a, b) {
        var length = Math.min(a.length, b.length)
        var buffer = new Buffer(length)

        for (var i = 0; i < length; ++i) {
          buffer[i] = a[i] ^ b[i]
        }

        return buffer
      }

    }).call(this, require("buffer").Buffer)
  }, { "buffer": 58 }], 58: [function (require, module, exports) {
    (function (Buffer) {
      /*!
       * The buffer module from node.js, for the browser.
       *
       * @author   Feross Aboukhadijeh <https://feross.org>
       * @license  MIT
       */
      /* eslint-disable no-proto */

      'use strict'

      var base64 = require('base64-js')
      var ieee754 = require('ieee754')

      exports.Buffer = Buffer
      exports.SlowBuffer = SlowBuffer
      exports.INSPECT_MAX_BYTES = 50

      var K_MAX_LENGTH = 0x7fffffff
      exports.kMaxLength = K_MAX_LENGTH

      /**
       * If `Buffer.TYPED_ARRAY_SUPPORT`:
       *   === true    Use Uint8Array implementation (fastest)
       *   === false   Print warning and recommend using `buffer` v4.x which has an Object
       *               implementation (most compatible, even IE6)
       *
       * Browsers that support typed arrays are IE 10+, Firefox 4+, Chrome 7+, Safari 5.1+,
       * Opera 11.6+, iOS 4.2+.
       *
       * We report that the browser does not support typed arrays if the are not subclassable
       * using __proto__. Firefox 4-29 lacks support for adding new properties to `Uint8Array`
       * (See: https://bugzilla.mozilla.org/show_bug.cgi?id=695438). IE 10 lacks support
       * for __proto__ and has a buggy typed array implementation.
       */
      Buffer.TYPED_ARRAY_SUPPORT = typedArraySupport()

      if (!Buffer.TYPED_ARRAY_SUPPORT && typeof console !== 'undefined' &&
        typeof console.error === 'function') {
        console.error(
          'This browser lacks typed array (Uint8Array) support which is required by ' +
          '`buffer` v5.x. Use `buffer` v4.x if you require old browser support.'
        )
      }

      function typedArraySupport() {
        // Can typed array instances can be augmented?
        try {
          var arr = new Uint8Array(1)
          arr.__proto__ = { __proto__: Uint8Array.prototype, foo: function () { return 42 } }
          return arr.foo() === 42
        } catch (e) {
          return false
        }
      }

      Object.defineProperty(Buffer.prototype, 'parent', {
        enumerable: true,
        get: function () {
          if (!Buffer.isBuffer(this)) return undefined
          return this.buffer
        }
      })

      Object.defineProperty(Buffer.prototype, 'offset', {
        enumerable: true,
        get: function () {
          if (!Buffer.isBuffer(this)) return undefined
          return this.byteOffset
        }
      })

      function createBuffer(length) {
        if (length > K_MAX_LENGTH) {
          throw new RangeError('The value "' + length + '" is invalid for option "size"')
        }
        // Return an augmented `Uint8Array` instance
        var buf = new Uint8Array(length)
        buf.__proto__ = Buffer.prototype
        return buf
      }

      /**
       * The Buffer constructor returns instances of `Uint8Array` that have their
       * prototype changed to `Buffer.prototype`. Furthermore, `Buffer` is a subclass of
       * `Uint8Array`, so the returned instances will have all the node `Buffer` methods
       * and the `Uint8Array` methods. Square bracket notation works as expected -- it
       * returns a single octet.
       *
       * The `Uint8Array` prototype remains unmodified.
       */

      function Buffer(arg, encodingOrOffset, length) {
        // Common case.
        if (typeof arg === 'number') {
          if (typeof encodingOrOffset === 'string') {
            throw new TypeError(
              'The "string" argument must be of type string. Received type number'
            )
          }
          return allocUnsafe(arg)
        }
        return from(arg, encodingOrOffset, length)
      }

      // Fix subarray() in ES2016. See: https://github.com/feross/buffer/pull/97
      if (typeof Symbol !== 'undefined' && Symbol.species != null &&
        Buffer[Symbol.species] === Buffer) {
        Object.defineProperty(Buffer, Symbol.species, {
          value: null,
          configurable: true,
          enumerable: false,
          writable: false
        })
      }

      Buffer.poolSize = 8192 // not used by this implementation

      function from(value, encodingOrOffset, length) {
        if (typeof value === 'string') {
          return fromString(value, encodingOrOffset)
        }

        if (ArrayBuffer.isView(value)) {
          return fromArrayLike(value)
        }

        if (value == null) {
          throw TypeError(
            'The first argument must be one of type string, Buffer, ArrayBuffer, Array, ' +
            'or Array-like Object. Received type ' + (typeof value)
          )
        }

        if (isInstance(value, ArrayBuffer) ||
          (value && isInstance(value.buffer, ArrayBuffer))) {
          return fromArrayBuffer(value, encodingOrOffset, length)
        }

        if (typeof value === 'number') {
          throw new TypeError(
            'The "value" argument must not be of type number. Received type number'
          )
        }

        var valueOf = value.valueOf && value.valueOf()
        if (valueOf != null && valueOf !== value) {
          return Buffer.from(valueOf, encodingOrOffset, length)
        }

        var b = fromObject(value)
        if (b) return b

        if (typeof Symbol !== 'undefined' && Symbol.toPrimitive != null &&
          typeof value[Symbol.toPrimitive] === 'function') {
          return Buffer.from(
            value[Symbol.toPrimitive]('string'), encodingOrOffset, length
          )
        }

        throw new TypeError(
          'The first argument must be one of type string, Buffer, ArrayBuffer, Array, ' +
          'or Array-like Object. Received type ' + (typeof value)
        )
      }

      /**
       * Functionally equivalent to Buffer(arg, encoding) but throws a TypeError
       * if value is a number.
       * Buffer.from(str[, encoding])
       * Buffer.from(array)
       * Buffer.from(buffer)
       * Buffer.from(arrayBuffer[, byteOffset[, length]])
       **/
      Buffer.from = function (value, encodingOrOffset, length) {
        return from(value, encodingOrOffset, length)
      }

      // Note: Change prototype *after* Buffer.from is defined to workaround Chrome bug:
      // https://github.com/feross/buffer/pull/148
      Buffer.prototype.__proto__ = Uint8Array.prototype
      Buffer.__proto__ = Uint8Array

      function assertSize(size) {
        if (typeof size !== 'number') {
          throw new TypeError('"size" argument must be of type number')
        } else if (size < 0) {
          throw new RangeError('The value "' + size + '" is invalid for option "size"')
        }
      }

      function alloc(size, fill, encoding) {
        assertSize(size)
        if (size <= 0) {
          return createBuffer(size)
        }
        if (fill !== undefined) {
          // Only pay attention to encoding if it's a string. This
          // prevents accidentally sending in a number that would
          // be interpretted as a start offset.
          return typeof encoding === 'string'
            ? createBuffer(size).fill(fill, encoding)
            : createBuffer(size).fill(fill)
        }
        return createBuffer(size)
      }

      /**
       * Creates a new filled Buffer instance.
       * alloc(size[, fill[, encoding]])
       **/
      Buffer.alloc = function (size, fill, encoding) {
        return alloc(size, fill, encoding)
      }

      function allocUnsafe(size) {
        assertSize(size)
        return createBuffer(size < 0 ? 0 : checked(size) | 0)
      }

      /**
       * Equivalent to Buffer(num), by default creates a non-zero-filled Buffer instance.
       * */
      Buffer.allocUnsafe = function (size) {
        return allocUnsafe(size)
      }
      /**
       * Equivalent to SlowBuffer(num), by default creates a non-zero-filled Buffer instance.
       */
      Buffer.allocUnsafeSlow = function (size) {
        return allocUnsafe(size)
      }

      function fromString(string, encoding) {
        if (typeof encoding !== 'string' || encoding === '') {
          encoding = 'utf8'
        }

        if (!Buffer.isEncoding(encoding)) {
          throw new TypeError('Unknown encoding: ' + encoding)
        }

        var length = byteLength(string, encoding) | 0
        var buf = createBuffer(length)

        var actual = buf.write(string, encoding)

        if (actual !== length) {
          // Writing a hex string, for example, that contains invalid characters will
          // cause everything after the first invalid character to be ignored. (e.g.
          // 'abxxcd' will be treated as 'ab')
          buf = buf.slice(0, actual)
        }

        return buf
      }

      function fromArrayLike(array) {
        var length = array.length < 0 ? 0 : checked(array.length) | 0
        var buf = createBuffer(length)
        for (var i = 0; i < length; i += 1) {
          buf[i] = array[i] & 255
        }
        return buf
      }

      function fromArrayBuffer(array, byteOffset, length) {
        if (byteOffset < 0 || array.byteLength < byteOffset) {
          throw new RangeError('"offset" is outside of buffer bounds')
        }

        if (array.byteLength < byteOffset + (length || 0)) {
          throw new RangeError('"length" is outside of buffer bounds')
        }

        var buf
        if (byteOffset === undefined && length === undefined) {
          buf = new Uint8Array(array)
        } else if (length === undefined) {
          buf = new Uint8Array(array, byteOffset)
        } else {
          buf = new Uint8Array(array, byteOffset, length)
        }

        // Return an augmented `Uint8Array` instance
        buf.__proto__ = Buffer.prototype
        return buf
      }

      function fromObject(obj) {
        if (Buffer.isBuffer(obj)) {
          var len = checked(obj.length) | 0
          var buf = createBuffer(len)

          if (buf.length === 0) {
            return buf
          }

          obj.copy(buf, 0, 0, len)
          return buf
        }

        if (obj.length !== undefined) {
          if (typeof obj.length !== 'number' || numberIsNaN(obj.length)) {
            return createBuffer(0)
          }
          return fromArrayLike(obj)
        }

        if (obj.type === 'Buffer' && Array.isArray(obj.data)) {
          return fromArrayLike(obj.data)
        }
      }

      function checked(length) {
        // Note: cannot use `length < K_MAX_LENGTH` here because that fails when
        // length is NaN (which is otherwise coerced to zero.)
        if (length >= K_MAX_LENGTH) {
          throw new RangeError('Attempt to allocate Buffer larger than maximum ' +
            'size: 0x' + K_MAX_LENGTH.toString(16) + ' bytes')
        }
        return length | 0
      }

      function SlowBuffer(length) {
        if (+length != length) { // eslint-disable-line eqeqeq
          length = 0
        }
        return Buffer.alloc(+length)
      }

      Buffer.isBuffer = function isBuffer(b) {
        return b != null && b._isBuffer === true &&
          b !== Buffer.prototype // so Buffer.isBuffer(Buffer.prototype) will be false
      }

      Buffer.compare = function compare(a, b) {
        if (isInstance(a, Uint8Array)) a = Buffer.from(a, a.offset, a.byteLength)
        if (isInstance(b, Uint8Array)) b = Buffer.from(b, b.offset, b.byteLength)
        if (!Buffer.isBuffer(a) || !Buffer.isBuffer(b)) {
          throw new TypeError(
            'The "buf1", "buf2" arguments must be one of type Buffer or Uint8Array'
          )
        }

        if (a === b) return 0

        var x = a.length
        var y = b.length

        for (var i = 0, len = Math.min(x, y); i < len; ++i) {
          if (a[i] !== b[i]) {
            x = a[i]
            y = b[i]
            break
          }
        }

        if (x < y) return -1
        if (y < x) return 1
        return 0
      }

      Buffer.isEncoding = function isEncoding(encoding) {
        switch (String(encoding).toLowerCase()) {
          case 'hex':
          case 'utf8':
          case 'utf-8':
          case 'ascii':
          case 'latin1':
          case 'binary':
          case 'base64':
          case 'ucs2':
          case 'ucs-2':
          case 'utf16le':
          case 'utf-16le':
            return true
          default:
            return false
        }
      }

      Buffer.concat = function concat(list, length) {
        if (!Array.isArray(list)) {
          throw new TypeError('"list" argument must be an Array of Buffers')
        }

        if (list.length === 0) {
          return Buffer.alloc(0)
        }

        var i
        if (length === undefined) {
          length = 0
          for (i = 0; i < list.length; ++i) {
            length += list[i].length
          }
        }

        var buffer = Buffer.allocUnsafe(length)
        var pos = 0
        for (i = 0; i < list.length; ++i) {
          var buf = list[i]
          if (isInstance(buf, Uint8Array)) {
            buf = Buffer.from(buf)
          }
          if (!Buffer.isBuffer(buf)) {
            throw new TypeError('"list" argument must be an Array of Buffers')
          }
          buf.copy(buffer, pos)
          pos += buf.length
        }
        return buffer
      }

      function byteLength(string, encoding) {
        if (Buffer.isBuffer(string)) {
          return string.length
        }
        if (ArrayBuffer.isView(string) || isInstance(string, ArrayBuffer)) {
          return string.byteLength
        }
        if (typeof string !== 'string') {
          throw new TypeError(
            'The "string" argument must be one of type string, Buffer, or ArrayBuffer. ' +
            'Received type ' + typeof string
          )
        }

        var len = string.length
        var mustMatch = (arguments.length > 2 && arguments[2] === true)
        if (!mustMatch && len === 0) return 0

        // Use a for loop to avoid recursion
        var loweredCase = false
        for (; ;) {
          switch (encoding) {
            case 'ascii':
            case 'latin1':
            case 'binary':
              return len
            case 'utf8':
            case 'utf-8':
              return utf8ToBytes(string).length
            case 'ucs2':
            case 'ucs-2':
            case 'utf16le':
            case 'utf-16le':
              return len * 2
            case 'hex':
              return len >>> 1
            case 'base64':
              return base64ToBytes(string).length
            default:
              if (loweredCase) {
                return mustMatch ? -1 : utf8ToBytes(string).length // assume utf8
              }
              encoding = ('' + encoding).toLowerCase()
              loweredCase = true
          }
        }
      }
      Buffer.byteLength = byteLength

      function slowToString(encoding, start, end) {
        var loweredCase = false

        // No need to verify that "this.length <= MAX_UINT32" since it's a read-only
        // property of a typed array.

        // This behaves neither like String nor Uint8Array in that we set start/end
        // to their upper/lower bounds if the value passed is out of range.
        // undefined is handled specially as per ECMA-262 6th Edition,
        // Section 13.3.3.7 Runtime Semantics: KeyedBindingInitialization.
        if (start === undefined || start < 0) {
          start = 0
        }
        // Return early if start > this.length. Done here to prevent potential uint32
        // coercion fail below.
        if (start > this.length) {
          return ''
        }

        if (end === undefined || end > this.length) {
          end = this.length
        }

        if (end <= 0) {
          return ''
        }

        // Force coersion to uint32. This will also coerce falsey/NaN values to 0.
        end >>>= 0
        start >>>= 0

        if (end <= start) {
          return ''
        }

        if (!encoding) encoding = 'utf8'

        while (true) {
          switch (encoding) {
            case 'hex':
              return hexSlice(this, start, end)

            case 'utf8':
            case 'utf-8':
              return utf8Slice(this, start, end)

            case 'ascii':
              return asciiSlice(this, start, end)

            case 'latin1':
            case 'binary':
              return latin1Slice(this, start, end)

            case 'base64':
              return base64Slice(this, start, end)

            case 'ucs2':
            case 'ucs-2':
            case 'utf16le':
            case 'utf-16le':
              return utf16leSlice(this, start, end)

            default:
              if (loweredCase) throw new TypeError('Unknown encoding: ' + encoding)
              encoding = (encoding + '').toLowerCase()
              loweredCase = true
          }
        }
      }

      // This property is used by `Buffer.isBuffer` (and the `is-buffer` npm package)
      // to detect a Buffer instance. It's not possible to use `instanceof Buffer`
      // reliably in a browserify context because there could be multiple different
      // copies of the 'buffer' package in use. This method works even for Buffer
      // instances that were created from another copy of the `buffer` package.
      // See: https://github.com/feross/buffer/issues/154
      Buffer.prototype._isBuffer = true

      function swap(b, n, m) {
        var i = b[n]
        b[n] = b[m]
        b[m] = i
      }

      Buffer.prototype.swap16 = function swap16() {
        var len = this.length
        if (len % 2 !== 0) {
          throw new RangeError('Buffer size must be a multiple of 16-bits')
        }
        for (var i = 0; i < len; i += 2) {
          swap(this, i, i + 1)
        }
        return this
      }

      Buffer.prototype.swap32 = function swap32() {
        var len = this.length
        if (len % 4 !== 0) {
          throw new RangeError('Buffer size must be a multiple of 32-bits')
        }
        for (var i = 0; i < len; i += 4) {
          swap(this, i, i + 3)
          swap(this, i + 1, i + 2)
        }
        return this
      }

      Buffer.prototype.swap64 = function swap64() {
        var len = this.length
        if (len % 8 !== 0) {
          throw new RangeError('Buffer size must be a multiple of 64-bits')
        }
        for (var i = 0; i < len; i += 8) {
          swap(this, i, i + 7)
          swap(this, i + 1, i + 6)
          swap(this, i + 2, i + 5)
          swap(this, i + 3, i + 4)
        }
        return this
      }

      Buffer.prototype.toString = function toString() {
        var length = this.length
        if (length === 0) return ''
        if (arguments.length === 0) return utf8Slice(this, 0, length)
        return slowToString.apply(this, arguments)
      }

      Buffer.prototype.toLocaleString = Buffer.prototype.toString

      Buffer.prototype.equals = function equals(b) {
        if (!Buffer.isBuffer(b)) throw new TypeError('Argument must be a Buffer')
        if (this === b) return true
        return Buffer.compare(this, b) === 0
      }

      Buffer.prototype.inspect = function inspect() {
        var str = ''
        var max = exports.INSPECT_MAX_BYTES
        str = this.toString('hex', 0, max).replace(/(.{2})/g, '$1 ').trim()
        if (this.length > max) str += ' ... '
        return '<Buffer ' + str + '>'
      }

      Buffer.prototype.compare = function compare(target, start, end, thisStart, thisEnd) {
        if (isInstance(target, Uint8Array)) {
          target = Buffer.from(target, target.offset, target.byteLength)
        }
        if (!Buffer.isBuffer(target)) {
          throw new TypeError(
            'The "target" argument must be one of type Buffer or Uint8Array. ' +
            'Received type ' + (typeof target)
          )
        }

        if (start === undefined) {
          start = 0
        }
        if (end === undefined) {
          end = target ? target.length : 0
        }
        if (thisStart === undefined) {
          thisStart = 0
        }
        if (thisEnd === undefined) {
          thisEnd = this.length
        }

        if (start < 0 || end > target.length || thisStart < 0 || thisEnd > this.length) {
          throw new RangeError('out of range index')
        }

        if (thisStart >= thisEnd && start >= end) {
          return 0
        }
        if (thisStart >= thisEnd) {
          return -1
        }
        if (start >= end) {
          return 1
        }

        start >>>= 0
        end >>>= 0
        thisStart >>>= 0
        thisEnd >>>= 0

        if (this === target) return 0

        var x = thisEnd - thisStart
        var y = end - start
        var len = Math.min(x, y)

        var thisCopy = this.slice(thisStart, thisEnd)
        var targetCopy = target.slice(start, end)

        for (var i = 0; i < len; ++i) {
          if (thisCopy[i] !== targetCopy[i]) {
            x = thisCopy[i]
            y = targetCopy[i]
            break
          }
        }

        if (x < y) return -1
        if (y < x) return 1
        return 0
      }

      // Finds either the first index of `val` in `buffer` at offset >= `byteOffset`,
      // OR the last index of `val` in `buffer` at offset <= `byteOffset`.
      //
      // Arguments:
      // - buffer - a Buffer to search
      // - val - a string, Buffer, or number
      // - byteOffset - an index into `buffer`; will be clamped to an int32
      // - encoding - an optional encoding, relevant is val is a string
      // - dir - true for indexOf, false for lastIndexOf
      function bidirectionalIndexOf(buffer, val, byteOffset, encoding, dir) {
        // Empty buffer means no match
        if (buffer.length === 0) return -1

        // Normalize byteOffset
        if (typeof byteOffset === 'string') {
          encoding = byteOffset
          byteOffset = 0
        } else if (byteOffset > 0x7fffffff) {
          byteOffset = 0x7fffffff
        } else if (byteOffset < -0x80000000) {
          byteOffset = -0x80000000
        }
        byteOffset = +byteOffset // Coerce to Number.
        if (numberIsNaN(byteOffset)) {
          // byteOffset: it it's undefined, null, NaN, "foo", etc, search whole buffer
          byteOffset = dir ? 0 : (buffer.length - 1)
        }

        // Normalize byteOffset: negative offsets start from the end of the buffer
        if (byteOffset < 0) byteOffset = buffer.length + byteOffset
        if (byteOffset >= buffer.length) {
          if (dir) return -1
          else byteOffset = buffer.length - 1
        } else if (byteOffset < 0) {
          if (dir) byteOffset = 0
          else return -1
        }

        // Normalize val
        if (typeof val === 'string') {
          val = Buffer.from(val, encoding)
        }

        // Finally, search either indexOf (if dir is true) or lastIndexOf
        if (Buffer.isBuffer(val)) {
          // Special case: looking for empty string/buffer always fails
          if (val.length === 0) {
            return -1
          }
          return arrayIndexOf(buffer, val, byteOffset, encoding, dir)
        } else if (typeof val === 'number') {
          val = val & 0xFF // Search for a byte value [0-255]
          if (typeof Uint8Array.prototype.indexOf === 'function') {
            if (dir) {
              return Uint8Array.prototype.indexOf.call(buffer, val, byteOffset)
            } else {
              return Uint8Array.prototype.lastIndexOf.call(buffer, val, byteOffset)
            }
          }
          return arrayIndexOf(buffer, [val], byteOffset, encoding, dir)
        }

        throw new TypeError('val must be string, number or Buffer')
      }

      function arrayIndexOf(arr, val, byteOffset, encoding, dir) {
        var indexSize = 1
        var arrLength = arr.length
        var valLength = val.length

        if (encoding !== undefined) {
          encoding = String(encoding).toLowerCase()
          if (encoding === 'ucs2' || encoding === 'ucs-2' ||
            encoding === 'utf16le' || encoding === 'utf-16le') {
            if (arr.length < 2 || val.length < 2) {
              return -1
            }
            indexSize = 2
            arrLength /= 2
            valLength /= 2
            byteOffset /= 2
          }
        }

        function read(buf, i) {
          if (indexSize === 1) {
            return buf[i]
          } else {
            return buf.readUInt16BE(i * indexSize)
          }
        }

        var i
        if (dir) {
          var foundIndex = -1
          for (i = byteOffset; i < arrLength; i++) {
            if (read(arr, i) === read(val, foundIndex === -1 ? 0 : i - foundIndex)) {
              if (foundIndex === -1) foundIndex = i
              if (i - foundIndex + 1 === valLength) return foundIndex * indexSize
            } else {
              if (foundIndex !== -1) i -= i - foundIndex
              foundIndex = -1
            }
          }
        } else {
          if (byteOffset + valLength > arrLength) byteOffset = arrLength - valLength
          for (i = byteOffset; i >= 0; i--) {
            var found = true
            for (var j = 0; j < valLength; j++) {
              if (read(arr, i + j) !== read(val, j)) {
                found = false
                break
              }
            }
            if (found) return i
          }
        }

        return -1
      }

      Buffer.prototype.includes = function includes(val, byteOffset, encoding) {
        return this.indexOf(val, byteOffset, encoding) !== -1
      }

      Buffer.prototype.indexOf = function indexOf(val, byteOffset, encoding) {
        return bidirectionalIndexOf(this, val, byteOffset, encoding, true)
      }

      Buffer.prototype.lastIndexOf = function lastIndexOf(val, byteOffset, encoding) {
        return bidirectionalIndexOf(this, val, byteOffset, encoding, false)
      }

      function hexWrite(buf, string, offset, length) {
        offset = Number(offset) || 0
        var remaining = buf.length - offset
        if (!length) {
          length = remaining
        } else {
          length = Number(length)
          if (length > remaining) {
            length = remaining
          }
        }

        var strLen = string.length

        if (length > strLen / 2) {
          length = strLen / 2
        }
        for (var i = 0; i < length; ++i) {
          var parsed = parseInt(string.substr(i * 2, 2), 16)
          if (numberIsNaN(parsed)) return i
          buf[offset + i] = parsed
        }
        return i
      }

      function utf8Write(buf, string, offset, length) {
        return blitBuffer(utf8ToBytes(string, buf.length - offset), buf, offset, length)
      }

      function asciiWrite(buf, string, offset, length) {
        return blitBuffer(asciiToBytes(string), buf, offset, length)
      }

      function latin1Write(buf, string, offset, length) {
        return asciiWrite(buf, string, offset, length)
      }

      function base64Write(buf, string, offset, length) {
        return blitBuffer(base64ToBytes(string), buf, offset, length)
      }

      function ucs2Write(buf, string, offset, length) {
        return blitBuffer(utf16leToBytes(string, buf.length - offset), buf, offset, length)
      }

      Buffer.prototype.write = function write(string, offset, length, encoding) {
        // Buffer#write(string)
        if (offset === undefined) {
          encoding = 'utf8'
          length = this.length
          offset = 0
          // Buffer#write(string, encoding)
        } else if (length === undefined && typeof offset === 'string') {
          encoding = offset
          length = this.length
          offset = 0
          // Buffer#write(string, offset[, length][, encoding])
        } else if (isFinite(offset)) {
          offset = offset >>> 0
          if (isFinite(length)) {
            length = length >>> 0
            if (encoding === undefined) encoding = 'utf8'
          } else {
            encoding = length
            length = undefined
          }
        } else {
          throw new Error(
            'Buffer.write(string, encoding, offset[, length]) is no longer supported'
          )
        }

        var remaining = this.length - offset
        if (length === undefined || length > remaining) length = remaining

        if ((string.length > 0 && (length < 0 || offset < 0)) || offset > this.length) {
          throw new RangeError('Attempt to write outside buffer bounds')
        }

        if (!encoding) encoding = 'utf8'

        var loweredCase = false
        for (; ;) {
          switch (encoding) {
            case 'hex':
              return hexWrite(this, string, offset, length)

            case 'utf8':
            case 'utf-8':
              return utf8Write(this, string, offset, length)

            case 'ascii':
              return asciiWrite(this, string, offset, length)

            case 'latin1':
            case 'binary':
              return latin1Write(this, string, offset, length)

            case 'base64':
              // Warning: maxLength not taken into account in base64Write
              return base64Write(this, string, offset, length)

            case 'ucs2':
            case 'ucs-2':
            case 'utf16le':
            case 'utf-16le':
              return ucs2Write(this, string, offset, length)

            default:
              if (loweredCase) throw new TypeError('Unknown encoding: ' + encoding)
              encoding = ('' + encoding).toLowerCase()
              loweredCase = true
          }
        }
      }

      Buffer.prototype.toJSON = function toJSON() {
        return {
          type: 'Buffer',
          data: Array.prototype.slice.call(this._arr || this, 0)
        }
      }

      function base64Slice(buf, start, end) {
        if (start === 0 && end === buf.length) {
          return base64.fromByteArray(buf)
        } else {
          return base64.fromByteArray(buf.slice(start, end))
        }
      }

      function utf8Slice(buf, start, end) {
        end = Math.min(buf.length, end)
        var res = []

        var i = start
        while (i < end) {
          var firstByte = buf[i]
          var codePoint = null
          var bytesPerSequence = (firstByte > 0xEF) ? 4
            : (firstByte > 0xDF) ? 3
              : (firstByte > 0xBF) ? 2
                : 1

          if (i + bytesPerSequence <= end) {
            var secondByte, thirdByte, fourthByte, tempCodePoint

            switch (bytesPerSequence) {
              case 1:
                if (firstByte < 0x80) {
                  codePoint = firstByte
                }
                break
              case 2:
                secondByte = buf[i + 1]
                if ((secondByte & 0xC0) === 0x80) {
                  tempCodePoint = (firstByte & 0x1F) << 0x6 | (secondByte & 0x3F)
                  if (tempCodePoint > 0x7F) {
                    codePoint = tempCodePoint
                  }
                }
                break
              case 3:
                secondByte = buf[i + 1]
                thirdByte = buf[i + 2]
                if ((secondByte & 0xC0) === 0x80 && (thirdByte & 0xC0) === 0x80) {
                  tempCodePoint = (firstByte & 0xF) << 0xC | (secondByte & 0x3F) << 0x6 | (thirdByte & 0x3F)
                  if (tempCodePoint > 0x7FF && (tempCodePoint < 0xD800 || tempCodePoint > 0xDFFF)) {
                    codePoint = tempCodePoint
                  }
                }
                break
              case 4:
                secondByte = buf[i + 1]
                thirdByte = buf[i + 2]
                fourthByte = buf[i + 3]
                if ((secondByte & 0xC0) === 0x80 && (thirdByte & 0xC0) === 0x80 && (fourthByte & 0xC0) === 0x80) {
                  tempCodePoint = (firstByte & 0xF) << 0x12 | (secondByte & 0x3F) << 0xC | (thirdByte & 0x3F) << 0x6 | (fourthByte & 0x3F)
                  if (tempCodePoint > 0xFFFF && tempCodePoint < 0x110000) {
                    codePoint = tempCodePoint
                  }
                }
            }
          }

          if (codePoint === null) {
            // we did not generate a valid codePoint so insert a
            // replacement char (U+FFFD) and advance only 1 byte
            codePoint = 0xFFFD
            bytesPerSequence = 1
          } else if (codePoint > 0xFFFF) {
            // encode to utf16 (surrogate pair dance)
            codePoint -= 0x10000
            res.push(codePoint >>> 10 & 0x3FF | 0xD800)
            codePoint = 0xDC00 | codePoint & 0x3FF
          }

          res.push(codePoint)
          i += bytesPerSequence
        }

        return decodeCodePointsArray(res)
      }

      // Based on http://stackoverflow.com/a/22747272/680742, the browser with
      // the lowest limit is Chrome, with 0x10000 args.
      // We go 1 magnitude less, for safety
      var MAX_ARGUMENTS_LENGTH = 0x1000

      function decodeCodePointsArray(codePoints) {
        var len = codePoints.length
        if (len <= MAX_ARGUMENTS_LENGTH) {
          return String.fromCharCode.apply(String, codePoints) // avoid extra slice()
        }

        // Decode in chunks to avoid "call stack size exceeded".
        var res = ''
        var i = 0
        while (i < len) {
          res += String.fromCharCode.apply(
            String,
            codePoints.slice(i, i += MAX_ARGUMENTS_LENGTH)
          )
        }
        return res
      }

      function asciiSlice(buf, start, end) {
        var ret = ''
        end = Math.min(buf.length, end)

        for (var i = start; i < end; ++i) {
          ret += String.fromCharCode(buf[i] & 0x7F)
        }
        return ret
      }

      function latin1Slice(buf, start, end) {
        var ret = ''
        end = Math.min(buf.length, end)

        for (var i = start; i < end; ++i) {
          ret += String.fromCharCode(buf[i])
        }
        return ret
      }

      function hexSlice(buf, start, end) {
        var len = buf.length

        if (!start || start < 0) start = 0
        if (!end || end < 0 || end > len) end = len

        var out = ''
        for (var i = start; i < end; ++i) {
          out += toHex(buf[i])
        }
        return out
      }

      function utf16leSlice(buf, start, end) {
        var bytes = buf.slice(start, end)
        var res = ''
        for (var i = 0; i < bytes.length; i += 2) {
          res += String.fromCharCode(bytes[i] + (bytes[i + 1] * 256))
        }
        return res
      }

      Buffer.prototype.slice = function slice(start, end) {
        var len = this.length
        start = ~~start
        end = end === undefined ? len : ~~end

        if (start < 0) {
          start += len
          if (start < 0) start = 0
        } else if (start > len) {
          start = len
        }

        if (end < 0) {
          end += len
          if (end < 0) end = 0
        } else if (end > len) {
          end = len
        }

        if (end < start) end = start

        var newBuf = this.subarray(start, end)
        // Return an augmented `Uint8Array` instance
        newBuf.__proto__ = Buffer.prototype
        return newBuf
      }

      /*
       * Need to make sure that buffer isn't trying to write out of bounds.
       */
      function checkOffset(offset, ext, length) {
        if ((offset % 1) !== 0 || offset < 0) throw new RangeError('offset is not uint')
        if (offset + ext > length) throw new RangeError('Trying to access beyond buffer length')
      }

      Buffer.prototype.readUIntLE = function readUIntLE(offset, byteLength, noAssert) {
        offset = offset >>> 0
        byteLength = byteLength >>> 0
        if (!noAssert) checkOffset(offset, byteLength, this.length)

        var val = this[offset]
        var mul = 1
        var i = 0
        while (++i < byteLength && (mul *= 0x100)) {
          val += this[offset + i] * mul
        }

        return val
      }

      Buffer.prototype.readUIntBE = function readUIntBE(offset, byteLength, noAssert) {
        offset = offset >>> 0
        byteLength = byteLength >>> 0
        if (!noAssert) {
          checkOffset(offset, byteLength, this.length)
        }

        var val = this[offset + --byteLength]
        var mul = 1
        while (byteLength > 0 && (mul *= 0x100)) {
          val += this[offset + --byteLength] * mul
        }

        return val
      }

      Buffer.prototype.readUInt8 = function readUInt8(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 1, this.length)
        return this[offset]
      }

      Buffer.prototype.readUInt16LE = function readUInt16LE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 2, this.length)
        return this[offset] | (this[offset + 1] << 8)
      }

      Buffer.prototype.readUInt16BE = function readUInt16BE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 2, this.length)
        return (this[offset] << 8) | this[offset + 1]
      }

      Buffer.prototype.readUInt32LE = function readUInt32LE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 4, this.length)

        return ((this[offset]) |
          (this[offset + 1] << 8) |
          (this[offset + 2] << 16)) +
          (this[offset + 3] * 0x1000000)
      }

      Buffer.prototype.readUInt32BE = function readUInt32BE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 4, this.length)

        return (this[offset] * 0x1000000) +
          ((this[offset + 1] << 16) |
            (this[offset + 2] << 8) |
            this[offset + 3])
      }

      Buffer.prototype.readIntLE = function readIntLE(offset, byteLength, noAssert) {
        offset = offset >>> 0
        byteLength = byteLength >>> 0
        if (!noAssert) checkOffset(offset, byteLength, this.length)

        var val = this[offset]
        var mul = 1
        var i = 0
        while (++i < byteLength && (mul *= 0x100)) {
          val += this[offset + i] * mul
        }
        mul *= 0x80

        if (val >= mul) val -= Math.pow(2, 8 * byteLength)

        return val
      }

      Buffer.prototype.readIntBE = function readIntBE(offset, byteLength, noAssert) {
        offset = offset >>> 0
        byteLength = byteLength >>> 0
        if (!noAssert) checkOffset(offset, byteLength, this.length)

        var i = byteLength
        var mul = 1
        var val = this[offset + --i]
        while (i > 0 && (mul *= 0x100)) {
          val += this[offset + --i] * mul
        }
        mul *= 0x80

        if (val >= mul) val -= Math.pow(2, 8 * byteLength)

        return val
      }

      Buffer.prototype.readInt8 = function readInt8(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 1, this.length)
        if (!(this[offset] & 0x80)) return (this[offset])
        return ((0xff - this[offset] + 1) * -1)
      }

      Buffer.prototype.readInt16LE = function readInt16LE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 2, this.length)
        var val = this[offset] | (this[offset + 1] << 8)
        return (val & 0x8000) ? val | 0xFFFF0000 : val
      }

      Buffer.prototype.readInt16BE = function readInt16BE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 2, this.length)
        var val = this[offset + 1] | (this[offset] << 8)
        return (val & 0x8000) ? val | 0xFFFF0000 : val
      }

      Buffer.prototype.readInt32LE = function readInt32LE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 4, this.length)

        return (this[offset]) |
          (this[offset + 1] << 8) |
          (this[offset + 2] << 16) |
          (this[offset + 3] << 24)
      }

      Buffer.prototype.readInt32BE = function readInt32BE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 4, this.length)

        return (this[offset] << 24) |
          (this[offset + 1] << 16) |
          (this[offset + 2] << 8) |
          (this[offset + 3])
      }

      Buffer.prototype.readFloatLE = function readFloatLE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 4, this.length)
        return ieee754.read(this, offset, true, 23, 4)
      }

      Buffer.prototype.readFloatBE = function readFloatBE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 4, this.length)
        return ieee754.read(this, offset, false, 23, 4)
      }

      Buffer.prototype.readDoubleLE = function readDoubleLE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 8, this.length)
        return ieee754.read(this, offset, true, 52, 8)
      }

      Buffer.prototype.readDoubleBE = function readDoubleBE(offset, noAssert) {
        offset = offset >>> 0
        if (!noAssert) checkOffset(offset, 8, this.length)
        return ieee754.read(this, offset, false, 52, 8)
      }

      function checkInt(buf, value, offset, ext, max, min) {
        if (!Buffer.isBuffer(buf)) throw new TypeError('"buffer" argument must be a Buffer instance')
        if (value > max || value < min) throw new RangeError('"value" argument is out of bounds')
        if (offset + ext > buf.length) throw new RangeError('Index out of range')
      }

      Buffer.prototype.writeUIntLE = function writeUIntLE(value, offset, byteLength, noAssert) {
        value = +value
        offset = offset >>> 0
        byteLength = byteLength >>> 0
        if (!noAssert) {
          var maxBytes = Math.pow(2, 8 * byteLength) - 1
          checkInt(this, value, offset, byteLength, maxBytes, 0)
        }

        var mul = 1
        var i = 0
        this[offset] = value & 0xFF
        while (++i < byteLength && (mul *= 0x100)) {
          this[offset + i] = (value / mul) & 0xFF
        }

        return offset + byteLength
      }

      Buffer.prototype.writeUIntBE = function writeUIntBE(value, offset, byteLength, noAssert) {
        value = +value
        offset = offset >>> 0
        byteLength = byteLength >>> 0
        if (!noAssert) {
          var maxBytes = Math.pow(2, 8 * byteLength) - 1
          checkInt(this, value, offset, byteLength, maxBytes, 0)
        }

        var i = byteLength - 1
        var mul = 1
        this[offset + i] = value & 0xFF
        while (--i >= 0 && (mul *= 0x100)) {
          this[offset + i] = (value / mul) & 0xFF
        }

        return offset + byteLength
      }

      Buffer.prototype.writeUInt8 = function writeUInt8(value, offset, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) checkInt(this, value, offset, 1, 0xff, 0)
        this[offset] = (value & 0xff)
        return offset + 1
      }

      Buffer.prototype.writeUInt16LE = function writeUInt16LE(value, offset, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) checkInt(this, value, offset, 2, 0xffff, 0)
        this[offset] = (value & 0xff)
        this[offset + 1] = (value >>> 8)
        return offset + 2
      }

      Buffer.prototype.writeUInt16BE = function writeUInt16BE(value, offset, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) checkInt(this, value, offset, 2, 0xffff, 0)
        this[offset] = (value >>> 8)
        this[offset + 1] = (value & 0xff)
        return offset + 2
      }

      Buffer.prototype.writeUInt32LE = function writeUInt32LE(value, offset, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) checkInt(this, value, offset, 4, 0xffffffff, 0)
        this[offset + 3] = (value >>> 24)
        this[offset + 2] = (value >>> 16)
        this[offset + 1] = (value >>> 8)
        this[offset] = (value & 0xff)
        return offset + 4
      }

      Buffer.prototype.writeUInt32BE = function writeUInt32BE(value, offset, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) checkInt(this, value, offset, 4, 0xffffffff, 0)
        this[offset] = (value >>> 24)
        this[offset + 1] = (value >>> 16)
        this[offset + 2] = (value >>> 8)
        this[offset + 3] = (value & 0xff)
        return offset + 4
      }

      Buffer.prototype.writeIntLE = function writeIntLE(value, offset, byteLength, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) {
          var limit = Math.pow(2, (8 * byteLength) - 1)

          checkInt(this, value, offset, byteLength, limit - 1, -limit)
        }

        var i = 0
        var mul = 1
        var sub = 0
        this[offset] = value & 0xFF
        while (++i < byteLength && (mul *= 0x100)) {
          if (value < 0 && sub === 0 && this[offset + i - 1] !== 0) {
            sub = 1
          }
          this[offset + i] = ((value / mul) >> 0) - sub & 0xFF
        }

        return offset + byteLength
      }

      Buffer.prototype.writeIntBE = function writeIntBE(value, offset, byteLength, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) {
          var limit = Math.pow(2, (8 * byteLength) - 1)

          checkInt(this, value, offset, byteLength, limit - 1, -limit)
        }

        var i = byteLength - 1
        var mul = 1
        var sub = 0
        this[offset + i] = value & 0xFF
        while (--i >= 0 && (mul *= 0x100)) {
          if (value < 0 && sub === 0 && this[offset + i + 1] !== 0) {
            sub = 1
          }
          this[offset + i] = ((value / mul) >> 0) - sub & 0xFF
        }

        return offset + byteLength
      }

      Buffer.prototype.writeInt8 = function writeInt8(value, offset, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) checkInt(this, value, offset, 1, 0x7f, -0x80)
        if (value < 0) value = 0xff + value + 1
        this[offset] = (value & 0xff)
        return offset + 1
      }

      Buffer.prototype.writeInt16LE = function writeInt16LE(value, offset, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) checkInt(this, value, offset, 2, 0x7fff, -0x8000)
        this[offset] = (value & 0xff)
        this[offset + 1] = (value >>> 8)
        return offset + 2
      }

      Buffer.prototype.writeInt16BE = function writeInt16BE(value, offset, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) checkInt(this, value, offset, 2, 0x7fff, -0x8000)
        this[offset] = (value >>> 8)
        this[offset + 1] = (value & 0xff)
        return offset + 2
      }

      Buffer.prototype.writeInt32LE = function writeInt32LE(value, offset, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) checkInt(this, value, offset, 4, 0x7fffffff, -0x80000000)
        this[offset] = (value & 0xff)
        this[offset + 1] = (value >>> 8)
        this[offset + 2] = (value >>> 16)
        this[offset + 3] = (value >>> 24)
        return offset + 4
      }

      Buffer.prototype.writeInt32BE = function writeInt32BE(value, offset, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) checkInt(this, value, offset, 4, 0x7fffffff, -0x80000000)
        if (value < 0) value = 0xffffffff + value + 1
        this[offset] = (value >>> 24)
        this[offset + 1] = (value >>> 16)
        this[offset + 2] = (value >>> 8)
        this[offset + 3] = (value & 0xff)
        return offset + 4
      }

      function checkIEEE754(buf, value, offset, ext, max, min) {
        if (offset + ext > buf.length) throw new RangeError('Index out of range')
        if (offset < 0) throw new RangeError('Index out of range')
      }

      function writeFloat(buf, value, offset, littleEndian, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) {
          checkIEEE754(buf, value, offset, 4, 3.4028234663852886e+38, -3.4028234663852886e+38)
        }
        ieee754.write(buf, value, offset, littleEndian, 23, 4)
        return offset + 4
      }

      Buffer.prototype.writeFloatLE = function writeFloatLE(value, offset, noAssert) {
        return writeFloat(this, value, offset, true, noAssert)
      }

      Buffer.prototype.writeFloatBE = function writeFloatBE(value, offset, noAssert) {
        return writeFloat(this, value, offset, false, noAssert)
      }

      function writeDouble(buf, value, offset, littleEndian, noAssert) {
        value = +value
        offset = offset >>> 0
        if (!noAssert) {
          checkIEEE754(buf, value, offset, 8, 1.7976931348623157E+308, -1.7976931348623157E+308)
        }
        ieee754.write(buf, value, offset, littleEndian, 52, 8)
        return offset + 8
      }

      Buffer.prototype.writeDoubleLE = function writeDoubleLE(value, offset, noAssert) {
        return writeDouble(this, value, offset, true, noAssert)
      }

      Buffer.prototype.writeDoubleBE = function writeDoubleBE(value, offset, noAssert) {
        return writeDouble(this, value, offset, false, noAssert)
      }

      // copy(targetBuffer, targetStart=0, sourceStart=0, sourceEnd=buffer.length)
      Buffer.prototype.copy = function copy(target, targetStart, start, end) {
        if (!Buffer.isBuffer(target)) throw new TypeError('argument should be a Buffer')
        if (!start) start = 0
        if (!end && end !== 0) end = this.length
        if (targetStart >= target.length) targetStart = target.length
        if (!targetStart) targetStart = 0
        if (end > 0 && end < start) end = start

        // Copy 0 bytes; we're done
        if (end === start) return 0
        if (target.length === 0 || this.length === 0) return 0

        // Fatal error conditions
        if (targetStart < 0) {
          throw new RangeError('targetStart out of bounds')
        }
        if (start < 0 || start >= this.length) throw new RangeError('Index out of range')
        if (end < 0) throw new RangeError('sourceEnd out of bounds')

        // Are we oob?
        if (end > this.length) end = this.length
        if (target.length - targetStart < end - start) {
          end = target.length - targetStart + start
        }

        var len = end - start

        if (this === target && typeof Uint8Array.prototype.copyWithin === 'function') {
          // Use built-in when available, missing from IE11
          this.copyWithin(targetStart, start, end)
        } else if (this === target && start < targetStart && targetStart < end) {
          // descending copy from end
          for (var i = len - 1; i >= 0; --i) {
            target[i + targetStart] = this[i + start]
          }
        } else {
          Uint8Array.prototype.set.call(
            target,
            this.subarray(start, end),
            targetStart
          )
        }

        return len
      }

      // Usage:
      //    buffer.fill(number[, offset[, end]])
      //    buffer.fill(buffer[, offset[, end]])
      //    buffer.fill(string[, offset[, end]][, encoding])
      Buffer.prototype.fill = function fill(val, start, end, encoding) {
        // Handle string cases:
        if (typeof val === 'string') {
          if (typeof start === 'string') {
            encoding = start
            start = 0
            end = this.length
          } else if (typeof end === 'string') {
            encoding = end
            end = this.length
          }
          if (encoding !== undefined && typeof encoding !== 'string') {
            throw new TypeError('encoding must be a string')
          }
          if (typeof encoding === 'string' && !Buffer.isEncoding(encoding)) {
            throw new TypeError('Unknown encoding: ' + encoding)
          }
          if (val.length === 1) {
            var code = val.charCodeAt(0)
            if ((encoding === 'utf8' && code < 128) ||
              encoding === 'latin1') {
              // Fast path: If `val` fits into a single byte, use that numeric value.
              val = code
            }
          }
        } else if (typeof val === 'number') {
          val = val & 255
        }

        // Invalid ranges are not set to a default, so can range check early.
        if (start < 0 || this.length < start || this.length < end) {
          throw new RangeError('Out of range index')
        }

        if (end <= start) {
          return this
        }

        start = start >>> 0
        end = end === undefined ? this.length : end >>> 0

        if (!val) val = 0

        var i
        if (typeof val === 'number') {
          for (i = start; i < end; ++i) {
            this[i] = val
          }
        } else {
          var bytes = Buffer.isBuffer(val)
            ? val
            : Buffer.from(val, encoding)
          var len = bytes.length
          if (len === 0) {
            throw new TypeError('The value "' + val +
              '" is invalid for argument "value"')
          }
          for (i = 0; i < end - start; ++i) {
            this[i + start] = bytes[i % len]
          }
        }

        return this
      }

      // HELPER FUNCTIONS
      // ================

      var INVALID_BASE64_RE = /[^+/0-9A-Za-z-_]/g

      function base64clean(str) {
        // Node takes equal signs as end of the Base64 encoding
        str = str.split('=')[0]
        // Node strips out invalid characters like \n and \t from the string, base64-js does not
        str = str.trim().replace(INVALID_BASE64_RE, '')
        // Node converts strings with length < 2 to ''
        if (str.length < 2) return ''
        // Node allows for non-padded base64 strings (missing trailing ===), base64-js does not
        while (str.length % 4 !== 0) {
          str = str + '='
        }
        return str
      }

      function toHex(n) {
        if (n < 16) return '0' + n.toString(16)
        return n.toString(16)
      }

      function utf8ToBytes(string, units) {
        units = units || Infinity
        var codePoint
        var length = string.length
        var leadSurrogate = null
        var bytes = []

        for (var i = 0; i < length; ++i) {
          codePoint = string.charCodeAt(i)

          // is surrogate component
          if (codePoint > 0xD7FF && codePoint < 0xE000) {
            // last char was a lead
            if (!leadSurrogate) {
              // no lead yet
              if (codePoint > 0xDBFF) {
                // unexpected trail
                if ((units -= 3) > -1) bytes.push(0xEF, 0xBF, 0xBD)
                continue
              } else if (i + 1 === length) {
                // unpaired lead
                if ((units -= 3) > -1) bytes.push(0xEF, 0xBF, 0xBD)
                continue
              }

              // valid lead
              leadSurrogate = codePoint

              continue
            }

            // 2 leads in a row
            if (codePoint < 0xDC00) {
              if ((units -= 3) > -1) bytes.push(0xEF, 0xBF, 0xBD)
              leadSurrogate = codePoint
              continue
            }

            // valid surrogate pair
            codePoint = (leadSurrogate - 0xD800 << 10 | codePoint - 0xDC00) + 0x10000
          } else if (leadSurrogate) {
            // valid bmp char, but last char was a lead
            if ((units -= 3) > -1) bytes.push(0xEF, 0xBF, 0xBD)
          }

          leadSurrogate = null

          // encode utf8
          if (codePoint < 0x80) {
            if ((units -= 1) < 0) break
            bytes.push(codePoint)
          } else if (codePoint < 0x800) {
            if ((units -= 2) < 0) break
            bytes.push(
              codePoint >> 0x6 | 0xC0,
              codePoint & 0x3F | 0x80
            )
          } else if (codePoint < 0x10000) {
            if ((units -= 3) < 0) break
            bytes.push(
              codePoint >> 0xC | 0xE0,
              codePoint >> 0x6 & 0x3F | 0x80,
              codePoint & 0x3F | 0x80
            )
          } else if (codePoint < 0x110000) {
            if ((units -= 4) < 0) break
            bytes.push(
              codePoint >> 0x12 | 0xF0,
              codePoint >> 0xC & 0x3F | 0x80,
              codePoint >> 0x6 & 0x3F | 0x80,
              codePoint & 0x3F | 0x80
            )
          } else {
            throw new Error('Invalid code point')
          }
        }

        return bytes
      }

      function asciiToBytes(str) {
        var byteArray = []
        for (var i = 0; i < str.length; ++i) {
          // Node's code seems to be doing this and not & 0x7F..
          byteArray.push(str.charCodeAt(i) & 0xFF)
        }
        return byteArray
      }

      function utf16leToBytes(str, units) {
        var c, hi, lo
        var byteArray = []
        for (var i = 0; i < str.length; ++i) {
          if ((units -= 2) < 0) break

          c = str.charCodeAt(i)
          hi = c >> 8
          lo = c % 256
          byteArray.push(lo)
          byteArray.push(hi)
        }

        return byteArray
      }

      function base64ToBytes(str) {
        return base64.toByteArray(base64clean(str))
      }

      function blitBuffer(src, dst, offset, length) {
        for (var i = 0; i < length; ++i) {
          if ((i + offset >= dst.length) || (i >= src.length)) break
          dst[i + offset] = src[i]
        }
        return i
      }

      // ArrayBuffer or Uint8Array objects from other contexts (i.e. iframes) do not pass
      // the `instanceof` check but they should be treated as of that type.
      // See: https://github.com/feross/buffer/issues/166
      function isInstance(obj, type) {
        return obj instanceof type ||
          (obj != null && obj.constructor != null && obj.constructor.name != null &&
            obj.constructor.name === type.name)
      }
      function numberIsNaN(obj) {
        // For IE11 support
        return obj !== obj // eslint-disable-line no-self-compare
      }

    }).call(this, require("buffer").Buffer)
  }, { "base64-js": 23, "buffer": 58, "ieee754": 568 }], 59: [function (require, module, exports) {
    module.exports = {
      "100": "Continue",
      "101": "Switching Protocols",
      "102": "Processing",
      "200": "OK",
      "201": "Created",
      "202": "Accepted",
      "203": "Non-Authoritative Information",
      "204": "No Content",
      "205": "Reset Content",
      "206": "Partial Content",
      "207": "Multi-Status",
      "208": "Already Reported",
      "226": "IM Used",
      "300": "Multiple Choices",
      "301": "Moved Permanently",
      "302": "Found",
      "303": "See Other",
      "304": "Not Modified",
      "305": "Use Proxy",
      "307": "Temporary Redirect",
      "308": "Permanent Redirect",
      "400": "Bad Request",
      "401": "Unauthorized",
      "402": "Payment Required",
      "403": "Forbidden",
      "404": "Not Found",
      "405": "Method Not Allowed",
      "406": "Not Acceptable",
      "407": "Proxy Authentication Required",
      "408": "Request Timeout",
      "409": "Conflict",
      "410": "Gone",
      "411": "Length Required",
      "412": "Precondition Failed",
      "413": "Payload Too Large",
      "414": "URI Too Long",
      "415": "Unsupported Media Type",
      "416": "Range Not Satisfiable",
      "417": "Expectation Failed",
      "418": "I'm a teapot",
      "421": "Misdirected Request",
      "422": "Unprocessable Entity",
      "423": "Locked",
      "424": "Failed Dependency",
      "425": "Unordered Collection",
      "426": "Upgrade Required",
      "428": "Precondition Required",
      "429": "Too Many Requests",
      "431": "Request Header Fields Too Large",
      "451": "Unavailable For Legal Reasons",
      "500": "Internal Server Error",
      "501": "Not Implemented",
      "502": "Bad Gateway",
      "503": "Service Unavailable",
      "504": "Gateway Timeout",
      "505": "HTTP Version Not Supported",
      "506": "Variant Also Negotiates",
      "507": "Insufficient Storage",
      "508": "Loop Detected",
      "509": "Bandwidth Limit Exceeded",
      "510": "Not Extended",
      "511": "Network Authentication Required"
    }

  }, {}], 60: [function (require, module, exports) {
    /**
     * chroma.js - JavaScript library for color conversions
     *
     * Copyright (c) 2011-2019, Gregor Aisch
     * All rights reserved.
     *
     * Redistribution and use in source and binary forms, with or without
     * modification, are permitted provided that the following conditions are met:
     *
     * 1. Redistributions of source code must retain the above copyright notice, this
     * list of conditions and the following disclaimer.
     *
     * 2. Redistributions in binary form must reproduce the above copyright notice,
     * this list of conditions and the following disclaimer in the documentation
     * and/or other materials provided with the distribution.
     *
     * 3. The name Gregor Aisch may not be used to endorse or promote products
     * derived from this software without specific prior written permission.
     *
     * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
     * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
     * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
     * DISCLAIMED. IN NO EVENT SHALL GREGOR AISCH OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
     * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
     * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
     * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
     * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
     * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
     * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
     *
     * -------------------------------------------------------
     *
     * chroma.js includes colors from colorbrewer2.org, which are released under
     * the following license:
     *
     * Copyright (c) 2002 Cynthia Brewer, Mark Harrower,
     * and The Pennsylvania State University.
     *
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing,
     * software distributed under the License is distributed on an
     * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
     * either express or implied. See the License for the specific
     * language governing permissions and limitations under the License.
     *
     * ------------------------------------------------------
     *
     * Named colors are taken from X11 Color Names.
     * http://www.w3.org/TR/css3-color/#svg-color
     *
     * @preserve
     */

    (function (global, factory) {
      typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
        typeof define === 'function' && define.amd ? define(factory) :
          (global.chroma = factory());
    }(this, (function () {
      'use strict';

      var limit = function (x, min, max) {
        if (min === void 0) min = 0;
        if (max === void 0) max = 1;

        return x < min ? min : x > max ? max : x;
      };

      var clip_rgb = function (rgb) {
        rgb._clipped = false;
        rgb._unclipped = rgb.slice(0);
        for (var i = 0; i <= 3; i++) {
          if (i < 3) {
            if (rgb[i] < 0 || rgb[i] > 255) { rgb._clipped = true; }
            rgb[i] = limit(rgb[i], 0, 255);
          } else if (i === 3) {
            rgb[i] = limit(rgb[i], 0, 1);
          }
        }
        return rgb;
      };

      // ported from jQuery's $.type
      var classToType = {};
      for (var i = 0, list = ['Boolean', 'Number', 'String', 'Function', 'Array', 'Date', 'RegExp', 'Undefined', 'Null']; i < list.length; i += 1) {
        var name = list[i];

        classToType[("[object " + name + "]")] = name.toLowerCase();
      }
      var type = function (obj) {
        return classToType[Object.prototype.toString.call(obj)] || "object";
      };

      var unpack = function (args, keyOrder) {
        if (keyOrder === void 0) keyOrder = null;

        // if called with more than 3 arguments, we return the arguments
        if (args.length >= 3) { return Array.prototype.slice.call(args); }
        // with less than 3 args we check if first arg is object
        // and use the keyOrder string to extract and sort properties
        if (type(args[0]) == 'object' && keyOrder) {
          return keyOrder.split('')
            .filter(function (k) { return args[0][k] !== undefined; })
            .map(function (k) { return args[0][k]; });
        }
        // otherwise we just return the first argument
        // (which we suppose is an array of args)
        return args[0];
      };

      var last = function (args) {
        if (args.length < 2) { return null; }
        var l = args.length - 1;
        if (type(args[l]) == 'string') { return args[l].toLowerCase(); }
        return null;
      };

      var PI = Math.PI;

      var utils = {
        clip_rgb: clip_rgb,
        limit: limit,
        type: type,
        unpack: unpack,
        last: last,
        PI: PI,
        TWOPI: PI * 2,
        PITHIRD: PI / 3,
        DEG2RAD: PI / 180,
        RAD2DEG: 180 / PI
      };

      var input = {
        format: {},
        autodetect: []
      };

      var last$1 = utils.last;
      var clip_rgb$1 = utils.clip_rgb;
      var type$1 = utils.type;


      var Color = function Color() {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var me = this;
        if (type$1(args[0]) === 'object' &&
          args[0].constructor &&
          args[0].constructor === this.constructor) {
          // the argument is already a Color instance
          return args[0];
        }

        // last argument could be the mode
        var mode = last$1(args);
        var autodetect = false;

        if (!mode) {
          autodetect = true;
          if (!input.sorted) {
            input.autodetect = input.autodetect.sort(function (a, b) { return b.p - a.p; });
            input.sorted = true;
          }
          // auto-detect format
          for (var i = 0, list = input.autodetect; i < list.length; i += 1) {
            var chk = list[i];

            mode = chk.test.apply(chk, args);
            if (mode) { break; }
          }
        }

        if (input.format[mode]) {
          var rgb = input.format[mode].apply(null, autodetect ? args : args.slice(0, -1));
          me._rgb = clip_rgb$1(rgb);
        } else {
          throw new Error('unknown format: ' + args);
        }

        // add alpha channel
        if (me._rgb.length === 3) { me._rgb.push(1); }
      };

      Color.prototype.toString = function toString() {
        if (type$1(this.hex) == 'function') { return this.hex(); }
        return ("[" + (this._rgb.join(',')) + "]");
      };

      var Color_1 = Color;

      var chroma = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(chroma.Color, [null].concat(args)));
      };

      chroma.Color = Color_1;
      chroma.version = '2.1.0';

      var chroma_1 = chroma;

      var unpack$1 = utils.unpack;
      var max = Math.max;

      var rgb2cmyk = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var ref = unpack$1(args, 'rgb');
        var r = ref[0];
        var g = ref[1];
        var b = ref[2];
        r = r / 255;
        g = g / 255;
        b = b / 255;
        var k = 1 - max(r, max(g, b));
        var f = k < 1 ? 1 / (1 - k) : 0;
        var c = (1 - r - k) * f;
        var m = (1 - g - k) * f;
        var y = (1 - b - k) * f;
        return [c, m, y, k];
      };

      var rgb2cmyk_1 = rgb2cmyk;

      var unpack$2 = utils.unpack;

      var cmyk2rgb = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        args = unpack$2(args, 'cmyk');
        var c = args[0];
        var m = args[1];
        var y = args[2];
        var k = args[3];
        var alpha = args.length > 4 ? args[4] : 1;
        if (k === 1) { return [0, 0, 0, alpha]; }
        return [
          c >= 1 ? 0 : 255 * (1 - c) * (1 - k), // r
          m >= 1 ? 0 : 255 * (1 - m) * (1 - k), // g
          y >= 1 ? 0 : 255 * (1 - y) * (1 - k), // b
          alpha
        ];
      };

      var cmyk2rgb_1 = cmyk2rgb;

      var unpack$3 = utils.unpack;
      var type$2 = utils.type;



      Color_1.prototype.cmyk = function () {
        return rgb2cmyk_1(this._rgb);
      };

      chroma_1.cmyk = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['cmyk'])));
      };

      input.format.cmyk = cmyk2rgb_1;

      input.autodetect.push({
        p: 2,
        test: function () {
          var args = [], len = arguments.length;
          while (len--) args[len] = arguments[len];

          args = unpack$3(args, 'cmyk');
          if (type$2(args) === 'array' && args.length === 4) {
            return 'cmyk';
          }
        }
      });

      var unpack$4 = utils.unpack;
      var last$2 = utils.last;
      var rnd = function (a) { return Math.round(a * 100) / 100; };

      /*
       * supported arguments:
       * - hsl2css(h,s,l)
       * - hsl2css(h,s,l,a)
       * - hsl2css([h,s,l], mode)
       * - hsl2css([h,s,l,a], mode)
       * - hsl2css({h,s,l,a}, mode)
       */
      var hsl2css = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var hsla = unpack$4(args, 'hsla');
        var mode = last$2(args) || 'lsa';
        hsla[0] = rnd(hsla[0] || 0);
        hsla[1] = rnd(hsla[1] * 100) + '%';
        hsla[2] = rnd(hsla[2] * 100) + '%';
        if (mode === 'hsla' || (hsla.length > 3 && hsla[3] < 1)) {
          hsla[3] = hsla.length > 3 ? hsla[3] : 1;
          mode = 'hsla';
        } else {
          hsla.length = 3;
        }
        return (mode + "(" + (hsla.join(',')) + ")");
      };

      var hsl2css_1 = hsl2css;

      var unpack$5 = utils.unpack;

      /*
       * supported arguments:
       * - rgb2hsl(r,g,b)
       * - rgb2hsl(r,g,b,a)
       * - rgb2hsl([r,g,b])
       * - rgb2hsl([r,g,b,a])
       * - rgb2hsl({r,g,b,a})
       */
      var rgb2hsl = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        args = unpack$5(args, 'rgba');
        var r = args[0];
        var g = args[1];
        var b = args[2];

        r /= 255;
        g /= 255;
        b /= 255;

        var min = Math.min(r, g, b);
        var max = Math.max(r, g, b);

        var l = (max + min) / 2;
        var s, h;

        if (max === min) {
          s = 0;
          h = Number.NaN;
        } else {
          s = l < 0.5 ? (max - min) / (max + min) : (max - min) / (2 - max - min);
        }

        if (r == max) { h = (g - b) / (max - min); }
        else if (g == max) { h = 2 + (b - r) / (max - min); }
        else if (b == max) { h = 4 + (r - g) / (max - min); }

        h *= 60;
        if (h < 0) { h += 360; }
        if (args.length > 3 && args[3] !== undefined) { return [h, s, l, args[3]]; }
        return [h, s, l];
      };

      var rgb2hsl_1 = rgb2hsl;

      var unpack$6 = utils.unpack;
      var last$3 = utils.last;


      var round = Math.round;

      /*
       * supported arguments:
       * - rgb2css(r,g,b)
       * - rgb2css(r,g,b,a)
       * - rgb2css([r,g,b], mode)
       * - rgb2css([r,g,b,a], mode)
       * - rgb2css({r,g,b,a}, mode)
       */
      var rgb2css = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var rgba = unpack$6(args, 'rgba');
        var mode = last$3(args) || 'rgb';
        if (mode.substr(0, 3) == 'hsl') {
          return hsl2css_1(rgb2hsl_1(rgba), mode);
        }
        rgba[0] = round(rgba[0]);
        rgba[1] = round(rgba[1]);
        rgba[2] = round(rgba[2]);
        if (mode === 'rgba' || (rgba.length > 3 && rgba[3] < 1)) {
          rgba[3] = rgba.length > 3 ? rgba[3] : 1;
          mode = 'rgba';
        }
        return (mode + "(" + (rgba.slice(0, mode === 'rgb' ? 3 : 4).join(',')) + ")");
      };

      var rgb2css_1 = rgb2css;

      var unpack$7 = utils.unpack;
      var round$1 = Math.round;

      var hsl2rgb = function () {
        var assign;

        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];
        args = unpack$7(args, 'hsl');
        var h = args[0];
        var s = args[1];
        var l = args[2];
        var r, g, b;
        if (s === 0) {
          r = g = b = l * 255;
        } else {
          var t3 = [0, 0, 0];
          var c = [0, 0, 0];
          var t2 = l < 0.5 ? l * (1 + s) : l + s - l * s;
          var t1 = 2 * l - t2;
          var h_ = h / 360;
          t3[0] = h_ + 1 / 3;
          t3[1] = h_;
          t3[2] = h_ - 1 / 3;
          for (var i = 0; i < 3; i++) {
            if (t3[i] < 0) { t3[i] += 1; }
            if (t3[i] > 1) { t3[i] -= 1; }
            if (6 * t3[i] < 1) { c[i] = t1 + (t2 - t1) * 6 * t3[i]; }
            else if (2 * t3[i] < 1) { c[i] = t2; }
            else if (3 * t3[i] < 2) { c[i] = t1 + (t2 - t1) * ((2 / 3) - t3[i]) * 6; }
            else { c[i] = t1; }
          }
          (assign = [round$1(c[0] * 255), round$1(c[1] * 255), round$1(c[2] * 255)], r = assign[0], g = assign[1], b = assign[2]);
        }
        if (args.length > 3) {
          // keep alpha channel
          return [r, g, b, args[3]];
        }
        return [r, g, b, 1];
      };

      var hsl2rgb_1 = hsl2rgb;

      var RE_RGB = /^rgb\(\s*(-?\d+),\s*(-?\d+)\s*,\s*(-?\d+)\s*\)$/;
      var RE_RGBA = /^rgba\(\s*(-?\d+),\s*(-?\d+)\s*,\s*(-?\d+)\s*,\s*([01]|[01]?\.\d+)\)$/;
      var RE_RGB_PCT = /^rgb\(\s*(-?\d+(?:\.\d+)?)%,\s*(-?\d+(?:\.\d+)?)%\s*,\s*(-?\d+(?:\.\d+)?)%\s*\)$/;
      var RE_RGBA_PCT = /^rgba\(\s*(-?\d+(?:\.\d+)?)%,\s*(-?\d+(?:\.\d+)?)%\s*,\s*(-?\d+(?:\.\d+)?)%\s*,\s*([01]|[01]?\.\d+)\)$/;
      var RE_HSL = /^hsl\(\s*(-?\d+(?:\.\d+)?),\s*(-?\d+(?:\.\d+)?)%\s*,\s*(-?\d+(?:\.\d+)?)%\s*\)$/;
      var RE_HSLA = /^hsla\(\s*(-?\d+(?:\.\d+)?),\s*(-?\d+(?:\.\d+)?)%\s*,\s*(-?\d+(?:\.\d+)?)%\s*,\s*([01]|[01]?\.\d+)\)$/;

      var round$2 = Math.round;

      var css2rgb = function (css) {
        css = css.toLowerCase().trim();
        var m;

        if (input.format.named) {
          try {
            return input.format.named(css);
          } catch (e) {
            // eslint-disable-next-line
          }
        }

        // rgb(250,20,0)
        if ((m = css.match(RE_RGB))) {
          var rgb = m.slice(1, 4);
          for (var i = 0; i < 3; i++) {
            rgb[i] = +rgb[i];
          }
          rgb[3] = 1;  // default alpha
          return rgb;
        }

        // rgba(250,20,0,0.4)
        if ((m = css.match(RE_RGBA))) {
          var rgb$1 = m.slice(1, 5);
          for (var i$1 = 0; i$1 < 4; i$1++) {
            rgb$1[i$1] = +rgb$1[i$1];
          }
          return rgb$1;
        }

        // rgb(100%,0%,0%)
        if ((m = css.match(RE_RGB_PCT))) {
          var rgb$2 = m.slice(1, 4);
          for (var i$2 = 0; i$2 < 3; i$2++) {
            rgb$2[i$2] = round$2(rgb$2[i$2] * 2.55);
          }
          rgb$2[3] = 1;  // default alpha
          return rgb$2;
        }

        // rgba(100%,0%,0%,0.4)
        if ((m = css.match(RE_RGBA_PCT))) {
          var rgb$3 = m.slice(1, 5);
          for (var i$3 = 0; i$3 < 3; i$3++) {
            rgb$3[i$3] = round$2(rgb$3[i$3] * 2.55);
          }
          rgb$3[3] = +rgb$3[3];
          return rgb$3;
        }

        // hsl(0,100%,50%)
        if ((m = css.match(RE_HSL))) {
          var hsl = m.slice(1, 4);
          hsl[1] *= 0.01;
          hsl[2] *= 0.01;
          var rgb$4 = hsl2rgb_1(hsl);
          rgb$4[3] = 1;
          return rgb$4;
        }

        // hsla(0,100%,50%,0.5)
        if ((m = css.match(RE_HSLA))) {
          var hsl$1 = m.slice(1, 4);
          hsl$1[1] *= 0.01;
          hsl$1[2] *= 0.01;
          var rgb$5 = hsl2rgb_1(hsl$1);
          rgb$5[3] = +m[4];  // default alpha = 1
          return rgb$5;
        }
      };

      css2rgb.test = function (s) {
        return RE_RGB.test(s) ||
          RE_RGBA.test(s) ||
          RE_RGB_PCT.test(s) ||
          RE_RGBA_PCT.test(s) ||
          RE_HSL.test(s) ||
          RE_HSLA.test(s);
      };

      var css2rgb_1 = css2rgb;

      var type$3 = utils.type;




      Color_1.prototype.css = function (mode) {
        return rgb2css_1(this._rgb, mode);
      };

      chroma_1.css = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['css'])));
      };

      input.format.css = css2rgb_1;

      input.autodetect.push({
        p: 5,
        test: function (h) {
          var rest = [], len = arguments.length - 1;
          while (len-- > 0) rest[len] = arguments[len + 1];

          if (!rest.length && type$3(h) === 'string' && css2rgb_1.test(h)) {
            return 'css';
          }
        }
      });

      var unpack$8 = utils.unpack;

      input.format.gl = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var rgb = unpack$8(args, 'rgba');
        rgb[0] *= 255;
        rgb[1] *= 255;
        rgb[2] *= 255;
        return rgb;
      };

      chroma_1.gl = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['gl'])));
      };

      Color_1.prototype.gl = function () {
        var rgb = this._rgb;
        return [rgb[0] / 255, rgb[1] / 255, rgb[2] / 255, rgb[3]];
      };

      var unpack$9 = utils.unpack;

      var rgb2hcg = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var ref = unpack$9(args, 'rgb');
        var r = ref[0];
        var g = ref[1];
        var b = ref[2];
        var min = Math.min(r, g, b);
        var max = Math.max(r, g, b);
        var delta = max - min;
        var c = delta * 100 / 255;
        var _g = min / (255 - delta) * 100;
        var h;
        if (delta === 0) {
          h = Number.NaN;
        } else {
          if (r === max) { h = (g - b) / delta; }
          if (g === max) { h = 2 + (b - r) / delta; }
          if (b === max) { h = 4 + (r - g) / delta; }
          h *= 60;
          if (h < 0) { h += 360; }
        }
        return [h, c, _g];
      };

      var rgb2hcg_1 = rgb2hcg;

      var unpack$a = utils.unpack;
      var floor = Math.floor;

      /*
       * this is basically just HSV with some minor tweaks
       *
       * hue.. [0..360]
       * chroma .. [0..1]
       * grayness .. [0..1]
       */

      var hcg2rgb = function () {
        var assign, assign$1, assign$2, assign$3, assign$4, assign$5;

        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];
        args = unpack$a(args, 'hcg');
        var h = args[0];
        var c = args[1];
        var _g = args[2];
        var r, g, b;
        _g = _g * 255;
        var _c = c * 255;
        if (c === 0) {
          r = g = b = _g;
        } else {
          if (h === 360) { h = 0; }
          if (h > 360) { h -= 360; }
          if (h < 0) { h += 360; }
          h /= 60;
          var i = floor(h);
          var f = h - i;
          var p = _g * (1 - c);
          var q = p + _c * (1 - f);
          var t = p + _c * f;
          var v = p + _c;
          switch (i) {
            case 0: (assign = [v, t, p], r = assign[0], g = assign[1], b = assign[2]); break
            case 1: (assign$1 = [q, v, p], r = assign$1[0], g = assign$1[1], b = assign$1[2]); break
            case 2: (assign$2 = [p, v, t], r = assign$2[0], g = assign$2[1], b = assign$2[2]); break
            case 3: (assign$3 = [p, q, v], r = assign$3[0], g = assign$3[1], b = assign$3[2]); break
            case 4: (assign$4 = [t, p, v], r = assign$4[0], g = assign$4[1], b = assign$4[2]); break
            case 5: (assign$5 = [v, p, q], r = assign$5[0], g = assign$5[1], b = assign$5[2]); break
          }
        }
        return [r, g, b, args.length > 3 ? args[3] : 1];
      };

      var hcg2rgb_1 = hcg2rgb;

      var unpack$b = utils.unpack;
      var type$4 = utils.type;






      Color_1.prototype.hcg = function () {
        return rgb2hcg_1(this._rgb);
      };

      chroma_1.hcg = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['hcg'])));
      };

      input.format.hcg = hcg2rgb_1;

      input.autodetect.push({
        p: 1,
        test: function () {
          var args = [], len = arguments.length;
          while (len--) args[len] = arguments[len];

          args = unpack$b(args, 'hcg');
          if (type$4(args) === 'array' && args.length === 3) {
            return 'hcg';
          }
        }
      });

      var unpack$c = utils.unpack;
      var last$4 = utils.last;
      var round$3 = Math.round;

      var rgb2hex = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var ref = unpack$c(args, 'rgba');
        var r = ref[0];
        var g = ref[1];
        var b = ref[2];
        var a = ref[3];
        var mode = last$4(args) || 'auto';
        if (a === undefined) { a = 1; }
        if (mode === 'auto') {
          mode = a < 1 ? 'rgba' : 'rgb';
        }
        r = round$3(r);
        g = round$3(g);
        b = round$3(b);
        var u = r << 16 | g << 8 | b;
        var str = "000000" + u.toString(16); //#.toUpperCase();
        str = str.substr(str.length - 6);
        var hxa = '0' + round$3(a * 255).toString(16);
        hxa = hxa.substr(hxa.length - 2);
        switch (mode.toLowerCase()) {
          case 'rgba': return ("#" + str + hxa);
          case 'argb': return ("#" + hxa + str);
          default: return ("#" + str);
        }
      };

      var rgb2hex_1 = rgb2hex;

      var RE_HEX = /^#?([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/;
      var RE_HEXA = /^#?([A-Fa-f0-9]{8}|[A-Fa-f0-9]{4})$/;

      var hex2rgb = function (hex) {
        if (hex.match(RE_HEX)) {
          // remove optional leading #
          if (hex.length === 4 || hex.length === 7) {
            hex = hex.substr(1);
          }
          // expand short-notation to full six-digit
          if (hex.length === 3) {
            hex = hex.split('');
            hex = hex[0] + hex[0] + hex[1] + hex[1] + hex[2] + hex[2];
          }
          var u = parseInt(hex, 16);
          var r = u >> 16;
          var g = u >> 8 & 0xFF;
          var b = u & 0xFF;
          return [r, g, b, 1];
        }

        // match rgba hex format, eg #FF000077
        if (hex.match(RE_HEXA)) {
          if (hex.length === 5 || hex.length === 9) {
            // remove optional leading #
            hex = hex.substr(1);
          }
          // expand short-notation to full eight-digit
          if (hex.length === 4) {
            hex = hex.split('');
            hex = hex[0] + hex[0] + hex[1] + hex[1] + hex[2] + hex[2] + hex[3] + hex[3];
          }
          var u$1 = parseInt(hex, 16);
          var r$1 = u$1 >> 24 & 0xFF;
          var g$1 = u$1 >> 16 & 0xFF;
          var b$1 = u$1 >> 8 & 0xFF;
          var a = Math.round((u$1 & 0xFF) / 0xFF * 100) / 100;
          return [r$1, g$1, b$1, a];
        }

        // we used to check for css colors here
        // if _input.css? and rgb = _input.css hex
        //     return rgb

        throw new Error(("unknown hex color: " + hex));
      };

      var hex2rgb_1 = hex2rgb;

      var type$5 = utils.type;




      Color_1.prototype.hex = function (mode) {
        return rgb2hex_1(this._rgb, mode);
      };

      chroma_1.hex = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['hex'])));
      };

      input.format.hex = hex2rgb_1;
      input.autodetect.push({
        p: 4,
        test: function (h) {
          var rest = [], len = arguments.length - 1;
          while (len-- > 0) rest[len] = arguments[len + 1];

          if (!rest.length && type$5(h) === 'string' && [3, 4, 5, 6, 7, 8, 9].indexOf(h.length) >= 0) {
            return 'hex';
          }
        }
      });

      var unpack$d = utils.unpack;
      var TWOPI = utils.TWOPI;
      var min = Math.min;
      var sqrt = Math.sqrt;
      var acos = Math.acos;

      var rgb2hsi = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        /*
        borrowed from here:
        http://hummer.stanford.edu/museinfo/doc/examples/humdrum/keyscape2/rgb2hsi.cpp
        */
        var ref = unpack$d(args, 'rgb');
        var r = ref[0];
        var g = ref[1];
        var b = ref[2];
        r /= 255;
        g /= 255;
        b /= 255;
        var h;
        var min_ = min(r, g, b);
        var i = (r + g + b) / 3;
        var s = i > 0 ? 1 - min_ / i : 0;
        if (s === 0) {
          h = NaN;
        } else {
          h = ((r - g) + (r - b)) / 2;
          h /= sqrt((r - g) * (r - g) + (r - b) * (g - b));
          h = acos(h);
          if (b > g) {
            h = TWOPI - h;
          }
          h /= TWOPI;
        }
        return [h * 360, s, i];
      };

      var rgb2hsi_1 = rgb2hsi;

      var unpack$e = utils.unpack;
      var limit$1 = utils.limit;
      var TWOPI$1 = utils.TWOPI;
      var PITHIRD = utils.PITHIRD;
      var cos = Math.cos;

      /*
       * hue [0..360]
       * saturation [0..1]
       * intensity [0..1]
       */
      var hsi2rgb = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        /*
        borrowed from here:
        http://hummer.stanford.edu/museinfo/doc/examples/humdrum/keyscape2/hsi2rgb.cpp
        */
        args = unpack$e(args, 'hsi');
        var h = args[0];
        var s = args[1];
        var i = args[2];
        var r, g, b;

        if (isNaN(h)) { h = 0; }
        if (isNaN(s)) { s = 0; }
        // normalize hue
        if (h > 360) { h -= 360; }
        if (h < 0) { h += 360; }
        h /= 360;
        if (h < 1 / 3) {
          b = (1 - s) / 3;
          r = (1 + s * cos(TWOPI$1 * h) / cos(PITHIRD - TWOPI$1 * h)) / 3;
          g = 1 - (b + r);
        } else if (h < 2 / 3) {
          h -= 1 / 3;
          r = (1 - s) / 3;
          g = (1 + s * cos(TWOPI$1 * h) / cos(PITHIRD - TWOPI$1 * h)) / 3;
          b = 1 - (r + g);
        } else {
          h -= 2 / 3;
          g = (1 - s) / 3;
          b = (1 + s * cos(TWOPI$1 * h) / cos(PITHIRD - TWOPI$1 * h)) / 3;
          r = 1 - (g + b);
        }
        r = limit$1(i * r * 3);
        g = limit$1(i * g * 3);
        b = limit$1(i * b * 3);
        return [r * 255, g * 255, b * 255, args.length > 3 ? args[3] : 1];
      };

      var hsi2rgb_1 = hsi2rgb;

      var unpack$f = utils.unpack;
      var type$6 = utils.type;






      Color_1.prototype.hsi = function () {
        return rgb2hsi_1(this._rgb);
      };

      chroma_1.hsi = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['hsi'])));
      };

      input.format.hsi = hsi2rgb_1;

      input.autodetect.push({
        p: 2,
        test: function () {
          var args = [], len = arguments.length;
          while (len--) args[len] = arguments[len];

          args = unpack$f(args, 'hsi');
          if (type$6(args) === 'array' && args.length === 3) {
            return 'hsi';
          }
        }
      });

      var unpack$g = utils.unpack;
      var type$7 = utils.type;






      Color_1.prototype.hsl = function () {
        return rgb2hsl_1(this._rgb);
      };

      chroma_1.hsl = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['hsl'])));
      };

      input.format.hsl = hsl2rgb_1;

      input.autodetect.push({
        p: 2,
        test: function () {
          var args = [], len = arguments.length;
          while (len--) args[len] = arguments[len];

          args = unpack$g(args, 'hsl');
          if (type$7(args) === 'array' && args.length === 3) {
            return 'hsl';
          }
        }
      });

      var unpack$h = utils.unpack;
      var min$1 = Math.min;
      var max$1 = Math.max;

      /*
       * supported arguments:
       * - rgb2hsv(r,g,b)
       * - rgb2hsv([r,g,b])
       * - rgb2hsv({r,g,b})
       */
      var rgb2hsl$1 = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        args = unpack$h(args, 'rgb');
        var r = args[0];
        var g = args[1];
        var b = args[2];
        var min_ = min$1(r, g, b);
        var max_ = max$1(r, g, b);
        var delta = max_ - min_;
        var h, s, v;
        v = max_ / 255.0;
        if (max_ === 0) {
          h = Number.NaN;
          s = 0;
        } else {
          s = delta / max_;
          if (r === max_) { h = (g - b) / delta; }
          if (g === max_) { h = 2 + (b - r) / delta; }
          if (b === max_) { h = 4 + (r - g) / delta; }
          h *= 60;
          if (h < 0) { h += 360; }
        }
        return [h, s, v]
      };

      var rgb2hsv = rgb2hsl$1;

      var unpack$i = utils.unpack;
      var floor$1 = Math.floor;

      var hsv2rgb = function () {
        var assign, assign$1, assign$2, assign$3, assign$4, assign$5;

        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];
        args = unpack$i(args, 'hsv');
        var h = args[0];
        var s = args[1];
        var v = args[2];
        var r, g, b;
        v *= 255;
        if (s === 0) {
          r = g = b = v;
        } else {
          if (h === 360) { h = 0; }
          if (h > 360) { h -= 360; }
          if (h < 0) { h += 360; }
          h /= 60;

          var i = floor$1(h);
          var f = h - i;
          var p = v * (1 - s);
          var q = v * (1 - s * f);
          var t = v * (1 - s * (1 - f));

          switch (i) {
            case 0: (assign = [v, t, p], r = assign[0], g = assign[1], b = assign[2]); break
            case 1: (assign$1 = [q, v, p], r = assign$1[0], g = assign$1[1], b = assign$1[2]); break
            case 2: (assign$2 = [p, v, t], r = assign$2[0], g = assign$2[1], b = assign$2[2]); break
            case 3: (assign$3 = [p, q, v], r = assign$3[0], g = assign$3[1], b = assign$3[2]); break
            case 4: (assign$4 = [t, p, v], r = assign$4[0], g = assign$4[1], b = assign$4[2]); break
            case 5: (assign$5 = [v, p, q], r = assign$5[0], g = assign$5[1], b = assign$5[2]); break
          }
        }
        return [r, g, b, args.length > 3 ? args[3] : 1];
      };

      var hsv2rgb_1 = hsv2rgb;

      var unpack$j = utils.unpack;
      var type$8 = utils.type;






      Color_1.prototype.hsv = function () {
        return rgb2hsv(this._rgb);
      };

      chroma_1.hsv = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['hsv'])));
      };

      input.format.hsv = hsv2rgb_1;

      input.autodetect.push({
        p: 2,
        test: function () {
          var args = [], len = arguments.length;
          while (len--) args[len] = arguments[len];

          args = unpack$j(args, 'hsv');
          if (type$8(args) === 'array' && args.length === 3) {
            return 'hsv';
          }
        }
      });

      var labConstants = {
        // Corresponds roughly to RGB brighter/darker
        Kn: 18,

        // D65 standard referent
        Xn: 0.950470,
        Yn: 1,
        Zn: 1.088830,

        t0: 0.137931034,  // 4 / 29
        t1: 0.206896552,  // 6 / 29
        t2: 0.12841855,   // 3 * t1 * t1
        t3: 0.008856452,  // t1 * t1 * t1
      };

      var unpack$k = utils.unpack;
      var pow = Math.pow;

      var rgb2lab = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var ref = unpack$k(args, 'rgb');
        var r = ref[0];
        var g = ref[1];
        var b = ref[2];
        var ref$1 = rgb2xyz(r, g, b);
        var x = ref$1[0];
        var y = ref$1[1];
        var z = ref$1[2];
        var l = 116 * y - 16;
        return [l < 0 ? 0 : l, 500 * (x - y), 200 * (y - z)];
      };

      var rgb_xyz = function (r) {
        if ((r /= 255) <= 0.04045) { return r / 12.92; }
        return pow((r + 0.055) / 1.055, 2.4);
      };

      var xyz_lab = function (t) {
        if (t > labConstants.t3) { return pow(t, 1 / 3); }
        return t / labConstants.t2 + labConstants.t0;
      };

      var rgb2xyz = function (r, g, b) {
        r = rgb_xyz(r);
        g = rgb_xyz(g);
        b = rgb_xyz(b);
        var x = xyz_lab((0.4124564 * r + 0.3575761 * g + 0.1804375 * b) / labConstants.Xn);
        var y = xyz_lab((0.2126729 * r + 0.7151522 * g + 0.0721750 * b) / labConstants.Yn);
        var z = xyz_lab((0.0193339 * r + 0.1191920 * g + 0.9503041 * b) / labConstants.Zn);
        return [x, y, z];
      };

      var rgb2lab_1 = rgb2lab;

      var unpack$l = utils.unpack;
      var pow$1 = Math.pow;

      /*
       * L* [0..100]
       * a [-100..100]
       * b [-100..100]
       */
      var lab2rgb = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        args = unpack$l(args, 'lab');
        var l = args[0];
        var a = args[1];
        var b = args[2];
        var x, y, z, r, g, b_;

        y = (l + 16) / 116;
        x = isNaN(a) ? y : y + a / 500;
        z = isNaN(b) ? y : y - b / 200;

        y = labConstants.Yn * lab_xyz(y);
        x = labConstants.Xn * lab_xyz(x);
        z = labConstants.Zn * lab_xyz(z);

        r = xyz_rgb(3.2404542 * x - 1.5371385 * y - 0.4985314 * z);  // D65 -> sRGB
        g = xyz_rgb(-0.9692660 * x + 1.8760108 * y + 0.0415560 * z);
        b_ = xyz_rgb(0.0556434 * x - 0.2040259 * y + 1.0572252 * z);

        return [r, g, b_, args.length > 3 ? args[3] : 1];
      };

      var xyz_rgb = function (r) {
        return 255 * (r <= 0.00304 ? 12.92 * r : 1.055 * pow$1(r, 1 / 2.4) - 0.055)
      };

      var lab_xyz = function (t) {
        return t > labConstants.t1 ? t * t * t : labConstants.t2 * (t - labConstants.t0)
      };

      var lab2rgb_1 = lab2rgb;

      var unpack$m = utils.unpack;
      var type$9 = utils.type;






      Color_1.prototype.lab = function () {
        return rgb2lab_1(this._rgb);
      };

      chroma_1.lab = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['lab'])));
      };

      input.format.lab = lab2rgb_1;

      input.autodetect.push({
        p: 2,
        test: function () {
          var args = [], len = arguments.length;
          while (len--) args[len] = arguments[len];

          args = unpack$m(args, 'lab');
          if (type$9(args) === 'array' && args.length === 3) {
            return 'lab';
          }
        }
      });

      var unpack$n = utils.unpack;
      var RAD2DEG = utils.RAD2DEG;
      var sqrt$1 = Math.sqrt;
      var atan2 = Math.atan2;
      var round$4 = Math.round;

      var lab2lch = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var ref = unpack$n(args, 'lab');
        var l = ref[0];
        var a = ref[1];
        var b = ref[2];
        var c = sqrt$1(a * a + b * b);
        var h = (atan2(b, a) * RAD2DEG + 360) % 360;
        if (round$4(c * 10000) === 0) { h = Number.NaN; }
        return [l, c, h];
      };

      var lab2lch_1 = lab2lch;

      var unpack$o = utils.unpack;



      var rgb2lch = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var ref = unpack$o(args, 'rgb');
        var r = ref[0];
        var g = ref[1];
        var b = ref[2];
        var ref$1 = rgb2lab_1(r, g, b);
        var l = ref$1[0];
        var a = ref$1[1];
        var b_ = ref$1[2];
        return lab2lch_1(l, a, b_);
      };

      var rgb2lch_1 = rgb2lch;

      var unpack$p = utils.unpack;
      var DEG2RAD = utils.DEG2RAD;
      var sin = Math.sin;
      var cos$1 = Math.cos;

      var lch2lab = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        /*
        Convert from a qualitative parameter h and a quantitative parameter l to a 24-bit pixel.
        These formulas were invented by David Dalrymple to obtain maximum contrast without going
        out of gamut if the parameters are in the range 0-1.

        A saturation multiplier was added by Gregor Aisch
        */
        var ref = unpack$p(args, 'lch');
        var l = ref[0];
        var c = ref[1];
        var h = ref[2];
        if (isNaN(h)) { h = 0; }
        h = h * DEG2RAD;
        return [l, cos$1(h) * c, sin(h) * c]
      };

      var lch2lab_1 = lch2lab;

      var unpack$q = utils.unpack;



      var lch2rgb = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        args = unpack$q(args, 'lch');
        var l = args[0];
        var c = args[1];
        var h = args[2];
        var ref = lch2lab_1(l, c, h);
        var L = ref[0];
        var a = ref[1];
        var b_ = ref[2];
        var ref$1 = lab2rgb_1(L, a, b_);
        var r = ref$1[0];
        var g = ref$1[1];
        var b = ref$1[2];
        return [r, g, b, args.length > 3 ? args[3] : 1];
      };

      var lch2rgb_1 = lch2rgb;

      var unpack$r = utils.unpack;


      var hcl2rgb = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var hcl = unpack$r(args, 'hcl').reverse();
        return lch2rgb_1.apply(void 0, hcl);
      };

      var hcl2rgb_1 = hcl2rgb;

      var unpack$s = utils.unpack;
      var type$a = utils.type;






      Color_1.prototype.lch = function () { return rgb2lch_1(this._rgb); };
      Color_1.prototype.hcl = function () { return rgb2lch_1(this._rgb).reverse(); };

      chroma_1.lch = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['lch'])));
      };
      chroma_1.hcl = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['hcl'])));
      };

      input.format.lch = lch2rgb_1;
      input.format.hcl = hcl2rgb_1;

      ['lch', 'hcl'].forEach(function (m) {
        return input.autodetect.push({
          p: 2,
          test: function () {
            var args = [], len = arguments.length;
            while (len--) args[len] = arguments[len];

            args = unpack$s(args, m);
            if (type$a(args) === 'array' && args.length === 3) {
              return m;
            }
          }
        });
      });

      /**
        X11 color names

        http://www.w3.org/TR/css3-color/#svg-color
      */

      var w3cx11 = {
        aliceblue: '#f0f8ff',
        antiquewhite: '#faebd7',
        aqua: '#00ffff',
        aquamarine: '#7fffd4',
        azure: '#f0ffff',
        beige: '#f5f5dc',
        bisque: '#ffe4c4',
        black: '#000000',
        blanchedalmond: '#ffebcd',
        blue: '#0000ff',
        blueviolet: '#8a2be2',
        brown: '#a52a2a',
        burlywood: '#deb887',
        cadetblue: '#5f9ea0',
        chartreuse: '#7fff00',
        chocolate: '#d2691e',
        coral: '#ff7f50',
        cornflower: '#6495ed',
        cornflowerblue: '#6495ed',
        cornsilk: '#fff8dc',
        crimson: '#dc143c',
        cyan: '#00ffff',
        darkblue: '#00008b',
        darkcyan: '#008b8b',
        darkgoldenrod: '#b8860b',
        darkgray: '#a9a9a9',
        darkgreen: '#006400',
        darkgrey: '#a9a9a9',
        darkkhaki: '#bdb76b',
        darkmagenta: '#8b008b',
        darkolivegreen: '#556b2f',
        darkorange: '#ff8c00',
        darkorchid: '#9932cc',
        darkred: '#8b0000',
        darksalmon: '#e9967a',
        darkseagreen: '#8fbc8f',
        darkslateblue: '#483d8b',
        darkslategray: '#2f4f4f',
        darkslategrey: '#2f4f4f',
        darkturquoise: '#00ced1',
        darkviolet: '#9400d3',
        deeppink: '#ff1493',
        deepskyblue: '#00bfff',
        dimgray: '#696969',
        dimgrey: '#696969',
        dodgerblue: '#1e90ff',
        firebrick: '#b22222',
        floralwhite: '#fffaf0',
        forestgreen: '#228b22',
        fuchsia: '#ff00ff',
        gainsboro: '#dcdcdc',
        ghostwhite: '#f8f8ff',
        gold: '#ffd700',
        goldenrod: '#daa520',
        gray: '#808080',
        green: '#008000',
        greenyellow: '#adff2f',
        grey: '#808080',
        honeydew: '#f0fff0',
        hotpink: '#ff69b4',
        indianred: '#cd5c5c',
        indigo: '#4b0082',
        ivory: '#fffff0',
        khaki: '#f0e68c',
        laserlemon: '#ffff54',
        lavender: '#e6e6fa',
        lavenderblush: '#fff0f5',
        lawngreen: '#7cfc00',
        lemonchiffon: '#fffacd',
        lightblue: '#add8e6',
        lightcoral: '#f08080',
        lightcyan: '#e0ffff',
        lightgoldenrod: '#fafad2',
        lightgoldenrodyellow: '#fafad2',
        lightgray: '#d3d3d3',
        lightgreen: '#90ee90',
        lightgrey: '#d3d3d3',
        lightpink: '#ffb6c1',
        lightsalmon: '#ffa07a',
        lightseagreen: '#20b2aa',
        lightskyblue: '#87cefa',
        lightslategray: '#778899',
        lightslategrey: '#778899',
        lightsteelblue: '#b0c4de',
        lightyellow: '#ffffe0',
        lime: '#00ff00',
        limegreen: '#32cd32',
        linen: '#faf0e6',
        magenta: '#ff00ff',
        maroon: '#800000',
        maroon2: '#7f0000',
        maroon3: '#b03060',
        mediumaquamarine: '#66cdaa',
        mediumblue: '#0000cd',
        mediumorchid: '#ba55d3',
        mediumpurple: '#9370db',
        mediumseagreen: '#3cb371',
        mediumslateblue: '#7b68ee',
        mediumspringgreen: '#00fa9a',
        mediumturquoise: '#48d1cc',
        mediumvioletred: '#c71585',
        midnightblue: '#191970',
        mintcream: '#f5fffa',
        mistyrose: '#ffe4e1',
        moccasin: '#ffe4b5',
        navajowhite: '#ffdead',
        navy: '#000080',
        oldlace: '#fdf5e6',
        olive: '#808000',
        olivedrab: '#6b8e23',
        orange: '#ffa500',
        orangered: '#ff4500',
        orchid: '#da70d6',
        palegoldenrod: '#eee8aa',
        palegreen: '#98fb98',
        paleturquoise: '#afeeee',
        palevioletred: '#db7093',
        papayawhip: '#ffefd5',
        peachpuff: '#ffdab9',
        peru: '#cd853f',
        pink: '#ffc0cb',
        plum: '#dda0dd',
        powderblue: '#b0e0e6',
        purple: '#800080',
        purple2: '#7f007f',
        purple3: '#a020f0',
        rebeccapurple: '#663399',
        red: '#ff0000',
        rosybrown: '#bc8f8f',
        royalblue: '#4169e1',
        saddlebrown: '#8b4513',
        salmon: '#fa8072',
        sandybrown: '#f4a460',
        seagreen: '#2e8b57',
        seashell: '#fff5ee',
        sienna: '#a0522d',
        silver: '#c0c0c0',
        skyblue: '#87ceeb',
        slateblue: '#6a5acd',
        slategray: '#708090',
        slategrey: '#708090',
        snow: '#fffafa',
        springgreen: '#00ff7f',
        steelblue: '#4682b4',
        tan: '#d2b48c',
        teal: '#008080',
        thistle: '#d8bfd8',
        tomato: '#ff6347',
        turquoise: '#40e0d0',
        violet: '#ee82ee',
        wheat: '#f5deb3',
        white: '#ffffff',
        whitesmoke: '#f5f5f5',
        yellow: '#ffff00',
        yellowgreen: '#9acd32'
      };

      var w3cx11_1 = w3cx11;

      var type$b = utils.type;





      Color_1.prototype.name = function () {
        var hex = rgb2hex_1(this._rgb, 'rgb');
        for (var i = 0, list = Object.keys(w3cx11_1); i < list.length; i += 1) {
          var n = list[i];

          if (w3cx11_1[n] === hex) { return n.toLowerCase(); }
        }
        return hex;
      };

      input.format.named = function (name) {
        name = name.toLowerCase();
        if (w3cx11_1[name]) { return hex2rgb_1(w3cx11_1[name]); }
        throw new Error('unknown color name: ' + name);
      };

      input.autodetect.push({
        p: 5,
        test: function (h) {
          var rest = [], len = arguments.length - 1;
          while (len-- > 0) rest[len] = arguments[len + 1];

          if (!rest.length && type$b(h) === 'string' && w3cx11_1[h.toLowerCase()]) {
            return 'named';
          }
        }
      });

      var unpack$t = utils.unpack;

      var rgb2num = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var ref = unpack$t(args, 'rgb');
        var r = ref[0];
        var g = ref[1];
        var b = ref[2];
        return (r << 16) + (g << 8) + b;
      };

      var rgb2num_1 = rgb2num;

      var type$c = utils.type;

      var num2rgb = function (num) {
        if (type$c(num) == "number" && num >= 0 && num <= 0xFFFFFF) {
          var r = num >> 16;
          var g = (num >> 8) & 0xFF;
          var b = num & 0xFF;
          return [r, g, b, 1];
        }
        throw new Error("unknown num color: " + num);
      };

      var num2rgb_1 = num2rgb;

      var type$d = utils.type;



      Color_1.prototype.num = function () {
        return rgb2num_1(this._rgb);
      };

      chroma_1.num = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['num'])));
      };

      input.format.num = num2rgb_1;

      input.autodetect.push({
        p: 5,
        test: function () {
          var args = [], len = arguments.length;
          while (len--) args[len] = arguments[len];

          if (args.length === 1 && type$d(args[0]) === 'number' && args[0] >= 0 && args[0] <= 0xFFFFFF) {
            return 'num';
          }
        }
      });

      var unpack$u = utils.unpack;
      var type$e = utils.type;
      var round$5 = Math.round;

      Color_1.prototype.rgb = function (rnd) {
        if (rnd === void 0) rnd = true;

        if (rnd === false) { return this._rgb.slice(0, 3); }
        return this._rgb.slice(0, 3).map(round$5);
      };

      Color_1.prototype.rgba = function (rnd) {
        if (rnd === void 0) rnd = true;

        return this._rgb.slice(0, 4).map(function (v, i) {
          return i < 3 ? (rnd === false ? v : round$5(v)) : v;
        });
      };

      chroma_1.rgb = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['rgb'])));
      };

      input.format.rgb = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var rgba = unpack$u(args, 'rgba');
        if (rgba[3] === undefined) { rgba[3] = 1; }
        return rgba;
      };

      input.autodetect.push({
        p: 3,
        test: function () {
          var args = [], len = arguments.length;
          while (len--) args[len] = arguments[len];

          args = unpack$u(args, 'rgba');
          if (type$e(args) === 'array' && (args.length === 3 ||
            args.length === 4 && type$e(args[3]) == 'number' && args[3] >= 0 && args[3] <= 1)) {
            return 'rgb';
          }
        }
      });

      /*
       * Based on implementation by Neil Bartlett
       * https://github.com/neilbartlett/color-temperature
       */

      var log = Math.log;

      var temperature2rgb = function (kelvin) {
        var temp = kelvin / 100;
        var r, g, b;
        if (temp < 66) {
          r = 255;
          g = -155.25485562709179 - 0.44596950469579133 * (g = temp - 2) + 104.49216199393888 * log(g);
          b = temp < 20 ? 0 : -254.76935184120902 + 0.8274096064007395 * (b = temp - 10) + 115.67994401066147 * log(b);
        } else {
          r = 351.97690566805693 + 0.114206453784165 * (r = temp - 55) - 40.25366309332127 * log(r);
          g = 325.4494125711974 + 0.07943456536662342 * (g = temp - 50) - 28.0852963507957 * log(g);
          b = 255;
        }
        return [r, g, b, 1];
      };

      var temperature2rgb_1 = temperature2rgb;

      /*
       * Based on implementation by Neil Bartlett
       * https://github.com/neilbartlett/color-temperature
       **/


      var unpack$v = utils.unpack;
      var round$6 = Math.round;

      var rgb2temperature = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        var rgb = unpack$v(args, 'rgb');
        var r = rgb[0], b = rgb[2];
        var minTemp = 1000;
        var maxTemp = 40000;
        var eps = 0.4;
        var temp;
        while (maxTemp - minTemp > eps) {
          temp = (maxTemp + minTemp) * 0.5;
          var rgb$1 = temperature2rgb_1(temp);
          if ((rgb$1[2] / rgb$1[0]) >= (b / r)) {
            maxTemp = temp;
          } else {
            minTemp = temp;
          }
        }
        return round$6(temp);
      };

      var rgb2temperature_1 = rgb2temperature;

      Color_1.prototype.temp =
        Color_1.prototype.kelvin =
        Color_1.prototype.temperature = function () {
          return rgb2temperature_1(this._rgb);
        };

      chroma_1.temp =
        chroma_1.kelvin =
        chroma_1.temperature = function () {
          var args = [], len = arguments.length;
          while (len--) args[len] = arguments[len];

          return new (Function.prototype.bind.apply(Color_1, [null].concat(args, ['temp'])));
        };

      input.format.temp =
        input.format.kelvin =
        input.format.temperature = temperature2rgb_1;

      var type$f = utils.type;

      Color_1.prototype.alpha = function (a, mutate) {
        if (mutate === void 0) mutate = false;

        if (a !== undefined && type$f(a) === 'number') {
          if (mutate) {
            this._rgb[3] = a;
            return this;
          }
          return new Color_1([this._rgb[0], this._rgb[1], this._rgb[2], a], 'rgb');
        }
        return this._rgb[3];
      };

      Color_1.prototype.clipped = function () {
        return this._rgb._clipped || false;
      };

      Color_1.prototype.darken = function (amount) {
        if (amount === void 0) amount = 1;

        var me = this;
        var lab = me.lab();
        lab[0] -= labConstants.Kn * amount;
        return new Color_1(lab, 'lab').alpha(me.alpha(), true);
      };

      Color_1.prototype.brighten = function (amount) {
        if (amount === void 0) amount = 1;

        return this.darken(-amount);
      };

      Color_1.prototype.darker = Color_1.prototype.darken;
      Color_1.prototype.brighter = Color_1.prototype.brighten;

      Color_1.prototype.get = function (mc) {
        var ref = mc.split('.');
        var mode = ref[0];
        var channel = ref[1];
        var src = this[mode]();
        if (channel) {
          var i = mode.indexOf(channel);
          if (i > -1) { return src[i]; }
          throw new Error(("unknown channel " + channel + " in mode " + mode));
        } else {
          return src;
        }
      };

      var type$g = utils.type;
      var pow$2 = Math.pow;

      var EPS = 1e-7;
      var MAX_ITER = 20;

      Color_1.prototype.luminance = function (lum) {
        if (lum !== undefined && type$g(lum) === 'number') {
          if (lum === 0) {
            // return pure black
            return new Color_1([0, 0, 0, this._rgb[3]], 'rgb');
          }
          if (lum === 1) {
            // return pure white
            return new Color_1([255, 255, 255, this._rgb[3]], 'rgb');
          }
          // compute new color using...
          var cur_lum = this.luminance();
          var mode = 'rgb';
          var max_iter = MAX_ITER;

          var test = function (low, high) {
            var mid = low.interpolate(high, 0.5, mode);
            var lm = mid.luminance();
            if (Math.abs(lum - lm) < EPS || !max_iter--) {
              // close enough
              return mid;
            }
            return lm > lum ? test(low, mid) : test(mid, high);
          };

          var rgb = (cur_lum > lum ? test(new Color_1([0, 0, 0]), this) : test(this, new Color_1([255, 255, 255]))).rgb();
          return new Color_1(rgb.concat([this._rgb[3]]));
        }
        return rgb2luminance.apply(void 0, (this._rgb).slice(0, 3));
      };


      var rgb2luminance = function (r, g, b) {
        // relative luminance
        // see http://www.w3.org/TR/2008/REC-WCAG20-20081211/#relativeluminancedef
        r = luminance_x(r);
        g = luminance_x(g);
        b = luminance_x(b);
        return 0.2126 * r + 0.7152 * g + 0.0722 * b;
      };

      var luminance_x = function (x) {
        x /= 255;
        return x <= 0.03928 ? x / 12.92 : pow$2((x + 0.055) / 1.055, 2.4);
      };

      var interpolator = {};

      var type$h = utils.type;


      var mix = function (col1, col2, f) {
        if (f === void 0) f = 0.5;
        var rest = [], len = arguments.length - 3;
        while (len-- > 0) rest[len] = arguments[len + 3];

        var mode = rest[0] || 'lrgb';
        if (!interpolator[mode] && !rest.length) {
          // fall back to the first supported mode
          mode = Object.keys(interpolator)[0];
        }
        if (!interpolator[mode]) {
          throw new Error(("interpolation mode " + mode + " is not defined"));
        }
        if (type$h(col1) !== 'object') { col1 = new Color_1(col1); }
        if (type$h(col2) !== 'object') { col2 = new Color_1(col2); }
        return interpolator[mode](col1, col2, f)
          .alpha(col1.alpha() + f * (col2.alpha() - col1.alpha()));
      };

      Color_1.prototype.mix =
        Color_1.prototype.interpolate = function (col2, f) {
          if (f === void 0) f = 0.5;
          var rest = [], len = arguments.length - 2;
          while (len-- > 0) rest[len] = arguments[len + 2];

          return mix.apply(void 0, [this, col2, f].concat(rest));
        };

      Color_1.prototype.premultiply = function (mutate) {
        if (mutate === void 0) mutate = false;

        var rgb = this._rgb;
        var a = rgb[3];
        if (mutate) {
          this._rgb = [rgb[0] * a, rgb[1] * a, rgb[2] * a, a];
          return this;
        } else {
          return new Color_1([rgb[0] * a, rgb[1] * a, rgb[2] * a, a], 'rgb');
        }
      };

      Color_1.prototype.saturate = function (amount) {
        if (amount === void 0) amount = 1;

        var me = this;
        var lch = me.lch();
        lch[1] += labConstants.Kn * amount;
        if (lch[1] < 0) { lch[1] = 0; }
        return new Color_1(lch, 'lch').alpha(me.alpha(), true);
      };

      Color_1.prototype.desaturate = function (amount) {
        if (amount === void 0) amount = 1;

        return this.saturate(-amount);
      };

      var type$i = utils.type;

      Color_1.prototype.set = function (mc, value, mutate) {
        if (mutate === void 0) mutate = false;

        var ref = mc.split('.');
        var mode = ref[0];
        var channel = ref[1];
        var src = this[mode]();
        if (channel) {
          var i = mode.indexOf(channel);
          if (i > -1) {
            if (type$i(value) == 'string') {
              switch (value.charAt(0)) {
                case '+': src[i] += +value; break;
                case '-': src[i] += +value; break;
                case '*': src[i] *= +(value.substr(1)); break;
                case '/': src[i] /= +(value.substr(1)); break;
                default: src[i] = +value;
              }
            } else if (type$i(value) === 'number') {
              src[i] = value;
            } else {
              throw new Error("unsupported value for Color.set");
            }
            var out = new Color_1(src, mode);
            if (mutate) {
              this._rgb = out._rgb;
              return this;
            }
            return out;
          }
          throw new Error(("unknown channel " + channel + " in mode " + mode));
        } else {
          return src;
        }
      };

      var rgb$1 = function (col1, col2, f) {
        var xyz0 = col1._rgb;
        var xyz1 = col2._rgb;
        return new Color_1(
          xyz0[0] + f * (xyz1[0] - xyz0[0]),
          xyz0[1] + f * (xyz1[1] - xyz0[1]),
          xyz0[2] + f * (xyz1[2] - xyz0[2]),
          'rgb'
        )
      };

      // register interpolator
      interpolator.rgb = rgb$1;

      var sqrt$2 = Math.sqrt;
      var pow$3 = Math.pow;

      var lrgb = function (col1, col2, f) {
        var ref = col1._rgb;
        var x1 = ref[0];
        var y1 = ref[1];
        var z1 = ref[2];
        var ref$1 = col2._rgb;
        var x2 = ref$1[0];
        var y2 = ref$1[1];
        var z2 = ref$1[2];
        return new Color_1(
          sqrt$2(pow$3(x1, 2) * (1 - f) + pow$3(x2, 2) * f),
          sqrt$2(pow$3(y1, 2) * (1 - f) + pow$3(y2, 2) * f),
          sqrt$2(pow$3(z1, 2) * (1 - f) + pow$3(z2, 2) * f),
          'rgb'
        )
      };

      // register interpolator
      interpolator.lrgb = lrgb;

      var lab$1 = function (col1, col2, f) {
        var xyz0 = col1.lab();
        var xyz1 = col2.lab();
        return new Color_1(
          xyz0[0] + f * (xyz1[0] - xyz0[0]),
          xyz0[1] + f * (xyz1[1] - xyz0[1]),
          xyz0[2] + f * (xyz1[2] - xyz0[2]),
          'lab'
        )
      };

      // register interpolator
      interpolator.lab = lab$1;

      var _hsx = function (col1, col2, f, m) {
        var assign, assign$1;

        var xyz0, xyz1;
        if (m === 'hsl') {
          xyz0 = col1.hsl();
          xyz1 = col2.hsl();
        } else if (m === 'hsv') {
          xyz0 = col1.hsv();
          xyz1 = col2.hsv();
        } else if (m === 'hcg') {
          xyz0 = col1.hcg();
          xyz1 = col2.hcg();
        } else if (m === 'hsi') {
          xyz0 = col1.hsi();
          xyz1 = col2.hsi();
        } else if (m === 'lch' || m === 'hcl') {
          m = 'hcl';
          xyz0 = col1.hcl();
          xyz1 = col2.hcl();
        }

        var hue0, hue1, sat0, sat1, lbv0, lbv1;
        if (m.substr(0, 1) === 'h') {
          (assign = xyz0, hue0 = assign[0], sat0 = assign[1], lbv0 = assign[2]);
          (assign$1 = xyz1, hue1 = assign$1[0], sat1 = assign$1[1], lbv1 = assign$1[2]);
        }

        var sat, hue, lbv, dh;

        if (!isNaN(hue0) && !isNaN(hue1)) {
          // both colors have hue
          if (hue1 > hue0 && hue1 - hue0 > 180) {
            dh = hue1 - (hue0 + 360);
          } else if (hue1 < hue0 && hue0 - hue1 > 180) {
            dh = hue1 + 360 - hue0;
          } else {
            dh = hue1 - hue0;
          }
          hue = hue0 + f * dh;
        } else if (!isNaN(hue0)) {
          hue = hue0;
          if ((lbv1 == 1 || lbv1 == 0) && m != 'hsv') { sat = sat0; }
        } else if (!isNaN(hue1)) {
          hue = hue1;
          if ((lbv0 == 1 || lbv0 == 0) && m != 'hsv') { sat = sat1; }
        } else {
          hue = Number.NaN;
        }

        if (sat === undefined) { sat = sat0 + f * (sat1 - sat0); }
        lbv = lbv0 + f * (lbv1 - lbv0);
        return new Color_1([hue, sat, lbv], m);
      };

      var lch$1 = function (col1, col2, f) {
        return _hsx(col1, col2, f, 'lch');
      };

      // register interpolator
      interpolator.lch = lch$1;
      interpolator.hcl = lch$1;

      var num$1 = function (col1, col2, f) {
        var c1 = col1.num();
        var c2 = col2.num();
        return new Color_1(c1 + f * (c2 - c1), 'num')
      };

      // register interpolator
      interpolator.num = num$1;

      var hcg$1 = function (col1, col2, f) {
        return _hsx(col1, col2, f, 'hcg');
      };

      // register interpolator
      interpolator.hcg = hcg$1;

      var hsi$1 = function (col1, col2, f) {
        return _hsx(col1, col2, f, 'hsi');
      };

      // register interpolator
      interpolator.hsi = hsi$1;

      var hsl$1 = function (col1, col2, f) {
        return _hsx(col1, col2, f, 'hsl');
      };

      // register interpolator
      interpolator.hsl = hsl$1;

      var hsv$1 = function (col1, col2, f) {
        return _hsx(col1, col2, f, 'hsv');
      };

      // register interpolator
      interpolator.hsv = hsv$1;

      var clip_rgb$2 = utils.clip_rgb;
      var pow$4 = Math.pow;
      var sqrt$3 = Math.sqrt;
      var PI$1 = Math.PI;
      var cos$2 = Math.cos;
      var sin$1 = Math.sin;
      var atan2$1 = Math.atan2;

      var average = function (colors, mode, weights) {
        if (mode === void 0) mode = 'lrgb';
        if (weights === void 0) weights = null;

        var l = colors.length;
        if (!weights) { weights = Array.from(new Array(l)).map(function () { return 1; }); }
        // normalize weights
        var k = l / weights.reduce(function (a, b) { return a + b; });
        weights.forEach(function (w, i) { weights[i] *= k; });
        // convert colors to Color objects
        colors = colors.map(function (c) { return new Color_1(c); });
        if (mode === 'lrgb') {
          return _average_lrgb(colors, weights)
        }
        var first = colors.shift();
        var xyz = first.get(mode);
        var cnt = [];
        var dx = 0;
        var dy = 0;
        // initial color
        for (var i = 0; i < xyz.length; i++) {
          xyz[i] = (xyz[i] || 0) * weights[0];
          cnt.push(isNaN(xyz[i]) ? 0 : weights[0]);
          if (mode.charAt(i) === 'h' && !isNaN(xyz[i])) {
            var A = xyz[i] / 180 * PI$1;
            dx += cos$2(A) * weights[0];
            dy += sin$1(A) * weights[0];
          }
        }

        var alpha = first.alpha() * weights[0];
        colors.forEach(function (c, ci) {
          var xyz2 = c.get(mode);
          alpha += c.alpha() * weights[ci + 1];
          for (var i = 0; i < xyz.length; i++) {
            if (!isNaN(xyz2[i])) {
              cnt[i] += weights[ci + 1];
              if (mode.charAt(i) === 'h') {
                var A = xyz2[i] / 180 * PI$1;
                dx += cos$2(A) * weights[ci + 1];
                dy += sin$1(A) * weights[ci + 1];
              } else {
                xyz[i] += xyz2[i] * weights[ci + 1];
              }
            }
          }
        });

        for (var i$1 = 0; i$1 < xyz.length; i$1++) {
          if (mode.charAt(i$1) === 'h') {
            var A$1 = atan2$1(dy / cnt[i$1], dx / cnt[i$1]) / PI$1 * 180;
            while (A$1 < 0) { A$1 += 360; }
            while (A$1 >= 360) { A$1 -= 360; }
            xyz[i$1] = A$1;
          } else {
            xyz[i$1] = xyz[i$1] / cnt[i$1];
          }
        }
        alpha /= l;
        return (new Color_1(xyz, mode)).alpha(alpha > 0.99999 ? 1 : alpha, true);
      };


      var _average_lrgb = function (colors, weights) {
        var l = colors.length;
        var xyz = [0, 0, 0, 0];
        for (var i = 0; i < colors.length; i++) {
          var col = colors[i];
          var f = weights[i] / l;
          var rgb = col._rgb;
          xyz[0] += pow$4(rgb[0], 2) * f;
          xyz[1] += pow$4(rgb[1], 2) * f;
          xyz[2] += pow$4(rgb[2], 2) * f;
          xyz[3] += rgb[3] * f;
        }
        xyz[0] = sqrt$3(xyz[0]);
        xyz[1] = sqrt$3(xyz[1]);
        xyz[2] = sqrt$3(xyz[2]);
        if (xyz[3] > 0.9999999) { xyz[3] = 1; }
        return new Color_1(clip_rgb$2(xyz));
      };

      // minimal multi-purpose interface

      // @requires utils color analyze


      var type$j = utils.type;

      var pow$5 = Math.pow;

      var scale = function (colors) {

        // constructor
        var _mode = 'rgb';
        var _nacol = chroma_1('#ccc');
        var _spread = 0;
        // const _fixed = false;
        var _domain = [0, 1];
        var _pos = [];
        var _padding = [0, 0];
        var _classes = false;
        var _colors = [];
        var _out = false;
        var _min = 0;
        var _max = 1;
        var _correctLightness = false;
        var _colorCache = {};
        var _useCache = true;
        var _gamma = 1;

        // private methods

        var setColors = function (colors) {
          colors = colors || ['#fff', '#000'];
          if (colors && type$j(colors) === 'string' && chroma_1.brewer &&
            chroma_1.brewer[colors.toLowerCase()]) {
            colors = chroma_1.brewer[colors.toLowerCase()];
          }
          if (type$j(colors) === 'array') {
            // handle single color
            if (colors.length === 1) {
              colors = [colors[0], colors[0]];
            }
            // make a copy of the colors
            colors = colors.slice(0);
            // convert to chroma classes
            for (var c = 0; c < colors.length; c++) {
              colors[c] = chroma_1(colors[c]);
            }
            // auto-fill color position
            _pos.length = 0;
            for (var c$1 = 0; c$1 < colors.length; c$1++) {
              _pos.push(c$1 / (colors.length - 1));
            }
          }
          resetCache();
          return _colors = colors;
        };

        var getClass = function (value) {
          if (_classes != null) {
            var n = _classes.length - 1;
            var i = 0;
            while (i < n && value >= _classes[i]) {
              i++;
            }
            return i - 1;
          }
          return 0;
        };

        var tMapLightness = function (t) { return t; };
        var tMapDomain = function (t) { return t; };

        // const classifyValue = function(value) {
        //     let val = value;
        //     if (_classes.length > 2) {
        //         const n = _classes.length-1;
        //         const i = getClass(value);
        //         const minc = _classes[0] + ((_classes[1]-_classes[0]) * (0 + (_spread * 0.5)));  // center of 1st class
        //         const maxc = _classes[n-1] + ((_classes[n]-_classes[n-1]) * (1 - (_spread * 0.5)));  // center of last class
        //         val = _min + ((((_classes[i] + ((_classes[i+1] - _classes[i]) * 0.5)) - minc) / (maxc-minc)) * (_max - _min));
        //     }
        //     return val;
        // };

        var getColor = function (val, bypassMap) {
          var col, t;
          if (bypassMap == null) { bypassMap = false; }
          if (isNaN(val) || (val === null)) { return _nacol; }
          if (!bypassMap) {
            if (_classes && (_classes.length > 2)) {
              // find the class
              var c = getClass(val);
              t = c / (_classes.length - 2);
            } else if (_max !== _min) {
              // just interpolate between min/max
              t = (val - _min) / (_max - _min);
            } else {
              t = 1;
            }
          } else {
            t = val;
          }

          // domain map
          t = tMapDomain(t);

          if (!bypassMap) {
            t = tMapLightness(t);  // lightness correction
          }

          if (_gamma !== 1) { t = pow$5(t, _gamma); }

          t = _padding[0] + (t * (1 - _padding[0] - _padding[1]));

          t = Math.min(1, Math.max(0, t));

          var k = Math.floor(t * 10000);

          if (_useCache && _colorCache[k]) {
            col = _colorCache[k];
          } else {
            if (type$j(_colors) === 'array') {
              //for i in [0.._pos.length-1]
              for (var i = 0; i < _pos.length; i++) {
                var p = _pos[i];
                if (t <= p) {
                  col = _colors[i];
                  break;
                }
                if ((t >= p) && (i === (_pos.length - 1))) {
                  col = _colors[i];
                  break;
                }
                if (t > p && t < _pos[i + 1]) {
                  t = (t - p) / (_pos[i + 1] - p);
                  col = chroma_1.interpolate(_colors[i], _colors[i + 1], t, _mode);
                  break;
                }
              }
            } else if (type$j(_colors) === 'function') {
              col = _colors(t);
            }
            if (_useCache) { _colorCache[k] = col; }
          }
          return col;
        };

        var resetCache = function () { return _colorCache = {}; };

        setColors(colors);

        // public interface

        var f = function (v) {
          var c = chroma_1(getColor(v));
          if (_out && c[_out]) { return c[_out](); } else { return c; }
        };

        f.classes = function (classes) {
          if (classes != null) {
            if (type$j(classes) === 'array') {
              _classes = classes;
              _domain = [classes[0], classes[classes.length - 1]];
            } else {
              var d = chroma_1.analyze(_domain);
              if (classes === 0) {
                _classes = [d.min, d.max];
              } else {
                _classes = chroma_1.limits(d, 'e', classes);
              }
            }
            return f;
          }
          return _classes;
        };


        f.domain = function (domain) {
          if (!arguments.length) {
            return _domain;
          }
          _min = domain[0];
          _max = domain[domain.length - 1];
          _pos = [];
          var k = _colors.length;
          if ((domain.length === k) && (_min !== _max)) {
            // update positions
            for (var i = 0, list = Array.from(domain); i < list.length; i += 1) {
              var d = list[i];

              _pos.push((d - _min) / (_max - _min));
            }
          } else {
            for (var c = 0; c < k; c++) {
              _pos.push(c / (k - 1));
            }
            if (domain.length > 2) {
              // set domain map
              var tOut = domain.map(function (d, i) { return i / (domain.length - 1); });
              var tBreaks = domain.map(function (d) { return (d - _min) / (_max - _min); });
              if (!tBreaks.every(function (val, i) { return tOut[i] === val; })) {
                tMapDomain = function (t) {
                  if (t <= 0 || t >= 1) { return t; }
                  var i = 0;
                  while (t >= tBreaks[i + 1]) { i++; }
                  var f = (t - tBreaks[i]) / (tBreaks[i + 1] - tBreaks[i]);
                  var out = tOut[i] + f * (tOut[i + 1] - tOut[i]);
                  return out;
                };
              }

            }
          }
          _domain = [_min, _max];
          return f;
        };

        f.mode = function (_m) {
          if (!arguments.length) {
            return _mode;
          }
          _mode = _m;
          resetCache();
          return f;
        };

        f.range = function (colors, _pos) {
          setColors(colors, _pos);
          return f;
        };

        f.out = function (_o) {
          _out = _o;
          return f;
        };

        f.spread = function (val) {
          if (!arguments.length) {
            return _spread;
          }
          _spread = val;
          return f;
        };

        f.correctLightness = function (v) {
          if (v == null) { v = true; }
          _correctLightness = v;
          resetCache();
          if (_correctLightness) {
            tMapLightness = function (t) {
              var L0 = getColor(0, true).lab()[0];
              var L1 = getColor(1, true).lab()[0];
              var pol = L0 > L1;
              var L_actual = getColor(t, true).lab()[0];
              var L_ideal = L0 + ((L1 - L0) * t);
              var L_diff = L_actual - L_ideal;
              var t0 = 0;
              var t1 = 1;
              var max_iter = 20;
              while ((Math.abs(L_diff) > 1e-2) && (max_iter-- > 0)) {
                (function () {
                  if (pol) { L_diff *= -1; }
                  if (L_diff < 0) {
                    t0 = t;
                    t += (t1 - t) * 0.5;
                  } else {
                    t1 = t;
                    t += (t0 - t) * 0.5;
                  }
                  L_actual = getColor(t, true).lab()[0];
                  return L_diff = L_actual - L_ideal;
                })();
              }
              return t;
            };
          } else {
            tMapLightness = function (t) { return t; };
          }
          return f;
        };

        f.padding = function (p) {
          if (p != null) {
            if (type$j(p) === 'number') {
              p = [p, p];
            }
            _padding = p;
            return f;
          } else {
            return _padding;
          }
        };

        f.colors = function (numColors, out) {
          // If no arguments are given, return the original colors that were provided
          if (arguments.length < 2) { out = 'hex'; }
          var result = [];

          if (arguments.length === 0) {
            result = _colors.slice(0);

          } else if (numColors === 1) {
            result = [f(0.5)];

          } else if (numColors > 1) {
            var dm = _domain[0];
            var dd = _domain[1] - dm;
            result = __range__(0, numColors, false).map(function (i) { return f(dm + ((i / (numColors - 1)) * dd)); });

          } else { // returns all colors based on the defined classes
            colors = [];
            var samples = [];
            if (_classes && (_classes.length > 2)) {
              for (var i = 1, end = _classes.length, asc = 1 <= end; asc ? i < end : i > end; asc ? i++ : i--) {
                samples.push((_classes[i - 1] + _classes[i]) * 0.5);
              }
            } else {
              samples = _domain;
            }
            result = samples.map(function (v) { return f(v); });
          }

          if (chroma_1[out]) {
            result = result.map(function (c) { return c[out](); });
          }
          return result;
        };

        f.cache = function (c) {
          if (c != null) {
            _useCache = c;
            return f;
          } else {
            return _useCache;
          }
        };

        f.gamma = function (g) {
          if (g != null) {
            _gamma = g;
            return f;
          } else {
            return _gamma;
          }
        };

        f.nodata = function (d) {
          if (d != null) {
            _nacol = chroma_1(d);
            return f;
          } else {
            return _nacol;
          }
        };

        return f;
      };

      function __range__(left, right, inclusive) {
        var range = [];
        var ascending = left < right;
        var end = !inclusive ? right : ascending ? right + 1 : right - 1;
        for (var i = left; ascending ? i < end : i > end; ascending ? i++ : i--) {
          range.push(i);
        }
        return range;
      }

      //
      // interpolates between a set of colors uzing a bezier spline
      //

      // @requires utils lab




      var bezier = function (colors) {
        var assign, assign$1, assign$2;

        var I, lab0, lab1, lab2;
        colors = colors.map(function (c) { return new Color_1(c); });
        if (colors.length === 2) {
          // linear interpolation
          (assign = colors.map(function (c) { return c.lab(); }), lab0 = assign[0], lab1 = assign[1]);
          I = function (t) {
            var lab = ([0, 1, 2].map(function (i) { return lab0[i] + (t * (lab1[i] - lab0[i])); }));
            return new Color_1(lab, 'lab');
          };
        } else if (colors.length === 3) {
          // quadratic bezier interpolation
          (assign$1 = colors.map(function (c) { return c.lab(); }), lab0 = assign$1[0], lab1 = assign$1[1], lab2 = assign$1[2]);
          I = function (t) {
            var lab = ([0, 1, 2].map(function (i) { return ((1 - t) * (1 - t) * lab0[i]) + (2 * (1 - t) * t * lab1[i]) + (t * t * lab2[i]); }));
            return new Color_1(lab, 'lab');
          };
        } else if (colors.length === 4) {
          // cubic bezier interpolation
          var lab3;
          (assign$2 = colors.map(function (c) { return c.lab(); }), lab0 = assign$2[0], lab1 = assign$2[1], lab2 = assign$2[2], lab3 = assign$2[3]);
          I = function (t) {
            var lab = ([0, 1, 2].map(function (i) { return ((1 - t) * (1 - t) * (1 - t) * lab0[i]) + (3 * (1 - t) * (1 - t) * t * lab1[i]) + (3 * (1 - t) * t * t * lab2[i]) + (t * t * t * lab3[i]); }));
            return new Color_1(lab, 'lab');
          };
        } else if (colors.length === 5) {
          var I0 = bezier(colors.slice(0, 3));
          var I1 = bezier(colors.slice(2, 5));
          I = function (t) {
            if (t < 0.5) {
              return I0(t * 2);
            } else {
              return I1((t - 0.5) * 2);
            }
          };
        }
        return I;
      };

      var bezier_1 = function (colors) {
        var f = bezier(colors);
        f.scale = function () { return scale(f); };
        return f;
      };

      /*
       * interpolates between a set of colors uzing a bezier spline
       * blend mode formulas taken from http://www.venture-ware.com/kevin/coding/lets-learn-math-photoshop-blend-modes/
       */




      var blend = function (bottom, top, mode) {
        if (!blend[mode]) {
          throw new Error('unknown blend mode ' + mode);
        }
        return blend[mode](bottom, top);
      };

      var blend_f = function (f) {
        return function (bottom, top) {
          var c0 = chroma_1(top).rgb();
          var c1 = chroma_1(bottom).rgb();
          return chroma_1.rgb(f(c0, c1));
        };
      };

      var each = function (f) {
        return function (c0, c1) {
          var out = [];
          out[0] = f(c0[0], c1[0]);
          out[1] = f(c0[1], c1[1]);
          out[2] = f(c0[2], c1[2]);
          return out;
        };
      };

      var normal = function (a) { return a; };
      var multiply = function (a, b) { return a * b / 255; };
      var darken$1 = function (a, b) { return a > b ? b : a; };
      var lighten = function (a, b) { return a > b ? a : b; };
      var screen = function (a, b) { return 255 * (1 - (1 - a / 255) * (1 - b / 255)); };
      var overlay = function (a, b) { return b < 128 ? 2 * a * b / 255 : 255 * (1 - 2 * (1 - a / 255) * (1 - b / 255)); };
      var burn = function (a, b) { return 255 * (1 - (1 - b / 255) / (a / 255)); };
      var dodge = function (a, b) {
        if (a === 255) { return 255; }
        a = 255 * (b / 255) / (1 - a / 255);
        return a > 255 ? 255 : a
      };

      // # add = (a,b) ->
      // #     if (a + b > 255) then 255 else a + b

      blend.normal = blend_f(each(normal));
      blend.multiply = blend_f(each(multiply));
      blend.screen = blend_f(each(screen));
      blend.overlay = blend_f(each(overlay));
      blend.darken = blend_f(each(darken$1));
      blend.lighten = blend_f(each(lighten));
      blend.dodge = blend_f(each(dodge));
      blend.burn = blend_f(each(burn));
      // blend.add = blend_f(each(add));

      var blend_1 = blend;

      // cubehelix interpolation
      // based on D.A. Green "A colour scheme for the display of astronomical intensity images"
      // http://astron-soc.in/bulletin/11June/289392011.pdf

      var type$k = utils.type;
      var clip_rgb$3 = utils.clip_rgb;
      var TWOPI$2 = utils.TWOPI;
      var pow$6 = Math.pow;
      var sin$2 = Math.sin;
      var cos$3 = Math.cos;


      var cubehelix = function (start, rotations, hue, gamma, lightness) {
        if (start === void 0) start = 300;
        if (rotations === void 0) rotations = -1.5;
        if (hue === void 0) hue = 1;
        if (gamma === void 0) gamma = 1;
        if (lightness === void 0) lightness = [0, 1];

        var dh = 0, dl;
        if (type$k(lightness) === 'array') {
          dl = lightness[1] - lightness[0];
        } else {
          dl = 0;
          lightness = [lightness, lightness];
        }

        var f = function (fract) {
          var a = TWOPI$2 * (((start + 120) / 360) + (rotations * fract));
          var l = pow$6(lightness[0] + (dl * fract), gamma);
          var h = dh !== 0 ? hue[0] + (fract * dh) : hue;
          var amp = (h * l * (1 - l)) / 2;
          var cos_a = cos$3(a);
          var sin_a = sin$2(a);
          var r = l + (amp * ((-0.14861 * cos_a) + (1.78277 * sin_a)));
          var g = l + (amp * ((-0.29227 * cos_a) - (0.90649 * sin_a)));
          var b = l + (amp * (+1.97294 * cos_a));
          return chroma_1(clip_rgb$3([r * 255, g * 255, b * 255, 1]));
        };

        f.start = function (s) {
          if ((s == null)) { return start; }
          start = s;
          return f;
        };

        f.rotations = function (r) {
          if ((r == null)) { return rotations; }
          rotations = r;
          return f;
        };

        f.gamma = function (g) {
          if ((g == null)) { return gamma; }
          gamma = g;
          return f;
        };

        f.hue = function (h) {
          if ((h == null)) { return hue; }
          hue = h;
          if (type$k(hue) === 'array') {
            dh = hue[1] - hue[0];
            if (dh === 0) { hue = hue[1]; }
          } else {
            dh = 0;
          }
          return f;
        };

        f.lightness = function (h) {
          if ((h == null)) { return lightness; }
          if (type$k(h) === 'array') {
            lightness = h;
            dl = h[1] - h[0];
          } else {
            lightness = [h, h];
            dl = 0;
          }
          return f;
        };

        f.scale = function () { return chroma_1.scale(f); };

        f.hue(hue);

        return f;
      };

      var digits = '0123456789abcdef';

      var floor$2 = Math.floor;
      var random = Math.random;

      var random_1 = function () {
        var code = '#';
        for (var i = 0; i < 6; i++) {
          code += digits.charAt(floor$2(random() * 16));
        }
        return new Color_1(code, 'hex');
      };

      var log$1 = Math.log;
      var pow$7 = Math.pow;
      var floor$3 = Math.floor;
      var abs = Math.abs;


      var analyze = function (data, key) {
        if (key === void 0) key = null;

        var r = {
          min: Number.MAX_VALUE,
          max: Number.MAX_VALUE * -1,
          sum: 0,
          values: [],
          count: 0
        };
        if (type(data) === 'object') {
          data = Object.values(data);
        }
        data.forEach(function (val) {
          if (key && type(val) === 'object') { val = val[key]; }
          if (val !== undefined && val !== null && !isNaN(val)) {
            r.values.push(val);
            r.sum += val;
            if (val < r.min) { r.min = val; }
            if (val > r.max) { r.max = val; }
            r.count += 1;
          }
        });

        r.domain = [r.min, r.max];

        r.limits = function (mode, num) { return limits(r, mode, num); };

        return r;
      };


      var limits = function (data, mode, num) {
        if (mode === void 0) mode = 'equal';
        if (num === void 0) num = 7;

        if (type(data) == 'array') {
          data = analyze(data);
        }
        var min = data.min;
        var max = data.max;
        var values = data.values.sort(function (a, b) { return a - b; });

        if (num === 1) { return [min, max]; }

        var limits = [];

        if (mode.substr(0, 1) === 'c') { // continuous
          limits.push(min);
          limits.push(max);
        }

        if (mode.substr(0, 1) === 'e') { // equal interval
          limits.push(min);
          for (var i = 1; i < num; i++) {
            limits.push(min + ((i / num) * (max - min)));
          }
          limits.push(max);
        }

        else if (mode.substr(0, 1) === 'l') { // log scale
          if (min <= 0) {
            throw new Error('Logarithmic scales are only possible for values > 0');
          }
          var min_log = Math.LOG10E * log$1(min);
          var max_log = Math.LOG10E * log$1(max);
          limits.push(min);
          for (var i$1 = 1; i$1 < num; i$1++) {
            limits.push(pow$7(10, min_log + ((i$1 / num) * (max_log - min_log))));
          }
          limits.push(max);
        }

        else if (mode.substr(0, 1) === 'q') { // quantile scale
          limits.push(min);
          for (var i$2 = 1; i$2 < num; i$2++) {
            var p = ((values.length - 1) * i$2) / num;
            var pb = floor$3(p);
            if (pb === p) {
              limits.push(values[pb]);
            } else { // p > pb
              var pr = p - pb;
              limits.push((values[pb] * (1 - pr)) + (values[pb + 1] * pr));
            }
          }
          limits.push(max);

        }

        else if (mode.substr(0, 1) === 'k') { // k-means clustering
          /*
          implementation based on
          http://code.google.com/p/figue/source/browse/trunk/figue.js#336
          simplified for 1-d input values
          */
          var cluster;
          var n = values.length;
          var assignments = new Array(n);
          var clusterSizes = new Array(num);
          var repeat = true;
          var nb_iters = 0;
          var centroids = null;

          // get seed values
          centroids = [];
          centroids.push(min);
          for (var i$3 = 1; i$3 < num; i$3++) {
            centroids.push(min + ((i$3 / num) * (max - min)));
          }
          centroids.push(max);

          while (repeat) {
            // assignment step
            for (var j = 0; j < num; j++) {
              clusterSizes[j] = 0;
            }
            for (var i$4 = 0; i$4 < n; i$4++) {
              var value = values[i$4];
              var mindist = Number.MAX_VALUE;
              var best = (void 0);
              for (var j$1 = 0; j$1 < num; j$1++) {
                var dist = abs(centroids[j$1] - value);
                if (dist < mindist) {
                  mindist = dist;
                  best = j$1;
                }
                clusterSizes[best]++;
                assignments[i$4] = best;
              }
            }

            // update centroids step
            var newCentroids = new Array(num);
            for (var j$2 = 0; j$2 < num; j$2++) {
              newCentroids[j$2] = null;
            }
            for (var i$5 = 0; i$5 < n; i$5++) {
              cluster = assignments[i$5];
              if (newCentroids[cluster] === null) {
                newCentroids[cluster] = values[i$5];
              } else {
                newCentroids[cluster] += values[i$5];
              }
            }
            for (var j$3 = 0; j$3 < num; j$3++) {
              newCentroids[j$3] *= 1 / clusterSizes[j$3];
            }

            // check convergence
            repeat = false;
            for (var j$4 = 0; j$4 < num; j$4++) {
              if (newCentroids[j$4] !== centroids[j$4]) {
                repeat = true;
                break;
              }
            }

            centroids = newCentroids;
            nb_iters++;

            if (nb_iters > 200) {
              repeat = false;
            }
          }

          // finished k-means clustering
          // the next part is borrowed from gabrielflor.it
          var kClusters = {};
          for (var j$5 = 0; j$5 < num; j$5++) {
            kClusters[j$5] = [];
          }
          for (var i$6 = 0; i$6 < n; i$6++) {
            cluster = assignments[i$6];
            kClusters[cluster].push(values[i$6]);
          }
          var tmpKMeansBreaks = [];
          for (var j$6 = 0; j$6 < num; j$6++) {
            tmpKMeansBreaks.push(kClusters[j$6][0]);
            tmpKMeansBreaks.push(kClusters[j$6][kClusters[j$6].length - 1]);
          }
          tmpKMeansBreaks = tmpKMeansBreaks.sort(function (a, b) { return a - b; });
          limits.push(tmpKMeansBreaks[0]);
          for (var i$7 = 1; i$7 < tmpKMeansBreaks.length; i$7 += 2) {
            var v = tmpKMeansBreaks[i$7];
            if (!isNaN(v) && (limits.indexOf(v) === -1)) {
              limits.push(v);
            }
          }
        }
        return limits;
      };

      var analyze_1 = { analyze: analyze, limits: limits };

      var contrast = function (a, b) {
        // WCAG contrast ratio
        // see http://www.w3.org/TR/2008/REC-WCAG20-20081211/#contrast-ratiodef
        a = new Color_1(a);
        b = new Color_1(b);
        var l1 = a.luminance();
        var l2 = b.luminance();
        return l1 > l2 ? (l1 + 0.05) / (l2 + 0.05) : (l2 + 0.05) / (l1 + 0.05);
      };

      var sqrt$4 = Math.sqrt;
      var atan2$2 = Math.atan2;
      var abs$1 = Math.abs;
      var cos$4 = Math.cos;
      var PI$2 = Math.PI;

      var deltaE = function (a, b, L, C) {
        if (L === void 0) L = 1;
        if (C === void 0) C = 1;

        // Delta E (CMC)
        // see http://www.brucelindbloom.com/index.html?Eqn_DeltaE_CMC.html
        a = new Color_1(a);
        b = new Color_1(b);
        var ref = Array.from(a.lab());
        var L1 = ref[0];
        var a1 = ref[1];
        var b1 = ref[2];
        var ref$1 = Array.from(b.lab());
        var L2 = ref$1[0];
        var a2 = ref$1[1];
        var b2 = ref$1[2];
        var c1 = sqrt$4((a1 * a1) + (b1 * b1));
        var c2 = sqrt$4((a2 * a2) + (b2 * b2));
        var sl = L1 < 16.0 ? 0.511 : (0.040975 * L1) / (1.0 + (0.01765 * L1));
        var sc = ((0.0638 * c1) / (1.0 + (0.0131 * c1))) + 0.638;
        var h1 = c1 < 0.000001 ? 0.0 : (atan2$2(b1, a1) * 180.0) / PI$2;
        while (h1 < 0) { h1 += 360; }
        while (h1 >= 360) { h1 -= 360; }
        var t = (h1 >= 164.0) && (h1 <= 345.0) ? (0.56 + abs$1(0.2 * cos$4((PI$2 * (h1 + 168.0)) / 180.0))) : (0.36 + abs$1(0.4 * cos$4((PI$2 * (h1 + 35.0)) / 180.0)));
        var c4 = c1 * c1 * c1 * c1;
        var f = sqrt$4(c4 / (c4 + 1900.0));
        var sh = sc * (((f * t) + 1.0) - f);
        var delL = L1 - L2;
        var delC = c1 - c2;
        var delA = a1 - a2;
        var delB = b1 - b2;
        var dH2 = ((delA * delA) + (delB * delB)) - (delC * delC);
        var v1 = delL / (L * sl);
        var v2 = delC / (C * sc);
        var v3 = sh;
        return sqrt$4((v1 * v1) + (v2 * v2) + (dH2 / (v3 * v3)));
      };

      // simple Euclidean distance
      var distance = function (a, b, mode) {
        if (mode === void 0) mode = 'lab';

        // Delta E (CIE 1976)
        // see http://www.brucelindbloom.com/index.html?Equations.html
        a = new Color_1(a);
        b = new Color_1(b);
        var l1 = a.get(mode);
        var l2 = b.get(mode);
        var sum_sq = 0;
        for (var i in l1) {
          var d = (l1[i] || 0) - (l2[i] || 0);
          sum_sq += d * d;
        }
        return Math.sqrt(sum_sq);
      };

      var valid = function () {
        var args = [], len = arguments.length;
        while (len--) args[len] = arguments[len];

        try {
          new (Function.prototype.bind.apply(Color_1, [null].concat(args)));
          return true;
        } catch (e) {
          return false;
        }
      };

      // some pre-defined color scales:




      var scales = {
        cool: function cool() { return scale([chroma_1.hsl(180, 1, .9), chroma_1.hsl(250, .7, .4)]) },
        hot: function hot() { return scale(['#000', '#f00', '#ff0', '#fff'], [0, .25, .75, 1]).mode('rgb') }
      };

      /**
          ColorBrewer colors for chroma.js

          Copyright (c) 2002 Cynthia Brewer, Mark Harrower, and The
          Pennsylvania State University.

          Licensed under the Apache License, Version 2.0 (the "License");
          you may not use this file except in compliance with the License.
          You may obtain a copy of the License at
          http://www.apache.org/licenses/LICENSE-2.0

          Unless required by applicable law or agreed to in writing, software distributed
          under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
          CONDITIONS OF ANY KIND, either express or implied. See the License for the
          specific language governing permissions and limitations under the License.
      */

      var colorbrewer = {
        // sequential
        OrRd: ['#fff7ec', '#fee8c8', '#fdd49e', '#fdbb84', '#fc8d59', '#ef6548', '#d7301f', '#b30000', '#7f0000'],
        PuBu: ['#fff7fb', '#ece7f2', '#d0d1e6', '#a6bddb', '#74a9cf', '#3690c0', '#0570b0', '#045a8d', '#023858'],
        BuPu: ['#f7fcfd', '#e0ecf4', '#bfd3e6', '#9ebcda', '#8c96c6', '#8c6bb1', '#88419d', '#810f7c', '#4d004b'],
        Oranges: ['#fff5eb', '#fee6ce', '#fdd0a2', '#fdae6b', '#fd8d3c', '#f16913', '#d94801', '#a63603', '#7f2704'],
        BuGn: ['#f7fcfd', '#e5f5f9', '#ccece6', '#99d8c9', '#66c2a4', '#41ae76', '#238b45', '#006d2c', '#00441b'],
        YlOrBr: ['#ffffe5', '#fff7bc', '#fee391', '#fec44f', '#fe9929', '#ec7014', '#cc4c02', '#993404', '#662506'],
        YlGn: ['#ffffe5', '#f7fcb9', '#d9f0a3', '#addd8e', '#78c679', '#41ab5d', '#238443', '#006837', '#004529'],
        Reds: ['#fff5f0', '#fee0d2', '#fcbba1', '#fc9272', '#fb6a4a', '#ef3b2c', '#cb181d', '#a50f15', '#67000d'],
        RdPu: ['#fff7f3', '#fde0dd', '#fcc5c0', '#fa9fb5', '#f768a1', '#dd3497', '#ae017e', '#7a0177', '#49006a'],
        Greens: ['#f7fcf5', '#e5f5e0', '#c7e9c0', '#a1d99b', '#74c476', '#41ab5d', '#238b45', '#006d2c', '#00441b'],
        YlGnBu: ['#ffffd9', '#edf8b1', '#c7e9b4', '#7fcdbb', '#41b6c4', '#1d91c0', '#225ea8', '#253494', '#081d58'],
        Purples: ['#fcfbfd', '#efedf5', '#dadaeb', '#bcbddc', '#9e9ac8', '#807dba', '#6a51a3', '#54278f', '#3f007d'],
        GnBu: ['#f7fcf0', '#e0f3db', '#ccebc5', '#a8ddb5', '#7bccc4', '#4eb3d3', '#2b8cbe', '#0868ac', '#084081'],
        Greys: ['#ffffff', '#f0f0f0', '#d9d9d9', '#bdbdbd', '#969696', '#737373', '#525252', '#252525', '#000000'],
        YlOrRd: ['#ffffcc', '#ffeda0', '#fed976', '#feb24c', '#fd8d3c', '#fc4e2a', '#e31a1c', '#bd0026', '#800026'],
        PuRd: ['#f7f4f9', '#e7e1ef', '#d4b9da', '#c994c7', '#df65b0', '#e7298a', '#ce1256', '#980043', '#67001f'],
        Blues: ['#f7fbff', '#deebf7', '#c6dbef', '#9ecae1', '#6baed6', '#4292c6', '#2171b5', '#08519c', '#08306b'],
        PuBuGn: ['#fff7fb', '#ece2f0', '#d0d1e6', '#a6bddb', '#67a9cf', '#3690c0', '#02818a', '#016c59', '#014636'],
        Viridis: ['#440154', '#482777', '#3f4a8a', '#31678e', '#26838f', '#1f9d8a', '#6cce5a', '#b6de2b', '#fee825'],

        // diverging

        Spectral: ['#9e0142', '#d53e4f', '#f46d43', '#fdae61', '#fee08b', '#ffffbf', '#e6f598', '#abdda4', '#66c2a5', '#3288bd', '#5e4fa2'],
        RdYlGn: ['#a50026', '#d73027', '#f46d43', '#fdae61', '#fee08b', '#ffffbf', '#d9ef8b', '#a6d96a', '#66bd63', '#1a9850', '#006837'],
        RdBu: ['#67001f', '#b2182b', '#d6604d', '#f4a582', '#fddbc7', '#f7f7f7', '#d1e5f0', '#92c5de', '#4393c3', '#2166ac', '#053061'],
        PiYG: ['#8e0152', '#c51b7d', '#de77ae', '#f1b6da', '#fde0ef', '#f7f7f7', '#e6f5d0', '#b8e186', '#7fbc41', '#4d9221', '#276419'],
        PRGn: ['#40004b', '#762a83', '#9970ab', '#c2a5cf', '#e7d4e8', '#f7f7f7', '#d9f0d3', '#a6dba0', '#5aae61', '#1b7837', '#00441b'],
        RdYlBu: ['#a50026', '#d73027', '#f46d43', '#fdae61', '#fee090', '#ffffbf', '#e0f3f8', '#abd9e9', '#74add1', '#4575b4', '#313695'],
        BrBG: ['#543005', '#8c510a', '#bf812d', '#dfc27d', '#f6e8c3', '#f5f5f5', '#c7eae5', '#80cdc1', '#35978f', '#01665e', '#003c30'],
        RdGy: ['#67001f', '#b2182b', '#d6604d', '#f4a582', '#fddbc7', '#ffffff', '#e0e0e0', '#bababa', '#878787', '#4d4d4d', '#1a1a1a'],
        PuOr: ['#7f3b08', '#b35806', '#e08214', '#fdb863', '#fee0b6', '#f7f7f7', '#d8daeb', '#b2abd2', '#8073ac', '#542788', '#2d004b'],

        // qualitative

        Set2: ['#66c2a5', '#fc8d62', '#8da0cb', '#e78ac3', '#a6d854', '#ffd92f', '#e5c494', '#b3b3b3'],
        Accent: ['#7fc97f', '#beaed4', '#fdc086', '#ffff99', '#386cb0', '#f0027f', '#bf5b17', '#666666'],
        Set1: ['#e41a1c', '#377eb8', '#4daf4a', '#984ea3', '#ff7f00', '#ffff33', '#a65628', '#f781bf', '#999999'],
        Set3: ['#8dd3c7', '#ffffb3', '#bebada', '#fb8072', '#80b1d3', '#fdb462', '#b3de69', '#fccde5', '#d9d9d9', '#bc80bd', '#ccebc5', '#ffed6f'],
        Dark2: ['#1b9e77', '#d95f02', '#7570b3', '#e7298a', '#66a61e', '#e6ab02', '#a6761d', '#666666'],
        Paired: ['#a6cee3', '#1f78b4', '#b2df8a', '#33a02c', '#fb9a99', '#e31a1c', '#fdbf6f', '#ff7f00', '#cab2d6', '#6a3d9a', '#ffff99', '#b15928'],
        Pastel2: ['#b3e2cd', '#fdcdac', '#cbd5e8', '#f4cae4', '#e6f5c9', '#fff2ae', '#f1e2cc', '#cccccc'],
        Pastel1: ['#fbb4ae', '#b3cde3', '#ccebc5', '#decbe4', '#fed9a6', '#ffffcc', '#e5d8bd', '#fddaec', '#f2f2f2'],
      };

      // add lowercase aliases for case-insensitive matches
      for (var i$1 = 0, list$1 = Object.keys(colorbrewer); i$1 < list$1.length; i$1 += 1) {
        var key = list$1[i$1];

        colorbrewer[key.toLowerCase()] = colorbrewer[key];
      }

      var colorbrewer_1 = colorbrewer;

      // feel free to comment out anything to rollup
      // a smaller chroma.js built

      // io --> convert colors















      // operators --> modify existing Colors










      // interpolators










      // generators -- > create new colors
      chroma_1.average = average;
      chroma_1.bezier = bezier_1;
      chroma_1.blend = blend_1;
      chroma_1.cubehelix = cubehelix;
      chroma_1.mix = chroma_1.interpolate = mix;
      chroma_1.random = random_1;
      chroma_1.scale = scale;

      // other utility methods
      chroma_1.analyze = analyze_1.analyze;
      chroma_1.contrast = contrast;
      chroma_1.deltaE = deltaE;
      chroma_1.distance = distance;
      chroma_1.limits = analyze_1.limits;
      chroma_1.valid = valid;

      // scale
      chroma_1.scales = scales;

      // colors
      chroma_1.colors = w3cx11_1;
      chroma_1.brewer = colorbrewer_1;

      var chroma_js = chroma_1;

      return chroma_js;

    })));

  }, {}], 61: [function (require, module, exports) {
    var Buffer = require('safe-buffer').Buffer
    var Transform = require('stream').Transform
    var StringDecoder = require('string_decoder').StringDecoder
    var inherits = require('inherits')

    function CipherBase(hashMode) {
      Transform.call(this)
      this.hashMode = typeof hashMode === 'string'
      if (this.hashMode) {
        this[hashMode] = this._finalOrDigest
      } else {
        this.final = this._finalOrDigest
      }
      if (this._final) {
        this.__final = this._final
        this._final = null
      }
      this._decoder = null
      this._encoding = null
    }
    inherits(CipherBase, Transform)

    CipherBase.prototype.update = function (data, inputEnc, outputEnc) {
      if (typeof data === 'string') {
        data = Buffer.from(data, inputEnc)
      }

      var outData = this._update(data)
      if (this.hashMode) return this

      if (outputEnc) {
        outData = this._toString(outData, outputEnc)
      }

      return outData
    }

    CipherBase.prototype.setAutoPadding = function () { }
    CipherBase.prototype.getAuthTag = function () {
      throw new Error('trying to get auth tag in unsupported state')
    }

    CipherBase.prototype.setAuthTag = function () {
      throw new Error('trying to set auth tag in unsupported state')
    }

    CipherBase.prototype.setAAD = function () {
      throw new Error('trying to set aad in unsupported state')
    }

    CipherBase.prototype._transform = function (data, _, next) {
      var err
      try {
        if (this.hashMode) {
          this._update(data)
        } else {
          this.push(this._update(data))
        }
      } catch (e) {
        err = e
      } finally {
        next(err)
      }
    }
    CipherBase.prototype._flush = function (done) {
      var err
      try {
        this.push(this.__final())
      } catch (e) {
        err = e
      }

      done(err)
    }
    CipherBase.prototype._finalOrDigest = function (outputEnc) {
      var outData = this.__final() || Buffer.alloc(0)
      if (outputEnc) {
        outData = this._toString(outData, outputEnc, true)
      }
      return outData
    }

    CipherBase.prototype._toString = function (value, enc, fin) {
      if (!this._decoder) {
        this._decoder = new StringDecoder(enc)
        this._encoding = enc
      }

      if (this._encoding !== enc) throw new Error('can\'t switch encodings')

      var out = this._decoder.write(value)
      if (fin) {
        out += this._decoder.end()
      }

      return out
    }

    module.exports = CipherBase

  }, { "inherits": 569, "safe-buffer": 643, "stream": 671, "string_decoder": 677 }], 62: [function (require, module, exports) {
    (function (Buffer) {
      var clone = (function () {
        'use strict';

        var nativeMap;
        try {
          nativeMap = Map;
        } catch (_) {
          // maybe a reference error because no `Map`. Give it a dummy value that no
          // value will ever be an instanceof.
          nativeMap = function () { };
        }

        var nativeSet;
        try {
          nativeSet = Set;
        } catch (_) {
          nativeSet = function () { };
        }

        var nativePromise;
        try {
          nativePromise = Promise;
        } catch (_) {
          nativePromise = function () { };
        }

        /**
         * Clones (copies) an Object using deep copying.
         *
         * This function supports circular references by default, but if you are certain
         * there are no circular references in your object, you can save some CPU time
         * by calling clone(obj, false).
         *
         * Caution: if `circular` is false and `parent` contains circular references,
         * your program may enter an infinite loop and crash.
         *
         * @param `parent` - the object to be cloned
         * @param `circular` - set to true if the object to be cloned may contain
         *    circular references. (optional - true by default)
         * @param `depth` - set to a number if the object is only to be cloned to
         *    a particular depth. (optional - defaults to Infinity)
         * @param `prototype` - sets the prototype to be used when cloning an object.
         *    (optional - defaults to parent prototype).
         * @param `includeNonEnumerable` - set to true if the non-enumerable properties
         *    should be cloned as well. Non-enumerable properties on the prototype
         *    chain will be ignored. (optional - false by default)
        */
        function clone(parent, circular, depth, prototype, includeNonEnumerable) {
          if (typeof circular === 'object') {
            depth = circular.depth;
            prototype = circular.prototype;
            includeNonEnumerable = circular.includeNonEnumerable;
            circular = circular.circular;
          }
          // maintain two arrays for circular references, where corresponding parents
          // and children have the same index
          var allParents = [];
          var allChildren = [];

          var useBuffer = typeof Buffer != 'undefined';

          if (typeof circular == 'undefined')
            circular = true;

          if (typeof depth == 'undefined')
            depth = Infinity;

          // recurse this function so we don't reset allParents and allChildren
          function _clone(parent, depth) {
            // cloning null always returns null
            if (parent === null)
              return null;

            if (depth === 0)
              return parent;

            var child;
            var proto;
            if (typeof parent != 'object') {
              return parent;
            }

            if (parent instanceof nativeMap) {
              child = new nativeMap();
            } else if (parent instanceof nativeSet) {
              child = new nativeSet();
            } else if (parent instanceof nativePromise) {
              child = new nativePromise(function (resolve, reject) {
                parent.then(function (value) {
                  resolve(_clone(value, depth - 1));
                }, function (err) {
                  reject(_clone(err, depth - 1));
                });
              });
            } else if (clone.__isArray(parent)) {
              child = [];
            } else if (clone.__isRegExp(parent)) {
              child = new RegExp(parent.source, __getRegExpFlags(parent));
              if (parent.lastIndex) child.lastIndex = parent.lastIndex;
            } else if (clone.__isDate(parent)) {
              child = new Date(parent.getTime());
            } else if (useBuffer && Buffer.isBuffer(parent)) {
              child = new Buffer(parent.length);
              parent.copy(child);
              return child;
            } else if (parent instanceof Error) {
              child = Object.create(parent);
            } else {
              if (typeof prototype == 'undefined') {
                proto = Object.getPrototypeOf(parent);
                child = Object.create(proto);
              }
              else {
                child = Object.create(prototype);
                proto = prototype;
              }
            }

            if (circular) {
              var index = allParents.indexOf(parent);

              if (index != -1) {
                return allChildren[index];
              }
              allParents.push(parent);
              allChildren.push(child);
            }

            if (parent instanceof nativeMap) {
              var keyIterator = parent.keys();
              while (true) {
                var next = keyIterator.next();
                if (next.done) {
                  break;
                }
                var keyChild = _clone(next.value, depth - 1);
                var valueChild = _clone(parent.get(next.value), depth - 1);
                child.set(keyChild, valueChild);
              }
            }
            if (parent instanceof nativeSet) {
              var iterator = parent.keys();
              while (true) {
                var next = iterator.next();
                if (next.done) {
                  break;
                }
                var entryChild = _clone(next.value, depth - 1);
                child.add(entryChild);
              }
            }

            for (var i in parent) {
              var attrs;
              if (proto) {
                attrs = Object.getOwnPropertyDescriptor(proto, i);
              }

              if (attrs && attrs.set == null) {
                continue;
              }
              child[i] = _clone(parent[i], depth - 1);
            }

            if (Object.getOwnPropertySymbols) {
              var symbols = Object.getOwnPropertySymbols(parent);
              for (var i = 0; i < symbols.length; i++) {
                // Don't need to worry about cloning a symbol because it is a primitive,
                // like a number or string.
                var symbol = symbols[i];
                var descriptor = Object.getOwnPropertyDescriptor(parent, symbol);
                if (descriptor && !descriptor.enumerable && !includeNonEnumerable) {
                  continue;
                }
                child[symbol] = _clone(parent[symbol], depth - 1);
                if (!descriptor.enumerable) {
                  Object.defineProperty(child, symbol, {
                    enumerable: false
                  });
                }
              }
            }

            if (includeNonEnumerable) {
              var allPropertyNames = Object.getOwnPropertyNames(parent);
              for (var i = 0; i < allPropertyNames.length; i++) {
                var propertyName = allPropertyNames[i];
                var descriptor = Object.getOwnPropertyDescriptor(parent, propertyName);
                if (descriptor && descriptor.enumerable) {
                  continue;
                }
                child[propertyName] = _clone(parent[propertyName], depth - 1);
                Object.defineProperty(child, propertyName, {
                  enumerable: false
                });
              }
            }

            return child;
          }

          return _clone(parent, depth);
        }

        /**
         * Simple flat clone using prototype, accepts only objects, usefull for property
         * override on FLAT configuration object (no nested props).
         *
         * USE WITH CAUTION! This may not behave as you wish if you do not know how this
         * works.
         */
        clone.clonePrototype = function clonePrototype(parent) {
          if (parent === null)
            return null;

          var c = function () { };
          c.prototype = parent;
          return new c();
        };

        // private utility functions

        function __objToStr(o) {
          return Object.prototype.toString.call(o);
        }
        clone.__objToStr = __objToStr;

        function __isDate(o) {
          return typeof o === 'object' && __objToStr(o) === '[object Date]';
        }
        clone.__isDate = __isDate;

        function __isArray(o) {
          return typeof o === 'object' && __objToStr(o) === '[object Array]';
        }
        clone.__isArray = __isArray;

        function __isRegExp(o) {
          return typeof o === 'object' && __objToStr(o) === '[object RegExp]';
        }
        clone.__isRegExp = __isRegExp;

        function __getRegExpFlags(re) {
          var flags = '';
          if (re.global) flags += 'g';
          if (re.ignoreCase) flags += 'i';
          if (re.multiline) flags += 'm';
          return flags;
        }
        clone.__getRegExpFlags = __getRegExpFlags;

        return clone;
      })();

      if (typeof module === 'object' && module.exports) {
        module.exports = clone;
      }

    }).call(this, require("buffer").Buffer)
  }, { "buffer": 58 }], 63: [function (require, module, exports) {
    module.exports = {
      "O_RDONLY": 0,
      "O_WRONLY": 1,
      "O_RDWR": 2,
      "S_IFMT": 61440,
      "S_IFREG": 32768,
      "S_IFDIR": 16384,
      "S_IFCHR": 8192,
      "S_IFBLK": 24576,
      "S_IFIFO": 4096,
      "S_IFLNK": 40960,
      "S_IFSOCK": 49152,
      "O_CREAT": 512,
      "O_EXCL": 2048,
      "O_NOCTTY": 131072,
      "O_TRUNC": 1024,
      "O_APPEND": 8,
      "O_DIRECTORY": 1048576,
      "O_NOFOLLOW": 256,
      "O_SYNC": 128,
      "O_SYMLINK": 2097152,
      "O_NONBLOCK": 4,
      "S_IRWXU": 448,
      "S_IRUSR": 256,
      "S_IWUSR": 128,
      "S_IXUSR": 64,
      "S_IRWXG": 56,
      "S_IRGRP": 32,
      "S_IWGRP": 16,
      "S_IXGRP": 8,
      "S_IRWXO": 7,
      "S_IROTH": 4,
      "S_IWOTH": 2,
      "S_IXOTH": 1,
      "E2BIG": 7,
      "EACCES": 13,
      "EADDRINUSE": 48,
      "EADDRNOTAVAIL": 49,
      "EAFNOSUPPORT": 47,
      "EAGAIN": 35,
      "EALREADY": 37,
      "EBADF": 9,
      "EBADMSG": 94,
      "EBUSY": 16,
      "ECANCELED": 89,
      "ECHILD": 10,
      "ECONNABORTED": 53,
      "ECONNREFUSED": 61,
      "ECONNRESET": 54,
      "EDEADLK": 11,
      "EDESTADDRREQ": 39,
      "EDOM": 33,
      "EDQUOT": 69,
      "EEXIST": 17,
      "EFAULT": 14,
      "EFBIG": 27,
      "EHOSTUNREACH": 65,
      "EIDRM": 90,
      "EILSEQ": 92,
      "EINPROGRESS": 36,
      "EINTR": 4,
      "EINVAL": 22,
      "EIO": 5,
      "EISCONN": 56,
      "EISDIR": 21,
      "ELOOP": 62,
      "EMFILE": 24,
      "EMLINK": 31,
      "EMSGSIZE": 40,
      "EMULTIHOP": 95,
      "ENAMETOOLONG": 63,
      "ENETDOWN": 50,
      "ENETRESET": 52,
      "ENETUNREACH": 51,
      "ENFILE": 23,
      "ENOBUFS": 55,
      "ENODATA": 96,
      "ENODEV": 19,
      "ENOENT": 2,
      "ENOEXEC": 8,
      "ENOLCK": 77,
      "ENOLINK": 97,
      "ENOMEM": 12,
      "ENOMSG": 91,
      "ENOPROTOOPT": 42,
      "ENOSPC": 28,
      "ENOSR": 98,
      "ENOSTR": 99,
      "ENOSYS": 78,
      "ENOTCONN": 57,
      "ENOTDIR": 20,
      "ENOTEMPTY": 66,
      "ENOTSOCK": 38,
      "ENOTSUP": 45,
      "ENOTTY": 25,
      "ENXIO": 6,
      "EOPNOTSUPP": 102,
      "EOVERFLOW": 84,
      "EPERM": 1,
      "EPIPE": 32,
      "EPROTO": 100,
      "EPROTONOSUPPORT": 43,
      "EPROTOTYPE": 41,
      "ERANGE": 34,
      "EROFS": 30,
      "ESPIPE": 29,
      "ESRCH": 3,
      "ESTALE": 70,
      "ETIME": 101,
      "ETIMEDOUT": 60,
      "ETXTBSY": 26,
      "EWOULDBLOCK": 35,
      "EXDEV": 18,
      "SIGHUP": 1,
      "SIGINT": 2,
      "SIGQUIT": 3,
      "SIGILL": 4,
      "SIGTRAP": 5,
      "SIGABRT": 6,
      "SIGIOT": 6,
      "SIGBUS": 10,
      "SIGFPE": 8,
      "SIGKILL": 9,
      "SIGUSR1": 30,
      "SIGSEGV": 11,
      "SIGUSR2": 31,
      "SIGPIPE": 13,
      "SIGALRM": 14,
      "SIGTERM": 15,
      "SIGCHLD": 20,
      "SIGCONT": 19,
      "SIGSTOP": 17,
      "SIGTSTP": 18,
      "SIGTTIN": 21,
      "SIGTTOU": 22,
      "SIGURG": 16,
      "SIGXCPU": 24,
      "SIGXFSZ": 25,
      "SIGVTALRM": 26,
      "SIGPROF": 27,
      "SIGWINCH": 28,
      "SIGIO": 23,
      "SIGSYS": 12,
      "SSL_OP_ALL": 2147486719,
      "SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION": 262144,
      "SSL_OP_CIPHER_SERVER_PREFERENCE": 4194304,
      "SSL_OP_CISCO_ANYCONNECT": 32768,
      "SSL_OP_COOKIE_EXCHANGE": 8192,
      "SSL_OP_CRYPTOPRO_TLSEXT_BUG": 2147483648,
      "SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS": 2048,
      "SSL_OP_EPHEMERAL_RSA": 0,
      "SSL_OP_LEGACY_SERVER_CONNECT": 4,
      "SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER": 32,
      "SSL_OP_MICROSOFT_SESS_ID_BUG": 1,
      "SSL_OP_MSIE_SSLV2_RSA_PADDING": 0,
      "SSL_OP_NETSCAPE_CA_DN_BUG": 536870912,
      "SSL_OP_NETSCAPE_CHALLENGE_BUG": 2,
      "SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG": 1073741824,
      "SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG": 8,
      "SSL_OP_NO_COMPRESSION": 131072,
      "SSL_OP_NO_QUERY_MTU": 4096,
      "SSL_OP_NO_SESSION_RESUMPTION_ON_RENEGOTIATION": 65536,
      "SSL_OP_NO_SSLv2": 16777216,
      "SSL_OP_NO_SSLv3": 33554432,
      "SSL_OP_NO_TICKET": 16384,
      "SSL_OP_NO_TLSv1": 67108864,
      "SSL_OP_NO_TLSv1_1": 268435456,
      "SSL_OP_NO_TLSv1_2": 134217728,
      "SSL_OP_PKCS1_CHECK_1": 0,
      "SSL_OP_PKCS1_CHECK_2": 0,
      "SSL_OP_SINGLE_DH_USE": 1048576,
      "SSL_OP_SINGLE_ECDH_USE": 524288,
      "SSL_OP_SSLEAY_080_CLIENT_DH_BUG": 128,
      "SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG": 0,
      "SSL_OP_TLS_BLOCK_PADDING_BUG": 512,
      "SSL_OP_TLS_D5_BUG": 256,
      "SSL_OP_TLS_ROLLBACK_BUG": 8388608,
      "ENGINE_METHOD_DSA": 2,
      "ENGINE_METHOD_DH": 4,
      "ENGINE_METHOD_RAND": 8,
      "ENGINE_METHOD_ECDH": 16,
      "ENGINE_METHOD_ECDSA": 32,
      "ENGINE_METHOD_CIPHERS": 64,
      "ENGINE_METHOD_DIGESTS": 128,
      "ENGINE_METHOD_STORE": 256,
      "ENGINE_METHOD_PKEY_METHS": 512,
      "ENGINE_METHOD_PKEY_ASN1_METHS": 1024,
      "ENGINE_METHOD_ALL": 65535,
      "ENGINE_METHOD_NONE": 0,
      "DH_CHECK_P_NOT_SAFE_PRIME": 2,
      "DH_CHECK_P_NOT_PRIME": 1,
      "DH_UNABLE_TO_CHECK_GENERATOR": 4,
      "DH_NOT_SUITABLE_GENERATOR": 8,
      "NPN_ENABLED": 1,
      "RSA_PKCS1_PADDING": 1,
      "RSA_SSLV23_PADDING": 2,
      "RSA_NO_PADDING": 3,
      "RSA_PKCS1_OAEP_PADDING": 4,
      "RSA_X931_PADDING": 5,
      "RSA_PKCS1_PSS_PADDING": 6,
      "POINT_CONVERSION_COMPRESSED": 2,
      "POINT_CONVERSION_UNCOMPRESSED": 4,
      "POINT_CONVERSION_HYBRID": 6,
      "F_OK": 0,
      "R_OK": 4,
      "W_OK": 2,
      "X_OK": 1,
      "UV_UDP_REUSEADDR": 4
    }

  }, {}], 64: [function (require, module, exports) {
    (function (Buffer) {
      /*!
       * content-disposition
       * Copyright(c) 2014 Douglas Christopher Wilson
       * MIT Licensed
       */

      'use strict'

      /**
       * Module exports.
       */

      module.exports = contentDisposition
      module.exports.parse = parse

      /**
       * Module dependencies.
       */

      var basename = require('path').basename

      /**
       * RegExp to match non attr-char, *after* encodeURIComponent (i.e. not including "%")
       */

      var encodeUriAttrCharRegExp = /[\x00-\x20"'\(\)*,\/:;<=>?@\[\\\]\{\}\x7f]/g

      /**
       * RegExp to match percent encoding escape.
       */

      var hexEscapeRegExp = /%[0-9A-Fa-f]{2}/
      var hexEscapeReplaceRegExp = /%([0-9A-Fa-f]{2})/g

      /**
       * RegExp to match non-latin1 characters.
       */

      var nonLatin1RegExp = /[^\x20-\x7e\xa0-\xff]/g

      /**
       * RegExp to match quoted-pair in RFC 2616
       *
       * quoted-pair = "\" CHAR
       * CHAR        = <any US-ASCII character (octets 0 - 127)>
       */

      var qescRegExp = /\\([\u0000-\u007f])/g;

      /**
       * RegExp to match chars that must be quoted-pair in RFC 2616
       */

      var quoteRegExp = /([\\"])/g

      /**
       * RegExp for various RFC 2616 grammar
       *
       * parameter     = token "=" ( token | quoted-string )
       * token         = 1*<any CHAR except CTLs or separators>
       * separators    = "(" | ")" | "<" | ">" | "@"
       *               | "," | ";" | ":" | "\" | <">
       *               | "/" | "[" | "]" | "?" | "="
       *               | "{" | "}" | SP | HT
       * quoted-string = ( <"> *(qdtext | quoted-pair ) <"> )
       * qdtext        = <any TEXT except <">>
       * quoted-pair   = "\" CHAR
       * CHAR          = <any US-ASCII character (octets 0 - 127)>
       * TEXT          = <any OCTET except CTLs, but including LWS>
       * LWS           = [CRLF] 1*( SP | HT )
       * CRLF          = CR LF
       * CR            = <US-ASCII CR, carriage return (13)>
       * LF            = <US-ASCII LF, linefeed (10)>
       * SP            = <US-ASCII SP, space (32)>
       * HT            = <US-ASCII HT, horizontal-tab (9)>
       * CTL           = <any US-ASCII control character (octets 0 - 31) and DEL (127)>
       * OCTET         = <any 8-bit sequence of data>
       */

      var paramRegExp = /; *([!#$%&'\*\+\-\.0-9A-Z\^_`a-z\|~]+) *= *("(?:[ !\x23-\x5b\x5d-\x7e\x80-\xff]|\\[\x20-\x7e])*"|[!#$%&'\*\+\-\.0-9A-Z\^_`a-z\|~]+) */g
      var textRegExp = /^[\x20-\x7e\x80-\xff]+$/
      var tokenRegExp = /^[!#$%&'\*\+\-\.0-9A-Z\^_`a-z\|~]+$/

      /**
       * RegExp for various RFC 5987 grammar
       *
       * ext-value     = charset  "'" [ language ] "'" value-chars
       * charset       = "UTF-8" / "ISO-8859-1" / mime-charset
       * mime-charset  = 1*mime-charsetc
       * mime-charsetc = ALPHA / DIGIT
       *               / "!" / "#" / "$" / "%" / "&"
       *               / "+" / "-" / "^" / "_" / "`"
       *               / "{" / "}" / "~"
       * language      = ( 2*3ALPHA [ extlang ] )
       *               / 4ALPHA
       *               / 5*8ALPHA
       * extlang       = *3( "-" 3ALPHA )
       * value-chars   = *( pct-encoded / attr-char )
       * pct-encoded   = "%" HEXDIG HEXDIG
       * attr-char     = ALPHA / DIGIT
       *               / "!" / "#" / "$" / "&" / "+" / "-" / "."
       *               / "^" / "_" / "`" / "|" / "~"
       */

      var extValueRegExp = /^([A-Za-z0-9!#$%&+\-^_`{}~]+)'(?:[A-Za-z]{2,3}(?:-[A-Za-z]{3}){0,3}|[A-Za-z]{4,8}|)'((?:%[0-9A-Fa-f]{2}|[A-Za-z0-9!#$&+\-\.^_`|~])+)$/

      /**
       * RegExp for various RFC 6266 grammar
       *
       * disposition-type = "inline" | "attachment" | disp-ext-type
       * disp-ext-type    = token
       * disposition-parm = filename-parm | disp-ext-parm
       * filename-parm    = "filename" "=" value
       *                  | "filename*" "=" ext-value
       * disp-ext-parm    = token "=" value
       *                  | ext-token "=" ext-value
       * ext-token        = <the characters in token, followed by "*">
       */

      var dispositionTypeRegExp = /^([!#$%&'\*\+\-\.0-9A-Z\^_`a-z\|~]+) *(?:$|;)/

      /**
       * Create an attachment Content-Disposition header.
       *
       * @param {string} [filename]
       * @param {object} [options]
       * @param {string} [options.type=attachment]
       * @param {string|boolean} [options.fallback=true]
       * @return {string}
       * @api public
       */

      function contentDisposition(filename, options) {
        var opts = options || {}

        // get type
        var type = opts.type || 'attachment'

        // get parameters
        var params = createparams(filename, opts.fallback)

        // format into string
        return format(new ContentDisposition(type, params))
      }

      /**
       * Create parameters object from filename and fallback.
       *
       * @param {string} [filename]
       * @param {string|boolean} [fallback=true]
       * @return {object}
       * @api private
       */

      function createparams(filename, fallback) {
        if (filename === undefined) {
          return
        }

        var params = {}

        if (typeof filename !== 'string') {
          throw new TypeError('filename must be a string')
        }

        // fallback defaults to true
        if (fallback === undefined) {
          fallback = true
        }

        if (typeof fallback !== 'string' && typeof fallback !== 'boolean') {
          throw new TypeError('fallback must be a string or boolean')
        }

        if (typeof fallback === 'string' && nonLatin1RegExp.test(fallback)) {
          throw new TypeError('fallback must be ISO-8859-1 string')
        }

        // restrict to file base name
        var name = basename(filename)

        // determine if name is suitable for quoted string
        var isQuotedString = textRegExp.test(name)

        // generate fallback name
        var fallbackName = typeof fallback !== 'string'
          ? fallback && getlatin1(name)
          : basename(fallback)
        var hasFallback = typeof fallbackName === 'string' && fallbackName !== name

        // set extended filename parameter
        if (hasFallback || !isQuotedString || hexEscapeRegExp.test(name)) {
          params['filename*'] = name
        }

        // set filename parameter
        if (isQuotedString || hasFallback) {
          params.filename = hasFallback
            ? fallbackName
            : name
        }

        return params
      }

      /**
       * Format object to Content-Disposition header.
       *
       * @param {object} obj
       * @param {string} obj.type
       * @param {object} [obj.parameters]
       * @return {string}
       * @api private
       */

      function format(obj) {
        var parameters = obj.parameters
        var type = obj.type

        if (!type || typeof type !== 'string' || !tokenRegExp.test(type)) {
          throw new TypeError('invalid type')
        }

        // start with normalized type
        var string = String(type).toLowerCase()

        // append parameters
        if (parameters && typeof parameters === 'object') {
          var param
          var params = Object.keys(parameters).sort()

          for (var i = 0; i < params.length; i++) {
            param = params[i]

            var val = param.substr(-1) === '*'
              ? ustring(parameters[param])
              : qstring(parameters[param])

            string += '; ' + param + '=' + val
          }
        }

        return string
      }

      /**
       * Decode a RFC 6987 field value (gracefully).
       *
       * @param {string} str
       * @return {string}
       * @api private
       */

      function decodefield(str) {
        var match = extValueRegExp.exec(str)

        if (!match) {
          throw new TypeError('invalid extended field value')
        }

        var charset = match[1].toLowerCase()
        var encoded = match[2]
        var value

        // to binary string
        var binary = encoded.replace(hexEscapeReplaceRegExp, pdecode)

        switch (charset) {
          case 'iso-8859-1':
            value = getlatin1(binary)
            break
          case 'utf-8':
            value = new Buffer(binary, 'binary').toString('utf8')
            break
          default:
            throw new TypeError('unsupported charset in extended field')
        }

        return value
      }

      /**
       * Get ISO-8859-1 version of string.
       *
       * @param {string} val
       * @return {string}
       * @api private
       */

      function getlatin1(val) {
        // simple Unicode -> ISO-8859-1 transformation
        return String(val).replace(nonLatin1RegExp, '?')
      }

      /**
       * Parse Content-Disposition header string.
       *
       * @param {string} string
       * @return {object}
       * @api private
       */

      function parse(string) {
        if (!string || typeof string !== 'string') {
          throw new TypeError('argument string is required')
        }

        var match = dispositionTypeRegExp.exec(string)

        if (!match) {
          throw new TypeError('invalid type format')
        }

        // normalize type
        var index = match[0].length
        var type = match[1].toLowerCase()

        var key
        var names = []
        var params = {}
        var value

        // calculate index to start at
        index = paramRegExp.lastIndex = match[0].substr(-1) === ';'
          ? index - 1
          : index

        // match parameters
        while (match = paramRegExp.exec(string)) {
          if (match.index !== index) {
            throw new TypeError('invalid parameter format')
          }

          index += match[0].length
          key = match[1].toLowerCase()
          value = match[2]

          if (names.indexOf(key) !== -1) {
            throw new TypeError('invalid duplicate parameter')
          }

          names.push(key)

          if (key.indexOf('*') + 1 === key.length) {
            // decode extended value
            key = key.slice(0, -1)
            value = decodefield(value)

            // overwrite existing value
            params[key] = value
            continue
          }

          if (typeof params[key] === 'string') {
            continue
          }

          if (value[0] === '"') {
            // remove quotes and escapes
            value = value
              .substr(1, value.length - 2)
              .replace(qescRegExp, '$1')
          }

          params[key] = value
        }

        if (index !== -1 && index !== string.length) {
          throw new TypeError('invalid parameter format')
        }

        return new ContentDisposition(type, params)
      }

      /**
       * Percent decode a single character.
       *
       * @param {string} str
       * @param {string} hex
       * @return {string}
       * @api private
       */

      function pdecode(str, hex) {
        return String.fromCharCode(parseInt(hex, 16))
      }

      /**
       * Percent encode a single character.
       *
       * @param {string} char
       * @return {string}
       * @api private
       */

      function pencode(char) {
        var hex = String(char)
          .charCodeAt(0)
          .toString(16)
          .toUpperCase()
        return hex.length === 1
          ? '%0' + hex
          : '%' + hex
      }

      /**
       * Quote a string for HTTP.
       *
       * @param {string} val
       * @return {string}
       * @api private
       */

      function qstring(val) {
        var str = String(val)

        return '"' + str.replace(quoteRegExp, '\\$1') + '"'
      }

      /**
       * Encode a Unicode string for HTTP (RFC 5987).
       *
       * @param {string} val
       * @return {string}
       * @api private
       */

      function ustring(val) {
        var str = String(val)

        // percent encode as UTF-8
        var encoded = encodeURIComponent(str)
          .replace(encodeUriAttrCharRegExp, pencode)

        return 'UTF-8\'\'' + encoded
      }

      /**
       * Class for parsed Content-Disposition header for v8 optimization
       */

      function ContentDisposition(type, parameters) {
        this.type = type
        this.parameters = parameters
      }

    }).call(this, require("buffer").Buffer)
  }, { "buffer": 58, "path": 606 }], 65: [function (require, module, exports) {
    /*!
     * content-type
     * Copyright(c) 2015 Douglas Christopher Wilson
     * MIT Licensed
     */

    'use strict'

    /**
     * RegExp to match *( ";" parameter ) in RFC 7231 sec 3.1.1.1
     *
     * parameter     = token "=" ( token / quoted-string )
     * token         = 1*tchar
     * tchar         = "!" / "#" / "$" / "%" / "&" / "'" / "*"
     *               / "+" / "-" / "." / "^" / "_" / "`" / "|" / "~"
     *               / DIGIT / ALPHA
     *               ; any VCHAR, except delimiters
     * quoted-string = DQUOTE *( qdtext / quoted-pair ) DQUOTE
     * qdtext        = HTAB / SP / %x21 / %x23-5B / %x5D-7E / obs-text
     * obs-text      = %x80-FF
     * quoted-pair   = "\" ( HTAB / SP / VCHAR / obs-text )
     */
    var PARAM_REGEXP = /; *([!#$%&'*+.^_`|~0-9A-Za-z-]+) *= *("(?:[\u000b\u0020\u0021\u0023-\u005b\u005d-\u007e\u0080-\u00ff]|\\[\u000b\u0020-\u00ff])*"|[!#$%&'*+.^_`|~0-9A-Za-z-]+) */g
    var TEXT_REGEXP = /^[\u000b\u0020-\u007e\u0080-\u00ff]+$/
    var TOKEN_REGEXP = /^[!#$%&'*+.^_`|~0-9A-Za-z-]+$/

    /**
     * RegExp to match quoted-pair in RFC 7230 sec 3.2.6
     *
     * quoted-pair = "\" ( HTAB / SP / VCHAR / obs-text )
     * obs-text    = %x80-FF
     */
    var QESC_REGEXP = /\\([\u000b\u0020-\u00ff])/g

    /**
     * RegExp to match chars that must be quoted-pair in RFC 7230 sec 3.2.6
     */
    var QUOTE_REGEXP = /([\\"])/g

    /**
     * RegExp to match type in RFC 7231 sec 3.1.1.1
     *
     * media-type = type "/" subtype
     * type       = token
     * subtype    = token
     */
    var TYPE_REGEXP = /^[!#$%&'*+.^_`|~0-9A-Za-z-]+\/[!#$%&'*+.^_`|~0-9A-Za-z-]+$/

    /**
     * Module exports.
     * @public
     */

    exports.format = format
    exports.parse = parse

    /**
     * Format object to media type.
     *
     * @param {object} obj
     * @return {string}
     * @public
     */

    function format(obj) {
      if (!obj || typeof obj !== 'object') {
        throw new TypeError('argument obj is required')
      }

      var parameters = obj.parameters
      var type = obj.type

      if (!type || !TYPE_REGEXP.test(type)) {
        throw new TypeError('invalid type')
      }

      var string = type

      // append parameters
      if (parameters && typeof parameters === 'object') {
        var param
        var params = Object.keys(parameters).sort()

        for (var i = 0; i < params.length; i++) {
          param = params[i]

          if (!TOKEN_REGEXP.test(param)) {
            throw new TypeError('invalid parameter name')
          }

          string += '; ' + param + '=' + qstring(parameters[param])
        }
      }

      return string
    }

    /**
     * Parse media type to object.
     *
     * @param {string|object} string
     * @return {Object}
     * @public
     */

    function parse(string) {
      if (!string) {
        throw new TypeError('argument string is required')
      }

      // support req/res-like objects as argument
      var header = typeof string === 'object'
        ? getcontenttype(string)
        : string

      if (typeof header !== 'string') {
        throw new TypeError('argument string is required to be a string')
      }

      var index = header.indexOf(';')
      var type = index !== -1
        ? header.substr(0, index).trim()
        : header.trim()

      if (!TYPE_REGEXP.test(type)) {
        throw new TypeError('invalid media type')
      }

      var obj = new ContentType(type.toLowerCase())

      // parse parameters
      if (index !== -1) {
        var key
        var match
        var value

        PARAM_REGEXP.lastIndex = index

        while ((match = PARAM_REGEXP.exec(header))) {
          if (match.index !== index) {
            throw new TypeError('invalid parameter format')
          }

          index += match[0].length
          key = match[1].toLowerCase()
          value = match[2]

          if (value[0] === '"') {
            // remove quotes and escapes
            value = value
              .substr(1, value.length - 2)
              .replace(QESC_REGEXP, '$1')
          }

          obj.parameters[key] = value
        }

        if (index !== header.length) {
          throw new TypeError('invalid parameter format')
        }
      }

      return obj
    }

    /**
     * Get content-type from req/res objects.
     *
     * @param {object}
     * @return {Object}
     * @private
     */

    function getcontenttype(obj) {
      var header

      if (typeof obj.getHeader === 'function') {
        // res-like
        header = obj.getHeader('content-type')
      } else if (typeof obj.headers === 'object') {
        // req-like
        header = obj.headers && obj.headers['content-type']
      }

      if (typeof header !== 'string') {
        throw new TypeError('content-type header is missing from object')
      }

      return header
    }

    /**
     * Quote a string if necessary.
     *
     * @param {string} val
     * @return {string}
     * @private
     */

    function qstring(val) {
      var str = String(val)

      // no need to quote tokens
      if (TOKEN_REGEXP.test(str)) {
        return str
      }

      if (str.length > 0 && !TEXT_REGEXP.test(str)) {
        throw new TypeError('invalid parameter value')
      }

      return '"' + str.replace(QUOTE_REGEXP, '\\$1') + '"'
    }

    /**
     * Class to represent a content type.
     * @private
     */
    function ContentType(type) {
      this.parameters = Object.create(null)
      this.type = type
    }

  }, {}], 66: [function (require, module, exports) {
    /**
     * Module dependencies.
     */

    var crypto = require('crypto');

    /**
     * Sign the given `val` with `secret`.
     *
     * @param {String} val
     * @param {String} secret
     * @return {String}
     * @api private
     */

    exports.sign = function (val, secret) {
      if ('string' != typeof val) throw new TypeError("Cookie value must be provided as a string.");
      if ('string' != typeof secret) throw new TypeError("Secret string must be provided.");
      return val + '.' + crypto
        .createHmac('sha256', secret)
        .update(val)
        .digest('base64')
        .replace(/\=+$/, '');
    };

    /**
     * Unsign and decode the given `val` with `secret`,
     * returning `false` if the signature is invalid.
     *
     * @param {String} val
     * @param {String} secret
     * @return {String|Boolean}
     * @api private
     */

    exports.unsign = function (val, secret) {
      if ('string' != typeof val) throw new TypeError("Signed cookie string must be provided.");
      if ('string' != typeof secret) throw new TypeError("Secret string must be provided.");
      var str = val.slice(0, val.lastIndexOf('.'))
        , mac = exports.sign(str, secret);

      return sha1(mac) == sha1(val) ? str : false;
    };

    /**
     * Private
     */

    function sha1(str) {
      return crypto.createHash('sha1').update(str).digest('hex');
    }

  }, { "crypto": 75 }], 67: [function (require, module, exports) {
    /*!
     * cookie
     * Copyright(c) 2012-2014 Roman Shtylman
     * Copyright(c) 2015 Douglas Christopher Wilson
     * MIT Licensed
     */

    'use strict';

    /**
     * Module exports.
     * @public
     */

    exports.parse = parse;
    exports.serialize = serialize;

    /**
     * Module variables.
     * @private
     */

    var decode = decodeURIComponent;
    var encode = encodeURIComponent;
    var pairSplitRegExp = /; */;

    /**
     * RegExp to match field-content in RFC 7230 sec 3.2
     *
     * field-content = field-vchar [ 1*( SP / HTAB ) field-vchar ]
     * field-vchar   = VCHAR / obs-text
     * obs-text      = %x80-FF
     */

    var fieldContentRegExp = /^[\u0009\u0020-\u007e\u0080-\u00ff]+$/;

    /**
     * Parse a cookie header.
     *
     * Parse the given cookie header string into an object
     * The object has the various cookies as keys(names) => values
     *
     * @param {string} str
     * @param {object} [options]
     * @return {object}
     * @public
     */

    function parse(str, options) {
      if (typeof str !== 'string') {
        throw new TypeError('argument str must be a string');
      }

      var obj = {}
      var opt = options || {};
      var pairs = str.split(pairSplitRegExp);
      var dec = opt.decode || decode;

      for (var i = 0; i < pairs.length; i++) {
        var pair = pairs[i];
        var eq_idx = pair.indexOf('=');

        // skip things that don't look like key=value
        if (eq_idx < 0) {
          continue;
        }

        var key = pair.substr(0, eq_idx).trim()
        var val = pair.substr(++eq_idx, pair.length).trim();

        // quoted values
        if ('"' == val[0]) {
          val = val.slice(1, -1);
        }

        // only assign once
        if (undefined == obj[key]) {
          obj[key] = tryDecode(val, dec);
        }
      }

      return obj;
    }

    /**
     * Serialize data into a cookie header.
     *
     * Serialize the a name value pair into a cookie string suitable for
     * http headers. An optional options object specified cookie parameters.
     *
     * serialize('foo', 'bar', { httpOnly: true })
     *   => "foo=bar; httpOnly"
     *
     * @param {string} name
     * @param {string} val
     * @param {object} [options]
     * @return {string}
     * @public
     */

    function serialize(name, val, options) {
      var opt = options || {};
      var enc = opt.encode || encode;

      if (typeof enc !== 'function') {
        throw new TypeError('option encode is invalid');
      }

      if (!fieldContentRegExp.test(name)) {
        throw new TypeError('argument name is invalid');
      }

      var value = enc(val);

      if (value && !fieldContentRegExp.test(value)) {
        throw new TypeError('argument val is invalid');
      }

      var str = name + '=' + value;

      if (null != opt.maxAge) {
        var maxAge = opt.maxAge - 0;
        if (isNaN(maxAge)) throw new Error('maxAge should be a Number');
        str += '; Max-Age=' + Math.floor(maxAge);
      }

      if (opt.domain) {
        if (!fieldContentRegExp.test(opt.domain)) {
          throw new TypeError('option domain is invalid');
        }

        str += '; Domain=' + opt.domain;
      }

      if (opt.path) {
        if (!fieldContentRegExp.test(opt.path)) {
          throw new TypeError('option path is invalid');
        }

        str += '; Path=' + opt.path;
      }

      if (opt.expires) {
        if (typeof opt.expires.toUTCString !== 'function') {
          throw new TypeError('option expires is invalid');
        }

        str += '; Expires=' + opt.expires.toUTCString();
      }

      if (opt.httpOnly) {
        str += '; HttpOnly';
      }

      if (opt.secure) {
        str += '; Secure';
      }

      if (opt.sameSite) {
        var sameSite = typeof opt.sameSite === 'string'
          ? opt.sameSite.toLowerCase() : opt.sameSite;

        switch (sameSite) {
          case true:
            str += '; SameSite=Strict';
            break;
          case 'lax':
            str += '; SameSite=Lax';
            break;
          case 'strict':
            str += '; SameSite=Strict';
            break;
          default:
            throw new TypeError('option sameSite is invalid');
        }
      }

      return str;
    }

    /**
     * Try decoding a string using a decoding function.
     *
     * @param {string} str
     * @param {function} decode
     * @private
     */

    function tryDecode(str, decode) {
      try {
        return decode(str);
      } catch (e) {
        return str;
      }
    }

  }, {}], 68: [function (require, module, exports) {
    /*!
     * cookies
     * Copyright(c) 2014 Jed Schmidt, http://jed.is/
     * Copyright(c) 2015-2016 Douglas Christopher Wilson
     * MIT Licensed
     */

    'use strict'

    var deprecate = require('depd')('cookies')
    var Keygrip = require('keygrip')
    var http = require('http')
    var cache = {}

    /**
     * RegExp to match field-content in RFC 7230 sec 3.2
     *
     * field-content = field-vchar [ 1*( SP / HTAB ) field-vchar ]
     * field-vchar   = VCHAR / obs-text
     * obs-text      = %x80-FF
     */

    var fieldContentRegExp = /^[\u0009\u0020-\u007e\u0080-\u00ff]+$/;

    /**
     * RegExp to match Same-Site cookie attribute value.
     */

    var sameSiteRegExp = /^(?:lax|strict)$/i

    function Cookies(request, response, options) {
      if (!(this instanceof Cookies)) return new Cookies(request, response, options)

      this.secure = undefined
      this.request = request
      this.response = response

      if (options) {
        if (Array.isArray(options)) {
          // array of key strings
          deprecate('"keys" argument; provide using options {"keys": [...]}')
          this.keys = new Keygrip(options)
        } else if (options.constructor && options.constructor.name === 'Keygrip') {
          // any keygrip constructor to allow different versions
          deprecate('"keys" argument; provide using options {"keys": keygrip}')
          this.keys = options
        } else {
          this.keys = Array.isArray(options.keys) ? new Keygrip(options.keys) : options.keys
          this.secure = options.secure
        }
      }
    }

    Cookies.prototype.get = function (name, opts) {
      var sigName = name + ".sig"
        , header, match, value, remote, data, index
        , signed = opts && opts.signed !== undefined ? opts.signed : !!this.keys

      header = this.request.headers["cookie"]
      if (!header) return

      match = header.match(getPattern(name))
      if (!match) return

      value = match[1]
      if (!opts || !signed) return value

      remote = this.get(sigName)
      if (!remote) return

      data = name + "=" + value
      if (!this.keys) throw new Error('.keys required for signed cookies');
      index = this.keys.index(data, remote)

      if (index < 0) {
        this.set(sigName, null, { path: "/", signed: false })
      } else {
        index && this.set(sigName, this.keys.sign(data), { signed: false })
        return value
      }
    };

    Cookies.prototype.set = function (name, value, opts) {
      var res = this.response
        , req = this.request
        , headers = res.getHeader("Set-Cookie") || []
        , secure = this.secure !== undefined ? !!this.secure : req.protocol === 'https' || req.connection.encrypted
        , cookie = new Cookie(name, value, opts)
        , signed = opts && opts.signed !== undefined ? opts.signed : !!this.keys

      if (typeof headers == "string") headers = [headers]

      if (!secure && opts && opts.secure) {
        throw new Error('Cannot send secure cookie over unencrypted connection')
      }

      cookie.secure = secure
      if (opts && "secure" in opts) cookie.secure = opts.secure

      if (opts && "secureProxy" in opts) {
        deprecate('"secureProxy" option; use "secure" option, provide "secure" to constructor if needed')
        cookie.secure = opts.secureProxy
      }

      pushCookie(headers, cookie)

      if (opts && signed) {
        if (!this.keys) throw new Error('.keys required for signed cookies');
        cookie.value = this.keys.sign(cookie.toString())
        cookie.name += ".sig"
        pushCookie(headers, cookie)
      }

      var setHeader = res.set ? http.OutgoingMessage.prototype.setHeader : res.setHeader
      setHeader.call(res, 'Set-Cookie', headers)
      return this
    };

    function Cookie(name, value, attrs) {
      if (!fieldContentRegExp.test(name)) {
        throw new TypeError('argument name is invalid');
      }

      if (value && !fieldContentRegExp.test(value)) {
        throw new TypeError('argument value is invalid');
      }

      value || (this.expires = new Date(0))

      this.name = name
      this.value = value || ""

      for (var name in attrs) {
        this[name] = attrs[name]
      }

      if (this.path && !fieldContentRegExp.test(this.path)) {
        throw new TypeError('option path is invalid');
      }

      if (this.domain && !fieldContentRegExp.test(this.domain)) {
        throw new TypeError('option domain is invalid');
      }

      if (this.sameSite && this.sameSite !== true && !sameSiteRegExp.test(this.sameSite)) {
        throw new TypeError('option sameSite is invalid')
      }
    }

    Cookie.prototype.path = "/";
    Cookie.prototype.expires = undefined;
    Cookie.prototype.domain = undefined;
    Cookie.prototype.httpOnly = true;
    Cookie.prototype.sameSite = false;
    Cookie.prototype.secure = false;
    Cookie.prototype.overwrite = false;

    Cookie.prototype.toString = function () {
      return this.name + "=" + this.value
    };

    Cookie.prototype.toHeader = function () {
      var header = this.toString()

      if (this.maxAge) this.expires = new Date(Date.now() + this.maxAge);

      if (this.path) header += "; path=" + this.path
      if (this.expires) header += "; expires=" + this.expires.toUTCString()
      if (this.domain) header += "; domain=" + this.domain
      if (this.sameSite) header += "; samesite=" + (this.sameSite === true ? 'strict' : this.sameSite.toLowerCase())
      if (this.secure) header += "; secure"
      if (this.httpOnly) header += "; httponly"

      return header
    };

    // back-compat so maxage mirrors maxAge
    Object.defineProperty(Cookie.prototype, 'maxage', {
      configurable: true,
      enumerable: true,
      get: function () { return this.maxAge },
      set: function (val) { return this.maxAge = val }
    });
    deprecate.property(Cookie.prototype, 'maxage', '"maxage"; use "maxAge" instead')

    function getPattern(name) {
      if (cache[name]) return cache[name]

      return cache[name] = new RegExp(
        "(?:^|;) *" +
        name.replace(/[-[\]{}()*+?.,\\^$|#\s]/g, "\\$&") +
        "=([^;]*)"
      )
    }

    function pushCookie(headers, cookie) {
      if (cookie.overwrite) {
        for (var i = headers.length - 1; i >= 0; i--) {
          if (headers[i].indexOf(cookie.name + '=') === 0) {
            headers.splice(i, 1)
          }
        }
      }

      headers.push(cookie.toHeader())
    }

    Cookies.connect = Cookies.express = function (keys) {
      return function (req, res, next) {
        req.cookies = res.cookies = new Cookies(req, res, {
          keys: keys
        })

        next()
      }
    }

    Cookies.Cookie = Cookie

    module.exports = Cookies

  }, { "depd": 77, "http": 672, "keygrip": 576 }], 69: [function (require, module, exports) {
    (function (Buffer) {
      // Copyright Joyent, Inc. and other Node contributors.
      //
      // Permission is hereby granted, free of charge, to any person obtaining a
      // copy of this software and associated documentation files (the
      // "Software"), to deal in the Software without restriction, including
      // without limitation the rights to use, copy, modify, merge, publish,
      // distribute, sublicense, and/or sell copies of the Software, and to permit
      // persons to whom the Software is furnished to do so, subject to the
      // following conditions:
      //
      // The above copyright notice and this permission notice shall be included
      // in all copies or substantial portions of the Software.
      //
      // THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
      // OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
      // MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
      // NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
      // DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
      // OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
      // USE OR OTHER DEALINGS IN THE SOFTWARE.

      // NOTE: These type checking functions intentionally don't use `instanceof`
      // because it is fragile and can be easily faked with `Object.create()`.

      function isArray(arg) {
        if (Array.isArray) {
          return Array.isArray(arg);
        }
        return objectToString(arg) === '[object Array]';
      }
      exports.isArray = isArray;

      function isBoolean(arg) {
        return typeof arg === 'boolean';
      }
      exports.isBoolean = isBoolean;

      function isNull(arg) {
        return arg === null;
      }
      exports.isNull = isNull;

      function isNullOrUndefined(arg) {
        return arg == null;
      }
      exports.isNullOrUndefined = isNullOrUndefined;

      function isNumber(arg) {
        return typeof arg === 'number';
      }
      exports.isNumber = isNumber;

      function isString(arg) {
        return typeof arg === 'string';
      }
      exports.isString = isString;

      function isSymbol(arg) {
        return typeof arg === 'symbol';
      }
      exports.isSymbol = isSymbol;

      function isUndefined(arg) {
        return arg === void 0;
      }
      exports.isUndefined = isUndefined;

      function isRegExp(re) {
        return objectToString(re) === '[object RegExp]';
      }
      exports.isRegExp = isRegExp;

      function isObject(arg) {
        return typeof arg === 'object' && arg !== null;
      }
      exports.isObject = isObject;

      function isDate(d) {
        return objectToString(d) === '[object Date]';
      }
      exports.isDate = isDate;

      function isError(e) {
        return (objectToString(e) === '[object Error]' || e instanceof Error);
      }
      exports.isError = isError;

      function isFunction(arg) {
        return typeof arg === 'function';
      }
      exports.isFunction = isFunction;

      function isPrimitive(arg) {
        return arg === null ||
          typeof arg === 'boolean' ||
          typeof arg === 'number' ||
          typeof arg === 'string' ||
          typeof arg === 'symbol' ||  // ES6 symbol
          typeof arg === 'undefined';
      }
      exports.isPrimitive = isPrimitive;

      exports.isBuffer = Buffer.isBuffer;

      function objectToString(o) {
        return Object.prototype.toString.call(o);
      }

    }).call(this, { "isBuffer": require("../../is-buffer/index.js") })
  }, { "../../is-buffer/index.js": 571 }], 70: [function (require, module, exports) {
    (function (Buffer) {
      var elliptic = require('elliptic')
      var BN = require('bn.js')

      module.exports = function createECDH(curve) {
        return new ECDH(curve)
      }

      var aliases = {
        secp256k1: {
          name: 'secp256k1',
          byteLength: 32
        },
        secp224r1: {
          name: 'p224',
          byteLength: 28
        },
        prime256v1: {
          name: 'p256',
          byteLength: 32
        },
        prime192v1: {
          name: 'p192',
          byteLength: 24
        },
        ed25519: {
          name: 'ed25519',
          byteLength: 32
        },
        secp384r1: {
          name: 'p384',
          byteLength: 48
        },
        secp521r1: {
          name: 'p521',
          byteLength: 66
        }
      }

      aliases.p224 = aliases.secp224r1
      aliases.p256 = aliases.secp256r1 = aliases.prime256v1
      aliases.p192 = aliases.secp192r1 = aliases.prime192v1
      aliases.p384 = aliases.secp384r1
      aliases.p521 = aliases.secp521r1

      function ECDH(curve) {
        this.curveType = aliases[curve]
        if (!this.curveType) {
          this.curveType = {
            name: curve
          }
        }
        this.curve = new elliptic.ec(this.curveType.name) // eslint-disable-line new-cap
        this.keys = void 0
      }

      ECDH.prototype.generateKeys = function (enc, format) {
        this.keys = this.curve.genKeyPair()
        return this.getPublicKey(enc, format)
      }

      ECDH.prototype.computeSecret = function (other, inenc, enc) {
        inenc = inenc || 'utf8'
        if (!Buffer.isBuffer(other)) {
          other = new Buffer(other, inenc)
        }
        var otherPub = this.curve.keyFromPublic(other).getPublic()
        var out = otherPub.mul(this.keys.getPrivate()).getX()
        return formatReturnValue(out, enc, this.curveType.byteLength)
      }

      ECDH.prototype.getPublicKey = function (enc, format) {
        var key = this.keys.getPublic(format === 'compressed', true)
        if (format === 'hybrid') {
          if (key[key.length - 1] % 2) {
            key[0] = 7
          } else {
            key[0] = 6
          }
        }
        return formatReturnValue(key, enc)
      }

      ECDH.prototype.getPrivateKey = function (enc) {
        return formatReturnValue(this.keys.getPrivate(), enc)
      }

      ECDH.prototype.setPublicKey = function (pub, enc) {
        enc = enc || 'utf8'
        if (!Buffer.isBuffer(pub)) {
          pub = new Buffer(pub, enc)
        }
        this.keys._importPublic(pub)
        return this
      }

      ECDH.prototype.setPrivateKey = function (priv, enc) {
        enc = enc || 'utf8'
        if (!Buffer.isBuffer(priv)) {
          priv = new Buffer(priv, enc)
        }

        var _priv = new BN(priv)
        _priv = _priv.toString(16)
        this.keys = this.curve.genKeyPair()
        this.keys._importPrivate(_priv)
        return this
      }

      function formatReturnValue(bn, enc, len) {
        if (!Array.isArray(bn)) {
          bn = bn.toArray()
        }
        var buf = new Buffer(bn)
        if (len && buf.length < len) {
          var zeros = new Buffer(len - buf.length)
          zeros.fill(0)
          buf = Buffer.concat([zeros, buf])
        }
        if (!enc) {
          return buf
        } else {
          return buf.toString(enc)
        }
      }

    }).call(this, require("buffer").Buffer)
  }, { "bn.js": 24, "buffer": 58, "elliptic": 463 }], 71: [function (require, module, exports) {
    'use strict'
    var inherits = require('inherits')
    var MD5 = require('md5.js')
    var RIPEMD160 = require('ripemd160')
    var sha = require('sha.js')
    var Base = require('cipher-base')

    function Hash(hash) {
      Base.call(this, 'digest')

      this._hash = hash
    }

    inherits(Hash, Base)

    Hash.prototype._update = function (data) {
      this._hash.update(data)
    }

    Hash.prototype._final = function () {
      return this._hash.digest()
    }

    module.exports = function createHash(alg) {
      alg = alg.toLowerCase()
      if (alg === 'md5') return new MD5()
      if (alg === 'rmd160' || alg === 'ripemd160') return new RIPEMD160()

      return new Hash(sha(alg))
    }

  }, { "cipher-base": 61, "inherits": 569, "md5.js": 580, "ripemd160": 642, "sha.js": 661 }], 72: [function (require, module, exports) {
    var MD5 = require('md5.js')

    module.exports = function (buffer) {
      return new MD5().update(buffer).digest()
    }

  }, { "md5.js": 580 }], 73: [function (require, module, exports) {
    'use strict'
    var inherits = require('inherits')
    var Legacy = require('./legacy')
    var Base = require('cipher-base')
    var Buffer = require('safe-buffer').Buffer
    var md5 = require('create-hash/md5')
    var RIPEMD160 = require('ripemd160')

    var sha = require('sha.js')

    var ZEROS = Buffer.alloc(128)

    function Hmac(alg, key) {
      Base.call(this, 'digest')
      if (typeof key === 'string') {
        key = Buffer.from(key)
      }

      var blocksize = (alg === 'sha512' || alg === 'sha384') ? 128 : 64

      this._alg = alg
      this._key = key
      if (key.length > blocksize) {
        var hash = alg === 'rmd160' ? new RIPEMD160() : sha(alg)
        key = hash.update(key).digest()
      } else if (key.length < blocksize) {
        key = Buffer.concat([key, ZEROS], blocksize)
      }

      var ipad = this._ipad = Buffer.allocUnsafe(blocksize)
      var opad = this._opad = Buffer.allocUnsafe(blocksize)

      for (var i = 0; i < blocksize; i++) {
        ipad[i] = key[i] ^ 0x36
        opad[i] = key[i] ^ 0x5C
      }
      this._hash = alg === 'rmd160' ? new RIPEMD160() : sha(alg)
      this._hash.update(ipad)
    }

    inherits(Hmac, Base)

    Hmac.prototype._update = function (data) {
      this._hash.update(data)
    }

    Hmac.prototype._final = function () {
      var h = this._hash.digest()
      var hash = this._alg === 'rmd160' ? new RIPEMD160() : sha(this._alg)
      return hash.update(this._opad).update(h).digest()
    }

    module.exports = function createHmac(alg, key) {
      alg = alg.toLowerCase()
      if (alg === 'rmd160' || alg === 'ripemd160') {
        return new Hmac('rmd160', key)
      }
      if (alg === 'md5') {
        return new Legacy(md5, key)
      }
      return new Hmac(alg, key)
    }

  }, { "./legacy": 74, "cipher-base": 61, "create-hash/md5": 72, "inherits": 569, "ripemd160": 642, "safe-buffer": 643, "sha.js": 661 }], 74: [function (require, module, exports) {
    'use strict'
    var inherits = require('inherits')
    var Buffer = require('safe-buffer').Buffer

    var Base = require('cipher-base')

    var ZEROS = Buffer.alloc(128)
    var blocksize = 64

    function Hmac(alg, key) {
      Base.call(this, 'digest')
      if (typeof key === 'string') {
        key = Buffer.from(key)
      }

      this._alg = alg
      this._key = key

      if (key.length > blocksize) {
        key = alg(key)
      } else if (key.length < blocksize) {
        key = Buffer.concat([key, ZEROS], blocksize)
      }

      var ipad = this._ipad = Buffer.allocUnsafe(blocksize)
      var opad = this._opad = Buffer.allocUnsafe(blocksize)

      for (var i = 0; i < blocksize; i++) {
        ipad[i] = key[i] ^ 0x36
        opad[i] = key[i] ^ 0x5C
      }

      this._hash = [ipad]
    }

    inherits(Hmac, Base)

    Hmac.prototype._update = function (data) {
      this._hash.push(data)
    }

    Hmac.prototype._final = function () {
      var h = this._alg(Buffer.concat(this._hash))
      return this._alg(Buffer.concat([this._opad, h]))
    }
    module.exports = Hmac

  }, { "cipher-base": 61, "inherits": 569, "safe-buffer": 643 }], 75: [function (require, module, exports) {
    'use strict'

    exports.randomBytes = exports.rng = exports.pseudoRandomBytes = exports.prng = require('randombytes')
    exports.createHash = exports.Hash = require('create-hash')
    exports.createHmac = exports.Hmac = require('create-hmac')

    var algos = require('browserify-sign/algos')
    var algoKeys = Object.keys(algos)
    var hashes = ['sha1', 'sha224', 'sha256', 'sha384', 'sha512', 'md5', 'rmd160'].concat(algoKeys)
    exports.getHashes = function () {
      return hashes
    }

    var p = require('pbkdf2')
    exports.pbkdf2 = p.pbkdf2
    exports.pbkdf2Sync = p.pbkdf2Sync

    var aes = require('browserify-cipher')

    exports.Cipher = aes.Cipher
    exports.createCipher = aes.createCipher
    exports.Cipheriv = aes.Cipheriv
    exports.createCipheriv = aes.createCipheriv
    exports.Decipher = aes.Decipher
    exports.createDecipher = aes.createDecipher
    exports.Decipheriv = aes.Decipheriv
    exports.createDecipheriv = aes.createDecipheriv
    exports.getCiphers = aes.getCiphers
    exports.listCiphers = aes.listCiphers

    var dh = require('diffie-hellman')

    exports.DiffieHellmanGroup = dh.DiffieHellmanGroup
    exports.createDiffieHellmanGroup = dh.createDiffieHellmanGroup
    exports.getDiffieHellman = dh.getDiffieHellman
    exports.createDiffieHellman = dh.createDiffieHellman
    exports.DiffieHellman = dh.DiffieHellman

    var sign = require('browserify-sign')

    exports.createSign = sign.createSign
    exports.Sign = sign.Sign
    exports.createVerify = sign.createVerify
    exports.Verify = sign.Verify

    exports.createECDH = require('create-ecdh')

    var publicEncrypt = require('public-encrypt')

    exports.publicEncrypt = publicEncrypt.publicEncrypt
    exports.privateEncrypt = publicEncrypt.privateEncrypt
    exports.publicDecrypt = publicEncrypt.publicDecrypt
    exports.privateDecrypt = publicEncrypt.privateDecrypt

    // the least I can do is make error messages for the rest of the node.js/crypto api.
    // ;[
    //   'createCredentials'
    // ].forEach(function (name) {
    //   exports[name] = function () {
    //     throw new Error([
    //       'sorry, ' + name + ' is not implemented yet',
    //       'we accept pull requests',
    //       'https://github.com/crypto-browserify/crypto-browserify'
    //     ].join('\n'))
    //   }
    // })

    var rf = require('randomfill')

    exports.randomFill = rf.randomFill
    exports.randomFillSync = rf.randomFillSync

    exports.createCredentials = function () {
      throw new Error([
        'sorry, createCredentials is not implemented yet',
        'we accept pull requests',
        'https://github.com/crypto-browserify/crypto-browserify'
      ].join('\n'))
    }

    exports.constants = {
      'DH_CHECK_P_NOT_SAFE_PRIME': 2,
      'DH_CHECK_P_NOT_PRIME': 1,
      'DH_UNABLE_TO_CHECK_GENERATOR': 4,
      'DH_NOT_SUITABLE_GENERATOR': 8,
      'NPN_ENABLED': 1,
      'ALPN_ENABLED': 1,
      'RSA_PKCS1_PADDING': 1,
      'RSA_SSLV23_PADDING': 2,
      'RSA_NO_PADDING': 3,
      'RSA_PKCS1_OAEP_PADDING': 4,
      'RSA_X931_PADDING': 5,
      'RSA_PKCS1_PSS_PADDING': 6,
      'POINT_CONVERSION_COMPRESSED': 2,
      'POINT_CONVERSION_UNCOMPRESSED': 4,
      'POINT_CONVERSION_HYBRID': 6
    }

  }, { "browserify-cipher": 44, "browserify-sign": 51, "browserify-sign/algos": 48, "create-ecdh": 70, "create-hash": 71, "create-hmac": 73, "diffie-hellman": 85, "pbkdf2": 608, "public-encrypt": 616, "randombytes": 626, "randomfill": 627 }], 76: [function (require, module, exports) {
    'use strict';
    var token = '%[a-f0-9]{2}';
    var singleMatcher = new RegExp(token, 'gi');
    var multiMatcher = new RegExp('(' + token + ')+', 'gi');

    function decodeComponents(components, split) {
      try {
        // Try to decode the entire string first
        return decodeURIComponent(components.join(''));
      } catch (err) {
        // Do nothing
      }

      if (components.length === 1) {
        return components;
      }

      split = split || 1;

      // Split the array in 2 parts
      var left = components.slice(0, split);
      var right = components.slice(split);

      return Array.prototype.concat.call([], decodeComponents(left), decodeComponents(right));
    }

    function decode(input) {
      try {
        return decodeURIComponent(input);
      } catch (err) {
        var tokens = input.match(singleMatcher);

        for (var i = 1; i < tokens.length; i++) {
          input = decodeComponents(tokens, i).join('');

          tokens = input.match(singleMatcher);
        }

        return input;
      }
    }

    function customDecodeURIComponent(input) {
      // Keep track of all the replacements and prefill the map with the `BOM`
      var replaceMap = {
        '%FE%FF': '\uFFFD\uFFFD',
        '%FF%FE': '\uFFFD\uFFFD'
      };

      var match = multiMatcher.exec(input);
      while (match) {
        try {
          // Decode as big chunks as possible
          replaceMap[match[0]] = decodeURIComponent(match[0]);
        } catch (err) {
          var result = decode(match[0]);

          if (result !== match[0]) {
            replaceMap[match[0]] = result;
          }
        }

        match = multiMatcher.exec(input);
      }

      // Add `%C2` at the end of the map to make sure it does not replace the combinator before everything else
      replaceMap['%C2'] = '\uFFFD';

      var entries = Object.keys(replaceMap);

      for (var i = 0; i < entries.length; i++) {
        // Replace all decoded components
        var key = entries[i];
        input = input.replace(new RegExp(key, 'g'), replaceMap[key]);
      }

      return input;
    }

    module.exports = function (encodedURI) {
      if (typeof encodedURI !== 'string') {
        throw new TypeError('Expected `encodedURI` to be of type `string`, got `' + typeof encodedURI + '`');
      }

      try {
        encodedURI = encodedURI.replace(/\+/g, ' ');

        // Try the built in decoder first
        return decodeURIComponent(encodedURI);
      } catch (err) {
        // Fallback to a more advanced decoder
        return customDecodeURIComponent(encodedURI);
      }
    };

  }, {}], 77: [function (require, module, exports) {
    /*!
     * depd
     * Copyright(c) 2015 Douglas Christopher Wilson
     * MIT Licensed
     */

    'use strict'

    /**
     * Module exports.
     * @public
     */

    module.exports = depd

    /**
     * Create deprecate for namespace in caller.
     */

    function depd(namespace) {
      if (!namespace) {
        throw new TypeError('argument namespace is required')
      }

      function deprecate(message) {
        // no-op in browser
      }

      deprecate._file = undefined
      deprecate._ignored = true
      deprecate._namespace = namespace
      deprecate._traced = false
      deprecate._warned = Object.create(null)

      deprecate.function = wrapfunction
      deprecate.property = wrapproperty

      return deprecate
    }

    /**
     * Return a wrapped function in a deprecation message.
     *
     * This is a no-op version of the wrapper, which does nothing but call
     * validation.
     */

    function wrapfunction(fn, message) {
      if (typeof fn !== 'function') {
        throw new TypeError('argument fn must be a function')
      }

      return fn
    }

    /**
     * Wrap property in a deprecation message.
     *
     * This is a no-op version of the wrapper, which does nothing but call
     * validation.
     */

    function wrapproperty(obj, prop, message) {
      if (!obj || (typeof obj !== 'object' && typeof obj !== 'function')) {
        throw new TypeError('argument obj must be object')
      }

      var descriptor = Object.getOwnPropertyDescriptor(obj, prop)

      if (!descriptor) {
        throw new TypeError('must call property on owner object')
      }

      if (!descriptor.configurable) {
        throw new TypeError('property must be configurable')
      }
    }

  }, {}], 78: [function (require, module, exports) {
    'use strict';

    exports.utils = require('./des/utils');
    exports.Cipher = require('./des/cipher');
    exports.DES = require('./des/des');
    exports.CBC = require('./des/cbc');
    exports.EDE = require('./des/ede');

  }, { "./des/cbc": 79, "./des/cipher": 80, "./des/des": 81, "./des/ede": 82, "./des/utils": 83 }], 79: [function (require, module, exports) {
    'use strict';

    var assert = require('minimalistic-assert');
    var inherits = require('inherits');

    var proto = {};

    function CBCState(iv) {
      assert.equal(iv.length, 8, 'Invalid IV length');

      this.iv = new Array(8);
      for (var i = 0; i < this.iv.length; i++)
        this.iv[i] = iv[i];
    }

    function instantiate(Base) {
      function CBC(options) {
        Base.call(this, options);
        this._cbcInit();
      }
      inherits(CBC, Base);

      var keys = Object.keys(proto);
      for (var i = 0; i < keys.length; i++) {
        var key = keys[i];
        CBC.prototype[key] = proto[key];
      }

      CBC.create = function create(options) {
        return new CBC(options);
      };

      return CBC;
    }

    exports.instantiate = instantiate;

    proto._cbcInit = function _cbcInit() {
      var state = new CBCState(this.options.iv);
      this._cbcState = state;
    };

    proto._update = function _update(inp, inOff, out, outOff) {
      var state = this._cbcState;
      var superProto = this.constructor.super_.prototype;

      var iv = state.iv;
      if (this.type === 'encrypt') {
        for (var i = 0; i < this.blockSize; i++)
          iv[i] ^= inp[inOff + i];

        superProto._update.call(this, iv, 0, out, outOff);

        for (var i = 0; i < this.blockSize; i++)
          iv[i] = out[outOff + i];
      } else {
        superProto._update.call(this, inp, inOff, out, outOff);

        for (var i = 0; i < this.blockSize; i++)
          out[outOff + i] ^= iv[i];

        for (var i = 0; i < this.blockSize; i++)
          iv[i] = inp[inOff + i];
      }
    };

  }, { "inherits": 569, "minimalistic-assert": 590 }], 80: [function (require, module, exports) {
    'use strict';

    var assert = require('minimalistic-assert');

    function Cipher(options) {
      this.options = options;

      this.type = this.options.type;
      this.blockSize = 8;
      this._init();

      this.buffer = new Array(this.blockSize);
      this.bufferOff = 0;
    }
    module.exports = Cipher;

    Cipher.prototype._init = function _init() {
      // Might be overrided
    };

    Cipher.prototype.update = function update(data) {
      if (data.length === 0)
        return [];

      if (this.type === 'decrypt')
        return this._updateDecrypt(data);
      else
        return this._updateEncrypt(data);
    };

    Cipher.prototype._buffer = function _buffer(data, off) {
      // Append data to buffer
      var min = Math.min(this.buffer.length - this.bufferOff, data.length - off);
      for (var i = 0; i < min; i++)
        this.buffer[this.bufferOff + i] = data[off + i];
      this.bufferOff += min;

      // Shift next
      return min;
    };

    Cipher.prototype._flushBuffer = function _flushBuffer(out, off) {
      this._update(this.buffer, 0, out, off);
      this.bufferOff = 0;
      return this.blockSize;
    };

    Cipher.prototype._updateEncrypt = function _updateEncrypt(data) {
      var inputOff = 0;
      var outputOff = 0;

      var count = ((this.bufferOff + data.length) / this.blockSize) | 0;
      var out = new Array(count * this.blockSize);

      if (this.bufferOff !== 0) {
        inputOff += this._buffer(data, inputOff);

        if (this.bufferOff === this.buffer.length)
          outputOff += this._flushBuffer(out, outputOff);
      }

      // Write blocks
      var max = data.length - ((data.length - inputOff) % this.blockSize);
      for (; inputOff < max; inputOff += this.blockSize) {
        this._update(data, inputOff, out, outputOff);
        outputOff += this.blockSize;
      }

      // Queue rest
      for (; inputOff < data.length; inputOff++, this.bufferOff++)
        this.buffer[this.bufferOff] = data[inputOff];

      return out;
    };

    Cipher.prototype._updateDecrypt = function _updateDecrypt(data) {
      var inputOff = 0;
      var outputOff = 0;

      var count = Math.ceil((this.bufferOff + data.length) / this.blockSize) - 1;
      var out = new Array(count * this.blockSize);

      // TODO(indutny): optimize it, this is far from optimal
      for (; count > 0; count--) {
        inputOff += this._buffer(data, inputOff);
        outputOff += this._flushBuffer(out, outputOff);
      }

      // Buffer rest of the input
      inputOff += this._buffer(data, inputOff);

      return out;
    };

    Cipher.prototype.final = function final(buffer) {
      var first;
      if (buffer)
        first = this.update(buffer);

      var last;
      if (this.type === 'encrypt')
        last = this._finalEncrypt();
      else
        last = this._finalDecrypt();

      if (first)
        return first.concat(last);
      else
        return last;
    };

    Cipher.prototype._pad = function _pad(buffer, off) {
      if (off === 0)
        return false;

      while (off < buffer.length)
        buffer[off++] = 0;

      return true;
    };

    Cipher.prototype._finalEncrypt = function _finalEncrypt() {
      if (!this._pad(this.buffer, this.bufferOff))
        return [];

      var out = new Array(this.blockSize);
      this._update(this.buffer, 0, out, 0);
      return out;
    };

    Cipher.prototype._unpad = function _unpad(buffer) {
      return buffer;
    };

    Cipher.prototype._finalDecrypt = function _finalDecrypt() {
      assert.equal(this.bufferOff, this.blockSize, 'Not enough data to decrypt');
      var out = new Array(this.blockSize);
      this._flushBuffer(out, 0);

      return this._unpad(out);
    };

  }, { "minimalistic-assert": 590 }], 81: [function (require, module, exports) {
    'use strict';

    var assert = require('minimalistic-assert');
    var inherits = require('inherits');

    var des = require('../des');
    var utils = des.utils;
    var Cipher = des.Cipher;

    function DESState() {
      this.tmp = new Array(2);
      this.keys = null;
    }

    function DES(options) {
      Cipher.call(this, options);

      var state = new DESState();
      this._desState = state;

      this.deriveKeys(state, options.key);
    }
    inherits(DES, Cipher);
    module.exports = DES;

    DES.create = function create(options) {
      return new DES(options);
    };

    var shiftTable = [
      1, 1, 2, 2, 2, 2, 2, 2,
      1, 2, 2, 2, 2, 2, 2, 1
    ];

    DES.prototype.deriveKeys = function deriveKeys(state, key) {
      state.keys = new Array(16 * 2);

      assert.equal(key.length, this.blockSize, 'Invalid key length');

      var kL = utils.readUInt32BE(key, 0);
      var kR = utils.readUInt32BE(key, 4);

      utils.pc1(kL, kR, state.tmp, 0);
      kL = state.tmp[0];
      kR = state.tmp[1];
      for (var i = 0; i < state.keys.length; i += 2) {
        var shift = shiftTable[i >>> 1];
        kL = utils.r28shl(kL, shift);
        kR = utils.r28shl(kR, shift);
        utils.pc2(kL, kR, state.keys, i);
      }
    };

    DES.prototype._update = function _update(inp, inOff, out, outOff) {
      var state = this._desState;

      var l = utils.readUInt32BE(inp, inOff);
      var r = utils.readUInt32BE(inp, inOff + 4);

      // Initial Permutation
      utils.ip(l, r, state.tmp, 0);
      l = state.tmp[0];
      r = state.tmp[1];

      if (this.type === 'encrypt')
        this._encrypt(state, l, r, state.tmp, 0);
      else
        this._decrypt(state, l, r, state.tmp, 0);

      l = state.tmp[0];
      r = state.tmp[1];

      utils.writeUInt32BE(out, l, outOff);
      utils.writeUInt32BE(out, r, outOff + 4);
    };

    DES.prototype._pad = function _pad(buffer, off) {
      var value = buffer.length - off;
      for (var i = off; i < buffer.length; i++)
        buffer[i] = value;

      return true;
    };

    DES.prototype._unpad = function _unpad(buffer) {
      var pad = buffer[buffer.length - 1];
      for (var i = buffer.length - pad; i < buffer.length; i++)
        assert.equal(buffer[i], pad);

      return buffer.slice(0, buffer.length - pad);
    };

    DES.prototype._encrypt = function _encrypt(state, lStart, rStart, out, off) {
      var l = lStart;
      var r = rStart;

      // Apply f() x16 times
      for (var i = 0; i < state.keys.length; i += 2) {
        var keyL = state.keys[i];
        var keyR = state.keys[i + 1];

        // f(r, k)
        utils.expand(r, state.tmp, 0);

        keyL ^= state.tmp[0];
        keyR ^= state.tmp[1];
        var s = utils.substitute(keyL, keyR);
        var f = utils.permute(s);

        var t = r;
        r = (l ^ f) >>> 0;
        l = t;
      }

      // Reverse Initial Permutation
      utils.rip(r, l, out, off);
    };

    DES.prototype._decrypt = function _decrypt(state, lStart, rStart, out, off) {
      var l = rStart;
      var r = lStart;

      // Apply f() x16 times
      for (var i = state.keys.length - 2; i >= 0; i -= 2) {
        var keyL = state.keys[i];
        var keyR = state.keys[i + 1];

        // f(r, k)
        utils.expand(l, state.tmp, 0);

        keyL ^= state.tmp[0];
        keyR ^= state.tmp[1];
        var s = utils.substitute(keyL, keyR);
        var f = utils.permute(s);

        var t = l;
        l = (r ^ f) >>> 0;
        r = t;
      }

      // Reverse Initial Permutation
      utils.rip(l, r, out, off);
    };

  }, { "../des": 78, "inherits": 569, "minimalistic-assert": 590 }], 82: [function (require, module, exports) {
    'use strict';

    var assert = require('minimalistic-assert');
    var inherits = require('inherits');

    var des = require('../des');
    var Cipher = des.Cipher;
    var DES = des.DES;

    function EDEState(type, key) {
      assert.equal(key.length, 24, 'Invalid key length');

      var k1 = key.slice(0, 8);
      var k2 = key.slice(8, 16);
      var k3 = key.slice(16, 24);

      if (type === 'encrypt') {
        this.ciphers = [
          DES.create({ type: 'encrypt', key: k1 }),
          DES.create({ type: 'decrypt', key: k2 }),
          DES.create({ type: 'encrypt', key: k3 })
        ];
      } else {
        this.ciphers = [
          DES.create({ type: 'decrypt', key: k3 }),
          DES.create({ type: 'encrypt', key: k2 }),
          DES.create({ type: 'decrypt', key: k1 })
        ];
      }
    }

    function EDE(options) {
      Cipher.call(this, options);

      var state = new EDEState(this.type, this.options.key);
      this._edeState = state;
    }
    inherits(EDE, Cipher);

    module.exports = EDE;

    EDE.create = function create(options) {
      return new EDE(options);
    };

    EDE.prototype._update = function _update(inp, inOff, out, outOff) {
      var state = this._edeState;

      state.ciphers[0]._update(inp, inOff, out, outOff);
      state.ciphers[1]._update(out, outOff, out, outOff);
      state.ciphers[2]._update(out, outOff, out, outOff);
    };

    EDE.prototype._pad = DES.prototype._pad;
    EDE.prototype._unpad = DES.prototype._unpad;

  }, { "../des": 78, "inherits": 569, "minimalistic-assert": 590 }], 83: [function (require, module, exports) {
    'use strict';

    exports.readUInt32BE = function readUInt32BE(bytes, off) {
      var res = (bytes[0 + off] << 24) |
        (bytes[1 + off] << 16) |
        (bytes[2 + off] << 8) |
        bytes[3 + off];
      return res >>> 0;
    };

    exports.writeUInt32BE = function writeUInt32BE(bytes, value, off) {
      bytes[0 + off] = value >>> 24;
      bytes[1 + off] = (value >>> 16) & 0xff;
      bytes[2 + off] = (value >>> 8) & 0xff;
      bytes[3 + off] = value & 0xff;
    };

    exports.ip = function ip(inL, inR, out, off) {
      var outL = 0;
      var outR = 0;

      for (var i = 6; i >= 0; i -= 2) {
        for (var j = 0; j <= 24; j += 8) {
          outL <<= 1;
          outL |= (inR >>> (j + i)) & 1;
        }
        for (var j = 0; j <= 24; j += 8) {
          outL <<= 1;
          outL |= (inL >>> (j + i)) & 1;
        }
      }

      for (var i = 6; i >= 0; i -= 2) {
        for (var j = 1; j <= 25; j += 8) {
          outR <<= 1;
          outR |= (inR >>> (j + i)) & 1;
        }
        for (var j = 1; j <= 25; j += 8) {
          outR <<= 1;
          outR |= (inL >>> (j + i)) & 1;
        }
      }

      out[off + 0] = outL >>> 0;
      out[off + 1] = outR >>> 0;
    };

    exports.rip = function rip(inL, inR, out, off) {
      var outL = 0;
      var outR = 0;

      for (var i = 0; i < 4; i++) {
        for (var j = 24; j >= 0; j -= 8) {
          outL <<= 1;
          outL |= (inR >>> (j + i)) & 1;
          outL <<= 1;
          outL |= (inL >>> (j + i)) & 1;
        }
      }
      for (var i = 4; i < 8; i++) {
        for (var j = 24; j >= 0; j -= 8) {
          outR <<= 1;
          outR |= (inR >>> (j + i)) & 1;
          outR <<= 1;
          outR |= (inL >>> (j + i)) & 1;
        }
      }

      out[off + 0] = outL >>> 0;
      out[off + 1] = outR >>> 0;
    };

    exports.pc1 = function pc1(inL, inR, out, off) {
      var outL = 0;
      var outR = 0;

      // 7, 15, 23, 31, 39, 47, 55, 63
      // 6, 14, 22, 30, 39, 47, 55, 63
      // 5, 13, 21, 29, 39, 47, 55, 63
      // 4, 12, 20, 28
      for (var i = 7; i >= 5; i--) {
        for (var j = 0; j <= 24; j += 8) {
          outL <<= 1;
          outL |= (inR >> (j + i)) & 1;
        }
        for (var j = 0; j <= 24; j += 8) {
          outL <<= 1;
          outL |= (inL >> (j + i)) & 1;
        }
      }
      for (var j = 0; j <= 24; j += 8) {
        outL <<= 1;
        outL |= (inR >> (j + i)) & 1;
      }

      // 1, 9, 17, 25, 33, 41, 49, 57
      // 2, 10, 18, 26, 34, 42, 50, 58
      // 3, 11, 19, 27, 35, 43, 51, 59
      // 36, 44, 52, 60
      for (var i = 1; i <= 3; i++) {
        for (var j = 0; j <= 24; j += 8) {
          outR <<= 1;
          outR |= (inR >> (j + i)) & 1;
        }
        for (var j = 0; j <= 24; j += 8) {
          outR <<= 1;
          outR |= (inL >> (j + i)) & 1;
        }
      }
      for (var j = 0; j <= 24; j += 8) {
        outR <<= 1;
        outR |= (inL >> (j + i)) & 1;
      }

      out[off + 0] = outL >>> 0;
      out[off + 1] = outR >>> 0;
    };

    exports.r28shl = function r28shl(num, shift) {
      return ((num << shift) & 0xfffffff) | (num >>> (28 - shift));
    };

    var pc2table = [
      // inL => outL
      14, 11, 17, 4, 27, 23, 25, 0,
      13, 22, 7, 18, 5, 9, 16, 24,
      2, 20, 12, 21, 1, 8, 15, 26,

      // inR => outR
      15, 4, 25, 19, 9, 1, 26, 16,
      5, 11, 23, 8, 12, 7, 17, 0,
      22, 3, 10, 14, 6, 20, 27, 24
    ];

    exports.pc2 = function pc2(inL, inR, out, off) {
      var outL = 0;
      var outR = 0;

      var len = pc2table.length >>> 1;
      for (var i = 0; i < len; i++) {
        outL <<= 1;
        outL |= (inL >>> pc2table[i]) & 0x1;
      }
      for (var i = len; i < pc2table.length; i++) {
        outR <<= 1;
        outR |= (inR >>> pc2table[i]) & 0x1;
      }

      out[off + 0] = outL >>> 0;
      out[off + 1] = outR >>> 0;
    };

    exports.expand = function expand(r, out, off) {
      var outL = 0;
      var outR = 0;

      outL = ((r & 1) << 5) | (r >>> 27);
      for (var i = 23; i >= 15; i -= 4) {
        outL <<= 6;
        outL |= (r >>> i) & 0x3f;
      }
      for (var i = 11; i >= 3; i -= 4) {
        outR |= (r >>> i) & 0x3f;
        outR <<= 6;
      }
      outR |= ((r & 0x1f) << 1) | (r >>> 31);

      out[off + 0] = outL >>> 0;
      out[off + 1] = outR >>> 0;
    };

    var sTable = [
      14, 0, 4, 15, 13, 7, 1, 4, 2, 14, 15, 2, 11, 13, 8, 1,
      3, 10, 10, 6, 6, 12, 12, 11, 5, 9, 9, 5, 0, 3, 7, 8,
      4, 15, 1, 12, 14, 8, 8, 2, 13, 4, 6, 9, 2, 1, 11, 7,
      15, 5, 12, 11, 9, 3, 7, 14, 3, 10, 10, 0, 5, 6, 0, 13,

      15, 3, 1, 13, 8, 4, 14, 7, 6, 15, 11, 2, 3, 8, 4, 14,
      9, 12, 7, 0, 2, 1, 13, 10, 12, 6, 0, 9, 5, 11, 10, 5,
      0, 13, 14, 8, 7, 10, 11, 1, 10, 3, 4, 15, 13, 4, 1, 2,
      5, 11, 8, 6, 12, 7, 6, 12, 9, 0, 3, 5, 2, 14, 15, 9,

      10, 13, 0, 7, 9, 0, 14, 9, 6, 3, 3, 4, 15, 6, 5, 10,
      1, 2, 13, 8, 12, 5, 7, 14, 11, 12, 4, 11, 2, 15, 8, 1,
      13, 1, 6, 10, 4, 13, 9, 0, 8, 6, 15, 9, 3, 8, 0, 7,
      11, 4, 1, 15, 2, 14, 12, 3, 5, 11, 10, 5, 14, 2, 7, 12,

      7, 13, 13, 8, 14, 11, 3, 5, 0, 6, 6, 15, 9, 0, 10, 3,
      1, 4, 2, 7, 8, 2, 5, 12, 11, 1, 12, 10, 4, 14, 15, 9,
      10, 3, 6, 15, 9, 0, 0, 6, 12, 10, 11, 1, 7, 13, 13, 8,
      15, 9, 1, 4, 3, 5, 14, 11, 5, 12, 2, 7, 8, 2, 4, 14,

      2, 14, 12, 11, 4, 2, 1, 12, 7, 4, 10, 7, 11, 13, 6, 1,
      8, 5, 5, 0, 3, 15, 15, 10, 13, 3, 0, 9, 14, 8, 9, 6,
      4, 11, 2, 8, 1, 12, 11, 7, 10, 1, 13, 14, 7, 2, 8, 13,
      15, 6, 9, 15, 12, 0, 5, 9, 6, 10, 3, 4, 0, 5, 14, 3,

      12, 10, 1, 15, 10, 4, 15, 2, 9, 7, 2, 12, 6, 9, 8, 5,
      0, 6, 13, 1, 3, 13, 4, 14, 14, 0, 7, 11, 5, 3, 11, 8,
      9, 4, 14, 3, 15, 2, 5, 12, 2, 9, 8, 5, 12, 15, 3, 10,
      7, 11, 0, 14, 4, 1, 10, 7, 1, 6, 13, 0, 11, 8, 6, 13,

      4, 13, 11, 0, 2, 11, 14, 7, 15, 4, 0, 9, 8, 1, 13, 10,
      3, 14, 12, 3, 9, 5, 7, 12, 5, 2, 10, 15, 6, 8, 1, 6,
      1, 6, 4, 11, 11, 13, 13, 8, 12, 1, 3, 4, 7, 10, 14, 7,
      10, 9, 15, 5, 6, 0, 8, 15, 0, 14, 5, 2, 9, 3, 2, 12,

      13, 1, 2, 15, 8, 13, 4, 8, 6, 10, 15, 3, 11, 7, 1, 4,
      10, 12, 9, 5, 3, 6, 14, 11, 5, 0, 0, 14, 12, 9, 7, 2,
      7, 2, 11, 1, 4, 14, 1, 7, 9, 4, 12, 10, 14, 8, 2, 13,
      0, 15, 6, 12, 10, 9, 13, 0, 15, 3, 3, 5, 5, 6, 8, 11
    ];

    exports.substitute = function substitute(inL, inR) {
      var out = 0;
      for (var i = 0; i < 4; i++) {
        var b = (inL >>> (18 - i * 6)) & 0x3f;
        var sb = sTable[i * 0x40 + b];

        out <<= 4;
        out |= sb;
      }
      for (var i = 0; i < 4; i++) {
        var b = (inR >>> (18 - i * 6)) & 0x3f;
        var sb = sTable[4 * 0x40 + i * 0x40 + b];

        out <<= 4;
        out |= sb;
      }
      return out >>> 0;
    };

    var permuteTable = [
      16, 25, 12, 11, 3, 20, 4, 15, 31, 17, 9, 6, 27, 14, 1, 22,
      30, 24, 8, 18, 0, 5, 29, 23, 13, 19, 2, 26, 10, 21, 28, 7
    ];

    exports.permute = function permute(num) {
      var out = 0;
      for (var i = 0; i < permuteTable.length; i++) {
        out <<= 1;
        out |= (num >>> permuteTable[i]) & 0x1;
      }
      return out >>> 0;
    };

    exports.padSplit = function padSplit(num, size, group) {
      var str = num.toString(2);
      while (str.length < size)
        str = '0' + str;

      var out = [];
      for (var i = 0; i < size; i += group)
        out.push(str.slice(i, i + group));
      return out.join(' ');
    };

  }, {}], 84: [function (require, module, exports) {
    /*!
     * destroy
     * Copyright(c) 2014 Jonathan Ong
     * MIT Licensed
     */

    'use strict'

    /**
     * Module dependencies.
     * @private
     */

    var ReadStream = require('fs').ReadStream
    var Stream = require('stream')

    /**
     * Module exports.
     * @public
     */

    module.exports = destroy

    /**
     * Destroy a stream.
     *
     * @param {object} stream
     * @public
     */

    function destroy(stream) {
      if (stream instanceof ReadStream) {
        return destroyReadStream(stream)
      }

      if (!(stream instanceof Stream)) {
        return stream
      }

      if (typeof stream.destroy === 'function') {
        stream.destroy()
      }

      return stream
    }

    /**
     * Destroy a ReadStream.
     *
     * @param {object} stream
     * @private
     */

    function destroyReadStream(stream) {
      stream.destroy()

      if (typeof stream.close === 'function') {
        // node.js core bug work-around
        stream.on('open', onOpenClose)
      }

      return stream
    }

    /**
     * On open handler to close stream.
     * @private
     */

    function onOpenClose() {
      if (typeof this.fd === 'number') {
        // actually close down the fd
        this.close()
      }
    }

  }, { "fs": 54, "stream": 671 }], 85: [function (require, module, exports) {
    (function (Buffer) {
      var generatePrime = require('./lib/generatePrime')
      var primes = require('./lib/primes.json')

      var DH = require('./lib/dh')

      function getDiffieHellman(mod) {
        var prime = new Buffer(primes[mod].prime, 'hex')
        var gen = new Buffer(primes[mod].gen, 'hex')

        return new DH(prime, gen)
      }

      var ENCODINGS = {
        'binary': true, 'hex': true, 'base64': true
      }

      function createDiffieHellman(prime, enc, generator, genc) {
        if (Buffer.isBuffer(enc) || ENCODINGS[enc] === undefined) {
          return createDiffieHellman(prime, 'binary', enc, generator)
        }

        enc = enc || 'binary'
        genc = genc || 'binary'
        generator = generator || new Buffer([2])

        if (!Buffer.isBuffer(generator)) {
          generator = new Buffer(generator, genc)
        }

        if (typeof prime === 'number') {
          return new DH(generatePrime(prime, generator), generator, true)
        }

        if (!Buffer.isBuffer(prime)) {
          prime = new Buffer(prime, enc)
        }

        return new DH(prime, generator, true)
      }

      exports.DiffieHellmanGroup = exports.createDiffieHellmanGroup = exports.getDiffieHellman = getDiffieHellman
      exports.createDiffieHellman = exports.DiffieHellman = createDiffieHellman

    }).call(this, require("buffer").Buffer)
  }, { "./lib/dh": 86, "./lib/generatePrime": 87, "./lib/primes.json": 88, "buffer": 58 }], 86: [function (require, module, exports) {
    (function (Buffer) {
      var BN = require('bn.js');
      var MillerRabin = require('miller-rabin');
      var millerRabin = new MillerRabin();
      var TWENTYFOUR = new BN(24);
      var ELEVEN = new BN(11);
      var TEN = new BN(10);
      var THREE = new BN(3);
      var SEVEN = new BN(7);
      var primes = require('./generatePrime');
      var randomBytes = require('randombytes');
      module.exports = DH;

      function setPublicKey(pub, enc) {
        enc = enc || 'utf8';
        if (!Buffer.isBuffer(pub)) {
          pub = new Buffer(pub, enc);
        }
        this._pub = new BN(pub);
        return this;
      }

      function setPrivateKey(priv, enc) {
        enc = enc || 'utf8';
        if (!Buffer.isBuffer(priv)) {
          priv = new Buffer(priv, enc);
        }
        this._priv = new BN(priv);
        return this;
      }

      var primeCache = {};
      function checkPrime(prime, generator) {
        var gen = generator.toString('hex');
        var hex = [gen, prime.toString(16)].join('_');
        if (hex in primeCache) {
          return primeCache[hex];
        }
        var error = 0;

        if (prime.isEven() ||
          !primes.simpleSieve ||
          !primes.fermatTest(prime) ||
          !millerRabin.test(prime)) {
          //not a prime so +1
          error += 1;

          if (gen === '02' || gen === '05') {
            // we'd be able to check the generator
            // it would fail so +8
            error += 8;
          } else {
            //we wouldn't be able to test the generator
            // so +4
            error += 4;
          }
          primeCache[hex] = error;
          return error;
        }
        if (!millerRabin.test(prime.shrn(1))) {
          //not a safe prime
          error += 2;
        }
        var rem;
        switch (gen) {
          case '02':
            if (prime.mod(TWENTYFOUR).cmp(ELEVEN)) {
              // unsuidable generator
              error += 8;
            }
            break;
          case '05':
            rem = prime.mod(TEN);
            if (rem.cmp(THREE) && rem.cmp(SEVEN)) {
              // prime mod 10 needs to equal 3 or 7
              error += 8;
            }
            break;
          default:
            error += 4;
        }
        primeCache[hex] = error;
        return error;
      }

      function DH(prime, generator, malleable) {
        this.setGenerator(generator);
        this.__prime = new BN(prime);
        this._prime = BN.mont(this.__prime);
        this._primeLen = prime.length;
        this._pub = undefined;
        this._priv = undefined;
        this._primeCode = undefined;
        if (malleable) {
          this.setPublicKey = setPublicKey;
          this.setPrivateKey = setPrivateKey;
        } else {
          this._primeCode = 8;
        }
      }
      Object.defineProperty(DH.prototype, 'verifyError', {
        enumerable: true,
        get: function () {
          if (typeof this._primeCode !== 'number') {
            this._primeCode = checkPrime(this.__prime, this.__gen);
          }
          return this._primeCode;
        }
      });
      DH.prototype.generateKeys = function () {
        if (!this._priv) {
          this._priv = new BN(randomBytes(this._primeLen));
        }
        this._pub = this._gen.toRed(this._prime).redPow(this._priv).fromRed();
        return this.getPublicKey();
      };

      DH.prototype.computeSecret = function (other) {
        other = new BN(other);
        other = other.toRed(this._prime);
        var secret = other.redPow(this._priv).fromRed();
        var out = new Buffer(secret.toArray());
        var prime = this.getPrime();
        if (out.length < prime.length) {
          var front = new Buffer(prime.length - out.length);
          front.fill(0);
          out = Buffer.concat([front, out]);
        }
        return out;
      };

      DH.prototype.getPublicKey = function getPublicKey(enc) {
        return formatReturnValue(this._pub, enc);
      };

      DH.prototype.getPrivateKey = function getPrivateKey(enc) {
        return formatReturnValue(this._priv, enc);
      };

      DH.prototype.getPrime = function (enc) {
        return formatReturnValue(this.__prime, enc);
      };

      DH.prototype.getGenerator = function (enc) {
        return formatReturnValue(this._gen, enc);
      };

      DH.prototype.setGenerator = function (gen, enc) {
        enc = enc || 'utf8';
        if (!Buffer.isBuffer(gen)) {
          gen = new Buffer(gen, enc);
        }
        this.__gen = gen;
        this._gen = new BN(gen);
        return this;
      };

      function formatReturnValue(bn, enc) {
        var buf = new Buffer(bn.toArray());
        if (!enc) {
          return buf;
        } else {
          return buf.toString(enc);
        }
      }

    }).call(this, require("buffer").Buffer)
  }, { "./generatePrime": 87, "bn.js": 24, "buffer": 58, "miller-rabin": 584, "randombytes": 626 }], 87: [function (require, module, exports) {
    var randomBytes = require('randombytes');
    module.exports = findPrime;
    findPrime.simpleSieve = simpleSieve;
    findPrime.fermatTest = fermatTest;
    var BN = require('bn.js');
    var TWENTYFOUR = new BN(24);
    var MillerRabin = require('miller-rabin');
    var millerRabin = new MillerRabin();
    var ONE = new BN(1);
    var TWO = new BN(2);
    var FIVE = new BN(5);
    var SIXTEEN = new BN(16);
    var EIGHT = new BN(8);
    var TEN = new BN(10);
    var THREE = new BN(3);
    var SEVEN = new BN(7);
    var ELEVEN = new BN(11);
    var FOUR = new BN(4);
    var TWELVE = new BN(12);
    var primes = null;

    function _getPrimes() {
      if (primes !== null)
        return primes;

      var limit = 0x100000;
      var res = [];
      res[0] = 2;
      for (var i = 1, k = 3; k < limit; k += 2) {
        var sqrt = Math.ceil(Math.sqrt(k));
        for (var j = 0; j < i && res[j] <= sqrt; j++)
          if (k % res[j] === 0)
            break;

        if (i !== j && res[j] <= sqrt)
          continue;

        res[i++] = k;
      }
      primes = res;
      return res;
    }

    function simpleSieve(p) {
      var primes = _getPrimes();

      for (var i = 0; i < primes.length; i++)
        if (p.modn(primes[i]) === 0) {
          if (p.cmpn(primes[i]) === 0) {
            return true;
          } else {
            return false;
          }
        }

      return true;
    }

    function fermatTest(p) {
      var red = BN.mont(p);
      return TWO.toRed(red).redPow(p.subn(1)).fromRed().cmpn(1) === 0;
    }

    function findPrime(bits, gen) {
      if (bits < 16) {
        // this is what openssl does
        if (gen === 2 || gen === 5) {
          return new BN([0x8c, 0x7b]);
        } else {
          return new BN([0x8c, 0x27]);
        }
      }
      gen = new BN(gen);

      var num, n2;

      while (true) {
        num = new BN(randomBytes(Math.ceil(bits / 8)));
        while (num.bitLength() > bits) {
          num.ishrn(1);
        }
        if (num.isEven()) {
          num.iadd(ONE);
        }
        if (!num.testn(1)) {
          num.iadd(TWO);
        }
        if (!gen.cmp(TWO)) {
          while (num.mod(TWENTYFOUR).cmp(ELEVEN)) {
            num.iadd(FOUR);
          }
        } else if (!gen.cmp(FIVE)) {
          while (num.mod(TEN).cmp(THREE)) {
            num.iadd(FOUR);
          }
        }
        n2 = num.shrn(1);
        if (simpleSieve(n2) && simpleSieve(num) &&
          fermatTest(n2) && fermatTest(num) &&
          millerRabin.test(n2) && millerRabin.test(num)) {
          return num;
        }
      }

    }

  }, { "bn.js": 24, "miller-rabin": 584, "randombytes": 626 }], 88: [function (require, module, exports) {
    module.exports = {
      "modp1": {
        "gen": "02",
        "prime": "ffffffffffffffffc90fdaa22168c234c4c6628b80dc1cd129024e088a67cc74020bbea63b139b22514a08798e3404ddef9519b3cd3a431b302b0a6df25f14374fe1356d6d51c245e485b576625e7ec6f44c42e9a63a3620ffffffffffffffff"
      },
      "modp2": {
        "gen": "02",
        "prime": "ffffffffffffffffc90fdaa22168c234c4c6628b80dc1cd129024e088a67cc74020bbea63b139b22514a08798e3404ddef9519b3cd3a431b302b0a6df25f14374fe1356d6d51c245e485b576625e7ec6f44c42e9a637ed6b0bff5cb6f406b7edee386bfb5a899fa5ae9f24117c4b1fe649286651ece65381ffffffffffffffff"
      },
      "modp5": {
        "gen": "02",
        "prime": "ffffffffffffffffc90fdaa22168c234c4c6628b80dc1cd129024e088a67cc74020bbea63b139b22514a08798e3404ddef9519b3cd3a431b302b0a6df25f14374fe1356d6d51c245e485b576625e7ec6f44c42e9a637ed6b0bff5cb6f406b7edee386bfb5a899fa5ae9f24117c4b1fe649286651ece45b3dc2007cb8a163bf0598da48361c55d39a69163fa8fd24cf5f83655d23dca3ad961c62f356208552bb9ed529077096966d670c354e4abc9804f1746c08ca237327ffffffffffffffff"
      },
      "modp14": {
        "gen": "02",
        "prime": "ffffffffffffffffc90fdaa22168c234c4c6628b80dc1cd129024e088a67cc74020bbea63b139b22514a08798e3404ddef9519b3cd3a431b302b0a6df25f14374fe1356d6d51c245e485b576625e7ec6f44c42e9a637ed6b0bff5cb6f406b7edee386bfb5a899fa5ae9f24117c4b1fe649286651ece45b3dc2007cb8a163bf0598da48361c55d39a69163fa8fd24cf5f83655d23dca3ad961c62f356208552bb9ed529077096966d670c354e4abc9804f1746c08ca18217c32905e462e36ce3be39e772c180e86039b2783a2ec07a28fb5c55df06f4c52c9de2bcbf6955817183995497cea956ae515d2261898fa051015728e5a8aacaa68ffffffffffffffff"
      },
      "modp15": {
        "gen": "02",
        "prime": "ffffffffffffffffc90fdaa22168c234c4c6628b80dc1cd129024e088a67cc74020bbea63b139b22514a08798e3404ddef9519b3cd3a431b302b0a6df25f14374fe1356d6d51c245e485b576625e7ec6f44c42e9a637ed6b0bff5cb6f406b7edee386bfb5a899fa5ae9f24117c4b1fe649286651ece45b3dc2007cb8a163bf0598da48361c55d39a69163fa8fd24cf5f83655d23dca3ad961c62f356208552bb9ed529077096966d670c354e4abc9804f1746c08ca18217c32905e462e36ce3be39e772c180e86039b2783a2ec07a28fb5c55df06f4c52c9de2bcbf6955817183995497cea956ae515d2261898fa051015728e5a8aaac42dad33170d04507a33a85521abdf1cba64ecfb850458dbef0a8aea71575d060c7db3970f85a6e1e4c7abf5ae8cdb0933d71e8c94e04a25619dcee3d2261ad2ee6bf12ffa06d98a0864d87602733ec86a64521f2b18177b200cbbe117577a615d6c770988c0bad946e208e24fa074e5ab3143db5bfce0fd108e4b82d120a93ad2caffffffffffffffff"
      },
      "modp16": {
        "gen": "02",
        "prime": "ffffffffffffffffc90fdaa22168c234c4c6628b80dc1cd129024e088a67cc74020bbea63b139b22514a08798e3404ddef9519b3cd3a431b302b0a6df25f14374fe1356d6d51c245e485b576625e7ec6f44c42e9a637ed6b0bff5cb6f406b7edee386bfb5a899fa5ae9f24117c4b1fe649286651ece45b3dc2007cb8a163bf0598da48361c55d39a69163fa8fd24cf5f83655d23dca3ad961c62f356208552bb9ed529077096966d670c354e4abc9804f1746c08ca18217c32905e462e36ce3be39e772c180e86039b2783a2ec07a28fb5c55df06f4c52c9de2bcbf6955817183995497cea956ae515d2261898fa051015728e5a8aaac42dad33170d04507a33a85521abdf1cba64ecfb850458dbef0a8aea71575d060c7db3970f85a6e1e4c7abf5ae8cdb0933d71e8c94e04a25619dcee3d2261ad2ee6bf12ffa06d98a0864d87602733ec86a64521f2b18177b200cbbe117577a615d6c770988c0bad946e208e24fa074e5ab3143db5bfce0fd108e4b82d120a92108011a723c12a787e6d788719a10bdba5b2699c327186af4e23c1a946834b6150bda2583e9ca2ad44ce8dbbbc2db04de8ef92e8efc141fbecaa6287c59474e6bc05d99b2964fa090c3a2233ba186515be7ed1f612970cee2d7afb81bdd762170481cd0069127d5b05aa993b4ea988d8fddc186ffb7dc90a6c08f4df435c934063199ffffffffffffffff"
      },
      "modp17": {
        "gen": "02",
        "prime": "ffffffffffffffffc90fdaa22168c234c4c6628b80dc1cd129024e088a67cc74020bbea63b139b22514a08798e3404ddef9519b3cd3a431b302b0a6df25f14374fe1356d6d51c245e485b576625e7ec6f44c42e9a637ed6b0bff5cb6f406b7edee386bfb5a899fa5ae9f24117c4b1fe649286651ece45b3dc2007cb8a163bf0598da48361c55d39a69163fa8fd24cf5f83655d23dca3ad961c62f356208552bb9ed529077096966d670c354e4abc9804f1746c08ca18217c32905e462e36ce3be39e772c180e86039b2783a2ec07a28fb5c55df06f4c52c9de2bcbf6955817183995497cea956ae515d2261898fa051015728e5a8aaac42dad33170d04507a33a85521abdf1cba64ecfb850458dbef0a8aea71575d060c7db3970f85a6e1e4c7abf5ae8cdb0933d71e8c94e04a25619dcee3d2261ad2ee6bf12ffa06d98a0864d87602733ec86a64521f2b18177b200cbbe117577a615d6c770988c0bad946e208e24fa074e5ab3143db5bfce0fd108e4b82d120a92108011a723c12a787e6d788719a10bdba5b2699c327186af4e23c1a946834b6150bda2583e9ca2ad44ce8dbbbc2db04de8ef92e8efc141fbecaa6287c59474e6bc05d99b2964fa090c3a2233ba186515be7ed1f612970cee2d7afb81bdd762170481cd0069127d5b05aa993b4ea988d8fddc186ffb7dc90a6c08f4df435c93402849236c3fab4d27c7026c1d4dcb2602646dec9751e763dba37bdf8ff9406ad9e530ee5db382f413001aeb06a53ed9027d831179727b0865a8918da3edbebcf9b14ed44ce6cbaced4bb1bdb7f1447e6cc254b332051512bd7af426fb8f401378cd2bf5983ca01c64b92ecf032ea15d1721d03f482d7ce6e74fef6d55e702f46980c82b5a84031900b1c9e59e7c97fbec7e8f323a97a7e36cc88be0f1d45b7ff585ac54bd407b22b4154aacc8f6d7ebf48e1d814cc5ed20f8037e0a79715eef29be32806a1d58bb7c5da76f550aa3d8a1fbff0eb19ccb1a313d55cda56c9ec2ef29632387fe8d76e3c0468043e8f663f4860ee12bf2d5b0b7474d6e694f91e6dcc4024ffffffffffffffff"
      },
      "modp18": {
        "gen": "02",
        "prime": "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"
      }
    }
  }, {}], 89: [function (require, module, exports) {
    const moment = require('moment-timezone');

    const utcOffsetRegex = /\+00:00$/;
    const supportedDateTimeFormats = [moment.ISO_8601, moment.RFC_2822];

    /**
     * Parses a date/time string and returns a moment object.
     * The returned moment will be in the original timezone if the string contains an offset.
     * If the string does not contain a timezone, then it is assumed to be UTC, for consistency.
     * @param {string} str date/time string
     */
    function parseDateTime(str) {
      if (typeof str === 'number') {
        return moment.tz(str, 'UTC');
      }
      if (typeof str !== 'string') {
        return moment(NaN);
      }
      return moment.parseZone(str, supportedDateTimeFormats, true);
    }

    /**
     * Formats a MomentJS object. This should always be used when a function returns a date/time
     * formatted string (expect when formatting in a user-defined format).
     * Returns null if the moment object is not valid.
     * @param {object} moment A moment object to format
     * @param {string} format optional format
     */
    function formatMoment(momentObj, format) {
      if (!momentObj.isValid()) {
        return null;
      }
      if (format) {
        return momentObj.format(format);
      }
      if (momentObj.milliseconds()) {
        // Has non-zero milliseconds
        return momentObj.format('YYYY-MM-DDTHH:mm:ss.SSSZ').replace(utcOffsetRegex, 'Z');
      }
      // Format in MomentJS default format (but be explicit about it)
      return momentObj.format('YYYY-MM-DDTHH:mm:ssZ').replace(utcOffsetRegex, 'Z');
    }

    module.exports.formatMoment = formatMoment;
    module.exports.parseDateTime = parseDateTime;

  }, { "moment-timezone": 457 }], 90: [function (require, module, exports) {
    function horner(arr, v) {
      let z = 0;
      // eslint-disable-next-line no-plusplus
      for (let i = 0; i < arr.length; ++i) z = v * z + arr[i]; return z;
    }
    function besselIter(x, n, f0, f1, sign) {
      if (n === 0) return f0;
      if (n === 1) return f1;
      const tdx = 2 / x;
      let f2 = f1;
      let f01 = f0;
      let f11 = f1;
      // eslint-disable-next-line no-plusplus
      for (let o = 1; o < n; ++o) {
        f2 = f11 * o * tdx + sign * f01;
        f01 = f11;
        f11 = f2;
      }
      return f2;
    }
    function besselWrap(bessel0, bessel1, nonzero, sign) {
      return function bessel(x, n) {
        if (nonzero) {
          if (x === 0) return (nonzero === 1 ? -Infinity : Infinity);
          if (x < 0) return NaN;
        }
        if (n === 0) return bessel0(x);
        if (n === 1) return bessel1(x);
        if (n < 0) return NaN;
        // eslint-disable-next-line no-bitwise
        const fn = n | 0;
        const b0 = bessel0(x);
        const b1 = bessel1(x);
        return besselIter(x, fn, b0, b1, sign);
      };
    }
    function besselj(x, n) {
      const W = 0.636619772; // 2 / Math.PI

      const b0a1a = [57568490574.0, -13362590354.0, 651619640.7,
        -11214424.18, 77392.33017, -184.9052456].reverse();
      const b0a2a = [57568490411.0, 1029532985.0, 9494680.718,
        59272.64853, 267.8532712, 1.0].reverse();
      const b0a1b = [1.0, -0.1098628627e-2, 0.2734510407e-4, -0.2073370639e-5,
        0.2093887211e-6].reverse();
      const b0a2b = [-0.1562499995e-1, 0.1430488765e-3, -0.6911147651e-5,
        0.7621095161e-6, -0.934935152e-7].reverse();

      const bessel0 = (z) => {
        let a = 0;
        let a1 = 0;
        let a2 = 0;
        let y = z * z;
        if (z < 8) {
          a1 = horner(b0a1a, y);
          a2 = horner(b0a2a, y);
          a = a1 / a2;
        } else {
          const xx = z - 0.785398164;
          y = 64 / y;
          a1 = horner(b0a1b, y);
          a2 = horner(b0a2b, y);
          a = Math.sqrt(W / z) * (Math.cos(xx) * a1 - Math.sin(xx) * a2 * 8 / z);
        }
        return a;
      };

      const b1a1a = [72362614232.0, -7895059235.0, 242396853.1, -2972611.439,
        15704.48260, -30.16036606].reverse();
      const b1a2a = [144725228442.0, 2300535178.0, 18583304.74, 99447.43394,
        376.9991397, 1.0].reverse();
      const b1a1b = [1.0, 0.183105e-2, -0.3516396496e-4, 0.2457520174e-5,
        -0.240337019e-6].reverse();
      const b1a2b = [0.04687499995, -0.2002690873e-3, 0.8449199096e-5,
        -0.88228987e-6, 0.105787412e-6].reverse();

      const bessel1 = (z) => {
        let a = 0; let a1 = 0; let a2 = 0; let y = z * z; const
          xx = Math.abs(z) - 2.356194491;
        if (Math.abs(z) < 8) {
          a1 = z * horner(b1a1a, y);
          a2 = horner(b1a2a, y);
          a = a1 / a2;
        } else {
          y = 64 / y;
          a1 = horner(b1a1b, y);
          a2 = horner(b1a2b, y);
          a = Math.sqrt(W / Math.abs(z)) * (Math.cos(xx) * a1 - Math.sin(xx) * a2 * 8 / Math.abs(z));
          if (z < 0) a = -a;
        }
        return a;
      };
      const fn = Math.round(n);
      if (!Number.isFinite(x)) return Number.isNaN(x) ? x : 0;
      if (fn < 0) return ((fn % 2) ? -1 : 1) * besselj(x, -fn);
      if (x < 0) return ((fn % 2) ? -1 : 1) * besselj(-x, fn);
      if (fn === 0) return bessel0(x);
      if (fn === 1) return bessel1(x);
      if (x === 0) return 0;

      let ret = 0.0;
      if (x > fn) {
        ret = besselIter(x, fn, bessel0(x), bessel1(x), -1);
      } else {
        const m = 2 * Math.floor((fn + Math.floor(Math.sqrt(40 * fn))) / 2);
        let jsum = false;
        let bjp = 0.0; let
          sum = 0.0;
        let bj = 1.0; let
          bjm = 0.0;
        const tox = 2 / x;
        // eslint-disable-next-line no-plusplus
        for (let j = m; j > 0; j--) {
          bjm = j * tox * bj - bjp;
          bjp = bj;
          bj = bjm;
          if (Math.abs(bj) > 1E10) {
            bj *= 1E-10;
            bjp *= 1E-10;
            ret *= 1E-10;
            sum *= 1E-10;
          }
          if (jsum) sum += bj;
          jsum = !jsum;
          if (j === fn) ret = bjp;
        }
        sum = 2.0 * sum - bj;
        ret /= sum;
      }
      return ret;
    }
    function bessely(x, n) {
      const W = 0.636619772;

      const b0a1a = [-2957821389.0, 7062834065.0, -512359803.6,
        10879881.29, -86327.92757, 228.4622733].reverse();
      const b0a2a = [40076544269.0, 745249964.8, 7189466.438, 47447.26470, 226.1030244, 1.0].reverse();
      const b0a1b = [1.0, -0.1098628627e-2, 0.2734510407e-4, -0.2073370639e-5,
        0.2093887211e-6].reverse();
      const b0a2b = [-0.1562499995e-1, 0.1430488765e-3, -0.6911147651e-5, 0.7621095161e-6,
      -0.934945152e-7].reverse();

      const bessel0 = (z) => {
        let a = 0;
        let a1 = 0;
        let a2 = 0;
        let y = z * z;
        const xx = z - 0.785398164;
        if (z < 8) {
          a1 = horner(b0a1a, y);
          a2 = horner(b0a2a, y);
          a = a1 / a2 + W * besselj(z, 0) * Math.log(z);
        } else {
          y = 64 / y;
          a1 = horner(b0a1b, y);
          a2 = horner(b0a2b, y);
          a = Math.sqrt(W / z) * (Math.sin(xx) * a1 + Math.cos(xx) * a2 * 8 / z);
        }
        return a;
      };

      const b1a1a = [-0.4900604943e13, 0.1275274390e13, -0.5153438139e11, 0.7349264551e9,
      -0.4237922726e7, 0.8511937935e4].reverse();
      const b1a2a = [0.2499580570e14, 0.4244419664e12, 0.3733650367e10, 0.2245904002e8,
        0.1020426050e6, 0.3549632885e3, 1].reverse();
      const b1a1b = [1.0, 0.183105e-2, -0.3516396496e-4, 0.2457520174e-5, -0.240337019e-6].reverse();
      const b1a2b = [0.04687499995, -0.2002690873e-3, 0.8449199096e-5, -0.88228987e-6,
        0.105787412e-6].reverse();

      const bessel1 = (z) => {
        let a = 0;
        let a1 = 0;
        let a2 = 0;
        let y = z * z;
        const xx = z - 2.356194491;
        if (z < 8) {
          a1 = z * horner(b1a1a, y);
          a2 = horner(b1a2a, y);
          a = a1 / a2 + W * (besselj(z, 1) * Math.log(z) - 1 / z);
        } else {
          y = 64 / y;
          a1 = horner(b1a1b, y);
          a2 = horner(b1a2b, y);
          a = Math.sqrt(W / z) * (Math.sin(xx) * a1 + Math.cos(xx) * a2 * 8 / z);
        }
        return a;
      };

      return besselWrap(bessel0, bessel1, 1, -1)(x, n);
    }
    function besseli(x, n) {
      const b0a = [1.0, 3.5156229, 3.0899424, 1.2067492, 0.2659732, 0.360768e-1, 0.45813e-2].reverse();
      const b0b = [0.39894228, 0.1328592e-1, 0.225319e-2, -0.157565e-2, 0.916281e-2, -0.2057706e-1,
        0.2635537e-1, -0.1647633e-1, 0.392377e-2].reverse();

      const bessel0 = (y) => {
        if (y <= 3.75) return horner(b0a, y * y / (3.75 * 3.75));
        return Math.exp(Math.abs(y)) / Math.sqrt(Math.abs(y)) * horner(b0b, 3.75 / Math.abs(y));
      };

      const b1a = [0.5, 0.87890594, 0.51498869, 0.15084934, 0.2658733e-1, 0.301532e-2,
        0.32411e-3].reverse();
      const b1b = [0.39894228, -0.3988024e-1, -0.362018e-2, 0.163801e-2, -0.1031555e-1, 0.2282967e-1,
        -0.2895312e-1, 0.1787654e-1, -0.420059e-2].reverse();

      const bessel1 = (y) => {
        if (y < 3.75) return y * horner(b1a, y * y / (3.75 * 3.75));
        return (y < 0 ? -1 : 1) * Math.exp(Math.abs(y))
          / Math.sqrt(Math.abs(y)) * horner(b1b, 3.75 / Math.abs(y));
      };
      const fn = Math.round(n);
      if (fn === 0) return bessel0(x);
      if (fn === 1) return bessel1(x);
      if (fn < 0) return NaN;
      if (Math.abs(x) === 0) return 0;
      if (x === Infinity) return Infinity;

      let ret = 0.0;
      let j;
      const tox = 2 / Math.abs(x);
      let bip = 0.0; let bi = 1.0;
      let bim = 0.0;
      const m = 2 * Math.round((fn + Math.round(Math.sqrt(40 * fn))) / 2);
      // eslint-disable-next-line no-plusplus
      for (j = m; j > 0; j--) {
        bim = j * tox * bi + bip;
        bip = bi; bi = bim;
        if (Math.abs(bi) > 1E10) {
          bi *= 1E-10;
          bip *= 1E-10;
          ret *= 1E-10;
        }
        if (j === fn) ret = bip;
      }
      ret *= besseli(x, 0) / bi;
      return x < 0 && (fn % 2) ? -ret : ret;
    }

    function besselk(x, n) {
      const b0a = [-0.57721566, 0.42278420, 0.23069756, 0.3488590e-1, 0.262698e-2, 0.10750e-3,
        0.74e-5].reverse();
      const b0b = [1.25331414, -0.7832358e-1, 0.2189568e-1, -0.1062446e-1, 0.587872e-2, -0.251540e-2,
        0.53208e-3].reverse();

      const bessel0 = (y) => {
        if (y <= 2) return -Math.log(y / 2) * besseli(y, 0) + horner(b0a, y * y / 4);
        return Math.exp(-y) / Math.sqrt(y) * horner(b0b, 2 / y);
      };

      const b1a = [1.0, 0.15443144, -0.67278579, -0.18156897, -0.1919402e-1, -0.110404e-2,
        -0.4686e-4].reverse();
      const b1b = [1.25331414, 0.23498619, -0.3655620e-1, 0.1504268e-1, -0.780353e-2, 0.325614e-2,
        -0.68245e-3].reverse();

      const bessel1 = (y) => {
        if (y <= 2) return Math.log(y / 2) * besseli(y, 1) + (1 / y) * horner(b1a, y * y / 4);
        return Math.exp(-y) / Math.sqrt(y) * horner(b1b, 2 / y);
      };

      return besselWrap(bessel0, bessel1, 2, 1)(x, n);
    }

    module.exports.besselk = besselk;
    module.exports.besselj = besselj;
    module.exports.bessely = bessely;
    module.exports.besseli = besseli;

  }, {}], 91: [function (require, module, exports) {

    /* eslint no-plusplus: ["error", { "allowForLoopAfterthoughts": true }] */

    const { parseDateTime } = require('./datetime');

    function FV(rate, periods, amount, current, isOnBeginning) {
      let result;
      if (rate === 0) {
        result = current + amount * periods;
      } else {
        const term = (1 + rate) ** periods;
        result = (isOnBeginning)
          ? current * term + amount * (1 + rate) * (term - 1) / rate
          : current * term + amount * (term - 1) / rate;
      }
      return -result;
    }

    function DDB(cost, salvage, life, period, factor) {
      let result;
      let percent;
      let oldValue;

      percent = factor / life;
      if (percent >= 1) {
        percent = 1;
        if (period === 1) {
          oldValue = cost;
        } else {
          oldValue = 0;
        }
      } else {
        oldValue = cost * ((1 - percent) ** (period - 1));
      }
      const newValue = cost * ((1 - percent) ** period);

      if (newValue < salvage) {
        result = oldValue - salvage;
      } else {
        result = oldValue - newValue;
      }
      if (result < 0) {
        result = 0;
      }
      return result;
    }

    function PMT(rate, periods, present, future, isOnBeginning) {
      let result = null;
      if (rate === 0) {
        result = (present + future) / periods;
      } else {
        const term = (1 + rate) ** periods;
        if (isOnBeginning) {
          result = (future * rate / (term - 1) + present * rate / (1 - 1 / term)) / (1 + rate);
        } else {
          result = future * rate / (term - 1) + present * rate / (1 - 1 / term);
        }
      }
      return -result;
    }

    function IPMT(rate, period, periods, present, future, isOnBeginning) {
      if (periods < 0) {
        return null;
      }

      if (period < 1 || period > periods) {
        return null;
      }

      const payment = PMT(rate, periods, present, future, isOnBeginning);

      let interest;
      if (period === 1) {
        interest = isOnBeginning ? 0 : -present;
      } else {
        interest = isOnBeginning
          ? FV(rate, period - 2, payment, present, true) - payment
          : FV(rate, period - 1, payment, present, false);
      }
      return interest * rate;
    }

    function ISPMT(rate, period, periods, value) {
      return value * rate * (period / periods - 1);
    }

    function MIRR(values, finrate, rinrate) {
      const { length: n } = values;
      let fvpcfs = 0;
      let pvncfs = 0;
      let incomeDetected = false;
      let outcomeDetected = false;

      values.forEach((item, i) => {
        if (item < 0) {
          pvncfs += item / ((1 + finrate) ** i);
          incomeDetected = true;
        } else {
          fvpcfs += item * ((1 + rinrate) ** (n - i - 1));
          outcomeDetected = true;
        }
      });

      if (!incomeDetected || !outcomeDetected) {
        return null;
      }

      const mirr = ((fvpcfs / -pvncfs) ** (1 / (n - 1))) - 1;
      return mirr;
    }

    function NPER(rate, payment, present, future, type) {
      const typeValue = (type) ? 1 : 0;

      const num = payment * (1 + rate * typeValue) - future * rate;
      const den = (present * rate + payment * (1 + rate * typeValue));

      return Math.log(num / den) / Math.log(1 + rate);
    }

    function NOMINAL(rate, periods) {
      const periodsRounded = Math.trunc(periods);
      if (rate <= 0 || periodsRounded < 1) {
        return null;
      }
      return (((rate + 1) ** (1 / periodsRounded)) - 1) * periodsRounded;
    }

    function NPV(rate, cashFlows) {
      return cashFlows.reduce((total, item, i) => total + (item / (((1 + rate) ** (i + 1)))), 0);
    }

    function INVESTMENT_PDURATION(rate, present, future) {
      if (rate <= 0) {
        return null;
      }
      return (Math.log(future) - Math.log(present)) / Math.log(1 + rate);
    }

    function PPMT(rate, period, periods, present, future, type) {
      if (periods < 0) {
        return null;
      }
      if (period < 1 || period > periods) {
        return null;
      }

      return PMT(rate, periods, present, future, type)
        - IPMT(rate, period, periods, present, future, type);
    }

    function PV(rate, periods, payment, future, type) {
      if (rate) {
        return (((1 - ((1 + rate) ** periods)) / rate)
          * payment * (1 + rate * type) - future)
          / ((1 + rate) ** periods);
      }
      return -payment * periods - future;
    }

    function RRI(periods, present, future) {
      if (periods === 0 || present <= 0 || future < 0) {
        return null;
      }
      return ((future / present) ** (1 / periods)) - 1;
    }

    function SLN(cost, salvage, life) {
      if (life === 0) {
        return null;
      }
      return (cost - salvage) / life;
    }

    function SYD(cost, salvage, life, period) {
      if (life === 0) {
        return null;
      }
      if (period < 1 || period > life) {
        return null;
      }

      const periodRounded = Math.trunc(period);
      return ((cost - salvage) * (life - periodRounded + 1) * 2) / (life * (life + 1));
    }

    function VDB(cost, salvage, life, life1, period, factor) {
      let result = 0;
      const fIntEnd = Math.ceil(period);
      const nLoopEnd = fIntEnd;

      let fTerm;
      let fLia = 0;
      let fRestwert = cost - salvage;
      let bNowLia = false;

      for (let i = 1; i <= nLoopEnd; i++) {
        if (!bNowLia) {
          const fDDB = DDB(cost, salvage, life, i, factor);
          fLia = fRestwert / (life1 - (i - 1));

          if (fLia > fDDB) {
            fTerm = fLia;
            bNowLia = true;
          } else {
            fTerm = fDDB;
            fRestwert -= fDDB;
          }
        } else {
          fTerm = fLia;
        }
        if (i === nLoopEnd) {
          fTerm *= (period + 1.0 - fIntEnd);
        }
        result += fTerm;
      }
      return result;
    }


    function XNPV(rate, amounts, dates) {
      if (amounts.length !== dates.length) {
        return null;
      }
      if (!amounts.length) {
        return null;
      }
      let result = 0;
      const baseDate = (typeof dates[0] === 'string') ? parseDateTime(dates[0]) : null;
      if (baseDate == null || !baseDate.isValid()) {
        return null;
      }

      for (let i = 0; i < amounts.length; i++) {
        const amount = amounts[i];
        if (!Number.isFinite(amount)) {
          return null;
        }
        const currentDate = (typeof dates[i] === 'string') ? parseDateTime(dates[i]) : null;
        if (currentDate == null || !currentDate.isValid()) {
          return null;
        }

        const diff = currentDate.startOf('day').diff(baseDate.startOf('day'), 'days', true);

        if (diff < 0) {
          return null;
        }
        result += amounts[i] / ((1 + rate) ** (diff / 365));
      }

      return result;
    }

    module.exports.DDB = DDB;
    module.exports.FV = FV;
    module.exports.IPMT = IPMT;
    module.exports.ISPMT = ISPMT;
    module.exports.MIRR = MIRR;
    module.exports.NOMINAL = NOMINAL;
    module.exports.NPER = NPER;
    module.exports.NPV = NPV;
    module.exports.INVESTMENT_PDURATION = INVESTMENT_PDURATION;
    module.exports.PMT = PMT;
    module.exports.PPMT = PPMT;
    module.exports.PV = PV;
    module.exports.RRI = RRI;
    module.exports.SLN = SLN;
    module.exports.SYD = SYD;
    module.exports.VDB = VDB;
    module.exports.XNPV = XNPV;

  }, { "./datetime": 89 }], 92: [function (require, module, exports) {
    /* eslint quote-props: ["error", "consistent-as-needed"] */
    const chroma = require('chroma-js');

    // Re-used regexp objects
    const REGEX_INTEGER = /^-?\d+$/;
    const REGEX_NUMBER = /^-?\d+(\.\d+)?$/;
    const REGEX_DECIMAL = /^-?\d+(\.\d+)?%$/;
    const REGEX_DATE = /^(\d\d\d\d)-(\d\d)-(\d\d)$/;
    const REGEX_DATE_TIME = /^\d\d\d\d-[0-1]\d-[0-3]\d[t\s](?:[0-2]\d:[0-5]\d:[0-5]\d|23:59:60)(?:\.\d+)?(?:z|[+-]\d\d:\d\d)$/i;
    // (sources from jsen validator):
    // http://stackoverflow.com/questions/201323/using-a-regular-expression-to-validate-an-email-address#answer-8829363
    // http://www.w3.org/TR/html5/forms.html#valid-e-mail-address (search for 'willful violation')
    const REGEX_EMAIL = /^[a-z0-9.!#$%&'*+/=?^_`{|}~-]+@[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?(?:\.[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?)*$/i;
    const REGEX_URI = /^(?:[a-z][a-z0-9+\-.]*:)(?:\/?\/(?:(?:[a-z0-9\-._~!$&'()*+,;=:]|%[0-9a-f]{2})*@)?(?:\[(?:(?:(?:(?:[0-9a-f]{1,4}:){6}|::(?:[0-9a-f]{1,4}:){5}|(?:[0-9a-f]{1,4})?::(?:[0-9a-f]{1,4}:){4}|(?:(?:[0-9a-f]{1,4}:){0,1}[0-9a-f]{1,4})?::(?:[0-9a-f]{1,4}:){3}|(?:(?:[0-9a-f]{1,4}:){0,2}[0-9a-f]{1,4})?::(?:[0-9a-f]{1,4}:){2}|(?:(?:[0-9a-f]{1,4}:){0,3}[0-9a-f]{1,4})?::[0-9a-f]{1,4}:|(?:(?:[0-9a-f]{1,4}:){0,4}[0-9a-f]{1,4})?::)(?:[0-9a-f]{1,4}:[0-9a-f]{1,4}|(?:(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\.){3}(?:25[0-5]|2[0-4]\d|[01]?\d\d?))|(?:(?:[0-9a-f]{1,4}:){0,5}[0-9a-f]{1,4})?::[0-9a-f]{1,4}|(?:(?:[0-9a-f]{1,4}:){0,6}[0-9a-f]{1,4})?::)|[Vv][0-9a-f]+\.[a-z0-9\-._~!$&'()*+,;=:]+)\]|(?:(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\.){3}(?:25[0-5]|2[0-4]\d|[01]?\d\d?)|(?:[a-z0-9\-._~!$&'()*+,;=]|%[0-9a-f]{2})*)(?::\d*)?(?:\/(?:[a-z0-9\-._~!$&'()*+,;=:@]|%[0-9a-f]{2})*)*|\/(?:(?:[a-z0-9\-._~!$&'()*+,;=:@]|%[0-9a-f]{2})+(?:\/(?:[a-z0-9\-._~!$&'()*+,;=:@]|%[0-9a-f]{2})*)*)?|(?:[a-z0-9\-._~!$&'()*+,;=:@]|%[0-9a-f]{2})+(?:\/(?:[a-z0-9\-._~!$&'()*+,;=:@]|%[0-9a-f]{2})*)*)(?:\?(?:[a-z0-9\-._~!$&'()*+,;=:@/?]|%[0-9a-f]{2})*)?(?:#(?:[a-z0-9\-._~!$&'()*+,;=:@/?]|%[0-9a-f]{2})*)?$/i;
    const REGEX_URL = /^[a-z][a-z0-9+.-]*:(?:\/\/(?:[a-z0-9._~:/?#[\]!$&'()*+,;%=-]+@)?[a-z0-9._~/?#[\]@!$&'()*+,;%=-]+(?::\d+)?)?[a-z0-9._~:/#[\]@!$&'()*+,;%=-]*(?:\?[a-z0-9._~:/?[\]@!$&'()*+,;%=-]*)?(?:#[a-z0-9._~:/?#[\]@!$&'()*+,;%=-]*)?$/i;
    const REGEX_WEB_URL = /^(?:(?:https?):\/\/)(?:\S+(?::\S*)?@)?(?:(?!10(?:\.\d{1,3}){3})(?!127(?:\.\d{1,3}){3})(?!169\.254(?:\.\d{1,3}){2})(?!192\.168(?:\.\d{1,3}){2})(?!172\.(?:1[6-9]|2\d|3[0-1])(?:\.\d{1,3}){2})(?:[1-9]\d?|1\d\d|2[01]\d|22[0-3])(?:\.(?:1?\d{1,2}|2[0-4]\d|25[0-5])){2}(?:\.(?:[1-9]\d?|1\d\d|2[0-4]\d|25[0-4]))|(?:(?:[a-z\u{00a1}-\u{ffff}0-9]+-?)*[a-z\u{00a1}-\u{ffff}0-9]+)(?:\.(?:[a-z\u{00a1}-\u{ffff}0-9]+-?)*[a-z\u{00a1}-\u{ffff}0-9]+)*(?:\.(?:[a-z\u{00a1}-\u{ffff}]{2,})))(?::\d{2,5})?(?:\/[^\s]*)?$/iu;
    const REGEX_UUID = /^(?:urn:uuid:)?[0-9a-f]{8}-(?:[0-9a-f]{4}-){3}[0-9a-f]{12}$/i;
    const REGEX_COLOR = /^(?:rgba?\(\d+(?:\.\d+)?%?(?:,\s*\d+(?:\.\d+)?%?){2}(?:,\s*\d+(?:\.\d+)?)?\)|hsla?\(\d+(?:\.\d+)?(?:,\s*\d+(?:\.\d+)?%){2}(?:,\s*\d+(?:\.\d+)?)?\)|#[0-9a-f]{3}(?:[0-9a-f]{3})?)$/i;

    /**
     * Object whose keys defines all the formats understood by the expression system.
     * All the formats defined here are applied e.g. constant values automatically.
     * Other format keywords may be used as well but the expression system does not
     * understand their structure.
     *
     * Some formats are based on the implementations here:
     * https://github.com/epoberezkin/ajv/blob/d5edde43752e8c1bdb26074402452a41fe0742cb/lib/compile/formats.js
     */
    const formats = {
      'blank': {
        title: 'Blank string',
        description: 'A string that has no characters.',
        examples: [''],
        matches: value => !value,
      },
      'integer': {
        title: 'Integer',
        description: 'A string that only consists of digits (at least one), optionally starting with a minus sign.',
        examples: ['3', '-5', '123'],
        matches: value => REGEX_INTEGER.test(value),
      },
      'number': {
        title: 'Number',
        description: 'A string that only consists of digits, and optionally a decimal dot and a minus sign.',
        examples: ['3', '4.5', '-0.013'],
        matches: value => REGEX_NUMBER.test(value),
      },
      'percentage': {
        title: 'Percentage',
        description: 'A string representing a number and a percentage character. The value may be negative or contain a decimal dot.',
        examples: ['3%', '4.5%', '-0.013%'],
        matches: value => REGEX_DECIMAL.test(value),
      },
      'date': {
        title: 'Date',
        description: 'A string representing a valid ISO-formatted date.',
        examples: ['2019-07-10'],
        matches: value => REGEX_DATE.test(value),
      },
      'date-time': {
        title: 'Date/time',
        description: 'A string representing a valid ISO-formatted date/time.',
        examples: ['2019-07-10T09:56:12.123Z'],
        matches: value => REGEX_DATE_TIME.test(value),
      },
      'email': {
        title: 'Email',
        description: 'A string representing a valid email address.',
        examples: ['john.smith@example.com'],
        matches: value => REGEX_EMAIL.test(value),
      },
      'uri': {
        title: 'URI',
        description: 'A string representing a valid Uniform Resource Identifier (URI)',
        examples: [
          'https://john.doe@www.example.com:123/forum/questions/?tag=networking&order=newest#top',
          'ldap://[2001:db8::7]/c=GB?objectClass?one',
          'tel:+1-816-555-1212',
        ],
        matches: value => REGEX_URI.test(value),
      },
      'url': {
        title: 'URL (any protocol)',
        examples: [
          'http://www.example.com/index.html',
          'ftp://ftp.example.com/files/data.zip',
          'mailto://john.smith@example.com',
        ],
        description: 'A string representing a valid Uniform Resource Locator (URL)',
        matches: value => REGEX_URL.test(value) || REGEX_WEB_URL.test(value),
      },
      'web-url': {
        title: 'Web URL (http or https)',
        examples: [
          'http://www.example.com/index.html',
        ],
        description: 'A string representing a valid web address, as an Uniform Resource Locator (URL), that has either `http` or `https` procotol.',
        matches: value => REGEX_WEB_URL.test(value),
      },
      'uuid': {
        title: 'UUID',
        examples: [
          'ec20edcb-ab7f-41f4-99fd-6604bab3502b',
        ],
        description: 'A string representing a valid universally unique identifier (UUID)',
        matches: value => REGEX_UUID.test(value),
      },
      'color': {
        title: 'Color',
        description: 'A string representing a valid color code',
        examples: [
          'rgb(200, 140, 120)',
          'rgba(10, 20, 10, 0.3)',
          'hsl(180, 100%, 25%)',
          'hsla(180, 100%, 10%, 0.9)',
          '#ff3399',
          '#FF3399',
          '#aa4',
          'black',
          'hotpink',
        ],
        matches: value => REGEX_COLOR.test(value) || chroma.colors[value] != null,
      },
    };

    module.exports.formats = formats;

  }, { "chroma-js": 60 }], 93: [function (require, module, exports) {
    const { resolveObjectMemberAccessor, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'BUILD_OBJECT',
      category: 'Array',
      title: 'Build an object from an array of key-value objects',
      description: `Given an array of objects and the names of two properties that the objects in the array have, constructs a dictionary object.

The algorithm runs as follows: start with an empty result object. For every object in the source array, pick the value of the **Key value** property, and assign it to the result object under a property whose name is the value of **Key name** property.

If multiple objects in the array have the same value for their **Key name** property, the dictionary will have the value of the **Value name** property of the last object in the array.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'The array of objects from which the object will be built',
        items: {
          type: 'object',
        },
      }, {
        type: 'string',
        title: 'Key name',
        description: 'Property name in the source array object whose value to use as dictionary object key',
      }, {
        type: 'string',
        title: 'Value name',
        description: 'Property name in the source array object whose value to use as dictionary object value',
      }],
      analyze: ([arrayParam, keyParam, valueParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const valueSchema = (valueParam && valueParam.schema) || {};
        const {
          schema: valuePropSchema,
          errors: valueErrors,
        } = resolveObjectMemberAccessor(itemsSchema, valueSchema);
        const {
          schema: keyPropSchema,
          errors: keyErrors,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        const errors = [];
        if ((keyErrors.length || valueErrors.length) && !parameterErrors.length) {
          errors.push(...keyErrors.map(error => ({ ...error, offset: keyParam.offset })));
          errors.push(...valueErrors.map(error => ({ ...error, offset: valueParam.offset })));
        }
        if (keyPropSchema && valuePropSchema) {
          if (keyPropSchema.enum) {
            return {
              schema: {
                type: 'object',
                properties: {
                  ...keyPropSchema.enum.reduce((acum, item) => {
                    const newItem = {};
                    newItem[item] = valuePropSchema;
                    return Object.assign(acum, newItem);
                  }, {}),
                },
              },
              errors,
            };
          }
        }
        return {
          schema: {
            type: 'object',
            properties: {},
            additionalProperties: errors.length || parameterErrors.length ? {} : valuePropSchema,
          },
          errors,
        };
      },
      execute: ([array, keyName, valueName]) => {
        if (Array.isArray(array) && typeof keyName === 'string' && typeof valueName === 'string') {
          if (array.some(item => !(typeof item === 'object')) || !array.length) {
            return null;
          }
          const validKeyArr = array.filter(item => (Object.keys(item).includes(keyName)));
          const validArr = validKeyArr.filter(item => (Object.keys(item).includes(valueName)));
          if (validArr.length) {
            const newArray = validArr.map((item) => {
              const newItem = {};
              const key = item[keyName];
              const value = item[valueName];
              newItem[key] = value;
              return newItem;
            });
            return newArray.reduce((sum, current) => Object.assign(sum, current));
          }
          return {};
        }
        return null;
      },
      examples: [
        {
          context: {
            array: [
              {
                id: 'abc',
                label: 'ABC label',
              },
              {
                id: 'def',
                label: 'DEF label',
              },
            ],
          },
          expression: 'BUILD_OBJECT(array, "id", "label")',
          result: { abc: 'ABC label', def: 'DEF label' },
        },
        {
          context: {
            animals: [
              {
                name: 'cat',
                age: 2,
                height: 0.4,
              },
              {
                name: 'dog',
                age: 3,
                height: 0.6,
              },
              {
                name: 'mouse',
                age: 1,
                height: 0.1,
              },
            ],
          },
          expression: 'BUILD_OBJECT(animals, "name", "age")',
          result: { cat: 2, dog: 3, mouse: 1 },
        },
        {
          context: {
            animals: [
              {
                name: 'cat',
                age: 2,
                height: 0.4,
              },
              {
                name: 'dog',
                age: 3,
                height: 0.6,
              },
              {
                name: 'mouse',
                age: 1,
                height: 0.1,
              },
            ],
          },
          expression: 'BUILD_OBJECT(animals, "name", "height")',
          result: { cat: 0.4, dog: 0.6, mouse: 0.1 },
        },
      ],
    };

  }, { "../../schema": 460 }], 94: [function (require, module, exports) {
    const { getUnionSchema } = require('../../schema');

    module.exports = {
      name: 'CONCAT',
      category: 'Array',
      title: 'Concatenate two arrays',
      description: `Concatenates two arrays, returning an array that has all items from the first array followed by all the items from the second array.

The order of the elements in the source arrays will not be changed, with the elements in the second array placed after the elements in the first array.`,
      parameters: [{
        type: 'array',
        title: 'Array 1',
        description: 'First array',
      }, {
        type: 'array',
        title: 'Array 1',
        description: 'Second array',
      }],
      analyze: (parameters) => {
        const results = parameters.slice(0, 2);
        const resultSchemas = results.map(({ schema }) => schema);
        const schema = getUnionSchema(resultSchemas) || { type: 'array' };
        return { schema };
      },
      execute: ([array1, array2]) => {
        if (Array.isArray(array1) && Array.isArray(array2)) {
          return array1.concat(array2);
        }
        return null;
      },
      examples: [
        { expression: 'CONCAT([100, 3, 20], [1, 2, 3])', result: [100, 3, 20, 1, 2, 3] },
        { expression: 'CONCAT([100, 3, 20], ["a", "b"])', result: [100, 3, 20, 'a', 'b'] },
        { expression: 'CONCAT([], [])', result: [] },
      ],
    };

  }, { "../../schema": 460 }], 95: [function (require, module, exports) {
    module.exports = {
      name: 'COUNT',
      category: 'Array',
      title: 'Size of an array',
      description: `Returns the number of items in the given array.

For measuring the length for a string, use \`LENGTH\` instead.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array whose items to count',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([value]) => (value == null ? value : value.length),
      examples: [
        { expression: 'COUNT([1, 2, 3])', result: 3 },
        { expression: 'COUNT([])', result: 0 },
      ],
    };

  }, {}], 96: [function (require, module, exports) {
    const isEqual = require('lodash/isEqual');
    const { resolveObjectMemberAccessor, getSchemaAssignmentErrors, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'EXCLUDE_BY_KEY',
      category: 'Array',
      title: 'Remove objects from an array by a key',
      description: `Removes objects from an array by a given key-value pair.

Returns a new array which doesn't contain the matched objects, i.e. ones whose **Key** property's value matches **Value**.

For example, given **Key** \`"id"\` and **Value** \`0\`, all objects in the source **Array** whose \`id\` property is \`0\` would be removed from the returned array.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to exclude objects from',
        items: {
          type: 'object',
        },
      }, {
        title: 'Key',
        description: 'Key whose value to match',
        type: 'string',
      }, {
        title: 'Value',
        description: 'Value to match',
      }],
      analyze: ([arrayParam, keyParam, itemParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyErrors,
          schema: keyItemSchema,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        const itemSchema = (itemParam && itemParam.schema) || {};
        const assignmentErrors = getSchemaAssignmentErrors(itemSchema, keyItemSchema);

        return {
          schema: arraySchema,
          errors: parameterErrors.length ? [] : [
            ...assignmentErrors.map(error => ({ message: error.message, offset: itemParam.offset })),
            ...keyErrors.map(error => ({ ...error, offset: keyParam.offset })),
          ],
        };
      },
      execute: ([array, key, item]) => {
        if (!Array.isArray(array) || key == null) {
          return null;
        }
        return array.filter(el => !isEqual(el[key], item));
      },
      examples: [
        {
          context: {
            array1: [{ a: 3 }, { a: 2 }, { a: 4, b: 3 }, { a: 2, b: 5 }],
            key1: 'a',
            value1: 2,
          },
          expression: 'EXCLUDE_BY_KEY(array1, key1, value1)',
          result: [{ a: 3 }, { a: 4, b: 3 }],
        },
        {
          context: {
            array2: [{ a: 3 }, { a: { b: 10 } }],
            key2: 'a',
            value2: { b: 10 },
          },
          expression: 'EXCLUDE_BY_KEY(array2, key2, value2)',
          result: [{ a: 3 }],
        },
      ],
    };

  }, { "../../schema": 460, "lodash/isEqual": 579 }], 97: [function (require, module, exports) {
    const { resolveObjectMemberAccessor, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'EXCLUDE_BY_REGEX',
      category: 'Array',
      title: 'Remove objects from an array by a key-regex match',
      description: `Removes objects from an array by matching the value of a given key to a regular expression pattern.

Iterates through the array, checking if the value of each object's **Key** property matches the regex **Pattern**. Returns a new array which doesn't contain matched objects.

For example, given **Key** \`"name"\` and **Value** \`/John/\`, all objects in the source **Array** whose \`name\` property contains the string \`"John"\` would be removed from the returned array.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to exclude objects from',
        items: {
          type: 'object',
        },
      }, {
        title: 'Key',
        description: 'Key whose value to match',
        type: 'string',
      }, {
        title: 'Pattern',
        description: 'Regex pattern to match',
        type: 'string',
      }],
      analyze: ([arrayParam, keyParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyErrors,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        return {
          schema: arraySchema,
          errors: parameterErrors.length
            ? []
            : keyErrors.map(error => ({ ...error, offset: keyParam.offset })),
        };
      },
      execute: ([array, key, pattern]) => {
        if (!Array.isArray(array) || key == null || pattern == null) {
          return null;
        }
        if (!(typeof pattern === 'string' || pattern instanceof RegExp)) {
          return null;
        }
        const regex = pattern instanceof RegExp ? pattern : new RegExp(pattern);
        return array.filter(el => !regex.test(el[key]));
      },
      examples: [
        {
          expression: 'EXCLUDE_BY_REGEX(array, key, stringPattern)',
          context: {
            array: [{ a: 2 }, { a: 'foo bar' }, { a: 'foo', b: 3 }],
            key: 'a',
            stringPattern: '^foo',
          },
          result: [{ a: 2 }],
        },
      ],
    };

  }, { "../../schema": 460 }], 98: [function (require, module, exports) {
    const { getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'FIND',
      category: 'Array',
      title: 'Finds an item from array',
      description: `Finds the first item from the given array that matches the given criteria. Returns that item, or \`null\` if not found.

The criteria is defined as a formula at the second parameter. The formula is evaluated with each item in the array.
The formula must result in a boolean value. If \`true\`, then item will be included to the result, otherwise it will be excluded.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array of items to search',
        items: {
          $id: '#item', // Make this a generic type
        },
      }, {
        type: 'function',
        title: 'Selection formula',
        description: 'Formula used to determine if an item is the one to find from the array',
        parameters: [{
          name: 'item',
          title: 'Current item',
          description: 'The current item from the original array',
          $ref: '#item', // Refers to the generic type
        }, {
          name: 'index',
          type: 'number',
          minimum: 0,
          title: 'Current index',
          description: 'The position of the current item in the array as an index number starting from 0',
          examples: [0, 1, 2],
        }],
        returnValue: {
          type: 'boolean',
          title: 'Whether the item matches',
          description: 'Whether or not the current item is the one to be found from the array',
        },
      }],
      analyze: ([arrayParam]) => ({
        schema: getArrayItemSchema(arrayParam && arrayParam.schema) || {},
      }),
      execute: ([array, func]) => {
        if (!Array.isArray(array) || typeof func !== 'function') {
          return null;
        }
        return array.find((item, index) => func(item, index));
      },
      examples: [
        {
          expression: 'FIND(animals, item.name == "cat")',
          context: {
            animals: [
              { name: 'dog', weight: 123 },
              { name: 'cat', weight: 48 },
              { name: 'rat', weight: 10 },
              { name: 'pig', weight: 230 },
            ],
          },
          result: { name: 'cat', weight: 48 },
        },
        {
          expression: 'FIND(animals, item.name == "horse")',
          context: {
            animals: [
              { name: 'dog', weight: 123 },
              { name: 'cat', weight: 48 },
              { name: 'rat', weight: 10 },
              { name: 'pig', weight: 230 },
            ],
          },
          result: undefined,
        },
        {
          expression: 'FIND(animals, index == 2)',
          context: {
            animals: [
              { name: 'dog', weight: 123 },
              { name: 'cat', weight: 48 },
              { name: 'rat', weight: 10 },
              { name: 'pig', weight: 230 },
            ],
          },
          result: { name: 'rat', weight: 10 },
        },
        {
          expression: 'FIND<product>(products, product.name == "Duct tape")',
          context: {
            products: [
              { name: 'Duct tape', category: 'Utility' },
              { name: 'Swiss army knife', category: 'Tools' },
              { name: 'Glue', category: 'Utility' },
            ],
          },
          result: { name: 'Duct tape', category: 'Utility' },
        },
        {
          expression: 'FIND<product, position>(products, position == 1)',
          context: {
            products: [
              { name: 'Duct tape', category: 'Utility' },
              { name: 'Swiss army knife', category: 'Tools' },
              { name: 'Glue', category: 'Utility' },
            ],
          },
          result: { name: 'Swiss army knife', category: 'Tools' },
        },
      ],
    };

  }, { "../../schema": 460 }], 99: [function (require, module, exports) {
    const isEqual = require('lodash/isEqual');
    const { resolveObjectMemberAccessor, getSchemaAssignmentErrors, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'FIND_BY_KEY',
      category: 'Array',
      title: 'Find an object in array by a key',
      description: `Finds an object in the given array by a key-value pair.

For example, given **Key** \`"id"\` and **Value** \`123\`, will return the object in the array whose \`id\` property is \`123\`, or \`null\` if such an object was not found.

If multiple matches exist, the first matching object is returned.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to find object in',
        items: {
          type: 'object',
        },
      }, {
        title: 'Key',
        description: 'Key whose value to match',
        type: 'string',
      }, {
        title: 'Value',
        description: 'Value to match',
      }],
      analyze: ([arrayParam, keyParam, itemParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || {};
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyErrors,
          schema: keyItemSchema,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        const itemSchema = (itemParam && itemParam.schema) || {};
        const assignmentErrors = getSchemaAssignmentErrors(itemSchema, keyItemSchema);
        return {
          schema: itemsSchema,
          errors: parameterErrors.length ? [] : [
            ...keyErrors.map(error => ({ ...error, offset: keyParam.offset })),
            ...assignmentErrors.map(error => ({ ...error, offset: itemParam.offset })),
          ],
        };
      },
      execute: ([array, key, item]) => {
        if (!Array.isArray(array) || key == null || typeof item === 'undefined') {
          return null;
        }
        return array.find(el => isEqual(el[key], item)) || null;
      },
      examples: [
        {
          context: {
            array1: [{ a: 3 }, { a: 2 }],
            key: 'a',
            value: 2,
          },
          expression: 'FIND_BY_KEY(array1, key, value)',
          result: { a: 2 },
        },
        {
          context: {
            array: [{ a: 3 }, { a: { b: 10 } }],
            key: 'a',
            objValue: { b: 10 },
          },
          expression: 'FIND_BY_KEY(array, key, objValue)',
          result: { a: { b: 10 } },
        },
      ],
    };

  }, { "../../schema": 460, "lodash/isEqual": 579 }], 100: [function (require, module, exports) {

    const { resolveObjectMemberAccessor, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'FIND_BY_REGEX',
      category: 'Array',
      title: 'Find an object in array by matching a key value with a regex pattern',
      description: `Finds an object in the given array by comparing the value of each object's **Key** property with the given regex.

For example, given **Key** \`"name"\` and **Value** \`Agnus\`, will return the object in the array whose \`name\` property contains \`Agnus\`, or \`null\` if such an object was not found.

If multiple matches exist, the first matching object is returned.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to find object in',
        items: {
          type: 'object',
        },
      }, {
        title: 'Key',
        description: 'Key whose value to match',
        type: 'string',
      }, {
        title: 'Pattern',
        description: 'Regex pattern to match',
        type: 'string',
      }],
      analyze: ([arrayParam, keyParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || {};
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyErrors,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        return {
          schema: itemsSchema,
          errors: parameterErrors.length ? [] : keyErrors.map(error => ({
            ...error, offset: keyParam.offset,
          })),
        };
      },
      execute: ([array, key, pattern]) => {
        if (!Array.isArray(array) || key == null || pattern == null) {
          return null;
        }
        if (!(typeof pattern === 'string' || pattern instanceof RegExp)) {
          return null;
        }
        const regex = pattern instanceof RegExp ? pattern : new RegExp(pattern);
        return array.find(el => regex.test(el[key])) || null;
      },
      examples: [
        {
          expression: 'FIND_BY_REGEX(array, "a", "^foo")',
          context: {
            array: [{ a: 'abar' }, { a: 'bar' }, { a: 'foo bar' }, { a: 1 }],
          },
          result: { a: 'foo bar' },
        },
        // {
        //   expression: 'FIND_BY_REGEX(array, "a", "1")',
        //   context: {
        //     array: [{ a: 'abar' }, { a: 'bar' }, { a: 'foo bar' }, { a: 1 }],
        //   },
        //   result: { a: 1 },
        // },
      ],
    };

  }, { "../../schema": 460 }], 101: [function (require, module, exports) {
    module.exports = {
      name: 'GENERATE_RANGE',
      category: 'Array',
      title: 'Constructs an array with generated numbers',
      description: 'Constructs an array with generated numbers. Start - first number for generation, end - limit range for numbers generation, step - step increment(decrement).',
      parameters: [{
        type: 'number',
        title: 'Start',
        description: 'Start number',
      }, {
        type: 'number',
        title: 'End',
        description: 'End number',
      }, {
        type: 'number',
        title: 'Step',
        description: 'Step number',
        default: 1,
      }],
      analyze: () => ({ schema: { type: 'array', items: { type: 'number' } } }),
      execute: ([start, end, step]) => {
        if (step === null || step === 0) {
          return null;
        }
        const saveStep = step || 1;
        if (typeof start === 'number' && Number.isFinite(start)
          && typeof end === 'number' && Number.isFinite(end)
          && typeof saveStep === 'number' && Number.isFinite(saveStep)) {
          if ((start > end && saveStep > 0) || (end > start && saveStep < 0)) {
            return null;
          }
          const result = [];
          const isIncrement = start <= end;
          let i = start;
          if (isIncrement) {
            for (; i <= end; i += saveStep) {
              result.push(i);
            }
          } else {
            for (; i >= end; i += saveStep) {
              result.push(i);
            }
          }
          return result;
        }
        return null;
      },
      examples: [
        { expression: 'GENERATE_RANGE(0,5,1)', result: [0, 1, 2, 3, 4, 5] },
        { expression: 'GENERATE_RANGE(5,0,-1)', result: [5, 4, 3, 2, 1, 0] },
        { expression: 'GENERATE_RANGE(1,12,3)', result: [1, 4, 7, 10] },
      ],
    };

  }, {}], 102: [function (require, module, exports) {
    module.exports = {
      name: 'GROUP',
      category: 'Array',
      title: 'Groups items in an array',
      description: `Groups items in the given array using two formulas: one that defines the grouping criteria for each item, and one that creates the group items returned by this function.

First the grouping criteria formula is evaluated for each \`item\` in the array. All the items that evaluate to the equal _group key_ are grouped together.

Then the latter formula is evaluated for each _group_. You can use \`key\` in the formula to get the grouping key of the current group, and \`items\` to access all the items belonging to the current group.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array of items to group',
        items: {
          $id: '#item', // Make this a generic type
        },
      }, {
        type: 'function',
        title: 'Group criteria',
        description: 'Formula that returns a group key for the current item',
        parameters: [{
          name: 'item',
          title: 'Current item',
          description: 'The current item from the original array',
          $ref: '#item', // Refers to the generic type
        }, {
          name: 'index',
          type: 'number',
          minimum: 0,
          title: 'Current index',
          description: 'The position of the current item in the array as an index number starting from 0',
          examples: [0, 1, 2],
        }],
        returnValue: {
          $id: '#key', // Make this a generic type
          title: 'Group key',
          description: 'Group key that determines to which group this item belongs',
          anyOf: [{
            type: 'string',
          }, {
            type: 'number',
          }, {
            type: 'boolean',
          }],
        },
      }, {
        type: 'function',
        title: 'Grouping formula',
        description: 'Formula that returns the final value for each "group" in the result array',
        parameters: [{
          name: 'key',
          title: 'Group key',
          description: 'The key for the current group',
          $ref: '#key', // Refers to the generic type
        }, {
          name: 'items',
          title: 'Grouped items',
          description: 'Array of items from the original array that belong to the current group',
          type: 'array',
          items: {
            $ref: '#item',
          },
        }],
        returnValue: {
          title: 'Group',
          description: 'The value for this "group" to be added to the final array',
        },
      }],
      analyze: ([, , grouperItem]) => {
        const groupSchema = grouperItem && grouperItem.schema;
        return {
          schema: !groupSchema ? { type: 'array' } : {
            type: 'array',
            items: groupSchema,
          },
        };
      },
      execute: ([array, groupKeyFunc, groupMapFunc]) => {
        if (!Array.isArray(array) || typeof groupKeyFunc !== 'function' || typeof groupMapFunc !== 'function') {
          return null;
        }
        const groupMapping = new Map();
        array.forEach((item, index) => {
          const key = groupKeyFunc(item, index);
          const items = groupMapping.get(key) || [];
          items.push(item);
          groupMapping.set(key, items);
        });
        const results = [];
        groupMapping.forEach((items, key) => {
          const group = groupMapFunc(key, items);
          results.push(group);
        });
        return results;
      },
      examples: [
        {
          expression: 'GROUP(allProducts, item.category, { category: key, products: items })',
          context: {
            allProducts: [
              { name: 'Duct tape', category: 'Utility' },
              { name: 'Swiss army knife', category: 'Tools' },
              { name: 'Glue', category: 'Utility' },
            ],
          },
          result: [{
            category: 'Utility',
            products: [
              { name: 'Duct tape', category: 'Utility' },
              { name: 'Glue', category: 'Utility' },
            ],
          }, {
            category: 'Tools',
            products: [
              { name: 'Swiss army knife', category: 'Tools' },
            ],
          }],
        },
        {
          expression: 'GROUP(allProducts, item.category, key + " category contains " + COUNT(items) + " product(s)")',
          context: {
            allProducts: [
              { name: 'Duct tape', category: 'Utility' },
              { name: 'Swiss army knife', category: 'Tools' },
              { name: 'Glue', category: 'Utility' },
            ],
          },
          result: [
            'Utility category contains 2 product(s)',
            'Tools category contains 1 product(s)',
          ],
        },
        {
          expression: 'GROUP<product, index, category, products>(allProducts, product.category, { category: category, count: COUNT(products) })',
          context: {
            allProducts: [
              { name: 'Duct tape', category: 'Utility' },
              { name: 'Swiss army knife', category: 'Tools' },
              { name: 'Glue', category: 'Utility' },
            ],
          },
          result: [{
            category: 'Utility',
            count: 2,
          }, {
            category: 'Tools',
            count: 1,
          }],
        },
      ],
    };

  }, {}], 103: [function (require, module, exports) {
    const isEqual = require('lodash/isEqual');

    module.exports = {
      name: 'INDEX_OF',
      category: 'Array',
      title: 'Get index of item in array',
      description: `Returns the first index at which a given item can be found in the source array.

If the given item cannot be found, \`null\` is returned instead.`,
      parameters: [{
        type: 'array',
        title: 'Values',
        description: 'Array',
        items: {
          $id: '#item', // Make this a generic type
        },
      }, {
        $ref: '#item', // Refer to the generic type
        title: 'Item',
        description: 'Item to locate in the array',
      }],
      analyze: () => ({
        schema: { type: 'number' },
      }),
      execute: ([array, item]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        const result = array.findIndex(arrItem => isEqual(arrItem, item));
        return (result === -1) ? null : result;
      },
      examples: [
        {
          context: {
            numbers: [1, 2, 3],
          },
          expression: 'INDEX_OF(numbers, 1)',
          result: 0,
        },
        {
          context: {
            numbers: [1, 2, 3],
          },
          expression: 'INDEX_OF(numbers, 2)',
          result: 1,
        },
        {
          context: {
            numbers: [1, 2, 3],
          },
          expression: 'INDEX_OF(numbers, null)',
          result: null,
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
          },
          expression: 'INDEX_OF(complexArray, 0)',
          result: 1,
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
          },
          expression: 'INDEX_OF(complexArray, "test")',
          result: 3,
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
          },
          expression: 'INDEX_OF(complexArray, true)',
          result: 11,
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
            emptyArray: [],
          },
          expression: 'INDEX_OF(complexArray, emptyArray)',
          result: 8,
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
            arraySmall: [1, 2, 3],
          },
          expression: 'INDEX_OF(complexArray, arraySmall)',
          result: 6,
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
            emptyObject: {},
          },
          expression: 'INDEX_OF(complexArray, emptyObject)',
          result: 9,
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
            object: { a: 10 },
          },
          expression: 'INDEX_OF(complexArray, object)',
          result: 7,
        },
      ],
    };

  }, { "lodash/isEqual": 579 }], 104: [function (require, module, exports) {
    const isEqual = require('lodash/isEqual');

    module.exports = {
      name: 'IS_IN_ARRAY',
      category: 'Array',
      title: 'Is the given item in the source array?',
      description: 'Given an array and an item whose schema matches the array\'s schema, return `true` if exactly that item can be found in the array, or `false` if it is not found.',
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to check item',
        items: {
          $id: '#item', // Make this a generic type
        },
      }, {
        $ref: '#item', // Refer to the generic type
        title: 'Item',
        description: 'Item to check',
      }],
      analyze: () => ({ schema: { type: 'boolean' } }),
      execute: ([array, item]) => {
        if (!Array.isArray(array)) {
          return false;
        }
        const pos = array.findIndex(el => isEqual(item, el));
        return (pos > -1);
      },
      examples: [
        {
          expression: 'IS_IN_ARRAY(array, 3)',
          context: {
            array: [100, 3, 20],
          },
          result: true,
        },
        {
          expression: 'IS_IN_ARRAY(array, 1)',
          context: {
            array: [100, 3, 20],
          },
          result: false,
        },
        {
          expression: 'IS_IN_ARRAY([100, 3, { a: 10 }], { a: 10 })',
          result: true,
        },
      ],
    };

  }, { "lodash/isEqual": 579 }], 105: [function (require, module, exports) {
    const { execute: isEmpty } = require('../utility/IS_EMPTY');
    const { execute: findByKey } = require('./FIND_BY_KEY');
    const { resolveObjectMemberAccessor, getSchemaAssignmentErrors, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'IS_IN_ARRAY_BY_KEY',
      category: 'Array',
      title: 'Is the given item (identified by a key-value pair) in the target array?',
      description: 'Given an array of objects, the name of a key present in the objects, and a value to match, returns `true` if a matching item is found in the array, or `false` if it is not.',
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to find object in',
        items: {
          type: 'object',
        },
      }, {
        title: 'Key',
        description: 'Key of object whose value to match',
        type: 'string',
      }, {
        title: 'Value',
        description: 'Value to match',
      }],
      analyze: ([arrayParam, keyParam, itemParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || {};
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyErrors,
          schema: keyItemSchema,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        const itemSchema = (itemParam && itemParam.schema) || {};
        const assignmentErrors = getSchemaAssignmentErrors(itemSchema, keyItemSchema);
        return {
          schema: { type: 'boolean' },
          errors: parameterErrors.length ? [] : [
            ...keyErrors.map(error => ({ ...error, offset: keyParam.offset })),
            ...assignmentErrors.map(error => ({ ...error, offset: itemParam.offset })),
          ],
        };
      },
      execute: ([array, key, item]) => {
        const foundItem = findByKey([array, key, item]);
        return !isEmpty([foundItem]);
      },
      examples: [
        {
          context: {
            array1: [{ a: 3 }, { a: 2 }],
          },
          expression: 'IS_IN_ARRAY_BY_KEY(array1, "a", 2)',
          result: true,
        },
        {
          context: {
            array2: [{ a: 3 }, { a: 4 }],
          },
          expression: 'IS_IN_ARRAY_BY_KEY(array2, "a", 2)',
          result: false,
        },
        {
          context: {
            array: [{ a: 3 }, { a: { b: 10 } }],
            key: 'a',
          },
          expression: 'IS_IN_ARRAY_BY_KEY(array, "a", { b: 10 })',
          result: true,
        },
      ],
    };

  }, { "../../schema": 460, "../utility/IS_EMPTY": 438, "./FIND_BY_KEY": 99 }], 106: [function (require, module, exports) {
    module.exports = {
      name: 'MAP',
      category: 'Array',
      title: 'Transforms each item in the array to another',
      description: `Goes through each item in the given array and evaluates an formula for each of them. Returns an array with the evaluated values.

The returned array has the same length than the original array, and the order of its items correspond the order in the original array.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array of items to transform',
        items: {
          $id: '#item', // Make this a generic type
        },
      }, {
        type: 'function',
        title: 'Transform formula',
        description: 'Formula to be evaluated for each item',
        parameters: [{
          name: 'item',
          title: 'Current item',
          description: 'The current item from the original array',
          $ref: '#item', // Refers to the generic type
        }, {
          name: 'index',
          type: 'number',
          minimum: 0,
          title: 'Current index',
          description: 'The position of the current item in the array as an index number starting from 0',
          examples: [0, 1, 2],
        }],
        returnValue: {
          title: 'Transformed item',
          description: 'Transformed value to be set to the result array to the corresponding index',
        },
      }],
      analyze: ([, transformedItem]) => {
        const transformedItemSchema = transformedItem && transformedItem.schema;
        return {
          schema: !transformedItemSchema ? { type: 'array' } : {
            type: 'array',
            items: transformedItemSchema,
          },
        };
      },
      execute: ([array, func]) => {
        if (!Array.isArray(array) || typeof func !== 'function') {
          return null;
        }
        return array.map((item, index) => func(item, index));
      },
      examples: [
        {
          expression: 'MAP(animals, item.name)',
          context: {
            animals: [
              { name: 'dog', weight: 123 },
              { name: 'cat', weight: 48 },
              { name: 'rat', weight: 10 },
            ],
          },
          result: ['dog', 'cat', 'rat'],
        },
        {
          expression: 'MAP(products, index + ": " + item.name)',
          context: {
            products: [
              { name: 'Duct tape', price: 5 },
              { name: 'Swiss army knife', price: 60 },
            ],
          },
          result: [
            '0: Duct tape',
            '1: Swiss army knife',
          ],
        },
        {
          expression: 'MAP<product>(products, product.name + " (" + product.price + "€)")',
          context: {
            products: [
              { name: 'Duct tape', price: 5 },
              { name: 'Swiss army knife', price: 60 },
            ],
          },
          result: [
            'Duct tape (5€)',
            'Swiss army knife (60€)',
          ],
        },
        {
          expression: 'MAP<product, position>(products, position + ": " + product.name + " (" + product.price + "€)")',
          context: {
            products: [
              { name: 'Duct tape', price: 5 },
              { name: 'Swiss army knife', price: 60 },
            ],
          },
          result: [
            '0: Duct tape (5€)',
            '1: Swiss army knife (60€)',
          ],
        },
      ],
    };

  }, {}], 107: [function (require, module, exports) {
    const { getArrayItemSchema } = require('../../schema');

    function compare(a, b) {
      if (a < b) {
        return -1;
      }
      if (a > b) {
        return 1;
      }
      return 0;
    }

    module.exports = {
      name: 'ORDER',
      category: 'Array',
      title: 'Sort an array',
      description: `Sorts an array using a formula that is evaluated for each item in the array. Returns the sorted array.

The formula should result to a comparable value (a string or a number) that is used to determine the order of the items.
By default the items are sorted in ascending order. By providing the 3rd parameter you can choose the direction.

If multiple values have the same ordering value, their original order will be preserved.

Note that for string values, the ordering is case-sensitive. If you want a case-insensitive ordering, you may for example wrap your ordering formula with \`LOWERCASE\` function.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array of items to sort',
        items: {
          $id: '#item', // Make this a generic type
        },
      }, {
        type: 'function',
        title: 'Sorting criteria',
        description: 'Formula evaluated for each item in the array, used to determine the ordering value for the item',
        parameters: [{
          name: 'item',
          title: 'Current item',
          description: 'The current item from the original array',
          $ref: '#item', // Refers to the generic type
        }, {
          name: 'index',
          type: 'number',
          minimum: 0,
          title: 'Current index',
          description: 'The original position of the current item in the array as an index number starting from 0',
          examples: [0, 1, 2],
        }],
        returnValue: {
          title: 'Ordering value',
          description: 'A value that is used to determine the position of the current item in the result array',
          anyOf: [{
            type: 'string',
          }, {
            type: 'number',
          }],
        },
      }, {
        type: 'string',
        enum: ['asc', 'desc'],
        default: 'asc',
        title: 'Sort direction',
        description: 'Direction of the ordering. Use either `"asc"` meaning ascending order (smallest first) or `"desc"` meaning descending order (greatest first).',
      }],
      analyze: ([arrayParam]) => {
        const itemSchema = arrayParam && getArrayItemSchema(arrayParam && arrayParam.schema);
        return {
          schema: !itemSchema ? { type: 'array' } : {
            type: 'array', items: itemSchema,
          },
        };
      },
      execute: ([array, func, direction]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        if (typeof func !== 'function') {
          return array;
        }
        // Get ordering value for each item (only do this once)
        const items = array.map((value, index) => ({ value, ordering: func(value, index), index }));
        // Sort using the native sort, and using the index to ensure stability
        const factor = direction === 'desc' ? -1 : 1;
        items.sort((a, b) => {
          const comparison = compare(a.ordering, b.ordering) * factor;
          return comparison === 0 ? a.index - b.index : comparison;
        });
        return items.map(item => item.value);
      },
      examples: [
        {
          expression: 'ORDER(animals, item.weight)',
          context: {
            animals: [
              { name: 'dog', weight: 123 },
              { name: 'cat', weight: 48 },
              { name: 'rat', weight: 10 },
              { name: 'pig', weight: 123 },
            ],
          },
          result: [
            { name: 'rat', weight: 10 },
            { name: 'cat', weight: 48 },
            { name: 'dog', weight: 123 },
            { name: 'pig', weight: 123 },
          ],
        },
        {
          expression: 'ORDER(animals, item.weight, "desc")',
          context: {
            animals: [
              { name: 'dog', weight: 123 },
              { name: 'cat', weight: 48 },
              { name: 'rat', weight: 10 },
              { name: 'pig', weight: 123 },
            ],
          },
          result: [
            { name: 'dog', weight: 123 },
            { name: 'pig', weight: 123 },
            { name: 'cat', weight: 48 },
            { name: 'rat', weight: 10 },
          ],
        },
        {
          expression: 'ORDER<animal>(animals, animal.weight, "asc")',
          context: {
            animals: [
              { name: 'dog', weight: 123 },
              { name: 'cat', weight: 48 },
              { name: 'rat', weight: 10 },
              { name: 'pig', weight: 123 },
            ],
          },
          result: [
            { name: 'rat', weight: 10 },
            { name: 'cat', weight: 48 },
            { name: 'dog', weight: 123 },
            { name: 'pig', weight: 123 },
          ],
        },
      ],
    };

  }, { "../../schema": 460 }], 108: [function (require, module, exports) {
    const { resolveObjectMemberAccessor } = require('../../schema');


    module.exports = {
      name: 'PICK_ITEM',
      category: 'Array',
      title: 'Pick item from an arry by its index',
      description: 'Returns an item in the source array by the given index. If **Index** is out of bounds for the array (i.e. negative or larger than the array length - 1, since indexing starts from 0), `null` is returned.',
      parameters: [{
        type: 'array',
        title: 'Values',
        description: 'Source array',
      }, {
        title: 'Index',
        description: 'Index of item to return',
      }],
      analyze: ([arrayParam, indexParam], offset, parameterErrors) => {
        const itemSchema = (arrayParam && arrayParam.schema) || {};
        const accessorSchema = (indexParam && indexParam.schema) || {};
        const { schema, errors } = resolveObjectMemberAccessor(itemSchema, accessorSchema);
        const propertySchema = schema || {};
        return {
          schema: propertySchema,
          errors: parameterErrors.length ? [] : errors.map(error => ({
            ...error, offset: indexParam.offset,
          })),
        };
      },
      execute: ([array, index]) => {
        if (!Array.isArray(array)
          || !(typeof index === 'number' && Number.isFinite(index))
        ) {
          return null;
        }
        return array[index];
      },
      examples: [
        {
          context: {
            numbers: [1, 2, 3],
          },
          expression: 'PICK_ITEM(numbers, 0)',
          result: 1,
        },
        {
          context: {
            numbers: [1, 2, 3],
          },
          expression: 'PICK_ITEM(numbers, 1)',
          result: 2,
        },
        {
          context: {
            numbers: [1, 2, 3],
          },
          expression: 'PICK_ITEM(numbers, 10)',
          result: undefined,
        },
        {
          context: {
            numbers: [1, 2, 3],
          },
          expression: 'PICK_ITEM(numbers, -10)',
          result: undefined,
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
          },
          expression: 'PICK_ITEM(complexArray, 0)',
          result: 1,
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
          },
          expression: 'PICK_ITEM(complexArray, 3)',
          result: 'test',
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
          },
          expression: 'PICK_ITEM(complexArray, 4)',
          result: null,
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
          },
          expression: 'PICK_ITEM(complexArray, 5)',
          result: undefined,
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
          },
          expression: 'PICK_ITEM(complexArray, 6)',
          result: [1, 2, 3],
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
          },
          expression: 'PICK_ITEM(complexArray, 7)',
          result: { a: 10 },
        },
        {
          context: {
            complexArray: [1, 0, 2, 'test', null, undefined, [1, 2, 3], { a: 10 }, [], {}, false, true],
          },
          expression: 'PICK_ITEM(complexArray, 10)',
          result: false,
        },
      ],
    };

  }, { "../../schema": 460 }], 109: [function (require, module, exports) {
    const { resolveObjectMemberAccessor, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'PLUCK',
      category: 'Array',
      title: 'Get a property of each object in an array',
      description: `Gets the values for each object in the given array by the given key, and returns an array with those values, in the corresponding order.

For example, if you have an array of people as objects, you can get the name of each person by providing the array as the first parameter, and the key name \`"name"\` as the second parameter.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array of objects whose properties to get',
        items: {
          type: 'object',
        },
      }, {
        type: 'string',
        title: 'Key',
        description: 'Name of the key whose value to take from each object',
      }],
      analyze: ([arrayParam, propNameParam], offset, parameterErrors) => {
        const itemSchema = getArrayItemSchema(arrayParam && arrayParam.schema) || {};
        const accessorSchema = (propNameParam && propNameParam.schema) || {};
        const { schema, errors } = resolveObjectMemberAccessor(itemSchema, accessorSchema);
        const arraySchema = schema ? { type: 'array', items: schema } : { type: 'array' };
        return {
          schema: arraySchema,
          errors: parameterErrors.length ? [] : errors.map(error => ({
            ...error, offset: propNameParam.offset,
          })),
        };
      },
      execute: ([array, prop]) => (array == null ? array : array.map(item => item[prop])),
      examples: [
        {
          expression: 'PLUCK(products, "price")',
          context: {
            products: [
              { name: 'Duct tape', price: 5 },
              { name: 'Swiss army knife', price: 60 },
            ],
          },
          result: [5, 60],
        },
        {
          expression: 'PLUCK(animals, "name")',
          context: {
            animals: [
              { name: 'dog', weight: 123 },
              { name: 'cat', weight: 48 },
              { name: 'rat', weight: 10 },
            ],
          },
          result: ['dog', 'cat', 'rat'],
        },
      ],
    };

  }, { "../../schema": 460 }], 110: [function (require, module, exports) {
    module.exports = {
      name: 'REDUCE',
      category: 'Array',
      title: 'Reduce an array to a single value',
      description: `Reduces an array to a single value by evaluating a formula for each item in the array, accumulating to a result value which is then passed to the next evaluation.

The iteration is done from the first item to the last item. The iteration starts by using the 1st item in the array as an initial accumulator value.
Alternatively you can use \`REDUCE_INIT\` function instead and provide an initial value.

If the array has only one item, that item is returned. If the array is empty, returns \`null\`.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array of items to reduce',
        items: {
          $id: '#item', // Make this a generic type
        },
      }, {
        type: 'function',
        title: 'Reducer formula',
        description: 'Formula that returns a reduced result using the current item and the previous accumulated result value',
        parameters: [{
          name: 'accumulator',
          title: 'Accumulator value',
          description: 'The current accumulated, reduced result value',
          anyOf: [{
            $ref: '#accumulator', // Refers to the generic type
          }, {
            $ref: '#item', // Refers to the generic type
          }],
        }, {
          name: 'item',
          title: 'Current item',
          description: 'The current item from the original array',
          $ref: '#item', // Refers to the generic type
        }, {
          name: 'index',
          type: 'number',
          minimum: 0,
          title: 'Current index',
          description: 'The position of the current item in the array as an index number starting from 0',
          examples: [0, 1, 2],
        }],
        returnValue: {
          $id: '#accumulator', // Make this a generic type
          title: 'Accumulated value',
          description: 'Accumulated value reduced from the current item and the previous accumulator',
        },
      }],
      analyze: ([, accumulatorParam]) => {
        const accumulatorSchema = (accumulatorParam && accumulatorParam.schema) || {};
        return { schema: accumulatorSchema };
      },
      execute: ([array, func]) => {
        if (!Array.isArray(array) || typeof func !== 'function') {
          return null;
        }
        if (!array.length) {
          // Won't reduce an empty array
          return null;
        }
        return array.reduce((accumulator, item, index) => func(accumulator, item, index));
      },
      examples: [
        {
          expression: 'REDUCE(products, IF(item.price < accumulator.price, item, accumulator))',
          context: {
            products: [
              { name: 'Swiss army knife', price: 60 },
              { name: 'Duct tape', price: 5 },
              { name: 'Glue', price: 6 },
            ],
          },
          result: { name: 'Duct tape', price: 5 },
        },
        {
          expression: 'REDUCE<cheapest, product>(products, IF(product.price < cheapest.price, product, cheapest))',
          context: {
            products: [
              { name: 'Swiss army knife', price: 60 },
              { name: 'Duct tape', price: 5 },
              { name: 'Glue', price: 6 },
            ],
          },
          result: { name: 'Duct tape', price: 5 },
        },
      ],
    };

  }, {}], 111: [function (require, module, exports) {
    module.exports = {
      name: 'REDUCE_INIT',
      category: 'Array',
      title: 'Reduce an array to a single value',
      description: `Reduces an array to a single value by evaluating a formula for each item in the array, accumulating to a result value which is then passed to the next evaluation.

The iteration is done from the first item to the last item. The iteration starts by using the 1st item in the array as an initial accumulator value.
Alternatively you can use \`REDUCE_INIT\` function instead and provide an initial value.

If the array has only one item, that item is returned. If the array is empty, returns \`null\`.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array of items to reduce',
        items: {
          $id: '#item', // Make this a generic type
        },
      }, {
        title: 'Initial accumulator',
        description: 'Initial value for the accumulator passed to the first evaluation.',
        $id: '#initial', // Make this a generic type
      }, {
        type: 'function',
        title: 'Reducer formula',
        description: 'Formula that returns a reduced result using the current item and the previous accumulated result value',
        parameters: [{
          name: 'accumulator',
          title: 'Accumulator value',
          description: 'The current accumulated, reduced result value',
          anyOf: [{
            $ref: '#accumulator', // Refers to the generic type
          }, {
            $ref: '#initial', // Refers to the generic type
          }],
        }, {
          name: 'item',
          title: 'Current item',
          description: 'The current item from the original array',
          $ref: '#item', // Refers to the generic type
        }, {
          name: 'index',
          type: 'number',
          minimum: 0,
          title: 'Current index',
          description: 'The position of the current item in the array as an index number starting from 0',
          examples: [0, 1, 2],
        }],
        returnValue: {
          $id: '#accumulator', // Make this a generic type
          title: 'Accumulated value',
          description: 'Accumulated value reduced from the current item and the previous accumulator',
        },
      }],
      analyze: ([, , accumulatorParam]) => {
        const accumulatorSchema = (accumulatorParam && accumulatorParam.schema) || {};
        return { schema: accumulatorSchema };
      },
      execute: ([array, init, func]) => {
        if (!Array.isArray(array) || typeof func !== 'function') {
          return null;
        }
        return array.reduce((accumulator, item, index) => func(accumulator, item, index), init);
      },
      examples: [
        {
          expression: 'REDUCE_INIT(products, 0, accumulator + item.price)',
          context: {
            products: [
              { name: 'Swiss army knife', price: 60 },
              { name: 'Duct tape', price: 5 },
              { name: 'Glue', price: 6 },
            ],
          },
          result: 71,
        },
        {
          expression: 'REDUCE_INIT<sum, product>(products, 0, sum + product.price)',
          context: {
            products: [
              { name: 'Swiss army knife', price: 60 },
              { name: 'Duct tape', price: 5 },
              { name: 'Glue', price: 6 },
            ],
          },
          result: 71,
        },
      ],
    };

  }, {}], 112: [function (require, module, exports) {
    const isEqual = require('lodash/isEqual');
    const { resolveObjectMemberAccessor, getSchemaAssignmentErrors, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'REMOVE_ITEM_BY_KEY',
      category: 'Array',
      title: 'Remove objects from array by a key',
      description: `Finds an object by a key-value pair and returns an array which doesn't contain the matched object.

For example, given **Key** \`"id"\` and **Value** \`123\`, will return an array without the object whose \`id\` property is \`123\`.

If a matching object cannot be found, the original array is returned unmodified.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to remove object from',
        items: {
          type: 'object',
        },
      }, {
        title: 'Key',
        description: 'Key whose value to match',
        type: 'string',
      }, {
        title: 'Value',
        description: 'Value to match',
      }],
      analyze: ([arrayParam, keyParam, itemParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyErrors,
          schema: keyItemSchema,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        const itemSchema = (itemParam && itemParam.schema) || {};
        const assignmentErrors = getSchemaAssignmentErrors(itemSchema, keyItemSchema);
        return {
          schema: arraySchema,
          errors: parameterErrors.length ? [] : [
            ...keyErrors.map(error => ({
              ...error, offset: keyParam.offset,
            })),
            ...assignmentErrors.map(error => ({
              message: error.message, offset: itemParam.offset,
            })),
          ],
        };
      },
      execute: ([array, key, item]) => {
        if (!Array.isArray(array) || key == null || typeof item === 'undefined') {
          return null;
        }
        const pos = array.findIndex(el => isEqual(el[key], item));
        const splicingArray = [...array];
        if (pos > -1) {
          splicingArray.splice(pos, 1);
        }
        return splicingArray;
      },
      examples: [
        {
          expression: 'REMOVE_ITEM_BY_KEY(array1, "a", 2)',
          context: {
            array1: [{ a: 1 }, { a: 2 }, { a: 3 }],
          },
          result: [{ a: 1 }, { a: 3 }],
        },
        {
          expression: 'REMOVE_ITEM_BY_KEY(array2, "a", { b: 10 })',
          context: {
            array2: [{ a: 3 }, { a: { b: 10 } }],
          },
          result: [{ a: 3 }],
        },
        {
          expression: 'REMOVE_ITEM_BY_KEY(array3, "a", 4)',
          context: {
            array3: [{ a: 3 }, { a: 2 }],
          },
          result: [{ a: 3 }, { a: 2 }],
        },
      ],
    };

  }, { "../../schema": 460, "lodash/isEqual": 579 }], 113: [function (require, module, exports) {
    module.exports = {
      name: 'REPEAT',
      category: 'Array',
      title: 'Repeat item',
      description: 'Constructs an array with a certain value repeated the given number of times',
      parameters: [{
        title: 'Item',
        description: 'Item to be repeated',
      }, {
        type: 'number',
        title: 'Count',
        description: 'Number of times the item will be in the array',
      }],
      analyze: ([item]) => {
        const { schema: itemSchema } = item;
        const schema = itemSchema != null ? { type: 'array', items: itemSchema } : { type: 'array' };
        return { schema };
      },
      execute: ([item, count]) => {
        const result = [];
        if (typeof count === 'number' && Number.isFinite(count)) {
          for (let i = 0; i < count; i += 1) {
            result.push(item);
          }
        }
        return result;
      },
      examples: [
        { expression: 'REPEAT("three", 3)', result: ['three', 'three', 'three'] },
        { expression: 'REPEAT({foo: "bar"}, 2)', result: [{ foo: 'bar' }, { foo: 'bar' }] },
        { expression: 'REPEAT("hello", 0)', result: [] },
      ],
    };

  }, {}], 114: [function (require, module, exports) {
    module.exports = {
      name: 'REVERSE',
      category: 'Array',
      title: 'Reverse an array',
      description: 'Returns an array containing all the values from the given array, but in reverse order.',
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to reverse',
      }],
      analyze: ([arrayParam]) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        return {
          schema: arraySchema,
        };
      },
      execute: ([array]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        return [...array].reverse();
      },
      examples: [
        {
          expression: 'REVERSE(charArray)',
          context: {
            charArray: ['a', 'b', 'c'],
          },
          result: ['c', 'b', 'a'],
        },
        {
          expression: 'REVERSE(numbersArray)',
          context: {
            numbersArray: [1, 2, 3],
          },
          result: [3, 2, 1],
        },
        {
          expression: 'REVERSE(simpleArray)',
          context: {
            simpleArray: [1],
          },
          result: [1],
        },
      ],
    };

  }, {}], 115: [function (require, module, exports) {
    const { getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'SELECT',
      category: 'Array',
      title: 'Select specific items from array',
      description: `Selects items from an array that match the given criteria. Returns an array containing only the matching items, in their corresponding order.

The criteria is defined as a formula at the second parameter. The formula is evaluated with each item in the array.
The formula must result in a boolean value. If \`true\`, then item will be included to the result, otherwise it will be excluded.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array of items to filter',
        items: {
          $id: '#item', // Make this a generic type
        },
      }, {
        type: 'function',
        title: 'Filtering formula',
        description: 'Formula to be evaluated for each item, deciding whether to include or not to the result array',
        parameters: [{
          name: 'item',
          title: 'Current item',
          description: 'The current item from the original array',
          $ref: '#item', // Refers to the generic type
        }, {
          name: 'index',
          type: 'number',
          minimum: 0,
          title: 'Current index',
          description: 'The position of the current item in the array as an index number starting from 0',
          examples: [0, 1, 2],
        }],
        returnValue: {
          type: 'boolean',
          title: 'Whether item is included',
          description: 'Whether or not the item is included to the result array',
        },
      }],
      analyze: ([arrayParam]) => {
        const arraySchema = arrayParam && arrayParam.schema;
        if (!arraySchema) {
          return { schema: { type: 'array' } };
        }
        // NOTE: Not using the parameter type directly, as it might contain enum or examples
        // that are no longer valid.
        return {
          schema: {
            type: 'array',
            items: getArrayItemSchema(arraySchema) || {},
          },
        };
      },
      execute: ([array, func]) => {
        if (!Array.isArray(array) || typeof func !== 'function') {
          return null;
        }
        return array.filter((item, index) => func(item, index));
      },
      examples: [
        {
          expression: 'SELECT(animals, item.weight > 100)',
          context: {
            animals: [
              { name: 'dog', weight: 123 },
              { name: 'cat', weight: 48 },
              { name: 'rat', weight: 10 },
              { name: 'pig', weight: 230 },
            ],
          },
          result: [
            { name: 'dog', weight: 123 },
            { name: 'pig', weight: 230 },
          ],
        },
        {
          expression: 'SELECT(animals, IS_ODD(index))',
          context: {
            animals: [
              { name: 'dog', weight: 123 },
              { name: 'cat', weight: 48 },
              { name: 'rat', weight: 10 },
              { name: 'pig', weight: 230 },
            ],
          },
          result: [
            { name: 'cat', weight: 48 },
            { name: 'pig', weight: 230 },
          ],
        },
        {
          expression: 'SELECT<product>(products, product.category == "Utility")',
          context: {
            products: [
              { name: 'Duct tape', category: 'Utility' },
              { name: 'Swiss army knife', category: 'Tools' },
              { name: 'Glue', category: 'Utility' },
            ],
          },
          result: [
            { name: 'Duct tape', category: 'Utility' },
            { name: 'Glue', category: 'Utility' },
          ],
        },
        {
          expression: 'SELECT<product, position>(products, position < 2 && product.category == "Utility")',
          context: {
            products: [
              { name: 'Duct tape', category: 'Utility' },
              { name: 'Swiss army knife', category: 'Tools' },
              { name: 'Glue', category: 'Utility' },
            ],
          },
          result: [
            { name: 'Duct tape', category: 'Utility' },
          ],
        },
      ],
    };

  }, { "../../schema": 460 }], 116: [function (require, module, exports) {
    const isEqual = require('lodash/isEqual');
    const { resolveObjectMemberAccessor, getSchemaAssignmentErrors, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'SELECT_BY_KEY',
      category: 'Array',
      title: 'Find objects from an array by a key',
      description: `Finds multiple objects from the given array by a key-value pair.

For example, given **Key** \`"age"\` and **Value** \`20\`, will return an array containing all objects in the array whose \`age\` property is \`20\`, or an empty array if no matching objects were found.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to find objects in',
        items: {
          type: 'object',
        },
      }, {
        title: 'Key',
        description: 'Key whose value to match',
        type: 'string',
      }, {
        title: 'Value',
        description: 'Value to match',
      }],
      analyze: ([arrayParam, keyParam, itemParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyErrors,
          schema: keyItemSchema,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        const itemSchema = (itemParam && itemParam.schema) || {};
        const assignmentErrors = getSchemaAssignmentErrors(itemSchema, keyItemSchema);
        return {
          schema: arraySchema,
          errors: parameterErrors.length ? [] : [
            ...keyErrors.map(error => ({ ...error, offset: keyParam.offset })),
            ...assignmentErrors.map(error => ({ message: error.message, offset: itemParam.offset }))],
        };
      },
      execute: ([array, key, item]) => {
        if (!Array.isArray(array) || key == null) {
          return null;
        }
        return array.filter(el => isEqual(el[key], item));
      },
      examples: [
        {
          expression: 'SELECT_BY_KEY(array1, "a", 2)',
          context: {
            array1: [{ a: 3 }, { a: 2 }, { a: 2, b: 3 }],
          },
          result: [{ a: 2 }, { a: 2, b: 3 }],
        },
        {
          expression: 'SELECT_BY_KEY(array2, "a", { b: 10 })',
          context: {
            array2: [{ a: 3 }, { a: { b: 10 } }],
          },
          result: [{ a: { b: 10 } }],
        },
        {
          expression: 'SELECT_BY_KEY(array, "a", 4)',
          context: {
            array: [{ a: 3 }, { a: 2 }],
          },
          result: [],
        },
      ],
    };

  }, { "../../schema": 460, "lodash/isEqual": 579 }], 117: [function (require, module, exports) {
    const { resolveObjectMemberAccessor, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'SELECT_BY_REGEX',
      category: 'Array',
      title: 'Find objects from array by a key-regex match',
      description: `Finds multiple objects from the given array by a key-regex pair.

For example, given **Key** \`"name"\` and **Pattern** \`/Agnus/\`, will return an array containing all objects in the array whose \`name\` property contains \`Agnus\`, or an empty array if no matching objects were found.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to find objects in',
        items: {
          type: 'object',
        },
      }, {
        title: 'Key',
        description: 'Key whose value to match',
        type: 'string',
      }, {
        title: 'Pattern',
        description: 'Regex pattern to match',
        type: 'string',
      }],
      analyze: ([arrayParam, keyParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyErrors,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        return {
          schema: arraySchema,
          errors: parameterErrors.length ? [] : keyErrors.map(error => ({
            ...error, offset: keyParam.offset,
          })),
        };
      },
      execute: ([array, key, pattern]) => {
        if (!Array.isArray(array) || key == null || pattern == null) {
          return null;
        }
        if (!(typeof pattern === 'string' || pattern instanceof RegExp)) {
          return null;
        }
        const regex = pattern instanceof RegExp ? pattern : new RegExp(pattern);
        return array.filter(el => regex.test(el[key]));
      },
      examples: [
        {
          expression: 'SELECT_BY_REGEX(array1, "a", "^foo")',
          context: {
            array1: [{ a: 2 }, { a: 'foo bar' }, { a: 'foo', b: 3 }],
          },
          result: [{ a: 'foo bar' }, { a: 'foo', b: 3 }],
        },
        {
          expression: 'SELECT_BY_REGEX(array1, "a", "^foo")',
          context: {
            array1: [{ a: 2 }, { a: 'foo bar' }, { a: 'foo', b: 3 }],
          },
          result: [{ a: 'foo bar' }, { a: 'foo', b: 3 }],
        },
      ],
    };

  }, { "../../schema": 460 }], 118: [function (require, module, exports) {
    module.exports = {
      name: 'SHUFFLE',
      category: 'Array',
      title: 'Shuffle an array',
      description: 'Returns an array containing all the values from the array parameter, but shuffled into a random order.',
      deterministic: false, // Result is in random order
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to shuffle',
      }],
      analyze: ([arrayParam]) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        return {
          schema: arraySchema,
        };
      },
      execute: ([array]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        if (array.length <= 1) {
          return array;
        }
        const result = new Array(array.length);
        for (let i = 0; i < array.length; i += 1) {
          const j = Math.floor(Math.random() * (i + 1));
          if (i !== j) {
            result[i] = result[j];
          }
          result[j] = array[i];
        }
        return result;
      },
      examples: [
        {
          expression: 'SHUFFLE([])',
          result: [],
        },
        {
          expression: 'SHUFFLE(array1)',
          context: {
            array1: ['a', 'b', 'c', 'd', 'e'],
          },
          result: ['b', 'c', 'a', 'e', 'd'],
        },
        {
          expression: 'SHUFFLE(array2)',
          context: {
            array2: ['a', 'b', 'c', 1, 2, 3],
          },
          result: ['1', 'b', '2', 'a', 'c', '3'],
        },
      ],
    };

  }, {}], 119: [function (require, module, exports) {
    const { getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'SLICE',
      category: 'Array',
      title: 'Select a range of items from array',
      description: `Selects the items starting at the given **Start** index and ending at the given **End** index.

Note that the element at the given end index is *not* included.

You can use a negative end index to select from the end of the array.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to slice',
      }, {
        title: 'Start',
        description: 'Index of the first item',
        type: 'number',
      }, {
        title: 'End',
        description: 'Index where the range ends, not including item at this index. Negative index will be calculated from the end of the array.',
        type: 'number',
        default: null,
      }],
      analyze: ([arrayParam]) => {
        const schema = { type: 'array' };
        const itemSchema = arrayParam && getArrayItemSchema(arrayParam.schema);
        if (itemSchema) {
          schema.items = itemSchema;
        }
        return { schema };
      },
      execute: ([array, start, end]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        return end == null ? array.slice(start) : array.slice(start, end);
      },
      examples: [
        {
          expression: 'SLICE(["a", "b", "c", "d"], 1, 3)',
          result: ['b', 'c'],
        },
        {
          expression: 'SLICE(["a", "b", "c", "d"], 0, -1)',
          result: ['a', 'b', 'c'],
        },
        {
          expression: 'SLICE(["a", "b", "c", "d"], 2)',
          result: ['c', 'd'],
        },
      ],
    };

  }, { "../../schema": 460 }], 120: [function (require, module, exports) {
    const comparator = (left, right, desc) => {
      if (typeof left === 'string' && typeof right === 'string') {
        const leftVal = left.toLocaleLowerCase();
        const rightVal = right.toLocaleLowerCase();
        const compared = desc ? rightVal.localeCompare(leftVal) : leftVal.localeCompare(rightVal);
        return desc ? !(compared >= 0) : compared <= 0;
      }
      return desc ? (left >= right) : (left <= right);
    };

    const merge = (left, right, desc) => {
      const result = [];
      let indexLeft = 0;
      let indexRight = 0;

      while (indexLeft < left.length && indexRight < right.length) {
        const leftVal = left[indexLeft];
        const rightVal = right[indexRight];
        if (comparator(leftVal, rightVal, desc)) {
          result.push(left[indexLeft]);
          indexLeft += 1;
        } else {
          result.push(right[indexRight]);
          indexRight += 1;
        }
      }
      return result.concat(left.slice(indexLeft)).concat(right.slice(indexRight));
    };

    const mergeSort = (arr, desc) => {
      if (arr.length === 1) {
        return arr;
      }
      const middle = Math.floor(arr.length / 2);
      const left = arr.slice(0, middle);
      const right = arr.slice(middle);
      return merge(
        mergeSort(left, desc),
        mergeSort(right, desc),
        desc,
      );
    };

    module.exports = {
      name: 'SORT',
      category: 'Array',
      title: 'Sort an array of strings or numbers',
      description: 'Returns an array containing all the values in the given array of strings or numbers, sorted in order. The function ignores letter case.',
      parameters: [{
        title: 'Array',
        description: 'Array to sort',
        anyOf: [
          {
            type: 'array',
            items: { type: 'number' },
          },
          {
            type: 'array',
            items: { type: 'string' },
          },
        ],
      }, {
        type: 'string',
        title: 'Order',
        default: 'asc',
        enum: ['asc', 'desc'],
        description: 'Order to sort in (`"asc"` for ascending, `"desc"` for descending)',
      }],
      analyze: ([arrayParam]) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        return { schema: arraySchema };
      },
      execute: ([array, order]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        if (!array.length) {
          return [];
        }
        const desc = order === 'desc';
        return (mergeSort(array, desc));
      },
      examples: [
        {
          expression: 'SORT(array)',
          context: {
            array: [10, 1, 2, 6, -1, 10, 12],
          },
          result: [-1, 1, 2, 6, 10, 10, 12],
        },
        {
          expression: 'SORT(array, "desc")',
          context: {
            array: [10, 1, 2, 6, -1, 10, 12],
          },
          result: [12, 10, 10, 6, 2, 1, -1],
        },
        {
          expression: 'SORT(textArray)',
          context: {
            textArray: ['ABCD', 'CC', 'bb', 'AA', 'a', 'aa', 'abcd'],
          },
          result: ['a', 'AA', 'aa', 'ABCD', 'abcd', 'bb', 'CC'],
        },
      ],
    };

  }, {}], 121: [function (require, module, exports) {
    const { resolveObjectMemberAccessor, getArrayItemSchema } = require('../../schema');

    const comparator = (left, right, desc) => {
      if (typeof left === 'string' && typeof right === 'string') {
        const leftVal = left.toLocaleLowerCase();
        const rightVal = right.toLocaleLowerCase();
        const compared = desc ? rightVal.localeCompare(leftVal) : leftVal.localeCompare(rightVal);
        return desc ? !(compared >= 0) : compared <= 0;
      }
      return desc ? (left >= right) : (left <= right);
    };

    const merge = (left, right, key, desc) => {
      const result = [];
      let indexLeft = 0;
      let indexRight = 0;

      while (indexLeft < left.length && indexRight < right.length) {
        const leftObj = left[indexLeft];
        const rightObj = right[indexRight];
        let leftVal = (typeof leftObj === 'object' && leftObj != null) ? leftObj[key] : null;
        let rightVal = (typeof rightObj === 'object' && rightObj != null) ? rightObj[key] : null;
        leftVal = (leftVal !== undefined) ? leftVal : null;
        rightVal = (rightVal !== undefined) ? rightVal : null;
        if (comparator(leftVal, rightVal, desc)) {
          result.push(left[indexLeft]);
          indexLeft += 1;
        } else {
          result.push(right[indexRight]);
          indexRight += 1;
        }
      }
      return result.concat(left.slice(indexLeft)).concat(right.slice(indexRight));
    };

    const mergeSort = (arr, key, desc) => {
      if (arr.length === 1) {
        return arr;
      }
      const middle = Math.floor(arr.length / 2);
      const left = arr.slice(0, middle);
      const right = arr.slice(middle);
      return merge(
        mergeSort(left, key, desc),
        mergeSort(right, key, desc),
        key,
        desc,
      );
    };

    module.exports = {
      name: 'SORT_BY_KEY',
      category: 'Array',
      title: 'Sort an array of objects by a key',
      description: `Sorts an array of objects by the values of the given key.

If the objects do not contain string or number values under the given key, the sort might not behave as expected.

The third parameter can be used to sort in ascending (default) or descending order.

Compared to \`SORT_BY_KEY_CASE_SENSITIVE\`, this function is case insensitive.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to sort',
        items: {
          type: 'object',
        },
      }, {
        title: 'Key',
        description: 'Key whose value to sort by',
        type: 'string',
      }, {
        type: 'string',
        title: 'Order',
        default: 'asc',
        enum: ['asc', 'desc'],
        description: 'Order to sort in (`"asc"` for ascending, `"desc"` for descending); defaults to ascending',
      }],
      analyze: ([arrayParam, keyParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyErrors,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        return {
          schema: arraySchema,
          errors: parameterErrors.length ? [] : keyErrors.map(error => ({
            ...error, offset: keyParam.offset,
          })),
        };
      },
      execute: ([array, key, order]) => {
        if (!Array.isArray(array) || key == null) {
          return null;
        }
        if (!array.length) {
          return [];
        }
        const desc = order === 'desc';
        return (mergeSort(array, key, desc));
      },
      examples: [
        {
          expression: 'SORT_BY_KEY(array, key)',
          context: {
            array: [
              { a: 10 },
              { a: 20, b: 1 },
              { a: 20 },
              { a: 20, b: 2 },
              { a: 1 },
              { a: -1, b: 2 },
              { a: 1, b: 2 },
              { a: 3 },
              { a: -1 },
            ],
            key: 'a',
          },
          result: [
            { a: -1, b: 2 },
            { a: -1 },
            { a: 1 },
            { a: 1, b: 2 },
            { a: 3 },
            { a: 10 },
            { a: 20, b: 1 },
            { a: 20 },
            { a: 20, b: 2 },
          ],
        },
        {
          expression: 'SORT_BY_KEY(array, key, order)',
          context: {
            array: [
              { a: 10 },
              { a: 20, b: 1 },
              { a: 20 },
              { a: 20, b: 2 },
              { a: 1 },
              { a: -1, b: 2 },
              { a: 1, b: 2 },
              { a: 3 },
              { a: -1 },
            ],
            key: 'a',
            order: 'desc',
          },
          result: [
            { a: 20, b: 1 },
            { a: 20 },
            { a: 20, b: 2 },
            { a: 10 },
            { a: 3 },
            { a: 1 },
            { a: 1, b: 2 },
            { a: -1, b: 2 },
            { a: -1 },
          ],
        },
      ],
    };

  }, { "../../schema": 460 }], 122: [function (require, module, exports) {
    const { resolveObjectMemberAccessor, getArrayItemSchema } = require('../../schema');

    const comparator = (left, right, desc) => {
      if (typeof left === 'string' && typeof right === 'string') {
        const compared = desc ? right.localeCompare(left) : left.localeCompare(right);
        return compared <= 0;
      }
      return desc ? (left >= right) : (left <= right);
    };

    const merge = (left, right, key, desc) => {
      const result = [];
      let indexLeft = 0;
      let indexRight = 0;

      while (indexLeft < left.length && indexRight < right.length) {
        const leftObj = left[indexLeft];
        const rightObj = right[indexRight];
        let leftVal = (typeof leftObj === 'object' && leftObj != null) ? leftObj[key] : null;
        let rightVal = (typeof rightObj === 'object' && rightObj != null) ? rightObj[key] : null;
        leftVal = (leftVal !== undefined) ? leftVal : null;
        rightVal = (rightVal !== undefined) ? rightVal : null;
        if (comparator(leftVal, rightVal, desc)) {
          result.push(left[indexLeft]);
          indexLeft += 1;
        } else {
          result.push(right[indexRight]);
          indexRight += 1;
        }
      }
      return result.concat(left.slice(indexLeft)).concat(right.slice(indexRight));
    };

    const mergeSort = (arr, key, desc) => {
      if (arr.length === 1) {
        return arr;
      }
      const middle = Math.floor(arr.length / 2);
      const left = arr.slice(0, middle);
      const right = arr.slice(middle);
      return merge(
        mergeSort(left, key, desc),
        mergeSort(right, key, desc),
        key,
        desc,
      );
    };

    module.exports = {
      name: 'SORT_BY_KEY_CASE_SENSITIVE',
      category: 'Array',
      title: 'Sort an array of objects by a key (case-sensitive)',
      description: `Sorts an array of objects by the values of the given key.

If the objects do not contain string or number values under the given key, the sort might not behave as expected.

The third parameter can be used to sort in ascending (default) or descending order.

Compared to \`SORT_BY_KEY\`, this function is case sensitive.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to sort',
        items: {
          type: 'object',
        },
      }, {
        title: 'Key',
        description: 'Key whose value to sort by',
        type: 'string',
      }, {
        type: 'string',
        title: 'Order',
        default: 'asc',
        enum: ['asc', 'desc'],
        description: 'Order to sort in (`"asc"` for ascending, `"desc"` for descending); defaults to ascending.',
      }],
      analyze: ([arrayParam, keyParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyErrors,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        return {
          schema: arraySchema,
          errors: parameterErrors.length ? [] : keyErrors.map(error => ({
            ...error, offset: keyParam.offset,
          })),
        };
      },
      execute: ([array, key, order]) => {
        if (!Array.isArray(array) || key == null) {
          return null;
        }
        if (!array.length) {
          return [];
        }
        const desc = order === 'desc';
        return (mergeSort(array, key, desc));
      },
      examples: [
        {
          expression: 'SORT_BY_KEY_CASE_SENSITIVE(array, key)',
          context: {
            array: [
              { a: 10 },
              { a: 20, b: 1 },
              { a: 20 },
              { a: 20, b: 2 },
              { a: 1 },
              { a: -1, b: 2 },
              { a: 1, b: 2 },
              { a: 3 },
              { a: -1 },
            ],
            key: 'a',
          },
          result: [
            { a: -1, b: 2 },
            { a: -1 },
            { a: 1 },
            { a: 1, b: 2 },
            { a: 3 },
            { a: 10 },
            { a: 20, b: 1 },
            { a: 20 },
            { a: 20, b: 2 },
          ],
        },
        {
          expression: 'SORT_BY_KEY_CASE_SENSITIVE(array, key, order)',
          context: {
            array: [
              { a: 10 },
              { a: 20, b: 1 },
              { a: 20 },
              { a: 20, b: 2 },
              { a: 1 },
              { a: -1, b: 2 },
              { a: 1, b: 2 },
              { a: 3 },
              { a: -1 },
            ],
            key: 'a',
            order: 'desc',
          },
          result: [
            { a: 20, b: 1 },
            { a: 20 },
            { a: 20, b: 2 },
            { a: 10 },
            { a: 3 },
            { a: 1 },
            { a: 1, b: 2 },
            { a: -1, b: 2 },
            { a: -1 },
          ],
        },
      ],
    };

  }, { "../../schema": 460 }], 123: [function (require, module, exports) {
    const comparator = (left, right, desc) => {
      if (typeof left === 'string' && typeof right === 'string') {
        const compared = desc ? right.localeCompare(left) : left.localeCompare(right);
        return compared <= 0;
      }
      return desc ? (left >= right) : (left <= right);
    };

    const merge = (left, right, desc) => {
      const result = [];
      let indexLeft = 0;
      let indexRight = 0;

      while (indexLeft < left.length && indexRight < right.length) {
        const leftVal = left[indexLeft];
        const rightVal = right[indexRight];
        if (comparator(leftVal, rightVal, desc)) {
          result.push(left[indexLeft]);
          indexLeft += 1;
        } else {
          result.push(right[indexRight]);
          indexRight += 1;
        }
      }
      return result.concat(left.slice(indexLeft)).concat(right.slice(indexRight));
    };

    const mergeSort = (arr, desc) => {
      if (arr.length === 1) {
        return arr;
      }
      const middle = Math.floor(arr.length / 2);
      const left = arr.slice(0, middle);
      const right = arr.slice(middle);
      return merge(
        mergeSort(left, desc),
        mergeSort(right, desc),
        desc,
      );
    };

    module.exports = {
      name: 'SORT_CASE_SENSITIVE',
      category: 'Array',
      title: 'Sort an array of strings or numbers (case-sensitive)',
      description: 'Returns an array containing all the values in the given array of strings or numbers, sorted in order. The function takes letter case into account.',
      parameters: [{
        title: 'Array',
        description: 'Array to sort',
        anyOf: [
          {
            type: 'array',
            items: { type: 'number' },
          },
          {
            type: 'array',
            items: { type: 'string' },
          },
        ],
      }, {
        type: 'string',
        title: 'Order',
        default: 'asc',
        enum: ['asc', 'desc'],
        description: 'Order to sort in (`"asc"` for ascending, `"desc"` for descending)',
      }],
      analyze: ([arrayParam]) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        return { schema: arraySchema };
      },
      execute: ([array, order]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        if (!array.length) {
          return [];
        }
        const desc = order === 'desc';
        return (mergeSort(array, desc));
      },
      examples: [
        {
          expression: 'SORT_CASE_SENSITIVE(array)',
          context: {
            array: [10, 1, 2, 6, -1, 10, 12],
          },
          result: [-1, 1, 2, 6, 10, 10, 12],
        },
        {
          expression: 'SORT_CASE_SENSITIVE(array, "desc")',
          context: {
            array: [10, 1, 2, 6, -1, 10, 12],
          },
          result: [12, 10, 10, 6, 2, 1, -1],
        },
        {
          expression: 'SORT_CASE_SENSITIVE(textArray)',
          context: {
            textArray: ['ABCD', 'CC', 'bb', 'AA', 'a', 'aa', 'abcd'],
          },
          result: ['a', 'aa', 'AA', 'abcd', 'ABCD', 'bb', 'CC'],
        },
      ],
    };

  }, {}], 124: [function (require, module, exports) {
    const { resolveObjectMemberAccessor, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'SUM_BY_KEY',
      category: 'Array',
      title: 'Calculate the sum of numeric key values of objects',
      description: 'Calculates the sum of all numeric key values of objects in the given array. Returns `0` if the array is empty or all key values are non-numeric.',
      parameters: [{
        type: 'array',
        title: 'Objects',
        description: 'Array of objects',
        items: {
          type: 'object',
        },
      }, {
        type: 'string',
        title: 'Key name',
        description: 'Name of the key whose values to sum',
      }],
      analyze: ([arrayParam, propNameParam], offset, parameterErrors) => {
        const itemSchema = getArrayItemSchema(arrayParam && arrayParam.schema) || {};
        const accessorSchema = (propNameParam && propNameParam.schema) || {};
        const { errors } = resolveObjectMemberAccessor(itemSchema, accessorSchema);
        return {
          schema: { type: 'number' },
          errors: parameterErrors.length ? [] : errors.map(error => ({
            ...error, offset: propNameParam.offset,
          })),
        };
      },
      execute: ([array, prop]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        return array.reduce((total, item) => {
          const value = item[prop];
          return (typeof value === 'number') ? value + total : total;
        }, 0);
      },
      examples: [
        {
          expression: 'SUM_BY_KEY(products, "price")',
          context: {
            products: [
              { name: 'Duct tape', price: 5 },
              { name: 'Swiss army knife', price: 60 },
            ],
          },
          result: 65,
        },
        {
          expression: 'SUM_BY_KEY(animals, "weight")',
          context: {
            animals: [
              { name: 'dog', weight: 123 },
              { name: 'cat', weight: 48 },
              { name: 'rat', weight: 10 },
            ],
          },
          result: 123 + 48 + 10,
        },
      ],
    };

  }, { "../../schema": 460 }], 125: [function (require, module, exports) {
    const isEqual = require('lodash/isEqual');
    const { resolveObjectMemberAccessor, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'UNIQUE_BY_KEY',
      category: 'Array',
      title: 'Find unique objects by a key',
      description: `Returns the original array where any object duplicates by the given key are removed.

For example, given an array of objects and **Key** \`id\`, would return an array where items that have duplicate \`id\` property values are removed.

The object with the first unique value is always picked into the returned array, with subsequent occurences of the same value treated as duplicates and discarded.`,
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to find objects',
        items: {
          type: 'object',
        },
      }, {
        title: 'Key',
        description: 'Key for compare',
        type: 'string',
      }],
      analyze: ([arrayParam, keyParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || {};
        const itemsSchema = getArrayItemSchema(arraySchema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyErrors,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        return {
          schema: arraySchema,
          errors: parameterErrors.length
            ? []
            : keyErrors.map(error => ({ ...error, offset: keyParam.offset })),
        };
      },
      execute: ([array, key]) => {
        if (!Array.isArray(array) || key == null) {
          return null;
        }
        const result = [];
        array.forEach((originItem) => {
          if (typeof originItem === 'object' && originItem != null) {
            const foundDuplicate = result.find(resultItem => isEqual(originItem[key], resultItem[key]));
            if (!foundDuplicate) {
              result.push(originItem);
            }
          }
        });
        return result;
      },
      examples: [
        {
          expression: 'UNIQUE_BY_KEY(array1, key)',
          context: {
            array1: [{ a: 3 }, { a: 2 }, { a: 2, b: 3 }, { a: 2, b: 5 }],
            key: 'a',
          },
          result: [{ a: 3 }, { a: 2 }],
        },
        {
          expression: 'UNIQUE_BY_KEY(array2, key)',
          context: {
            array2: [{ a: { b: 20 } }, { a: { b: 10 } }, { a: { b: 10 } }],
            key: 'a',
          },
          result: [{ a: { b: 20 } }, { a: { b: 10 } }],
        },
      ],
    };

  }, { "../../schema": 460, "lodash/isEqual": 579 }], 126: [function (require, module, exports) {
    const isEqual = require('lodash/isEqual');

    module.exports = {
      name: 'WITHOUT_ITEM',
      category: 'Array',
      title: 'Remove item from array',
      description: 'Returns an array which doesn\'t contain the given item.',
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to remove item from',
        items: {
          $id: '#item', // Make this a generic type
        },
      }, {
        $ref: '#item', // Refer to the generic type
        title: 'Item',
        description: 'Item to remove',
      }],
      analyze: ([arrayParam]) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        return {
          schema: arraySchema,
        };
      },
      execute: ([array, item]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        const pos = array.findIndex(el => isEqual(item, el));
        const splicingArray = [...array];
        if (pos > -1) {
          splicingArray.splice(pos, 1);
        }
        return splicingArray;
      },
      examples: [
        { expression: 'WITHOUT_ITEM([100, 3, 20], 3)', result: [100, 20] },
        { expression: 'WITHOUT_ITEM([[1, 2], [3, 4], [1, 2]], [1, 2])', result: [[3, 4], [1, 2]] },
        { expression: 'WITHOUT_ITEM([1, "2", 3, { a: 4 }], 3)', result: [1, '2', { a: 4 }] },
      ],
    };

  }, { "lodash/isEqual": 579 }], 127: [function (require, module, exports) {
    const { getUnionSchema, getArrayItemSchema } = require('../../schema');
    const { getSchemaAssignmentErrors } = require('../../schema');

    module.exports = {
      name: 'WITH_ITEM',
      category: 'Array',
      title: 'Add item to array',
      description: 'Returns new array which contains the given item at the end of the array. The item must match the schema of the array.',
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to add item to',
      }, {
        title: 'Item',
        description: 'Item to add',
      }],
      analyze: ([arrayParam, itemParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        const arrayItemsSchema = getArrayItemSchema(arraySchema) || {};
        const itemSchema = (itemParam && itemParam.schema) || {};
        const unionSchema = getUnionSchema([arrayItemsSchema, itemSchema]);
        const errors = getSchemaAssignmentErrors(itemSchema, arrayItemsSchema);
        return {
          schema: { type: 'array', items: unionSchema },
          errors: parameterErrors.length ? [] : errors.map(error => ({
            message: error.message, offset: itemParam.offset,
          })),
        };
      },
      execute: ([array, item]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        const updatedArray = [...array];
        if (typeof item !== 'undefined') {
          updatedArray.push(item);
        }
        return updatedArray;
      },
      examples: [
        {
          expression: 'WITH_ITEM(numbers, 4)',
          context: {
            numbers: [100, 3, 20],
          },
          result: [100, 3, 20, 4],
        },
        {
          expression: 'WITH_ITEM(animals, "pig")',
          context: {
            animals: ['cat', 'dog'],
          },
          result: ['cat', 'dog', 'pig'],
        },
        {
          expression: 'WITH_ITEM(empty, 4)',
          context: {
            empty: [],
          },
          result: [4],
        },
      ],
    };

  }, { "../../schema": 460 }], 128: [function (require, module, exports) {
    const isEqual = require('lodash/isEqual');
    const { getUnionSchema, getArrayItemSchema } = require('../../schema');
    const { getSchemaAssignmentErrors } = require('../../schema');

    module.exports = {
      name: 'WITH_UNIQUE_ITEM',
      category: 'Array',
      title: 'Add unique item to the array',
      description: 'Returns a new array which contains the given item at the end of the array, but only if the exact item does not already exist in the array. If it does, the original array is returned.',
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to add item to',
      }, {
        title: 'Item',
        description: 'Item to add',
      }],
      analyze: ([arrayParam, itemParam], offset, parameterErrors) => {
        const arraySchema = (arrayParam && arrayParam.schema) || { type: 'array' };
        const arrayItemsSchema = getArrayItemSchema(arraySchema) || {};
        const itemSchema = (itemParam && itemParam.schema) || {};
        const unionSchema = getUnionSchema([arrayItemsSchema, itemSchema]);
        const errors = getSchemaAssignmentErrors(itemSchema, arrayItemsSchema);
        return {
          schema: { type: 'array', items: unionSchema },
          errors: parameterErrors.length ? [] : errors.map(error => ({
            message: error.message, offset: itemParam.offset,
          })),
        };
      },
      execute: ([array, item]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        const updatedArray = [...array];
        if (typeof item !== 'undefined') {
          const copyIndex = updatedArray.findIndex(arrItem => (isEqual(item, arrItem)));
          if (copyIndex === -1) {
            updatedArray.push(item);
          }
        }
        return updatedArray;
      },
      examples: [
        {
          expression: 'WITH_UNIQUE_ITEM(numbers, 4)',
          context: {
            numbers: [100, 3, 20],
          },
          result: [100, 3, 20, 4],
        },
        {
          expression: 'WITH_UNIQUE_ITEM(numbers, 3)',
          context: {
            numbers: [100, 3, 20],
          },
          result: [100, 3, 20],
        },
        {
          expression: 'WITH_UNIQUE_ITEM(animals, "pig")',
          context: {
            animals: ['cat', 'dog'],
          },
          result: ['cat', 'dog', 'pig'],
        },
        {
          expression: 'WITH_UNIQUE_ITEM(empty, 4)',
          context: {
            empty: [],
          },
          result: [4],
        },
      ],
    };

  }, { "../../schema": 460, "lodash/isEqual": 579 }], 129: [function (require, module, exports) {
    const isEqual = require('lodash/isEqual');
    const { resolveObjectMemberAccessor, getUnionSchema, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'WITH_UNIQUE_ITEM_BY_KEY',
      category: 'Array',
      title: 'Add item to the array, unique by given key',
      description: 'Returns the array with the given item added, but only if the array doesn\'t already have an object whose **Key** value matches the given item\'s **Key** value.',
      parameters: [{
        type: 'array',
        title: 'Array',
        description: 'Array to add item to',
        items: {
          type: 'object',
        },
      }, {
        title: 'Item',
        description: 'Item to add',
        type: 'object',
      }, {
        title: 'Key',
        description: 'Key whose values to compare to ensure uniqueness',
        type: 'string',
      }],
      analyze: ([arrayParam, itemParam, keyParam], offset, parameterErrors) => {
        const itemsSchema = getArrayItemSchema(arrayParam && arrayParam.schema) || { type: 'object' };
        const itemSchema = (itemParam && itemParam.schema) || { type: 'object' };
        const keySchema = (keyParam && keyParam.schema) || {};
        const {
          errors: keyArrItemErrors,
        } = resolveObjectMemberAccessor(itemsSchema, keySchema);
        const {
          errors: keyNewItemErrors,
        } = resolveObjectMemberAccessor(itemSchema, keySchema);
        const unionSchema = getUnionSchema([itemsSchema, itemSchema]);
        return {
          schema: { type: 'array', items: unionSchema },
          errors: parameterErrors.length ? [] : [
            ...keyNewItemErrors.map(error => ({ ...error, offset: keyParam.offset })),
            ...keyArrItemErrors.map(error => ({ ...error, offset: keyParam.offset })),
          ],
        };
      },

      execute: ([array, item, key]) => {
        if (!Array.isArray(array)
          || item == null
          || typeof item !== 'object'
          || key == null) {
          return null;
        }
        const copyIndex = array.findIndex(el => isEqual(el[key], item[key]));

        if (copyIndex === -1) {
          return [...array, item];
        }
        return [...array];
      },
      examples: [
        {
          expression: 'WITH_UNIQUE_ITEM_BY_KEY([{ a: 3, b: 10 }, { a: 2, b: 11 }], { a: 10 } , "a")',
          result: [{ a: 3, b: 10 }, { a: 2, b: 11 }, { a: 10 }],
        },
        {
          expression: 'WITH_UNIQUE_ITEM_BY_KEY([{ a: 3, b: 10 }, { a: 2, b: 11 }], { a: 2 } , "a")',
          result: [{ a: 3, b: 10 }, { a: 2, b: 11 }],
        },
      ],
    };

  }, { "../../schema": 460, "lodash/isEqual": 579 }], 130: [function (require, module, exports) {
    const { getUnionSchema, getArrayItemSchema } = require('../../schema');

    module.exports = {
      name: 'ZIP',
      category: 'Array',
      title: 'Zip two arrays together',
      description: `Given two arrays, returns an array of arrays, where the first array contains the first items of the source arrays, the second array contains the second items and so on.

If the arrays don't have an equal number of items, the extra items are discarded.`,
      parameters: [{
        type: 'array',
        title: 'Array 1',
        description: 'First array to zip',
      }, {
        type: 'array',
        title: 'Array 2',
        description: 'Second array to zip',
      }],
      analyze: ([arrayParam1, arrayParam2]) => {
        const arrayParamSchema1 = (arrayParam1 && arrayParam1.schema) || { type: 'array' };
        const arrayParamSchema2 = (arrayParam2 && arrayParam2.schema) || { type: 'array' };
        const arrayItems1 = getArrayItemSchema(arrayParamSchema1) || {};
        const arrayItems2 = getArrayItemSchema(arrayParamSchema2) || {};
        const unionItemsSchema = getUnionSchema([arrayItems1, arrayItems2]);
        return { schema: { type: 'array', items: unionItemsSchema } };
      },
      execute: ([array1, array2]) => {
        if (array1 == null || array2 == null) {
          return null;
        }
        if (!Array.isArray(array1) || !Array.isArray(array2)) {
          return null;
        }
        const result = [];
        const args = [array1, array2];
        const length = Math.min(...args.map((arg) => {
          if (Array.isArray(arg)) {
            return arg.length;
          }
          return 0;
        }));
        for (let i = 0; i < length; i += 1) {
          const tuple = args.map(arg => arg[i]);
          result.push(tuple);
        }
        return result;
      },
      examples: [
        {
          expression: 'ZIP(numbers, numbers2)',
          context: {
            numbers: [100, 3, 20, 50],
            numbers2: [1, 2, 3],
          },
          result: [[100, 1], [3, 2], [20, 3]],
        },
        {
          expression: 'ZIP(numbers, strings)',
          context: {
            numbers: [100, 3, 20, 50],
            strings: ['a', 'b'],
          },
          result: [[100, 'a'], [3, 'b']],
        },
        {
          expression: 'ZIP(numbers, emptyArray)',
          context: {
            numbers: [100, 3, 20, 50],
            emptyArray: [],
          },
          result: [],
        },
      ],
    };

  }, { "../../schema": 460 }], 131: [function (require, module, exports) {
    module.exports = {
      name: 'BIT_AND',
      category: 'Bitwise',
      title: 'Bitwise AND',
      description: 'Returns a 1 in each bit position for which the corresponding bits of both operands are 1s.',
      parameters: [{
        type: 'number',
        title: 'Number',
        description: 'Number A',
      }, {
        type: 'number',
        title: 'Number',
        description: 'Number B',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([numA, numB]) => {
        if (typeof numA === 'number' && typeof numB === 'number'
          && Number.isFinite(numA) && Number.isFinite(numB)) {
          // eslint-disable-next-line no-bitwise
          return numA & numB;
        }
        return null;
      },
      examples: [
        { expression: 'BIT_AND(14, 9)', result: 8 },
        { expression: 'BIT_AND(23, 10)', result: 2 },
      ],
    };

  }, {}], 132: [function (require, module, exports) {
    module.exports = {
      name: 'BIT_NOT',
      category: 'Bitwise',
      title: 'Bitwise NOT',
      description: 'Inverts the bits of its operand.',
      parameters: [{
        type: 'number',
        title: 'Number',
        description: 'Number A',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([num]) => {
        if (typeof num === 'number' && Number.isFinite(num)) {
          // eslint-disable-next-line no-bitwise
          return ~num;
        }
        return null;
      },
      examples: [
        { expression: 'BIT_NOT(9)', result: -10 },
        { expression: 'BIT_NOT(10)', result: -11 },
      ],
    };

  }, {}], 133: [function (require, module, exports) {
    module.exports = {
      name: 'BIT_OR',
      category: 'Bitwise',
      title: 'Bitwise OR',
      description: 'Returns a 1 in each bit position for which the corresponding bits of either or both operands are 1s.',
      parameters: [{
        type: 'number',
        title: 'Number',
        description: 'Number A',
      }, {
        type: 'number',
        title: 'Number',
        description: 'Number B',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([numA, numB]) => {
        if (typeof numA === 'number' && typeof numB === 'number'
          && Number.isFinite(numA) && Number.isFinite(numB)) {
          // eslint-disable-next-line no-bitwise
          return numA | numB;
        }
        return null;
      },
      examples: [
        { expression: 'BIT_OR(14, 9)', result: 15 },
        { expression: 'BIT_OR(23, 10)', result: 31 },
      ],
    };

  }, {}], 134: [function (require, module, exports) {
    module.exports = {
      name: 'BIT_SHIFT_LEFT',
      category: 'Bitwise',
      title: 'Bitwise left logical shift of bits',
      description: 'Shifts a in binary representation b (< 32) bits to the left, shifting in 0s from the right',
      parameters: [{
        type: 'number',
        title: 'Number',
        description: 'Number A',
      }, {
        type: 'number',
        title: 'Number',
        description: 'Number B',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([numA, numB]) => {
        if (typeof numA === 'number' && typeof numB === 'number'
          && Number.isFinite(numA) && Number.isFinite(numB)) {
          // eslint-disable-next-line no-bitwise
          return numA << numB;
        }
        return null;
      },
      examples: [
        { expression: 'BIT_SHIFT_LEFT(14, 9)', result: 7168 },
        { expression: 'BIT_SHIFT_LEFT(23, 10)', result: 23552 },
      ],
    };

  }, {}], 135: [function (require, module, exports) {
    module.exports = {
      name: 'BIT_SHIFT_RIGHT',
      category: 'Bitwise',
      title: 'Bitwise right logical shift of bits',
      description: 'Shifts A in binary representation B (< 32) bits to the right, discarding bits shifted off, and shifting in 0s from the left. (A >>> B)',
      parameters: [{
        type: 'number',
        title: 'Number',
        description: 'Number A',
      }, {
        type: 'number',
        title: 'Number',
        description: 'Number B',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([numA, numB]) => {
        if (typeof numA === 'number' && typeof numB === 'number'
          && Number.isFinite(numA) && Number.isFinite(numB)) {
          // eslint-disable-next-line no-bitwise
          return numA >>> numB;
        }
        return null;
      },
      examples: [
        { expression: 'BIT_SHIFT_RIGHT(14, 1)', result: 7 },
        { expression: 'BIT_SHIFT_RIGHT(56, 4)', result: 3 },
      ],
    };

  }, {}], 136: [function (require, module, exports) {
    module.exports = {
      name: 'BIT_SHIFT_RIGHT_ARITH',
      category: 'Bitwise',
      title: 'Bitwise right logical shift of bits',
      description: 'Shifts A in binary representation B (< 32) bits to the right, discarding bits shifted off. (A >> B)',
      parameters: [{
        type: 'number',
        title: 'Number',
        description: 'Number A',
      }, {
        type: 'number',
        title: 'Number',
        description: 'Number B',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([numA, numB]) => {
        if (typeof numA === 'number' && typeof numB === 'number'
          && Number.isFinite(numA) && Number.isFinite(numB)) {
          // eslint-disable-next-line no-bitwise
          return numA >> numB;
        }
        return null;
      },
      examples: [
        { expression: 'BIT_SHIFT_RIGHT_ARITH(14, 1)', result: 7 },
        { expression: 'BIT_SHIFT_RIGHT_ARITH(56, 4)', result: 3 },
      ],
    };

  }, {}], 137: [function (require, module, exports) {
    module.exports = {
      name: 'BIT_XOR',
      category: 'Bitwise',
      title: 'Bitwise XOR',
      description: 'Returns a 1 in each bit position for which the corresponding bits of either but not both operands are 1s.',
      parameters: [{
        type: 'number',
        title: 'Number',
        description: 'Number A',
      }, {
        type: 'number',
        title: 'Number',
        description: 'Number B',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([numA, numB]) => {
        if (typeof numA === 'number' && typeof numB === 'number'
          && Number.isFinite(numA) && Number.isFinite(numB)) {
          // eslint-disable-next-line no-bitwise
          return numA ^ numB;
        }
        return null;
      },
      examples: [
        { expression: 'BIT_XOR(14, 9)', result: 7 },
        { expression: 'BIT_XOR(23, 10)', result: 29 },
      ],
    };

  }, {}], 138: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'ALPHA',
      category: 'Color',
      title: 'Returns the alpha component of the color',
      description: 'Returns the alpha component of the color (0…255)',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([color]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          return chroma(color).get('rgba.a');
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'ALPHA("#ff0")', result: 1 },
        { expression: 'ALPHA("rgba(10, 20, 40, 0.5)")', result: 0.5 },
        { expression: 'ALPHA("hsl(80, 20%, 40%)")', result: 1 },
        { expression: 'ALPHA("red")', result: 1 },
      ],
    };

  }, { "chroma-js": 60 }], 139: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'BLUE',
      category: 'Color',
      title: 'Returns the blue component of the color',
      description: 'Returns the blue component of the color (0…255)',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([color]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          return chroma(color).get('rgb.b');
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'BLUE("#ff0")', result: 0 },
        { expression: 'BLUE("rgb(10, 20, 40)")', result: 40 },
        { expression: 'BLUE("hsl(80, 20%, 40%)")', result: 82 },
        { expression: 'BLUE("red")', result: 0 },
      ],
    };

  }, { "chroma-js": 60 }], 140: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'COLOR',
      category: 'Color',
      title: 'Convert to color',
      description: 'Parse the given string as a color and return a string representing a valid color, '
        + 'or null if the color cannot be parsed.',
      parameters: [{
        title: 'String',
        description: 'String to parse as a color',
        type: 'string',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          // Parse and format as css
          return chroma(color).css();
        } catch (error) {
          // Could not parse
          return null;
        }
      },
      examples: [
        { expression: 'COLOR("#ff0")', result: 'rgb(255,255,0)' },
        { expression: 'COLOR("rgb(0, 20, 40)")', result: 'rgb(0,20,40)' },
        { expression: 'COLOR("hsl(80, 20%, 40%)")', result: 'rgb(109,122,82)' },
        { expression: 'COLOR("red")', result: 'rgb(255,0,0)' },
      ],
    };

  }, { "chroma-js": 60 }], 141: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'DARKEN',
      category: 'Color',
      title: 'Decreases the lightness of the color',
      description: 'Returns color with the decreased lightness component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Darkness percentage',
        description: 'Percentage to increase darkness (0...1)',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, percentage]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof percentage === 'number' && Number.isFinite(percentage) && percentage > 0) {
            const hsla = chroma(color).hsl();
            hsla[2] -= percentage;
            return chroma.hsl(hsla).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'DARKEN("#fff", 0.5)', result: 'rgb(128,128,128)' },
        { expression: 'DARKEN("rgb(10, 20, 40)", 0.5)', result: 'rgb(0,0,0)' },
        { expression: 'DARKEN("hsl(80, 20%, 40%)", 0.3)', result: 'rgb(27,31,21)' },
        { expression: 'DARKEN("red", 0.2)', result: 'rgb(153,0,0)' },
      ],
    };

  }, { "chroma-js": 60 }], 142: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'DESATURATE',
      category: 'Color',
      title: 'Decrease the saturation of the color',
      description: 'Returns color with the decreased saturation component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Saturation percentage',
        description: 'Percentage to decrease saturation (0...1)',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, percentage]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof percentage === 'number' && Number.isFinite(percentage) && percentage > 0) {
            const hsla = chroma(color).hsl();
            hsla[1] -= percentage;
            return chroma.hsl(hsla).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'DESATURATE("#0bc", 0.5)', result: 'rgb(51,145,153)' },
        { expression: 'DESATURATE("rgb(10, 20, 40)", 0.5)', result: 'rgb(22,24,28)' },
        { expression: 'DESATURATE("hsl(80, 20%, 40%)", 0.3)', result: 'rgb(98,91,113)' },
        { expression: 'DESATURATE("red", 0.2)', result: 'rgb(230,25,25)' },
      ],
    };

  }, { "chroma-js": 60 }], 143: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'FADE_IN',
      category: 'Color',
      title: 'Increase the alpha component of the color',
      description: 'Returns color with increased alpha component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Alpha',
        description: 'Alpha value to increase (0..1)',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, alpha]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof alpha === 'number' && Number.isFinite(alpha) && alpha > 0) {
            const chromaColor = chroma(color);
            const newAlpha = chromaColor.alpha() + alpha;
            return chromaColor.alpha(newAlpha).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'FADE_IN("#ff0", 0.5)', result: 'rgb(255,255,0)' },
        { expression: 'FADE_IN("rgb(10, 20, 40)", 0.5)', result: 'rgb(10,20,40)' },
        { expression: 'FADE_IN("hsl(80, 20%, 40%)", 0.5)', result: 'rgb(109,122,82)' },
        { expression: 'FADE_IN("red", 0.5)', result: 'rgb(255,0,0)' },
      ],
    };

  }, { "chroma-js": 60 }], 144: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'FADE_OUT',
      category: 'Color',
      title: 'Decrease the alpha component of the color',
      description: 'Returns color with decreased alpha component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Alpha',
        description: 'Alpha value to decrease (0..1)',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, alpha]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof alpha === 'number' && Number.isFinite(alpha) && alpha > 0) {
            const chromaColor = chroma(color);
            const newAlpha = chromaColor.alpha() - alpha;
            return chromaColor.alpha(newAlpha).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'FADE_OUT("#ff0", 0.5)', result: 'rgba(255,255,0,0.5)' },
        { expression: 'FADE_OUT("rgb(10, 20, 40)", 0.5)', result: 'rgba(10,20,40,0.5)' },
        { expression: 'FADE_OUT("hsl(80, 20%, 40%)", 0.5)', result: 'rgba(109,122,82,0.5)' },
        { expression: 'FADE_OUT("red", 0.5)', result: 'rgba(255,0,0,0.5)' },
      ],
    };

  }, { "chroma-js": 60 }], 145: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'GREEN',
      category: 'Color',
      title: 'Returns the green component of the color',
      description: 'Returns the green component of the color (0…255)',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([color]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          return chroma(color).get('rgb.g');
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'GREEN("#ff0")', result: 255 },
        { expression: 'GREEN("rgb(10, 20, 40)")', result: 20 },
        { expression: 'GREEN("hsl(80, 20%, 40%)")', result: 122 },
        { expression: 'GREEN("red")', result: 0 },
      ],
    };

  }, { "chroma-js": 60 }], 146: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'HSL',
      category: 'Color',
      title: 'Generate a color based on given of hue, saturation, lightness and alpha components',
      description: 'Generate a color based on the values of the given hue, saturation, lightness '
        + 'and alpha components. Returns a string representing a valid color, or `null` if the color '
        + 'cannot be generated.',
      parameters: [{
        title: 'Hue',
        description: 'Hue component (`0…360`)',
        type: 'number',
      }, {
        title: 'Saturation',
        description: 'Saturation component (`0…1`)',
        type: 'number',
      }, {
        title: 'Lightness',
        description: 'Lightness component (`0…1`)',
        type: 'number',
      }, {
        title: 'Alpha',
        description: 'Alpha component (`0…1`)',
        type: 'number',
        default: 1,
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([hue, saturation, lightness, alpha]) => {
        if (typeof hue !== 'number' || Number.isNaN(hue)) {
          return null;
        }
        if (typeof saturation !== 'number' || Number.isNaN(saturation)) {
          return null;
        }
        if (typeof lightness !== 'number' || Number.isNaN(lightness)) {
          return null;
        }
        let validAlpha = (typeof alpha === 'number' && !Number.isNaN(alpha)) ? alpha : 1;
        if (validAlpha < 0) {
          validAlpha = 0;
        }
        if (validAlpha > 1) {
          validAlpha = 1;
        }
        try {
          return chroma.hsl(hue, saturation, lightness, validAlpha).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'HSL(10, 0.2, 0.3)', result: 'rgb(92,66,61)' },
        { expression: 'HSL(10, 0.2, 0.3, 0.5)', result: 'rgba(92,66,61,0.5)' },
        { expression: 'HSL(500, 500, 500, 500)', result: 'rgb(255,0,255)' },
      ],
    };

  }, { "chroma-js": 60 }], 147: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'HUE',
      category: 'Color',
      title: 'Returns the hue component of the color',
      description: 'Returns the hue component of the color (0…360)',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([color]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          const result = chroma(color).get('hsl.h');
          return Number.isFinite(result) ? result : null;
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'HUE("#ff0")', result: 60 },
        { expression: 'HUE("rgb(10, 20, 40)")', result: 220 },
        { expression: 'HUE("hsl(80, 20%, 40%)")', result: 79.5 },
        { expression: 'HUE("red")', result: 0 },
      ],
    };

  }, { "chroma-js": 60 }], 148: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'LIGHTEN',
      category: 'Color',
      title: 'Increases the lightness of the color',
      description: 'Returns color with the increased lightness component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Lightness percentage',
        description: 'Percentage to increase lightness (0...1)',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, percentage]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof percentage === 'number' && Number.isFinite(percentage) && percentage > 0) {
            const hsla = chroma(color).hsl();
            hsla[2] += percentage;
            return chroma.hsl(hsla).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'LIGHTEN("#000", 0.5)', result: 'rgb(128,128,128)' },
        { expression: 'LIGHTEN("rgb(10, 20, 40)", 0.5)', result: 'rgb(91,132,214)' },
        { expression: 'LIGHTEN("hsl(80, 20%, 40%)", 0.5)', result: 'rgb(231,235,225)' },
        { expression: 'LIGHTEN("red", 0.2)', result: 'rgb(255,102,102)' },
      ],
    };

  }, { "chroma-js": 60 }], 149: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'LIGHTNESS',
      category: 'Color',
      title: 'Returns the lightness component of the color',
      description: 'Returns the lightness component of the color (0..1)',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([color]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          const result = chroma(color).get('hsl.l');
          return Number.isFinite(result) ? result : null;
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'LIGHTNESS("#ff0")', result: 0.5 },
        { expression: 'LIGHTNESS("rgb(10, 20, 40)")', result: 0.09803922 },
        { expression: 'LIGHTNESS("hsl(80, 20%, 40%)")', result: 0.4 },
        { expression: 'LIGHTNESS("red")', result: 0.5 },
      ],
    };

  }, { "chroma-js": 60 }], 150: [function (require, module, exports) {
    const chroma = require('chroma-js');

    const defaultRatio = 0.5;

    module.exports = {
      name: 'MIX',
      category: 'Color',
      title: 'Mixes the two colors together',
      description: 'Returns a color that is a mix of the given two colors. By default the color is the average of the two colors. You may optionally define a ratio how much the returned color resembles the second color.',
      parameters: [{
        title: 'Color 1',
        description: 'The first color to mix',
        type: 'string',
        format: 'color',
      }, {
        title: 'Color 2',
        description: 'The second color to mix',
        type: 'string',
        format: 'color',
      }, {
        title: 'Ratio',
        description: 'Mix ratio (0...1)',
        type: 'number',
        default: defaultRatio,
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color1, color2, ratio]) => {
        if (typeof color1 !== 'string' || typeof color2 !== 'string') {
          return null;
        }
        try {
          let validRatio = (typeof ratio === 'number' && Number.isFinite(ratio)) ? ratio : defaultRatio;
          validRatio = (validRatio < 0) ? 0 : validRatio;
          validRatio = (validRatio > 1) ? 1 : validRatio;
          return chroma.mix(color1, color2, validRatio).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'MIX("#fff", "#000", 0.5)', result: 'rgb(180,180,180)' },
        { expression: 'MIX("rgb(10, 20, 40)", "rgb(100, 100, 100)",  0.5)', result: 'rgb(71,72,76)' },
        { expression: 'MIX("hsl(80, 20%, 40%)", "hsl(80, 40%, 20%)")', result: 'rgb(87,100,62)' },
        { expression: 'MIX("red", "blue", 0.5)', result: 'rgb(180,0,180)' },
      ],
    };

  }, { "chroma-js": 60 }], 151: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'RED',
      category: 'Color',
      title: 'Returns the red component of the color',
      description: 'Returns the red component of the color (0…255)',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([color]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          return chroma(color).get('rgb.r');
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'RED("#ff0")', result: 255 },
        { expression: 'RED("rgb(10, 20, 40)")', result: 10 },
        { expression: 'RED("hsl(80, 20%, 40%)")', result: 109 },
        { expression: 'RED("red")', result: 255 },
      ],
    };

  }, { "chroma-js": 60 }], 152: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'RGB',
      category: 'Color',
      title: 'Generate color based on values of red, green, blue colors and alpha',
      description: 'Generate color based on values of red, green, blue colors and alpha '
        + 'and return a string representing a valid color, or null if the color cannot be created.',
      parameters: [{
        title: 'Red',
        description: 'Red component',
        type: 'number',
      }, {
        title: 'Green',
        description: 'Green component',
        type: 'number',
      }, {
        title: 'Blue',
        description: 'Red component',
        type: 'number',
      }, {
        title: 'Alpha',
        description: 'Alpha component',
        type: 'number',
        default: 1,
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([red, green, blue, alpha]) => {
        if (typeof red !== 'number' || Number.isNaN(red)) {
          return null;
        }
        if (typeof green !== 'number' || Number.isNaN(green)) {
          return null;
        }
        if (typeof blue !== 'number' || Number.isNaN(blue)) {
          return null;
        }
        let validAlpha = (typeof alpha === 'number' && !Number.isNaN(alpha)) ? alpha : 1;
        if (validAlpha < 0) {
          validAlpha = 0;
        }
        if (validAlpha > 1) {
          validAlpha = 1;
        }
        try {
          return chroma(red, green, blue, validAlpha).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'RGB(125, 125, 125)', result: 'rgb(125,125,125)' },
        { expression: 'RGB(125, 125, 125, 0.5)', result: 'rgba(125,125,125,0.5)' },
        { expression: 'RGB(333, -333, 0, -1)', result: 'rgba(255,0,0,0)' },
      ],
    };

  }, { "chroma-js": 60 }], 153: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'SATURATE',
      category: 'Color',
      title: 'Increase the saturation of the color',
      description: 'Returns color with the increased saturation component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Saturation percentage',
        description: 'Percentage to increase saturation (0...1)',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, percentage]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof percentage === 'number' && Number.isFinite(percentage) && percentage > 0) {
            const hsla = chroma(color).hsl();
            hsla[1] += percentage;
            return chroma.hsl(hsla).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'SATURATE("#0bc", 0.5)', result: 'rgb(0,230,255)' },
        { expression: 'SATURATE("rgb(10, 20, 40)", 0.5)', result: 'rgb(0,16,53)' },
        { expression: 'SATURATE("hsl(80, 20%, 40%)", 0.3)', result: 'rgb(120,153,51)' },
        { expression: 'SATURATE("red", 0.2)', result: 'rgb(255,0,0)' },
      ],
    };

  }, { "chroma-js": 60 }], 154: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'SATURATION',
      category: 'Color',
      title: 'Returns the saturation component of the color',
      description: 'Returns the saturation component of the color (0…1)',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([color]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          const result = chroma(color).get('hsl.s');
          return Number.isFinite(result) ? result : null;
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'SATURATION("hsl(80, 20%, 40%)")', result: 0.19607843 },
        { expression: 'SATURATION("rgb(10, 20, 40)")', result: 0.6 },
        { expression: 'SATURATION("#ff0")', result: 1 },
        { expression: 'SATURATION("red")', result: 1 },
      ],
    };

  }, { "chroma-js": 60 }], 155: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'SET_ALPHA',
      category: 'Color',
      title: 'Change the alpha component of the color',
      description: 'Returns color with the new alpha component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Alpha',
        description: 'Alpha value',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, alpha]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof alpha === 'number' && Number.isFinite(alpha)) {
            return chroma(color).alpha(alpha).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'SET_ALPHA("#ffaacc", 0.5)', result: 'rgba(255,170,204,0.5)' },
        { expression: 'SET_ALPHA("rgb(10, 20, 40)", 0.5)', result: 'rgba(10,20,40,0.5)' },
      ],
    };

  }, { "chroma-js": 60 }], 156: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'SET_BLUE',
      category: 'Color',
      title: 'Change the blue component of the color',
      description: 'Returns color with the new blue component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Blue',
        description: 'Blue color',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, blue]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof blue === 'number' && Number.isFinite(blue)) {
            const rgba = chroma(color).rgba();
            rgba[2] = blue;
            return chroma(rgba).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'SET_BLUE("rgb(10, 20, 40)", 100)', result: 'rgb(10,20,100)' },
        { expression: 'SET_BLUE("#ffaacc", 100)', result: 'rgb(255,170,100)' },
      ],
    };

  }, { "chroma-js": 60 }], 157: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'SET_GREEN',
      category: 'Color',
      title: 'Change the green component of the color',
      description: 'Returns color with the new blue component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Green',
        description: 'Green color',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, green]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof green === 'number' && Number.isFinite(green)) {
            const rgba = chroma(color).rgba();
            rgba[1] = green;
            return chroma(rgba).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'SET_GREEN("#ffaacc", 100)', result: 'rgb(255,100,204)' },
        { expression: 'SET_GREEN("rgb(10, 20, 40)", 100)', result: 'rgb(10,100,40)' },
      ],
    };

  }, { "chroma-js": 60 }], 158: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'SET_HUE',
      category: 'Color',
      title: 'Change the hue component of the color',
      description: 'Returns color with the new hue component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Hue',
        description: 'Hue component (0...360)',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, hue]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof hue === 'number' && Number.isFinite(hue)) {
            const hsla = chroma(color).hsl();
            hsla[0] = hue;
            return chroma.hsl(hsla).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'SET_HUE("rgb(10, 20, 40)", 100)', result: 'rgb(20,40,10)' },
        { expression: 'SET_HUE("#ffaacc", 100)', result: 'rgb(198,255,170)' },
      ],
    };

  }, { "chroma-js": 60 }], 159: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'SET_LIGHTNESS',
      category: 'Color',
      title: 'Change the lightness component of the color',
      description: 'Returns color with the new lightness component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Lightness',
        description: 'Lightness component (0...1)',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, lightness]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof lightness === 'number' && Number.isFinite(lightness)) {
            const hsla = chroma(color).hsl();
            hsla[2] = lightness;
            return chroma.hsl(hsla).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'SET_LIGHTNESS("#ffaacc", 0.5)', result: 'rgb(255,0,102)' },
        { expression: 'SET_LIGHTNESS("rgb(10, 20, 40)", 0.5)', result: 'rgb(51,102,204)' },
      ],
    };

  }, { "chroma-js": 60 }], 160: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'SET_RED',
      category: 'Color',
      title: 'Change the red component of the color',
      description: 'Returns color with the new red component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Red',
        description: 'Red color',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, red]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof red === 'number' && Number.isFinite(red)) {
            const rgba = chroma(color).rgba();
            rgba[0] = red;
            return chroma(rgba).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'SET_RED("rgb(10, 20, 40)", 100)', result: 'rgb(100,20,40)' },
        { expression: 'SET_RED("#ffaacc", 100)', result: 'rgb(100,170,204)' },
      ],
    };

  }, { "chroma-js": 60 }], 161: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'SET_SATURATION',
      category: 'Color',
      title: 'Change the saturation component of the color',
      description: 'Returns color with the new saturation component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Saturation',
        description: 'Saturation component (0...1)',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, saturation]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof saturation === 'number' && Number.isFinite(saturation)) {
            const hsla = chroma(color).hsl();
            hsla[1] = saturation;
            return chroma.hsl(hsla).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'SET_SATURATION("rgb(10, 20, 40)", 0.5)', result: 'rgb(12,21,38)' },
        { expression: 'SET_SATURATION("#ffaacc", 0.5)', result: 'rgb(234,191,208)' },
      ],
    };

  }, { "chroma-js": 60 }], 162: [function (require, module, exports) {
    const chroma = require('chroma-js');

    module.exports = {
      name: 'SHIFT_HUE',
      category: 'Color',
      title: 'Rotate hue with by given angle',
      description: 'Returns color with rotated hue component',
      parameters: [{
        title: 'Color',
        description: 'Color',
        type: 'string',
        format: 'color',
      }, {
        title: 'Angle',
        description: 'Angle to rotate hue',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'color' } }),
      execute: ([color, percentage]) => {
        if (typeof color !== 'string') {
          return null;
        }
        try {
          if (typeof percentage === 'number' && Number.isFinite(percentage)) {
            const hsla = chroma(color).hsl();
            hsla[0] += percentage;
            return chroma.hsl(hsla).css();
          }
          return chroma(color).css();
        } catch (error) {
          return null;
        }
      },
      examples: [
        { expression: 'SHIFT_HUE("rgb(10, 20, 40)", 90)', result: 'rgb(40,10,35)' },
        { expression: 'SHIFT_HUE("#ffaa77", 90)', result: 'rgb(136,255,119)' },
      ],
    };

  }, { "chroma-js": 60 }], 163: [function (require, module, exports) {
    const { parseDateTime, formatMoment } = require('../../datetime');

    const defaultUnit = 'milliseconds';

    module.exports = {
      name: 'ADD_DURATION',
      category: 'Date',
      title: 'Add duration to date',
      description: 'Return new date by adding units to original date.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date',
        description: 'Date',
      }, {
        type: 'number',
        title: 'Duration',
        description: 'Duration to be added to date',
      }, {
        type: 'string',
        title: 'Units',
        description: 'Units of duration("years", "months", "weeks", "quarters", "days", "hours", "minutes", "seconds", "milliseconds"). Default is "milliseconds".',
        enum: ['years', 'months', 'weeks', 'quarters', 'days', 'hours', 'minutes', 'seconds', 'milliseconds', null],
        default: defaultUnit,
      }],
      analyze: () => ({ schema: { type: 'string' } }),
      execute: ([date, duration, units]) => {
        const parsedDate = parseDateTime(date);
        if (!parsedDate.isValid()) {
          return null;
        }
        if (typeof duration === 'number'
          && Number.isFinite(duration)
          && duration > 0 && (units == null || typeof units === 'string')) {
          return formatMoment(parsedDate.add(duration, units));
        }
        return formatMoment(parsedDate);
      },
      examples: [
        { expression: 'ADD_DURATION("2019-07-11T13:06:15+03:00", 0)', result: '2019-07-11T13:06:15+03:00' },
        { expression: 'ADD_DURATION("2019-07-11T13:06:15+03:00", 1, "days")', result: '2019-07-12T13:06:15+03:00' },
      ],
    };

  }, { "../../datetime": 89 }], 164: [function (require, module, exports) {
    const moment = require('moment-timezone');
    const { formatMoment } = require('../../datetime');

    const parseDateTime = (str, dateFormat) => {
      if (typeof str === 'number') {
        return moment.tz(str, 'UTC');
      }
      if (typeof str !== 'string') {
        return moment(NaN);
      }
      if (typeof dateFormat === 'string' && !dateFormat.includes('Z')) {
        return moment.utc(str, dateFormat, true);
      }
      return moment.parseZone(str, dateFormat, true);
    };
    module.exports = {
      name: 'DATETIME',
      category: 'Date',
      title: 'Convert to date/time',
      description: 'Convert date string or number representing milliseconds since the Unix Epoch (1 January, 1970 UTC)'
        + ' to a ISO 8601 formatted string representation.',
      parameters: [{
        title: 'Timestamp',
        description: 'Date string or number of milliseconds since 1 January, 1970 UTC',
        anyOf: [{ type: 'number' }, { type: 'string' }],
      }, {
        title: 'Date Format',
        description: 'If you know the format of an input string, you can use that to parse a date',
        type: 'string',
        default: '',
      }],
      analyze: () => ({ schema: { type: 'string', format: 'date-time' } }),
      execute: ([timestamp, dateFormat]) => {
        if (timestamp == null) {
          return null;
        }
        const dateFormatSave = dateFormat || [moment.ISO_8601, moment.RFC_2822];
        if ((typeof timestamp === 'number' || typeof timestamp === 'string')
          && (typeof dateFormatSave === 'string' || Array.isArray(dateFormatSave))) {
          const dateIsParse = parseDateTime(timestamp, dateFormatSave);
          if (dateIsParse.isValid()) {
            return formatMoment(dateIsParse);
          }
          return null;
        }
        return null;
      },
      examples: [
        { expression: 'DATETIME(0)', result: '1970-01-01T00:00:00Z' },
        { expression: 'DATETIME(5000)', result: '1970-01-01T00:00:05Z' },
        { expression: 'DATETIME("2019-12-31T23:59:55+05:00")', result: '2019-12-31T23:59:55+05:00' },
      ],
    };

  }, { "../../datetime": 89, "moment-timezone": 457 }], 165: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    const defaultUnit = 'milliseconds';

    module.exports = {
      name: 'DATETIME_DIFFERENCE',
      category: 'Date',
      title: 'Difference between 2 dates',
      description: 'Difference between 2 dates in provided units',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date 1',
        description: 'Date 1',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Date 2',
        description: 'Date 2',
      }, {
        type: 'string',
        title: 'Units',
        description: 'Returned units ("years", "quarters", "months", "weeks", "days", "hours", "minutes", "seconds" , "milliseconds")',
        default: defaultUnit,
        enum: ['years', 'quarters', 'months', 'weeks', 'days', 'hours', 'minutes', 'seconds', 'milliseconds'],
      }, {
        type: 'boolean',
        title: 'In decimal',
        description: 'Prevent truncate of returned result (true, false)',
        default: false,
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([date1, date2, units, inDecimal]) => {
        const moment1 = parseDateTime(date1);
        const moment2 = parseDateTime(date2);
        if (!(moment1.isValid() && moment2.isValid())) {
          return null;
        }
        return moment1.diff(moment2, units, inDecimal);
      },
      examples: [
        { expression: 'DATETIME_DIFFERENCE("2019-07-11T13:06:15+03:00", "2019-07-11T13:06:15+03:00")', result: 0 },
        { expression: 'DATETIME_DIFFERENCE("2019-07-11T13:06:15+03:00", "2018-07-11T13:06:15+03:00", "hours")', result: 8760 },
        { expression: 'DATETIME_DIFFERENCE("2019-07-11T13:06:15+03:00", "2018-07-11T13:06:15+03:00", "weeks", true)', result: 52.14285714 },
      ],
    };

  }, { "../../datetime": 89 }], 166: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'DATETIME_IS_AFTER',
      category: 'Date',
      title: 'Check if a date is after another',
      description: 'Compare 2 dates and returns false or true.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date 1',
        description: 'Date 1',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Date 2',
        description: 'Date 2',
      }, {
        type: 'string',
        title: 'Precision',
        description: 'Precision for comparing ("year","month","week","isoWeek","day","hour","minute","second")',
        enum: ['year', 'month', 'week', 'isoWeek', 'day', 'hour', 'minute', 'second', null],
        default: null,
      }],
      analyze: () => ({ schema: { type: 'boolean' } }),
      execute: ([date1, date2, precision]) => {
        if (date1 == null || date2 == null) {
          return false;
        }
        return parseDateTime(date1).isAfter(parseDateTime(date2), precision);
      },
      examples: [
        { expression: 'DATETIME_IS_AFTER("2011-10-10T12:13:14Z", "2011-10-09T12:13:14Z")', result: true },
        { expression: 'DATETIME_IS_AFTER("2011-10-09T12:13:14Z", "2011-10-10T12:13:14Z")', result: false },
        { expression: 'DATETIME_IS_AFTER("2011-10-10T12:13:14Z", "2011-10-10T12:13:14Z")', result: false },
        { expression: 'DATETIME_IS_AFTER("2012-10-10T12:13:14Z", "2011-10-09T12:13:14Z", "year")', result: true },
      ],
    };

  }, { "../../datetime": 89 }], 167: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'DATETIME_IS_BEFORE',
      category: 'Date',
      title: 'Check if a date is before another',
      description: `Compares the given two dates. If the first date is before the second date, returns \`true\`, otherwise returns \`false\`.

By default, the comparison is done on the millisecond level. The optional **Precision** parameter allows you to override this, comparing e.g. if a date's month is before the other date's month.'`,
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date 1',
        description: 'First date',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Date 2',
        description: 'Second date',
      }, {
        type: 'string',
        title: 'Precision',
        description: 'Optional precision for the comparison, one of `year`, `month`, `week`, `isoWeek`, `day`, `hour`, `minute` or `second`',
        enum: ['year', 'month', 'week', 'isoWeek', 'day', 'hour', 'minute', 'second', null],
        default: null,
      }],
      analyze: () => ({ schema: { type: 'boolean' } }),
      execute: ([date1, date2, precision]) => {
        if (date1 == null || date2 == null) {
          return false;
        }
        return parseDateTime(date1).isBefore(parseDateTime(date2), precision);
      },
      examples: [
        { expression: 'DATETIME_IS_BEFORE("2011-10-10T12:13:14Z", "2011-10-09T12:13:14Z")', result: false },
        { expression: 'DATETIME_IS_BEFORE("2011-10-09T12:13:14Z", "2011-10-10T12:13:14Z")', result: true },
        { expression: 'DATETIME_IS_BEFORE("2011-10-10T12:13:14Z", "2011-10-10T12:13:14Z")', result: false },
        { expression: 'DATETIME_IS_BEFORE("2012-10-10T12:13:14Z", "2011-10-09T12:13:14Z", "year")', result: false },
      ],
    };

  }, { "../../datetime": 89 }], 168: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'DATETIME_IS_BETWEEN',
      category: 'Date',
      title: 'Check if a date is in range',
      description: 'Check if a date/time is between two other dates. By default returns false if the date is exactly one of the boundary date/times. You may modify this behavior by providing the "Borders type" parameter.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date',
        description: 'Date to check',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Date border 1',
        description: 'The 1st date border',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Date border 2',
        description: 'The 2nd date border',
      }, {
        type: 'string',
        title: 'Precision',
        default: null,
        enum: ['year', 'month', 'week', 'isoWeek', 'day', 'hour', 'minute', 'second', null],
        description: 'Precision for comparing ("year","month","week","isoWeek","day","hour","minute","second")',
      }, {
        type: 'string',
        title: 'Borders type',
        default: '()',
        enum: ['()', '(]', '[)', '[]'],
        description: 'Type of border inclusion ("()", "(]", "[)", "[]")',
      }],
      analyze: () => ({ schema: { type: 'boolean' } }),
      execute: ([baseDate, dateBorder1, dateBorder2, precision, bordersType]) => {
        if (baseDate == null || dateBorder1 == null || dateBorder2 == null) {
          return false;
        }
        return parseDateTime(baseDate).isBetween(
          parseDateTime(dateBorder1), parseDateTime(dateBorder2), precision, bordersType,
        );
      },
      examples: [
        { expression: 'DATETIME_IS_BETWEEN("2019-06-20T12:13:14Z", "2019-06-19T12:13:14Z", "2019-06-21T12:13:14Z")', result: true },
        { expression: 'DATETIME_IS_BETWEEN("2019-06-20T12:13:14Z", "2019-06-19T12:13:14Z", "2021-06-21T12:13:14Z", "year")', result: false },
        { expression: 'DATETIME_IS_BETWEEN("2019-06-20T12:13:14Z", "2019-06-20T12:13:14Z", "2019-06-21T12:13:14Z", null , "[]")', result: true },
      ],
    };

  }, { "../../datetime": 89 }], 169: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'DATETIME_IS_SAME',
      category: 'Date',
      title: 'Check if dates are equal',
      description: 'Compare whether or not the given two dates represent the same moments',
      parameters: [{
        title: 'Date 1',
        description: 'Date 1',
        type: 'string',
        format: 'date-time',
      }, {
        title: 'Date 2',
        description: 'Date 2',
        type: 'string',
        format: 'date-time',
      }, {
        type: 'string',
        title: 'Precision',
        default: null,
        description: 'Precision for comparing ("year","month","week","isoWeek","day","hour","minute","second", null)',
        enum: ['year', 'month', 'week', 'isoWeek', 'day', 'hour', 'minute', 'second', null],
      }],
      analyze: () => ({ schema: { type: 'boolean' } }),
      execute: ([date1, date2, precision]) => {
        if (date1 == null || date2 == null) {
          return false;
        }
        return parseDateTime(date1).isSame(parseDateTime(date2), precision);
      },
      examples: [
        { expression: 'DATETIME_IS_SAME("2011-10-10T12:13:14Z", "2011-10-09T12:13:14Z")', result: false },
        { expression: 'DATETIME_IS_SAME("2011-10-10T12:13:14Z", "2011-10-10T12:13:14Z")', result: true },
        { expression: 'DATETIME_IS_SAME("2011-10-10T12:13:14Z", "2011-10-09T12:13:14Z", "year")', result: true },
      ],
    };

  }, { "../../datetime": 89 }], 170: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'DATETIME_IS_SAME_OR_AFTER',
      category: 'Date',
      title: 'Check if a date is same or after another',
      description: 'Compare 2 dates and returns false or true.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date 1',
        description: 'Date 1',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Date 2',
        description: 'Date 2',
      }, {
        type: 'string',
        title: 'Precision',
        description: 'Precision for comparing ("year","month","week","isoWeek","day","hour","minute","second", null)',
        enum: ['year', 'month', 'week', 'isoWeek', 'day', 'hour', 'minute', 'second', null],
        default: null,
      }],
      analyze: () => ({ schema: { type: 'boolean' } }),
      execute: ([date1, date2, precision]) => {
        if (date1 == null || date2 == null) {
          return false;
        }
        return parseDateTime(date1).isSameOrAfter(parseDateTime(date2), precision);
      },
      examples: [
        { expression: 'DATETIME_IS_SAME_OR_AFTER("2011-10-10T12:13:14Z", "2011-10-09T12:13:14Z")', result: true },
        { expression: 'DATETIME_IS_SAME_OR_AFTER("2011-10-09T12:13:14Z", "2011-10-10T12:13:14Z")', result: false },
        { expression: 'DATETIME_IS_SAME_OR_AFTER("2011-10-10T12:13:14Z", "2011-10-10T12:13:14Z")', result: true },
        { expression: 'DATETIME_IS_SAME_OR_AFTER("2012-10-10T12:13:14Z", "2011-10-09T12:13:14Z", "year")', result: true },
      ],
    };

  }, { "../../datetime": 89 }], 171: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'DATETIME_IS_SAME_OR_BEFORE',
      category: 'Date',
      title: 'Check if a date is same or before another',
      description: 'Compare 2 dates and returns false or true.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date 1',
        description: 'Date 1',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Date 2',
        description: 'Date 2',
      }, {
        type: 'string',
        title: 'Precision',
        description: 'Precision for comparing ("year","month","week","isoWeek","day","hour","minute","second", null)',
        enum: ['year', 'month', 'week', 'isoWeek', 'day', 'hour', 'minute', 'second', null],
        default: null,
      }],
      analyze: () => ({ schema: { type: 'boolean' } }),
      execute: ([date1, date2, precision]) => {
        if (date1 == null || date2 == null) {
          return false;
        }
        return parseDateTime(date1).isSameOrBefore(parseDateTime(date2), precision);
      },
      examples: [
        { expression: 'DATETIME_IS_SAME_OR_BEFORE("2011-10-10T12:13:14Z", "2011-10-09T12:13:14Z")', result: false },
        { expression: 'DATETIME_IS_SAME_OR_BEFORE("2011-10-09T12:13:14Z", "2011-10-10T12:13:14Z")', result: true },
        { expression: 'DATETIME_IS_SAME_OR_BEFORE("2011-10-10T12:13:14Z", "2011-10-10T12:13:14Z")', result: true },
        { expression: 'DATETIME_IS_SAME_OR_BEFORE("2012-10-10T12:13:14Z", "2011-10-09T12:13:14Z", "year")', result: false },
      ],
    };

  }, { "../../datetime": 89 }], 172: [function (require, module, exports) {
    const moment = require('moment-timezone');
    const { parseDateTime, formatMoment } = require('../../datetime');

    module.exports = {
      name: 'FORMAT_DATETIME_LOCAL',
      category: 'Date',
      title: 'Format date in local timezone',
      description: 'Return formatted date/time string in the local timezone. If the provided date/time is in different timezone, it is convert to the local timezone.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date/time',
        description: 'Date/time to format',
      }, {
        type: 'string',
        title: 'Format',
        description: 'Format in which the date/time is returned',
      }],
      analyze: () => ({ schema: { type: 'string' } }),
      execute: ([date, format]) => {
        if (date == null || typeof date !== 'string') {
          return null;
        }
        const momentDate = parseDateTime(date);
        if (!momentDate.isValid()) {
          return null;
        }
        return formatMoment(momentDate.tz(moment.tz.guess()), format);
      },
      examples: [
        // NOTE: The examples assume the local timezone US/Central! See functions.test.js
        { expression: 'FORMAT_DATETIME_LOCAL("2019-07-11T13:06:15+03:00", "DD.MM.YYYY HH:mm:ss")', result: '11.07.2019 05:06:15' },
        { expression: 'FORMAT_DATETIME_LOCAL("2019-02-11T13:06:15Z", "DD.MM.YYYY HH:mm:ss")', result: '11.02.2019 07:06:15' },
      ],
    };

  }, { "../../datetime": 89, "moment-timezone": 457 }], 173: [function (require, module, exports) {
    const moment = require('moment-timezone');
    const { parseDateTime, formatMoment } = require('../../datetime');

    const timezones = moment.tz.names();

    module.exports = {
      name: 'FORMAT_DATETIME_WITH_TIMEZONE',
      category: 'Date',
      title: 'Format date using a time zone',
      description: 'Return formatted date string in its own zone. '
        + 'Alternatively, if a timezone name is provided as a 3rd argument, the timezone is first converted to that timezone before formatting.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date',
        description: 'Date',
      }, {
        type: 'string',
        title: 'Format',
        description: 'Format to display date',
      }, {
        type: 'string',
        title: 'Timezone',
        description: 'Timezone in which the date will be formatted. Defaults to the timezone defined in the date/time itself, or UTC if undefined.',
        enum: [null, ...timezones],
        default: null,
      }],
      analyze: () => ({ schema: { type: 'string' } }),
      execute: ([date, format, timezone]) => {
        if (date == null || typeof date !== 'string') {
          return null;
        }
        const momentDate = parseDateTime(date);
        if (!momentDate.isValid()) {
          return null;
        }
        const convertedDate = timezones.indexOf(timezone) >= 0
          ? momentDate.tz(timezone) : momentDate;
        return formatMoment(convertedDate, format);
      },
      examples: [
        { expression: 'FORMAT_DATETIME_WITH_TIMEZONE("2019-07-11T13:06:15+03:00", "DD-MM-YYYY HH:mm:ss")', result: '11-07-2019 13:06:15' },
        { expression: 'FORMAT_DATETIME_WITH_TIMEZONE("2019-07-11T13:06:15Z", "DD-MM-YYYY HH:mm:ss", "Europe/Helsinki")', result: '11-07-2019 16:06:15' },
      ],
    };

  }, { "../../datetime": 89, "moment-timezone": 457 }], 174: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'GET_DATETIME_COMPONENT',
      category: 'Date',
      title: 'Get a date component',
      description: 'Returns a component of a date. You can get one of the following components: year, month, date, hour, minute, second, or millisecond.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date',
        description: 'Date',
      }, {
        type: 'string',
        title: 'Component',
        description: 'Component to get from the date',
        enum: ['year', 'month', 'date', 'hour', 'minute', 'second', 'millisecond'],
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([date, component]) => {
        if (date == null || !component) {
          return null;
        }
        if (typeof date === 'string') {
          if (Number.isNaN((Number(date)))) {
            const parsedDate = parseDateTime(date);
            if (parsedDate.isValid()) {
              const value = parsedDate.get(component);
              if (component.toLowerCase() === 'month' || component.toLowerCase() === 'm') {
                // The month numbering starts with 1 instead of 0 in expressions
                return value + 1;
              }
              return value;
            }
          }
        }
        return null;
      },
      examples: [
        { expression: 'GET_DATETIME_COMPONENT("2012-07-11T13:06:15+03:00", "year")', result: 2012 },
        { expression: 'GET_DATETIME_COMPONENT("2012-07-11T13:06:15+03:00", "month")', result: 7 },
        { expression: 'GET_DATETIME_COMPONENT("2012-07-11T13:06:15+03:00", "date")', result: 11 },
        { expression: 'GET_DATETIME_COMPONENT("2012-07-11T13:06:15+02:00", "hour")', result: 13 },
        { expression: 'GET_DATETIME_COMPONENT("2012-07-11T13:06:15+01:00", "minute")', result: 6 },
        { expression: 'GET_DATETIME_COMPONENT("2012-07-11T13:06:15+01:00", "second")', result: 15 },
        { expression: 'GET_DATETIME_COMPONENT("2015-12-02T21:45:22.279Z", "millisecond")', result: 279 },
      ],
    };

  }, { "../../datetime": 89 }], 175: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'IS_DATETIME',
      category: 'Date',
      title: 'Check if given string is a date',
      description: 'Returns true if the given string is a date',
      parameters: [{
        title: 'Value',
        description: 'Value to check',
        type: 'string',
      }],
      analyze: () => ({ schema: { type: 'boolean' } }),
      execute: ([value]) => {
        if (typeof value === 'string') {
          return parseDateTime(value).isValid();
        }
        return false;
      },
      examples: [
        { expression: 'IS_DATETIME("2019-08-09T18:31:42+03:30")', result: true },
        { expression: 'IS_DATETIME("10-10-2019")', result: false },
        { expression: 'IS_DATETIME("2001-60-60")', result: false },
      ],
    };

  }, { "../../datetime": 89 }], 176: [function (require, module, exports) {
    const moment = require('moment-timezone');

    module.exports = {
      name: 'LOCAL_TIMEZONE',
      category: 'Date',
      title: 'Get local timezone',
      description: 'Returns the local timezone of the current environment',
      parameters: [],
      deterministic: false,
      analyze: () => ({
        schema: {
          type: 'string',
          examples: ['Europe/Helsinki', 'US/Central'],
        },
      }),
      execute: () => moment.tz.guess(),
      examples: [
        { expression: 'LOCAL_TIMEZONE()', result: 'US/Central' },
      ],
    };

  }, { "moment-timezone": 457 }], 177: [function (require, module, exports) {
    const moment = require('moment-timezone');
    const { formatMoment } = require('../../datetime');

    const exampleMoment = moment.parseZone('2019-06-27T15:46:12.240Z');

    module.exports = {
      name: 'NOW',
      category: 'Date',
      title: 'Returns the current date and time',
      description: 'Generates the current date and returns it as a formatted string.',
      deterministic: false, // Depends on the current time
      parameters: [
        {
          type: 'string',
          title: 'Format',
          description: 'Format to display date',
          default: null,
        },
      ],
      analyze: ([formatParam]) => {
        const formatSchema = formatParam && formatParam.schema;
        const schema = { type: 'string' };
        if (!formatParam) {
          schema.format = 'date-time';
        }
        let formatExamples = (formatSchema && (formatSchema.examples || formatSchema.enum)) || [];
        formatExamples = formatExamples.filter(example => typeof example === 'string');
        if (!formatExamples.length) {
          formatExamples = [null];
        }
        schema.examples = [];
        formatExamples.forEach((format) => {
          try {
            schema.examples.push(exampleMoment.format(format));
          } catch (error) {
            // Ignore formatting errors
          }
        });
        return { schema };
      },
      execute: ([format]) => {
        const now = moment().utc();
        return formatMoment(now, format);
      },
      examples: [
        { expression: 'NOW("DD-MM-YYYY")', result: '13-08-2019' },
        { expression: 'NOW("")', result: '2019-08-13T17:33:41Z' },
      ],
    };

  }, { "../../datetime": 89, "moment-timezone": 457 }], 178: [function (require, module, exports) {
    const { parseDateTime, formatMoment } = require('../../datetime');

    module.exports = {
      name: 'SET_DATETIME_COMPONENT',
      category: 'Date',
      title: 'Set a date component',
      description: 'Returns a date where the given date component has been changed to another value. You can change either year, month, date, hour, minute, second, or millisecond.'
        + '\n\nNote that te replacement is done in the timezone in which the date/time is specified!',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date',
        description: 'Date',
      }, {
        type: 'number',
        title: 'Value',
        description: 'New value for the component',
      }, {
        type: 'string',
        title: 'Component',
        description: 'Component to change',
        enum: ['year', 'month', 'date', 'hour', 'minute', 'second', 'millisecond'],
      }],
      analyze: () => ({ schema: { type: 'string', format: 'date-time' } }),
      execute: ([timestamp, value, component]) => {
        if (timestamp == null || !component) {
          return null;
        }
        if (Number.isFinite(value) && typeof value === 'number') {
          if (typeof timestamp === 'string') {
            if (Number.isNaN((Number(timestamp)))) {
              const parsedDate = parseDateTime(timestamp);
              if (parsedDate.isValid()) {
                // The month numbering starts with 1 instead of 0 in expressions
                const newValue = (component.toLowerCase() === 'month' || component.toLowerCase() === 'm')
                  ? value - 1 : value;
                return formatMoment(parsedDate.set(component, newValue));
              }
            }
          }
        }
        return null;
      },
      examples: [
        { expression: 'SET_DATETIME_COMPONENT("2012-07-11T13:06:15+03:00", 2013, "year")', result: '2013-07-11T13:06:15+03:00' },
        { expression: 'SET_DATETIME_COMPONENT("2012-07-11T13:06:15+03:00", 3, "month")', result: '2012-03-11T13:06:15+03:00' },
        { expression: 'SET_DATETIME_COMPONENT("2012-07-11T13:06:15+03:00", 5, "date")', result: '2012-07-05T13:06:15+03:00' },
        { expression: 'SET_DATETIME_COMPONENT("2012-07-11T13:06:15+02:00", 10, "hour")', result: '2012-07-11T10:06:15+02:00' },
        { expression: 'SET_DATETIME_COMPONENT("2012-07-11T13:06:15+01:00", 30, "minute")', result: '2012-07-11T13:30:15+01:00' },
        { expression: 'SET_DATETIME_COMPONENT("2015-12-02T21:45:22.279Z", 12, "millisecond")', result: '2015-12-02T21:45:22.012Z' },
      ],
    };

  }, { "../../datetime": 89 }], 179: [function (require, module, exports) {
    const moment = require('moment-timezone');
    const { parseDateTime, formatMoment } = require('../../datetime');

    const timezones = moment.tz.names();

    module.exports = {
      name: 'SET_TIMEZONE',
      category: 'Date',
      title: 'Convert date to timezone',
      description: 'Convert date to another timezone. Returns new date that represents the same moment of time, but in different timezone.',
      parameters: [{
        title: 'Date',
        description: 'Date to convert',
        type: 'string',
        format: 'date-time',
      }, {
        title: 'Timezone',
        description: 'Timezone to convert.',
        type: 'string',
        enum: timezones,
      }],
      analyze: () => ({ schema: { type: 'string', format: 'date-time' } }),
      execute: ([date, timezone]) => {
        if (date == null) {
          return null;
        }
        if (typeof date !== 'string') {
          return null;
        }
        if (!timezones.includes(timezone)) {
          return date;
        }
        return formatMoment(parseDateTime(date).tz(timezone));
      },
      examples: [
        { expression: 'SET_TIMEZONE("2019-07-11T13:06:15+03:00", "America/Los_Angeles")', result: '2019-07-11T03:06:15-07:00' },
        { expression: 'SET_TIMEZONE("2019-07-11T13:06:15+03:00", "Europe/Madrid")', result: '2019-07-11T12:06:15+02:00' },
        { expression: 'SET_TIMEZONE("2019-07-11T13:06:15+03:00", "Etc/GMT+3")', result: '2019-07-11T07:06:15-03:00' },
        { expression: 'SET_TIMEZONE("2019-07-11T13:06:15+03:00", "Etc/GMT-3")', result: '2019-07-11T13:06:15+03:00' },
      ],
    };

  }, { "../../datetime": 89, "moment-timezone": 457 }], 180: [function (require, module, exports) {
    const { parseDateTime, formatMoment } = require('../../datetime');

    const defaultUnit = 'milliseconds';

    module.exports = {
      name: 'SUBTRACT_DURATION',
      category: 'Date',
      title: 'Subtract duration from date',
      description: 'Return new date by subtracting units from original date.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date',
        description: 'Date',
      }, {
        type: 'number',
        title: 'Duration',
        description: 'Duration to be subtracted from date',
      }, {
        type: 'string',
        title: 'Units',
        description: 'Units of duration("years", "months", "weeks", "quarters", "days", "hours", "minutes", "seconds", "milliseconds"). Default is "milliseconds".',
        enum: ['years', 'months', 'weeks', 'quarters', 'days', 'hours', 'minutes', 'seconds', 'milliseconds', null],
        default: defaultUnit,
      }],
      analyze: () => ({ schema: { type: 'string' } }),
      execute: ([date, duration, units]) => {
        if (date == null) {
          return null;
        }
        const parsedDate = parseDateTime(date);
        if (!parsedDate.isValid()) {
          return null;
        }
        if (typeof duration === 'number'
          && Number.isFinite(duration)
          && duration > 0 && (units == null || typeof units === 'string')) {
          return formatMoment(parsedDate.subtract(duration, units));
        }
        return formatMoment(parsedDate);
      },
      examples: [
        { expression: 'SUBTRACT_DURATION("2019-07-11T13:06:15+03:00", 3600000)', result: '2019-07-11T12:06:15+03:00' },
        { expression: 'SUBTRACT_DURATION("2019-07-11T13:06:15+03:00", 1, "hours")', result: '2019-07-11T12:06:15+03:00' },
        { expression: 'SUBTRACT_DURATION("2019-07-11T13:06:15+03:00", 1, "months")', result: '2019-06-11T13:06:15+03:00' },
      ],
    };

  }, { "../../datetime": 89 }], 181: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'TIMESTAMP',
      category: 'Date',
      title: 'Returns the number of milliseconds from the Unix Epoch to the given date',
      description: 'For the given date, returns the number of milliseconds since the Unix Epoch (1st January, 1970 UTC).',
      deterministic: false,
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date',
        description: 'Date to convert',
        default: null,
      }],
      analyze: () => ({ schema: { type: 'number', examples: [1568190050182] } }),
      execute: ([timestamp]) => {
        if (timestamp == null) {
          const currentDate = new Date();
          return currentDate.getTime();
        }
        if (typeof timestamp === 'string') {
          if (Number.isNaN(Number(timestamp))) {
            const date = parseDateTime(timestamp);
            return date.isValid() ? date.valueOf() : null;
          }
        }
        return null;
      },
      examples: [
        { expression: 'TIMESTAMP()', result: 1568190050182 },
        { expression: 'TIMESTAMP("2019-08-09T18:31:12Z")', result: 1565375460000 },
        { expression: 'TIMESTAMP("2019-08-09T18:31:42.120-03:30")', result: 1565388102000 },
      ],
    };

  }, { "../../datetime": 89 }], 182: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'TIMEZONE',
      category: 'Date',
      title: 'Get timezone from date',
      description: 'Return timezone by given date',
      parameters: [{
        title: 'Date',
        description: 'Date to receive timezone',
        type: 'string',
        format: 'date-time',
      }],
      analyze: () => ({ schema: { type: 'string' } }),
      execute: ([date]) => {
        if (date == null) {
          return null;
        }
        const momentDate = parseDateTime(date);
        return momentDate.isValid() ? momentDate.format('Z') : null;
      },
      examples: [
        { expression: 'TIMEZONE("2019-07-11T13:06:15+03:00")', result: '+03:00' },
        { expression: 'TIMEZONE("2019-07-11T13:06:15-04:00")', result: '-04:00' },
        { expression: 'TIMEZONE("2019-07-11T13:06:15Z")', result: '+00:00' },
        { expression: 'TIMEZONE("2019-07-11T13:06:15.969Z")', result: '+00:00' },

      ],
    };

  }, { "../../datetime": 89 }], 183: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    const isLeapYear = year => new Date(year, 1, 29).getMonth() === 1;
    const daysBetween = (startDate, endDate) => Math.ceil((endDate - startDate) / 1000 / 60 / 60 / 24);
    const feb29Between = (date1, date2) => {
      const year1 = date1.get('year');
      const mar1year1 = new Date(year1, 2, 1);

      if (isLeapYear(year1) && date1 < mar1year1 && date2 >= mar1year1) {
        return true;
      }
      const year2 = date2.get('year');
      const mar1year2 = new Date(year2, 2, 1);
      return (year2 && date2 >= mar1year2 && date1 < mar1year2);
    };

    module.exports = {
      name: 'YEARFRAC',
      category: 'Date',
      title: 'Number of years between dates',
      description: 'Returns the number of years, including fractional years, between two dates using a specified day count convention.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date start',
        description: 'The start date to consider in the calculation. Must be a reference to a cell containing a date, a function returning a date type, or a number.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Date end',
        description: 'The end date to consider in the calculation. Must be a reference to a cell containing a date, a function returning a date type, or a number.',
      }, {
        type: 'string',
        title: 'Day counting convention',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([startDate, endDate, convention]) => {
        const startMoment = parseDateTime(startDate);
        const endMoment = parseDateTime(endDate);
        if (startMoment.isValid() && endMoment.isValid() && typeof convention === 'string'
          && typeof startDate === 'string' && typeof endDate === 'string') {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          if (valuesBasis.includes(basis)) {
            let sd = startMoment.get('date');
            const sm = startMoment.get('month') + 1;
            const sy = startMoment.get('year');
            let ed = endMoment.get('date');
            const em = endMoment.get('month') + 1;
            const ey = endMoment.get('year');
            const years = (ey - sy) + 1;
            const days = (new Date(ey + 1, 0, 1) - new Date(sy, 0, 1)) / 1000 / 60 / 60 / 24;
            const average = days / years;
            let ylength = 365;

            // eslint-disable-next-line default-case
            switch (basis) {
              case 'us_nasd':
                if (sd === 31 && ed === 31) {
                  sd = 30;
                  ed = 30;
                } else if (sd === 31) {
                  sd = 30;
                } else if (sd === 30 && ed === 31) {
                  ed = 30;
                }
                return Math.abs(((ed + em * 30 + ey * 360) - (sd + sm * 30 + sy * 360)) / 360);
              case 'actual':
                if ((sy === ey || ((sy + 1) === ey)) && ((sm > em) || ((sm === em) && (sd >= ed)))) {
                  if ((sy === ey && isLeapYear(sy))
                    || feb29Between(startMoment, endMoment)
                    || (em === 1 && ed === 29)) {
                    ylength = 366;
                  }
                  return Math.abs(daysBetween(startMoment, endMoment) / ylength);
                }
                return Math.abs(daysBetween(startMoment, endMoment) / average);
              case 'actual_360':
                return Math.abs(daysBetween(startMoment, endMoment) / 360);
              case 'actual_365':
                return Math.abs(daysBetween(startMoment, endMoment) / 365);
              case 'european':
                return Math.abs(((ed + em * 30 + ey * 360) - (sd + sm * 30 + sy * 360)) / 360);
            }
          }
          return null;
        }
        return null;
      },
      examples: [
        { expression: 'YEARFRAC("2010-01-22T13:06:15+03:00", "2000-01-01T13:06:15+03:00")', result: 10.05833333 },
        { expression: 'YEARFRAC("2013-01-23T13:06:15+03:00", "2004-02-03T13:06:15+03:00", "actual_360")', result: 9.10277778 },
        { expression: 'YEARFRAC("2016-01-25T13:06:15+03:00", "2008-03-07T13:06:15+03:00", "actual_365")', result: 7.89041096 },
        { expression: 'YEARFRAC("2019-01-26T13:06:15+03:00", "2012-04-09T13:06:15+03:00", "european")', result: 6.79722222 },
      ],
    };

  }, { "../../datetime": 89 }], 184: [function (require, module, exports) {
    const DATETIME = require('../date/DATETIME');

    module.exports = {
      ...DATETIME,
      name: 'DATE',
      category: 'Deprecated',
      deprecated: true,
      examples: [],
    };

  }, { "../date/DATETIME": 164 }], 185: [function (require, module, exports) {
    module.exports = {
      name: 'EVAL',
      category: 'Deprecated',
      title: 'Evaluates an evaluable returning its result',
      description: 'Executes a given special evaluable expression object with the given context and returns its result, or passing through any error.',
      // This function exists for internal usage only for now, so therefore mark it as "deprecated"
      // that discourages anyone else from using it and hides it from auto-completions etc.
      deprecated: true,
      parameters: [{
        title: 'Evaluable',
        type: 'evaluable',
        description: 'Evaluable expression object',
      }, {
        title: 'Context',
        type: 'object',
        description: 'Object whose keys are made available to the execution context',
      }],
      analyze: () => ({ schema: {} /* unknown type */ }),
      execute: ([evaluable, context]) => {
        if (evaluable == null) {
          return null;
        }
        // NOTE: Need a lazy import to avoid circular reference
        // eslint-disable-next-line global-require
        const { prepareExecutable } = require('../../runtime');
        const executable = prepareExecutable(evaluable, context);
        return executable.execute();
      },
      examples: [],
    };

  }, { "../../runtime": 459 }], 186: [function (require, module, exports) {
    const moment = require('moment-timezone');
    const { formatMoment } = require('../../datetime');

    module.exports = {
      name: 'FORMAT_DATE',
      category: 'Deprecated',
      title: 'Format date',
      description: 'Return formatted date. **This function is deprecated and should not be used any more!**',
      deprecated: true,
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date',
        description: 'Date',
      }, {
        type: 'string',
        title: 'Format',
        description: 'Format to display date',
      }],
      analyze: () => ({ schema: { type: 'string' } }),
      execute: ([date, format]) => {
        if (date == null || typeof date !== 'string') {
          return null;
        }
        const dateISORegex = /(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})[+-](\d{2}):(\d{2})/;
        const momentDate = dateISORegex.test(date)
          ? moment.parseZone(date)
          : moment.tz(date, 'Etc/Greenwich');
        return momentDate.isValid() ? formatMoment(momentDate, format) : null;
      },
      examples: [
        { expression: 'FORMAT_DATE("2019-07-11T13:06:15Z", "DD.MM.YYYY")', result: '11.07.2019' },
        { expression: 'FORMAT_DATE("2019-07-11T13:06:15Z", "DD-MM-YYYY HH:mm:ss")', result: '11-07-2019 13:06:15' },
        { expression: 'FORMAT_DATE("2019-07-11T13:06:15+03:00",null)', result: '2019-07-11T13:06:15+03:00' },
      ],
    };

  }, { "../../datetime": 89, "moment-timezone": 457 }], 187: [function (require, module, exports) {
    module.exports = {
      name: 'LOCALIZE_NUMBER',
      category: 'Deprecated',
      title: 'Returns localized number representation',
      description: 'Returns a string with a language-sensitive representation of this number.',
      deprecated: true, // Not shown in the editor, but still supported by the runtime until removed!
      parameters: [{
        type: 'number',
        title: 'Number',
        description: 'Number to localize',
      }, {
        type: 'string',
        title: 'Locale',
        description: 'A string with a BCP 47 language tag. '
          + 'Examples: "en", "hi", "de-AT", "zh-Hans-CN"',
      }, {
        type: 'object',
        title: 'Options',
        default: null,
        description: 'An object with some or all of the following properties:\n'
          + '\n- localeMatcher: The locale matching algorithm to use. Possible values are "lookup" and "best fit"; the default is "best fit".'
          + '\n- style: '
          + 'The formatting style to use. Possible values are "decimal" for plain number formatting, '
          + '"currency" for currency formatting, and "percent" for percent formatting; the default is "decimal".'
          + '\n- currency: '
          + 'The currency to use in currency formatting. '
          + 'Possible values are the ISO 4217 currency codes, such as "USD" for the US dollar, "EUR" for the euro. '
          + 'There is no default value; if the style is "currency", the currency property must be provided.'
          + '\n- currencyDisplay: '
          + 'How to display the currency in currency formatting. '
          + 'Possible values are "symbol" to use a localized currency symbol such as €, '
          + '"code" to use the ISO currency code, "name" to use a localized currency name such as "dollar"; the default is "symbol".'
          + '\n- useGrouping: '
          + 'Whether to use grouping separators, such as thousands separators or thousand/lakh/crore separators.'
          + 'Possible values are true and false; the default is true.'
          + '\n- minimumIntegerDigits: '
          + 'The minimum number of integer digits to use. Possible values are from 1 to 21; the default is 1.'
          + '\n- minimumFractionDigits: '
          + 'The minimum number of fraction digits to use. Possible values are from 0 to 20; '
          + 'the default for plain number and percent formatting is 0; '
          + 'the default for currency formatting is the number of minor unit digits provided by the ISO 4217 currency code list '
          + '(2 if the list doesn\'t provide that information).'
          + '\n- maximumFractionDigits: '
          + 'The maximum number of fraction digits to use. '
          + 'Possible values are from 0 to 20; the default for plain number formatting is the larger of minimumFractionDigits and 3; '
          + 'the default for currency formatting is the larger of minimumFractionDigits '
          + 'and the number of minor unit digits provided by the ISO 4217 currency code list '
          + '(2 if the list doesn\'t provide that information); '
          + 'the default for percent formatting is the larger of minimumFractionDigits and 0.'
          + '\n- minimumSignificantDigits: '
          + 'The minimum number of significant digits to use. Possible values are from 1 to 21; the default is 1.'
          + '\n- maximumSignificantDigits: '
          + 'The maximum number of significant digits to use. Possible values are from 1 to 21; the default is 21.',
      }],
      analyze: () => ({ schema: { type: 'string' } }),
      execute: ([value, locale, options]) => {
        if (value == null) {
          return null;
        }
        if (typeof value === 'number') {
          try {
            const opt = options || {};
            return value.toLocaleString(locale, opt);
          } catch (e) {
            return null;
          }
        }
        return null;
      },
      examples: [
        { expression: 'LOCALIZE_NUMBER(12345.67, "ru-RU")', result: '12,345.67' },
        {
          expression: 'LOCALIZE_NUMBER(value, locale, options)',
          context: {
            value: 12345.67,
            locale: 'ru-RU',
            options: {
              style: 'currency',
              currency: 'RUB',
            },
          },
          result: 'RUB 12,345.67',
        },
      ],
    };

  }, {}], 188: [function (require, module, exports) {
    const { besseli } = require('../../engineering');

    module.exports = {
      name: 'BESSELI',
      category: 'Engineering',
      title: 'BESSELI',
      description: 'Returns the modified Bessel function, which is equivalent to the Bessel function evaluated for purely imaginary arguments.',
      parameters: [{
        title: 'X',
        description: 'The value at which to evaluate the function.',
        type: 'number',
      }, {
        title: 'N',
        description: 'The order of the function. If n is not an integer, it is truncated.',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([x, n]) => {
        if (
          typeof n === 'number'
          && Number.isFinite(n)
          && typeof x === 'number'
          && Number.isFinite(x)
        ) {
          if (n < 0) {
            return null;
          }
          const result = besseli(x, n);
          return Number.isNaN(result) ? null : result;
        }
        return null;
      },
      examples: [
        { expression: 'BESSELI(1, 3)', result: 0.02216842 },
        { expression: 'BESSELI(1.5, 1)', result: 0.98166643 },
        { expression: 'BESSELI(3, 1)', result: 3.95337022 },
        { expression: 'BESSELI(2.5, 1)', result: 2.51671624 },
      ],
    };

  }, { "../../engineering": 90 }], 189: [function (require, module, exports) {
    const { besselj } = require('../../engineering');

    module.exports = {
      name: 'BESSELJ',
      category: 'Engineering',
      title: 'BESSELJ',
      description: 'Returns the Bessel function.',
      parameters: [{
        title: 'X',
        description: 'The value at which to evaluate the function.',
        type: 'number',
      }, {
        title: 'N',
        description: 'The order of the function. If n is not an integer, it is truncated.',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([x, n]) => {
        if (
          typeof n === 'number'
          && Number.isFinite(n)
          && typeof x === 'number'
          && Number.isFinite(x)
        ) {
          if (n < 0) {
            return null;
          }
          const result = besselj(x, n);
          return Number.isNaN(result) ? null : result;
        }
        return null;
      },
      examples: [
        { expression: 'BESSELJ(1, 3)', result: 0.01956335 },
        { expression: 'BESSELJ(1.5, 1)', result: 0.55793651 },
        { expression: 'BESSELJ(3, 1)', result: 0.33905896 },
        { expression: 'BESSELJ(2.5, 1)', result: 0.4970941 },
      ],
    };

  }, { "../../engineering": 90 }], 190: [function (require, module, exports) {
    const { besselk } = require('../../engineering');

    module.exports = {
      name: 'BESSELK',
      category: 'Engineering',
      title: 'BESSELK',
      description: 'Returns the modified Bessel function, which is equivalent to the Bessel functions evaluated for purely imaginary arguments.',
      parameters: [{
        title: 'X',
        description: 'The value at which to evaluate the function.',
        type: 'number',
      }, {
        title: 'N',
        description: 'The order of the function. If n is not an integer, it is truncated.',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([x, n]) => {
        if (
          typeof n === 'number'
          && Number.isFinite(n)
          && typeof x === 'number'
          && Number.isFinite(x)
        ) {
          if (x <= 0 || n < 0) {
            return null;
          }
          const result = besselk(x, n);
          return Number.isNaN(result) ? null : result;
        }
        return null;
      },
      examples: [
        { expression: 'BESSELK(1, 3)', result: 7.10126277 },
        { expression: 'BESSELK(1.5, 1)', result: 0.2773878 },
        { expression: 'BESSELK(3, 1)', result: 0.04015643 },
        { expression: 'BESSELK(2.5, 1)', result: 0.07389082 },
      ],
    };

  }, { "../../engineering": 90 }], 191: [function (require, module, exports) {
    const { bessely } = require('../../engineering');

    module.exports = {
      name: 'BESSELY',
      category: 'Engineering',
      title: 'BESSELY',
      description: 'Returns the Bessel function, which is also called the Weber function or the Neumann function.',
      parameters: [{
        title: 'X',
        description: 'The value at which to evaluate the function.',
        type: 'number',
      }, {
        title: 'N',
        description: 'The order of the function. If n is not an integer, it is truncated.',
        type: 'number',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([x, n]) => {
        if (
          typeof n === 'number'
          && Number.isFinite(n)
          && typeof x === 'number'
          && Number.isFinite(x)
        ) {
          if (x < 1 || n < 0) {
            return null;
          }
          const result = bessely(x, n);
          return Number.isNaN(result) ? null : result;
        }
        return null;
      },
      examples: [
        { expression: 'BESSELY(1, 3)', result: -5.82151763 },
        { expression: 'BESSELY(1.5, 1)', result: -0.41230863 },
        { expression: 'BESSELY(3, 1)', result: 0.32467442 },
        { expression: 'BESSELY(2.5, 1)', result: 0.145918138 },
      ],
    };

  }, { "../../engineering": 90 }], 192: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');
    const { execute: YEARFRAC } = require('../date/YEARFRAC');

    module.exports = {
      name: 'ACCRINT',
      category: 'Financial',
      title: 'Calculates the accrued interest of a security',
      description: 'Calculates the accrued interest of a security that has periodic payments.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Date start',
        description: 'The date the security was initially issued.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Date first payment',
        description: 'The first date interest will be paid.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Date end',
        description: 'The settlement date of the security, the date after issuance when the security is delivered to the buyer.',
      }, {
        type: 'number',
        title: 'Rate',
        description: 'The annualized rate of interest.',
      }, {
        type: 'number',
        title: 'Redemption',
        description: 'The redemption amount per 100 face value, or par.',
      }, {
        type: 'number',
        title: 'Frequency',
        description: 'The number of interest or coupon payments per year (1, 2, or 4).',
        enum: [1, 2, 4],
      }, {
        type: 'string',
        title: 'basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([issue, first, settlement, rate, redemption, frequency, convention]) => {
        if (typeof issue !== 'string' && typeof settlement !== 'string' && typeof first !== 'string') {
          return null;
        }
        const startDate = parseDateTime(issue);
        const endDate = parseDateTime(settlement);
        const firstDate = parseDateTime(first);
        if (startDate.isValid() && endDate.isValid() && firstDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          if (!valuesBasis.includes(basis)) {
            return null;
          }
          if (![1, 2, 4].includes(frequency)) {
            return null;
          }
          if (rate <= 0 || redemption <= 0) {
            return null;
          }
          if (endDate <= startDate) {
            return null;
          }
          return redemption * rate * YEARFRAC([startDate.format(), endDate.format(), basis]);
        }
        return null;
      },
      examples: [
        {
          context: {
            startDate: '2000-01-01T13:06:15+03:00',
            firstDate: '2009-01-22T13:06:15+03:00',
            endDate: '2010-01-22T13:06:15+03:00',
          },
          expression: 'ACCRINT(startDate, firstDate, endDate, 0.8, 200, 2)',
          result: 1609.33333333,
        },
        {
          context: {
            startDate: '2000-01-01T13:06:15+03:00',
            firstDate: '2009-01-22T13:06:15+03:00',
            endDate: '2010-01-22T13:06:15+03:00',
          },
          expression: 'ACCRINT(startDate, firstDate, endDate, 0.8, 200, 2, "us_nasd")',
          result: 1609.33333333,
        },
        {
          context: {
            startDate: '2000-01-01T13:06:15+03:00',
            firstDate: '2009-01-22T13:06:15+03:00',
            endDate: '2010-01-22T13:06:15+03:00',
          },
          expression: 'ACCRINT(startDate, firstDate, endDate, 0.8, 200, 2, "actual_365")',
          result: 1610.52054795,
        },
      ],
    };

  }, { "../../datetime": 89, "../date/YEARFRAC": 183 }], 193: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');
    const { execute: YEARFRAC } = require('../date/YEARFRAC');

    module.exports = {
      name: 'AMORLINC',
      category: 'Financial',
      title: 'Calculates AMORLINC',
      description: 'The AMORLINC function returns the depreciation for an accounting period, or the prorated depreciation if the asset was purchased in the middle of a period.This function is available for users of the French accounting system.',
      parameters: [{
        type: 'number',
        title: 'Cost',
        description: 'The asset\'s purchase cost',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Purchase date',
        description: 'The date the asset was purchased',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'First period end',
        description: 'The end date of the first period.',
      }, {
        type: 'number',
        title: 'Salvage',
        description: 'The asset\'s value at the end of its life (i.e. its salvage value)',
      }, {
        type: 'number',
        title: 'Period',
        description: 'The period for which to calculate depreciation',
      }, {
        type: 'number',
        title: 'Rate',
        description: 'The annual depreciation rate.',
      }, {
        type: 'string',
        title: 'basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([cost, purchaseDate, firstPeriodEnd, salvage, perion, rate, convention]) => {
        if (typeof purchaseDate !== 'string' && typeof firstPeriodEnd !== 'string') {
          return null;
        }
        const startDate = parseDateTime(purchaseDate);
        const endDate = parseDateTime(firstPeriodEnd);
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          if (!valuesBasis.includes(basis)) {
            return null;
          }
          if (startDate > endDate || rate <= 0
            || salvage > cost || cost <= 0
            || salvage < 0 || perion < 0) {
            return null;
          }
          if (typeof cost === 'number' && Number.isFinite(cost)
            && typeof salvage === 'number' && Number.isFinite(salvage)
            && typeof perion === 'number' && Number.isFinite(perion)
            && typeof rate === 'number' && Number.isFinite(rate)) {
            const nPer = Math.trunc(perion);
            const fOneRate = cost * rate;
            const costDelta = cost - salvage;
            const f0Rate = YEARFRAC([purchaseDate, firstPeriodEnd, basis]) * rate * cost;
            const nNumOfFullPeriods = Math.trunc((cost - salvage - f0Rate) / fOneRate);
            let fResult = 0.0;
            if (nPer === 0) {
              fResult = f0Rate;
            } else if (nPer <= nNumOfFullPeriods) {
              fResult = fOneRate;
            } else if (nPer === nNumOfFullPeriods + 1) {
              fResult = costDelta - fOneRate * nNumOfFullPeriods - f0Rate;
            }
            if (fResult > 0.0) { return fResult; }
            return 0.0;
          }
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '1969-07-20T13:06:15+03:00',
            date2: '1969-08-20T13:06:15+03:00',
          },
          expression: 'AMORLINC(1000, date1, date2, 100, 1.9, 0.15, "us_nasd")',
          result: 150,
        },
        {
          context: {
            date3: '2000-07-20T13:06:15+03:00',
            date4: '2002-08-20T13:06:15+03:00',
          },
          expression: 'AMORLINC(2000, date3, date4, 300, 1.9, 0.12, "actual")',
          result: 240,
        },
      ],
    };

  }, { "../../datetime": 89, "../date/YEARFRAC": 183 }], 194: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');
    const { execute: COUPPCD } = require('./COUPPCD');

    const daysBetween = (startDate, endDate) => Math.ceil((endDate - startDate) / 1000 / 60 / 60 / 24);

    module.exports = {
      name: 'COUPDAYBS',
      category: 'Financial',
      title: 'Calculates the number of days in the coupon period containg the settlement date',
      description: 'The COUPDAYBS function calculates the number of days in the coupon, or interest payment, period that contains the specified settlement date.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Settlement',
        description: 'The settlement date of the security, the date after issuance when the security is delivered to the buyer.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Maturity',
        description: 'The maturity or end date of the security, when it can be redeemed at face or par value.',
      }, {
        type: 'number',
        title: 'Frequency',
        description: 'The number of interest or coupon payments per year (1, 2, or 4).',
        enum: [1, 2, 4],
      }, {
        type: 'string',
        title: 'Basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({
        schema: { type: 'number' },
      }),
      execute: ([settlement, maturity, frequency, convention]) => {
        if (typeof settlement !== 'string' && typeof maturity !== 'string') {
          return null;
        }
        if (typeof frequency !== 'number' && !Number.isFinite(frequency)) {
          return null;
        }
        const startDate = parseDateTime(settlement);
        const endDate = parseDateTime(maturity);
        if (startDate >= endDate) {
          return null;
        }
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          const valuesPeriod = [1, 2, 4];
          if (!valuesBasis.includes(basis) || !valuesPeriod.includes(frequency)) {
            return null;
          }
          const aDate = COUPPCD([settlement, maturity, frequency]);
          const aDateMoment = parseDateTime(aDate);
          let sd = startDate.get('date');
          const sm = startDate.get('month') + 1;
          const sy = startDate.get('year');
          let ed = aDateMoment.get('date');
          const em = aDateMoment.get('month') + 1;
          const ey = aDateMoment.get('year');
          if (basis === 'us_nasd' || basis === 'european') {
            if (sd === 31 && ed === 31) {
              sd = 30;
              ed = 30;
            } else if (sd === 31) {
              sd = 30;
            } else if (ed === 31) {
              ed = 30;
            }
            return Math.abs((ed + em * 30 + ey * 360) - (sd + sm * 30 + sy * 360));
          }
          return daysBetween(parseDateTime(aDate), startDate);
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '2020-02-01T13:06:15+03:00',
            date2: '2023-12-31T13:06:15+03:00',
          },
          expression: 'COUPDAYBS(date1, date2, 1, "european")',
          result: 31,
        },
        {
          context: {
            date1: '2020-02-01T13:06:15+03:00',
            date2: '2023-12-31T13:06:15+03:00',
          },
          expression: 'COUPDAYBS(date1, date2, 2, "actual")',
          result: 32,
        },
        {
          context: {
            date3: '2020-01-31T13:06:15+03:00',
            date4: '2023-10-31T13:06:15+03:00',
          },
          expression: 'COUPDAYBS(date3, date4, 2, "us_nasd")',
          result: 90,
        },
        {
          context: {
            date5: '2020-01-30T13:06:15+03:00',
            date6: '2023-10-30T13:06:15+03:00',
          },
          expression: 'COUPDAYBS(date5, date6, 2)',
          result: 90,
        },
        {
          context: {
            date7: '2020-01-31T13:06:15+03:00',
            date8: '2023-10-30T13:06:15+03:00',
          },
          expression: 'COUPDAYBS(date7, date8, 4, "european")',
          result: 0,
        },
      ],
    };

  }, { "../../datetime": 89, "./COUPPCD": 199 }], 195: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');
    const { execute: COUPPCD } = require('./COUPPCD');

    const daysBetween = (startDate, endDate) => Math.ceil((endDate - startDate) / 1000 / 60 / 60 / 24);

    module.exports = {
      name: 'COUPDAYS',
      category: 'Financial',
      title: 'Calculates the number of days in the coupon period that contains the specified settlement date',
      description: 'The COUPDAYS function calculates the number of days in the coupon, or interest payment, period that contains the specified settlement date.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Settlement',
        description: 'The settlement date of the security, the date after issuance when the security is delivered to the buyer.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Maturity',
        description: 'The maturity or end date of the security, when it can be redeemed at face or par value.',
      }, {
        type: 'number',
        title: 'Frequency',
        description: 'The number of interest or coupon payments per year (1, 2, or 4).',
        enum: [1, 2, 4],
      }, {
        type: 'string',
        title: 'Basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({
        schema: { type: 'number' },
      }),
      execute: ([settlement, maturity, frequency, convention]) => {
        if (typeof settlement !== 'string' && typeof maturity !== 'string') {
          return null;
        }
        if (typeof frequency !== 'number' && !Number.isFinite(frequency)) {
          return null;
        }
        const startDate = parseDateTime(settlement);
        const endDate = parseDateTime(maturity);
        if (startDate >= endDate) {
          return null;
        }
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          const valuesPeriod = [1, 2, 4];
          if (!valuesBasis.includes(basis) || !valuesPeriod.includes(frequency)) {
            return null;
          }
          if (basis === 'actual') {
            const aDate = COUPPCD([settlement, maturity, frequency, basis]);
            const aNextDate = parseDateTime(aDate);
            aNextDate.add(12 / frequency, 'month');
            return daysBetween(parseDateTime(aDate), aNextDate);
          }
          const yLength = basis === 'actual_365' ? 365 : 360;
          return yLength / frequency;
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '2021-02-01T13:06:15+03:00',
            date2: '2023-12-31T13:06:15+03:00',
          },
          expression: 'COUPDAYS(date1, date2, 2, "actual")',
          result: 181,
        },
        {
          context: {
            date1: '2021-02-01T13:06:15+03:00',
            date2: '2023-12-31T13:06:15+03:00',
          },
          expression: 'COUPDAYS(date1, date2, 4, "actual")',
          result: 90,
        },
        {
          context: {
            date1: '2021-02-01T13:06:15+03:00',
            date2: '2023-12-31T13:06:15+03:00',
          },
          expression: 'COUPDAYS(date1, date2, 2)',
          result: 180,
        },
      ],
    };

  }, { "../../datetime": 89, "./COUPPCD": 199 }], 196: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');
    const { execute: COUPNCD } = require('./COUPNCD');
    const { execute: COUPDAYS } = require('./COUPDAYS');
    const { execute: COUPDAYBS } = require('./COUPDAYBS');

    const daysBetween = (startDate, endDate) => Math.ceil((endDate - startDate) / 1000 / 60 / 60 / 24);

    module.exports = {
      name: 'COUPDAYSNC',
      category: 'Financial',
      title: 'Calculates the number of days from the settlement date until the next coupon',
      description: 'The COUPDAYSNC function calculates the number of days from the settlement date until the next coupon, or interest payment.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Settlement',
        description: 'The settlement date of the security, the date after issuance when the security is delivered to the buyer.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Maturity',
        description: 'The maturity or end date of the security, when it can be redeemed at face or par value.',
      }, {
        type: 'number',
        title: 'Frequency',
        description: 'The number of interest or coupon payments per year (1, 2, or 4).',
        enum: [1, 2, 4],
      }, {
        type: 'string',
        title: 'Basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({
        schema: { type: 'number' },
      }),
      execute: ([settlement, maturity, frequency, convention]) => {
        if (typeof settlement !== 'string' && typeof maturity !== 'string') {
          return null;
        }
        if (typeof frequency !== 'number' && !Number.isFinite(frequency)) {
          return null;
        }
        const startDate = parseDateTime(settlement);
        const endDate = parseDateTime(maturity);
        if (startDate >= endDate) {
          return null;
        }
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          const valuesPeriod = [1, 2, 4];
          if (!valuesBasis.includes(basis) || !valuesPeriod.includes(frequency)) {
            return null;
          }
          if (basis === 'us_nasd' || basis === 'european') {
            const resultCOUPDAYS = COUPDAYS([settlement, maturity, frequency, convention]);
            const resultCOUPDAYBS = COUPDAYBS([settlement, maturity, frequency, convention]);

            return resultCOUPDAYS - resultCOUPDAYBS;
          }
          const aDate = COUPNCD([settlement, maturity, frequency]);
          return daysBetween(startDate, parseDateTime(aDate));
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '2020-02-01T13:06:15+03:00',
            date2: '2023-12-31T13:06:15+03:00',
          },
          expression: 'COUPDAYSNC(date1, date2, 2, "actual")',
          result: 150,
        },
        {
          context: {
            date3: '2020-01-31T13:06:15+03:00',
            date4: '2023-10-31T13:06:15+03:00',
          },
          expression: 'COUPDAYSNC(date3, date4, 1, "us_nasd")',
          result: 270,
        },
        {
          context: {
            date5: '2020-01-30T13:06:15+03:00',
            date6: '2023-10-30T13:06:15+03:00',
          },
          expression: 'COUPDAYSNC(date5, date6, 1)',
          result: 270,
        },
        {
          context: {
            date7: '2020-01-31T13:06:15+03:00',
            date8: '2023-10-30T13:06:15+03:00',
          },
          expression: 'COUPDAYSNC(date7, date8, 4, "european")',
          result: 90,
        },
      ],
    };

  }, { "../../datetime": 89, "./COUPDAYBS": 194, "./COUPDAYS": 195, "./COUPNCD": 197 }], 197: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'COUPNCD',
      category: 'Financial',
      title: 'Calculates next coupon date after the settlement date',
      description: 'The COUPNCD function calculates next coupon, or interest payment, date after the settlement date.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Settlement',
        description: 'The settlement date of the security, the date after issuance when the security is delivered to the buyer.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Maturity',
        description: 'The maturity or end date of the security, when it can be redeemed at face or par value.',
      }, {
        type: 'number',
        title: 'Frequency',
        description: 'The number of interest or coupon payments per year (1, 2, or 4).',
        enum: [1, 2, 4],
      }, {
        type: 'string',
        title: 'Basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({
        schema: { type: 'string', format: 'date-time' },
      }),
      execute: ([settlement, maturity, frequency, convention]) => {
        if (typeof settlement !== 'string' && typeof maturity !== 'string') {
          return null;
        }
        const startDate = parseDateTime(settlement);
        const endDate = parseDateTime(maturity);
        if (startDate >= endDate) {
          return null;
        }
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          const valuesPeriod = [1, 2, 4];
          if (!valuesBasis.includes(basis) || !valuesPeriod.includes(frequency)) {
            return null;
          }
          if (typeof frequency === 'number' && Number.isFinite(frequency)) {
            const rDate = parseDateTime(maturity);
            rDate.set('year', startDate.get('year'));
            if (rDate > startDate) {
              rDate.add(-1, 'year');
            }
            while (rDate <= startDate) {
              rDate.add(12 / frequency, 'month');
            }
            if (new Date(rDate.get('year'), rDate.get('month'), endDate.get('date')).getMonth() === rDate.get('month')) {
              rDate.set('date', endDate.get('date'));
            }
            return rDate.format();
          }
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '2010-02-01T13:06:15+03:00',
            date2: '2019-12-31T13:06:15+03:00',
          },
          expression: 'COUPNCD(date1, date2, 1)',
          result: '2010-12-31T13:06:15+03:00',
        },
        {
          context: {
            date1: '2010-02-01T13:06:15+03:00',
            date2: '2019-12-31T13:06:15+03:00',
          },
          expression: 'COUPNCD(date1, date2, 1, "us_nasd")',
          result: '2010-12-31T13:06:15+03:00',
        },
        {
          context: {
            date3: '2014-01-10T13:06:15+03:00',
            date4: '2017-05-20T13:06:15+03:00',
          },
          expression: 'COUPNCD(date3, date4, 4, "actual_360")',
          result: '2014-02-20T13:06:15+03:00',
        },
      ],
    };

  }, { "../../datetime": 89 }], 198: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');
    const { execute: COUPPCD } = require('./COUPPCD');

    module.exports = {
      name: 'COUPNUM',
      category: 'Financial',
      title: 'Calculates the number of coupons between dates',
      description: 'The COUPNUM function calculates the number of coupons, or interest payments, between the settlement date and the maturity date of the investment.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Settlement',
        description: 'The settlement date of the security, the date after issuance when the security is delivered to the buyer.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Maturity',
        description: 'The maturity or end date of the security, when it can be redeemed at face or par value.',
      }, {
        type: 'number',
        title: 'Frequency',
        description: 'The number of interest or coupon payments per year (1, 2, or 4).',
        enum: [1, 2, 4],
      }, {
        type: 'string',
        title: 'Basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({
        schema: { type: 'number' },
      }),
      execute: ([settlement, maturity, frequency, convention]) => {
        if (typeof settlement !== 'string' && typeof maturity !== 'string') {
          return null;
        }
        const startDate = parseDateTime(settlement);
        const endDate = parseDateTime(maturity);
        if (startDate >= endDate) {
          return null;
        }
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          const valuesPeriod = [1, 2, 4];
          if (!valuesBasis.includes(basis) || !valuesPeriod.includes(frequency)) {
            return null;
          }
          if (typeof frequency === 'number' && Number.isFinite(frequency)) {
            const aDate = COUPPCD([settlement, maturity, frequency]);
            const mDate = parseDateTime(aDate);
            const nMonth = (endDate.get('year') - mDate.get('year')) * 12 + endDate.get('month') - mDate.get('month');
            return nMonth * frequency / 12;
          }
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '2020-02-01T13:06:15+03:00',
            date2: '2023-12-31T13:06:15+03:00',
          },
          expression: 'COUPNUM(date1, date2, 2, "actual")',
          result: 8,
        },
        {
          context: {
            date3: '2020-01-31T13:06:15+03:00',
            date4: '2023-10-31T13:06:15+03:00',
          },
          expression: 'COUPNUM(date3, date4, 1, "us_nasd")',
          result: 4,
        },
        {
          context: {
            date5: '2020-01-30T13:06:15+03:00',
            date6: '2023-10-30T13:06:15+03:00',
          },
          expression: 'COUPNUM(date5, date6, 1)',
          result: 4,
        },
        {
          context: {
            date7: '2020-01-31T13:06:15+03:00',
            date8: '2023-10-30T13:06:15+03:00',
          },
          expression: 'COUPNUM(date7, date8, 4, "european")',
          result: 15,
        },
      ],
    };

  }, { "../../datetime": 89, "./COUPPCD": 199 }], 199: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');

    module.exports = {
      name: 'COUPPCD',
      category: 'Financial',
      title: 'Calculates last coupon date before the settlement date',
      description: 'The COUPPCD function calculates last coupon, or interest payment, date before the settlement date.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Settlement',
        description: 'The settlement date of the security, the date after issuance when the security is delivered to the buyer.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Maturity',
        description: 'The maturity or end date of the security, when it can be redeemed at face or par value.',
      }, {
        type: 'number',
        title: 'Frequency',
        description: 'The number of interest or coupon payments per year (1, 2, or 4).',
        enum: [1, 2, 4],
      }, {
        type: 'string',
        title: 'Basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({
        schema: { type: 'string', format: 'date-time' },
      }),
      execute: ([settlement, maturity, frequency, convention]) => {
        if (typeof settlement !== 'string' && typeof maturity !== 'string') {
          return null;
        }
        const startDate = parseDateTime(settlement);
        const endDate = parseDateTime(maturity);
        if (startDate >= endDate) {
          return null;
        }
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          const valuesPeriod = [1, 2, 4];
          if (!valuesBasis.includes(basis) || !valuesPeriod.includes(frequency)) {
            return null;
          }
          if (typeof frequency === 'number' && Number.isFinite(frequency)) {
            const rDate = parseDateTime(maturity);
            rDate.set('year', startDate.get('year'));
            if (rDate < startDate) {
              rDate.add(1, 'year');
            }
            while (rDate > startDate) {
              rDate.add(-12 / frequency, 'month');
            }
            if (new Date(rDate.get('year'), rDate.get('month'), endDate.get('date')).getMonth() === rDate.get('month')) {
              rDate.set('date', endDate.get('date'));
            }
            return rDate.format();
          }
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '2010-02-01T13:06:15+03:00',
            date2: '2019-12-31T13:06:15+03:00',
          },
          expression: 'COUPPCD(date1, date2, 1)',
          result: '2009-12-31T13:06:15+03:00',
        },
        {
          context: {
            date1: '2010-02-01T13:06:15+03:00',
            date2: '2019-12-31T13:06:15+03:00',
          },
          expression: 'COUPPCD(date1, date2, 1, "us_nasd")',
          result: '2009-12-31T13:06:15+03:00',
        },
        {
          context: {
            date3: '2014-01-10T13:06:15+03:00',
            date4: '2017-05-20T13:06:15+03:00',
          },
          expression: 'COUPPCD(date3, date4, 4, "actual_360")',
          result: '2013-11-20T13:06:15+03:00',
        },
      ],
    };

  }, { "../../datetime": 89 }], 200: [function (require, module, exports) {
    const { IPMT } = require('../../financial');

    module.exports = {
      name: 'CUMIPMT',
      category: 'Financial',
      title: 'CUMIPMT',
      description: 'Calculates the cumulative interest over a range of payment periods for an investment based on constant-amount periodic payments and a constant interest rate.',
      parameters: [{
        type: 'number',
        title: 'Rate',
        description: 'The interest rate',
      }, {
        type: 'number',
        title: 'Number of periods',
        description: 'The number of payments to be made',
      }, {
        type: 'number',
        title: 'Present value',
        description: 'The current value of the annuity.',
      }, {
        type: 'number',
        title: 'First period',
        description: 'The number of the payment period to begin the cumulative calculation.',
      }, {
        type: 'number',
        title: 'Last period',
        description: 'The number of the payment period to end the cumulative calculation.',
      }, {
        type: 'boolean',
        title: 'Payments at the beginning of period',
        description: 'Whether payments are due at the end (false) or beginning (true) of each period',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, periods, present, first, last, isOnBeginning]) => {
        if (rate == null || periods == null || present == null) {
          return null;
        }

        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || typeof periods !== 'number'
          || !Number.isFinite(periods)
          || typeof present !== 'number'
          || !Number.isFinite(present)
          || typeof first !== 'number'
          || !Number.isFinite(first)
          || typeof last !== 'number'
          || !Number.isFinite(last)
          || typeof isOnBeginning !== 'boolean'
        ) {
          return null;
        }
        if (rate <= 0 || periods <= 0 || present <= 0) {
          return null;
        }
        if (first < 1 || last < 1 || first > last) {
          return null;
        }
        let value = 0;
        // eslint-disable-next-line no-plusplus
        for (let per = first; per <= last; per++) {
          value += IPMT(rate, per, periods, present, 0, isOnBeginning);
        }

        return value;
      },
      examples: [
        { expression: 'CUMIPMT(0.012, 12, 100, 1, 5, false)', result: -5.05299881 },
        { expression: 'CUMIPMT(0.02, 12, 100, 1, 5, false)', result: -8.47868480 },
        { expression: 'CUMIPMT(0.01, 12, 100, 1, 5, true)', result: -3.17186904 },
      ],
    };

  }, { "../../financial": 91 }], 201: [function (require, module, exports) {
    const { PMT, FV } = require('../../financial');

    module.exports = {
      name: 'CUMPRINC',
      category: 'Financial',
      title: 'CUMPRINC',
      description: 'Calculates the cumulative principal paid over a range of payment periods for an investment based on constant-amount periodic payments and a constant interest rate.',
      parameters: [{
        type: 'number',
        title: 'Rate',
        description: 'The interest rate',
      }, {
        type: 'number',
        title: 'Number of periods',
        description: 'The number of payments to be made',
      }, {
        type: 'number',
        title: 'Present value',
        description: 'The current value of the annuity.',
      }, {
        type: 'number',
        title: 'First period',
        description: 'The number of the payment period to begin the cumulative calculation.',
      }, {
        type: 'number',
        title: 'Last period',
        description: 'The number of the payment period to end the cumulative calculation.',
      }, {
        type: 'boolean',
        title: 'Payments at the beginning of period',
        description: 'Whether payments are due at the end (false) or beginning (true) of each period',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, periods, present, first, last, isOnBeginning]) => {
        if (rate == null || periods == null || present == null) {
          return null;
        }

        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || typeof periods !== 'number'
          || !Number.isFinite(periods)
          || typeof present !== 'number'
          || !Number.isFinite(present)
          || typeof first !== 'number'
          || !Number.isFinite(first)
          || typeof last !== 'number'
          || !Number.isFinite(last)
          || typeof isOnBeginning !== 'boolean'
        ) {
          return null;
        }
        if (rate <= 0 || periods <= 0 || present <= 0) {
          return null;
        }
        if (first < 1 || last < 1 || first > last) {
          return null;
        }
        const payment = PMT(rate, periods, present, 0, isOnBeginning);
        let start = first;
        let principal = 0;
        if (first === 1) {
          if (isOnBeginning === false) {
            principal = payment + present * rate;
          } else {
            principal = payment;
          }
          // eslint-disable-next-line no-plusplus
          start++;
        }
        // eslint-disable-next-line no-plusplus
        for (let i = start; i <= last; i++) {
          if (isOnBeginning === true) {
            principal += payment - (FV(rate, i - 2, payment, present, 1) - payment) * rate;
          } else {
            principal += payment - FV(rate, i - 1, payment, present, 0) * rate;
          }
        }

        return principal;
      },
      examples: [
        { expression: 'CUMPRINC(0.01, 12, 100, 1, 5, true)', result: -40.81267981 },
        { expression: 'CUMPRINC(0.01, 12, 100, 1, 5, false)', result: -40.22080661 },
        { expression: 'CUMPRINC(0.12, 12, 100, 1, 5, false)', result: -26.32417137 },
      ],
    };

  }, { "../../financial": 91 }], 202: [function (require, module, exports) {
    module.exports = {
      name: 'CURRENCY_DECIMAL',
      category: 'Financial',
      title: 'Currency decimal',
      description: 'Converts a price quotation given as a decimal fraction into a decimal value. '
        + 'Function is used to convert from prices specified on certain securities exchanges with minimal increments '
        + '(e.g. 1/8 or 1/32) into decimal format. These prices may be transmitted on electronic exchanges as 100.01 for 100 '
        + '1/8 or 100 1/32 depending on the system and minimal increment. ',
      parameters: [{
        type: 'number',
        title: 'Fractional price',
        description: 'The price quotation given using fractional decimal conventions.',
      }, {
        type: 'number',
        title: 'Unit',
        description: 'The units of the fraction, e.g. 8 for 1/8ths or 32 for 1/32nds',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([price, unit]) => {
        if (price == null || unit == null) {
          return null;
        }
        if (typeof price !== 'number'
          || !Number.isFinite(price)
          || typeof unit !== 'number'
          || !Number.isFinite(unit)
        ) {
          return null;
        }

        const unitRounded = Math.trunc(unit);

        if (unitRounded < 1) {
          return null;
        }

        const priceInteger = Math.trunc(price);
        const converted = priceInteger
          + (price % 1) * (10 ** Math.ceil(Math.log(unitRounded) / Math.LN10))
          / unitRounded;
        const power = (10 ** (Math.ceil(Math.log(unitRounded) / Math.LN2) + 1));
        const result = Math.round(converted * power) / power;
        return Number.isFinite(result) ? result : null;
      },
      examples: [
        { expression: 'CURRENCY_DECIMAL(4.2, 4)', result: 4.5 },
        { expression: 'CURRENCY_DECIMAL(4.05, 16)', result: 4.3125 },
        { expression: 'CURRENCY_DECIMAL(4, 4)', result: 4 },
      ],
    };

  }, {}], 203: [function (require, module, exports) {
    module.exports = {
      name: 'CURRENCY_FRACTIONAL',
      category: 'Financial',
      title: 'Currency fractional',
      description: 'Converts a price quotation given as a decimal value into a decimal fraction. '
        + 'Function is is used to convert from decimal format to prices specified on certain securities exchanges with minimal increments '
        + '(e.g. 1/8 or 1/32). These prices may be transmitted on electronic exchanges as 100.01 for 100 1/8 or 100 1/32 '
        + 'depending on the system and minimal increment.',
      parameters: [{
        type: 'number',
        title: 'Decimal price',
        description: 'The price quotation given as a decimal value',
      }, {
        type: 'number',
        title: 'Unit',
        description: 'The units of the desired fraction, e.g. 8 for 1/8ths or 32 for 1/32nds',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([price, unit]) => {
        if (price == null || unit == null) {
          return null;
        }
        if (typeof price !== 'number'
          || !Number.isFinite(price)
          || typeof unit !== 'number'
          || !Number.isFinite(unit)
        ) {
          return null;
        }

        const unitRounded = Math.trunc(unit);

        if (unitRounded < 1) {
          return null;
        }

        const priceInteger = Math.trunc(price);
        const result = priceInteger
          + (price % 1) * (10 ** -Math.ceil(Math.log(unitRounded) / Math.LN10))
          * unitRounded;
        return Number.isFinite(result) ? result : null;
      },
      examples: [
        { expression: 'CURRENCY_FRACTIONAL(4.5, 4)', result: 4.2 },
        { expression: 'CURRENCY_FRACTIONAL(4.3125, 16)', result: 4.05 },
        { expression: 'CURRENCY_FRACTIONAL(4, 4)', result: 4 },
      ],
    };

  }, {}], 204: [function (require, module, exports) {
    /* eslint no-plusplus: ["error", { "allowForLoopAfterthoughts": true }] */

    const months = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12];

    module.exports = {
      name: 'DECLINING_BALANCE',
      category: 'Financial',
      title: 'Arithmetic declining balance method',
      description: 'Calculates the depreciation of an asset for a specified period using the arithmetic declining balance method.',
      parameters: [{
        type: 'number',
        title: 'Cost',
        description: 'The initial cost of the asset',
      }, {
        type: 'number',
        title: 'Salvage',
        description: 'The value of the asset at the end of depreciation',
      }, {
        type: 'number',
        title: 'Life',
        description: 'The number of periods over which the asset is depreciated',
      }, {
        type: 'number',
        title: 'Period',
        description: 'The single period within life for which to calculate depreciation',
      }, {
        type: 'number',
        title: 'Month',
        description: 'The number of months in the first year of depreciation',
        default: 12,
        enum: months,
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([cost, salvage, life, period, month]) => {
        if (cost == null || salvage == null || life == null || period == null) {
          return null;
        }

        const monthSaved = (month == null) ? 12 : month;

        if (typeof cost !== 'number'
          || !Number.isFinite(cost)
          || typeof salvage !== 'number'
          || !Number.isFinite(salvage)
          || typeof life !== 'number'
          || !Number.isFinite(life)
          || typeof period !== 'number'
          || !Number.isFinite(period)
          || typeof monthSaved !== 'number'
          || !Number.isFinite(monthSaved)
        ) {
          return null;
        }

        if (months.indexOf(monthSaved) === -1) {
          return null;
        }

        if (cost < 0 || salvage < 0 || life <= 0 || period <= 0) {
          return null;
        }

        if (period > life) {
          return null;
        }

        let balance;

        const rate = (1 - ((salvage / cost) ** (1 / life))).toFixed(3);

        const initial = cost * rate * month / 12;

        let total = initial;
        let current = 0;
        const ceiling = (period === life) ? life - 1 : period;
        for (let i = 2; i <= ceiling; i++) {
          current = (cost - total) * rate;
          total += current;
        }

        if (period === 1) {
          balance = initial;
        } if (period === life) {
          balance = (cost - total) * rate;
        } else {
          balance = current;
        }

        return Number.isFinite(balance) ? Number(balance.toFixed(2)) : null;
      },
      examples: [
        { expression: 'DECLINING_BALANCE(750, 60, 5, 4)', result: 65.28 },
        { expression: 'DECLINING_BALANCE(750, 60, 5, 5)', result: 39.37 },
        { expression: 'DECLINING_BALANCE(750, 60, 6, 5)', result: 47.78 },
        { expression: 'DECLINING_BALANCE(750, 100, 5, 4)', result: 74.22 },
        { expression: 'DECLINING_BALANCE(1000, 100, 5, 4)', result: 92.71 },
        { expression: 'DECLINING_BALANCE(1000, 100, 5, 4, 12)', result: 92.71 },
        { expression: 'DECLINING_BALANCE(1000, 100, 5, 4, 1)', result: 142.40 },
      ],
    };

  }, {}], 205: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');
    const { execute: YEARFRAC } = require('../date/YEARFRAC');

    module.exports = {
      name: 'DISC',
      category: 'Financial',
      title: 'Calculates the discount rate of a security',
      description: 'The DISC function calculates the discount rate of a security based on price.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Settlement',
        description: 'The settlement date of the security, the date after issuance when the security is delivered to the buyer.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Maturity',
        description: 'The maturity or end date of the security, when it can be redeemed at face or par value.',
      }, {
        type: 'number',
        title: 'Price',
        description: 'The price at which the security is bought.',
      }, {
        type: 'number',
        title: 'Redemption',
        description: 'The redemption value of the security.',
      }, {
        type: 'string',
        title: 'Basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({
        schema: { type: 'number' },
      }),
      execute: ([settlement, maturity, price, redemption, convention]) => {
        if (typeof settlement !== 'string' && typeof maturity !== 'string') {
          return null;
        }
        if (typeof price !== 'number' && !Number.isFinite(price)
          && typeof redemption !== 'number' && !Number.isFinite(redemption)) {
          return null;
        }
        if (price <= 0 || redemption <= 0) {
          return null;
        }
        const startDate = parseDateTime(settlement);
        const endDate = parseDateTime(maturity);
        if (startDate >= endDate) {
          return null;
        }
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          if (!valuesBasis.includes(basis)) {
            return null;
          }
          return (1.0 - price / redemption) / YEARFRAC([settlement, maturity, basis]);
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '2010-01-02T13:06:15+03:00',
            date2: '2039-12-31T13:06:15+03:00',
          },
          expression: 'DISC(date1, date2, 90, 100, "us_nasd")',
          result: 0.00333364,
        },
        {
          context: {
            date1: '2010-01-02T13:06:15+03:00',
            date2: '2039-12-31T13:06:15+03:00',
          },
          expression: 'DISC(date1, date2, 90, 100)',
          result: 0.00333364,
        },
        {
          context: {
            date3: '2009-01-31T13:06:15+03:00',
            date4: '2023-10-31T13:06:15+03:00',
          },
          expression: 'DISC(date3, date4, 280, 300, "us_nasd")',
          result: 0.00451977,
        },
        {
          context: {
            date5: '2009-01-30T13:06:15+03:00',
            date6: '2023-10-30T13:06:15+03:00',
          },
          expression: 'DISC(date5, date6, 280, 300, "actual_360")',
          result: 0.004456,
        },
        {
          context: {
            date7: '2009-01-31T13:06:15+03:00',
            date8: '2023-10-30T13:06:15+03:00',
          },
          expression: 'DISC(date7, date8, 250, 300, "actual")',
          result: 0.011303,
        },
      ],
    };

  }, { "../../datetime": 89, "../date/YEARFRAC": 183 }], 206: [function (require, module, exports) {
    /* eslint no-plusplus: ["error", { "allowForLoopAfterthoughts": true }] */

    module.exports = {
      name: 'DOUBLE_DECLINING_BALANCE',
      category: 'Financial',
      title: 'Double-declining balance method',
      description: 'Calculates the depreciation of an asset for a specified period using the double-declining balance method.',
      parameters: [{
        type: 'number',
        title: 'Cost',
        description: 'The initial cost of the asset',
      }, {
        type: 'number',
        title: 'Salvage',
        description: 'The value of the asset at the end of depreciation',
      }, {
        type: 'number',
        title: 'Life',
        description: 'The number of periods over which the asset is depreciated',
      }, {
        type: 'number',
        title: 'Period',
        description: 'The single period within life for which to calculate depreciation',
      }, {
        type: 'number',
        title: 'Factor',
        description: 'The factor by which depreciation decreases (default is 2)',
        default: 2,
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([cost, salvage, life, period, factor]) => {
        if (cost == null || salvage == null || life == null || period == null) {
          return null;
        }

        if (typeof cost !== 'number'
          || !Number.isFinite(cost)
          || typeof salvage !== 'number'
          || !Number.isFinite(salvage)
          || typeof life !== 'number'
          || !Number.isFinite(life)
          || typeof period !== 'number'
          || !Number.isFinite(period)
          || typeof factor !== 'number'
          || !Number.isFinite(factor)
        ) {
          return null;
        }

        if (cost < 0 || salvage < 0 || life <= 0 || period <= 0 || factor <= 0) {
          return null;
        }

        if (period > life) {
          return null;
        }

        let total = 0;
        let current = 0;
        for (let i = 1; i <= period; i++) {
          current = Math.min((cost - total) * (factor / life), (cost - salvage - total));
          total += current;
        }

        return Number.isFinite(current) ? Number(current.toFixed(2)) : null;
      },
      examples: [
        { expression: 'DOUBLE_DECLINING_BALANCE(750, 60, 5, 4)', result: 64.8 },
        { expression: 'DOUBLE_DECLINING_BALANCE(750, 60, 5, 5)', result: 37.20 },
        { expression: 'DOUBLE_DECLINING_BALANCE(750, 60, 6, 5)', result: 49.38 },
        { expression: 'DOUBLE_DECLINING_BALANCE(750, 100, 5, 4)', result: 62 },
        { expression: 'DOUBLE_DECLINING_BALANCE(1000, 100, 5, 4)', result: 86.40 },
        { expression: 'DOUBLE_DECLINING_BALANCE(1000, 100, 5, 4, 2)', result: 86.40 },
        { expression: 'DOUBLE_DECLINING_BALANCE(1000, 100, 5, 4, 1)', result: 102.40 },
      ],
    };

  }, {}], 207: [function (require, module, exports) {
    module.exports = {
      name: 'EFFECT',
      category: 'Financial',
      title: 'Annual effective interest rate',
      description: 'Calculates the annual effective interest rate given the nominal rate and number of compounding periods per year.',
      parameters: [{
        type: 'number',
        title: 'Nominal rate',
        description: 'The nominal interest rate per year',
      }, {
        type: 'number',
        title: 'Periods per year',
        description: 'The number of compounding periods per year',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, periods]) => {
        if (rate == null || periods == null) {
          return null;
        }
        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || typeof periods !== 'number'
          || !Number.isFinite(periods)
        ) {
          return null;
        }

        const periodsRounded = Math.trunc(periods);

        if (periodsRounded < 1 || rate <= 0) {
          return null;
        }
        const result = ((1 + rate / periodsRounded) ** periodsRounded) - 1;
        return Number.isFinite(result) ? result : null;
      },
      examples: [
        { expression: 'EFFECT(0.001, 4)', result: 0.001000375063 },
        { expression: 'EFFECT(0.1, 4)', result: 0.1038128906 },
        { expression: 'EFFECT(0.5, 4)', result: 0.6018066406 },
        { expression: 'EFFECT(1, 4)', result: 1.44140625 },
        { expression: 'EFFECT(0.5, 6)', result: 0.6164885679 },
      ],
    };

  }, {}], 208: [function (require, module, exports) {
    const { FV: futureValue } = require('../../financial');

    module.exports = {
      name: 'FUTURE_VALUE',
      category: 'Financial',
      title: 'Future value',
      description: 'Calculates the future value of an annuity investment based on constant-amount periodic payments and a constant interest rate',
      parameters: [{
        type: 'number',
        title: 'Rate',
        description: 'The interest rate',
      }, {
        type: 'number',
        title: 'Periods',
        description: 'The number of payments to be made',
      }, {
        type: 'number',
        title: 'Payment amount',
        description: 'The amount per period to be paid',
      }, {
        type: 'number',
        title: 'Present value',
        description: 'The current value of the annuity. Default is 0.',
        default: 0,
      }, {
        type: 'boolean',
        title: 'Payments at the beginning of period',
        description: 'Whether payments are due at the end (false) or beginning (true) of each period',
        default: false,
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, periods, amount, current, isOnBeginning]) => {
        if (rate == null || periods == null || amount === null) {
          return null;
        }
        const currentSaved = current || 0;
        const isOnBeginningSaved = isOnBeginning || false;

        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || typeof periods !== 'number'
          || !Number.isFinite(periods)
          || typeof amount !== 'number'
          || !Number.isFinite(amount)
          || typeof currentSaved !== 'number'
          || !Number.isFinite(currentSaved)
          || typeof isOnBeginningSaved !== 'boolean'
        ) {
          return null;
        }

        const result = futureValue(rate, periods, amount, currentSaved, isOnBeginningSaved);

        if (Number.isFinite(result)) {
          return Number(result.toFixed(2));
        }
        return null;
      },
      examples: [
        { expression: 'FUTURE_VALUE(0.1, 4, 100)', result: -464.10 },
        { expression: 'FUTURE_VALUE(0.5, 4, 100)', result: -812.50 },
        { expression: 'FUTURE_VALUE(0.5, 6, 100)', result: -2078.13 },
        { expression: 'FUTURE_VALUE(0.5, 6, 100, 200)', result: -4356.25 },
        { expression: 'FUTURE_VALUE(0.5, 6, 100, 200, true)', result: -5395.31 },
      ],
    };

  }, { "../../financial": 91 }], 209: [function (require, module, exports) {
    module.exports = {
      name: 'FUTURE_VALUE_SCHEDULE',
      category: 'Financial',
      title: 'Future value by schedule',
      description: 'Calculates the future value of some principal based on a specified series of potentially varying interest rates.',
      parameters: [{
        type: 'number',
        title: 'Principal',
        description: 'The amount of initial capital or value to compound against',
      }, {
        type: 'array',
        title: 'Rate schedule',
        description: 'A series of interest rates to compound against the principal',
        items: {
          type: 'number',
        },
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([principal, schedule]) => {
        if (principal == null || schedule == null) {
          return null;
        }
        if (typeof principal !== 'number'
          || !Number.isFinite(principal)
          || !Array.isArray(schedule)
        ) {
          return null;
        }
        const scheduleValid = schedule.filter(item => (typeof item === 'number' && Number.isFinite(item)));
        const result = scheduleValid.reduce((total, item) => total * (1 + item), principal);
        return Number.isFinite(result) ? result : null;
      },
      examples: [
        { expression: 'FUTURE_VALUE_SCHEDULE(1, [1, 2, 3, 4, 5])', result: 720 },
        { expression: 'FUTURE_VALUE_SCHEDULE(2, [1, 2, 3, 4, 5])', result: 1440 },
        { expression: 'FUTURE_VALUE_SCHEDULE(-1, [1, 2, 3, 4, 5])', result: -720 },
        { expression: 'FUTURE_VALUE_SCHEDULE(1, [1, -2, 3, 4, 5])', result: -240 },
      ],
    };

  }, {}], 210: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');
    const { execute: YEARFRAC } = require('../date/YEARFRAC');

    module.exports = {
      name: 'INTRATE',
      category: 'Financial',
      title: 'Calculates the effective interest rate',
      description: 'The INTRATE function calculates the effective interest rate generated when an investment is purchased at one price and sold at another with no interest or dividends generated by the investment itself.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Buy date',
        description: 'The date of purchase of the investment.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Sell date',
        description: 'The date of sale of the investment.',
      }, {
        type: 'number',
        title: 'Buy price',
        description: 'The price at which the investment was purchased.',
      }, {
        type: 'number',
        title: 'Sell price',
        description: 'The price at which the investment was sold.',
      }, {
        type: 'string',
        title: 'Basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({
        schema: { type: 'number' },
      }),
      execute: ([buyDate, sellDate, buyPrice, sellPrice, convention]) => {
        if (typeof buyDate !== 'string' && typeof sellDate !== 'string') {
          return null;
        }
        if (typeof buyPrice !== 'number' && !Number.isFinite(buyPrice)
          && typeof sellPrice !== 'number' && !Number.isFinite(sellPrice)) {
          return null;
        }
        if (buyPrice <= 0 || sellPrice <= 0) {
          return null;
        }
        const startDate = parseDateTime(buyDate);
        const endDate = parseDateTime(sellDate);
        if (startDate >= endDate) {
          return null;
        }
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          if (!valuesBasis.includes(basis)) {
            return null;
          }
          const result = ((sellPrice / buyPrice) - 1) / YEARFRAC([buyDate, sellDate, basis]);
          return result;
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '2010-01-02T13:06:15+03:00',
            date2: '2019-12-31T13:06:15+03:00',
          },
          expression: 'INTRATE(date1, date2, 90, 140, "us_nasd")',
          result: 0.05557099,
        },
        {
          context: {
            date1: '2010-01-02T13:06:15+03:00',
            date2: '2019-12-31T13:06:15+03:00',
          },
          expression: 'INTRATE(date1, date2, 90, 140)',
          result: 0.05557099,
        },
        {
          context: {
            date1: '2010-01-02T13:06:15+03:00',
            date2: '2019-12-31T13:06:15+03:00',
          },
          expression: 'INTRATE(date1, date2, 90, 140, "actual")',
          result: 0.055586,
        },
        {
          context: {
            date3: '2009-01-31T13:06:15+03:00',
            date4: '2023-10-31T13:06:15+03:00',
          },
          expression: 'INTRATE(date3, date4, 290, 300, "us_nasd")',
          result: 0.00233781,
        },
        {
          context: {
            date5: '2009-01-30T13:06:15+03:00',
            date6: '2023-10-30T13:06:15+03:00',
          },
          expression: 'INTRATE(date5, date6, 290, 300, "actual_360")',
          result: 0.00230483,
        },
        {
          context: {
            date7: '2009-01-31T13:06:15+03:00',
            date8: '2023-10-30T13:06:15+03:00',
          },
          expression: 'INTRATE(date7, date8, 290, 300, "actual")',
          result: 0.00233855,
        },
      ],
    };

  }, { "../../datetime": 89, "../date/YEARFRAC": 183 }], 211: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');
    const { execute: YEARFRAC } = require('../date/YEARFRAC');
    const { execute: COUPNUM } = require('./COUPNUM');

    module.exports = {
      name: 'INVESTMENT_DURATION',
      category: 'Financial',
      title: 'Calculates the number of compounding periods required for an investment',
      description: 'The INVESTMENT_DURATION function calculates the number of compounding periods required for an investment of a specified present value appreciating at a given rate to reach a target value.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Settlement',
        description: 'The settlement date of the security, the date after issuance when the security is delivered to the buyer.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Maturity',
        description: 'The maturity or end date of the security, when it can be redeemed at face or par value.',
      }, {
        type: 'number',
        title: 'Rate',
        description: 'The annualized rate of interest.',
      }, {
        type: 'number',
        title: 'Yield',
        description: 'The expected annual yield of the security.',
      }, {
        type: 'number',
        title: 'Frequency',
        description: 'The number of interest or coupon payments per year (1, 2, or 4).',
        enum: [1, 2, 4],
      }, {
        type: 'string',
        title: 'Basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({
        schema: { type: 'number' },
      }),
      execute: ([settlement, maturity, rate, fYield, frequency, convention]) => {
        if (typeof settlement !== 'string' && typeof maturity !== 'string') {
          return null;
        }
        if (typeof frequency !== 'number' && !Number.isFinite(frequency)
          && typeof rate !== 'number' && !Number.isFinite(rate)
          && typeof fYield !== 'number' && !Number.isFinite(fYield)) {
          return null;
        }
        if (rate < 0 || fYield < 0) {
          return null;
        }
        const startDate = parseDateTime(settlement);
        const endDate = parseDateTime(maturity);
        if (startDate >= endDate) {
          return null;
        }
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          const valuesPeriod = [1, 2, 4];
          if (!valuesBasis.includes(basis) || !valuesPeriod.includes(frequency)) {
            return null;
          }
          const fYearfrac = YEARFRAC([settlement, maturity, basis]);
          const fNumOfCoups = COUPNUM([settlement, maturity, frequency, basis]);
          let fDur = 0;
          const f100 = 100;
          const fCoup = rate * f100 / frequency;
          let funcYield = fYield / frequency;
          funcYield += 1;
          const nDiff = fYearfrac * frequency - fNumOfCoups;
          // eslint-disable-next-line no-plusplus
          for (let t = 1; t < fNumOfCoups; t++) {
            fDur += (t + nDiff) * fCoup / (funcYield ** (t + nDiff));
          }
          fDur += (fNumOfCoups + nDiff) * (fCoup + f100) / (funcYield ** (fNumOfCoups + nDiff));
          let p = 0;
          // eslint-disable-next-line no-plusplus
          for (let t = 1; t < fNumOfCoups; t++) {
            p += fCoup / (funcYield ** (t + nDiff));
          }
          p += (fCoup + f100) / (funcYield ** (fNumOfCoups + nDiff));
          fDur /= p;
          fDur /= (frequency);

          return fDur;
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '2010-01-01T13:06:15+03:00',
            date2: '2039-01-01T13:06:15+03:00',
          },
          expression: 'INVESTMENT_DURATION(date1, date2, 0.03, 0.12, 2, "us_nasd")',
          result: 10.4300076,
        },
        {
          context: {
            date1: '2010-01-01T13:06:15+03:00',
            date2: '2039-01-01T13:06:15+03:00',
          },
          expression: 'INVESTMENT_DURATION(date1, date2, 0.03, 0.12, 2)',
          result: 10.4300076,
        },
        {
          context: {
            date3: '2009-01-30T13:06:15+03:00',
            date4: '2023-07-30T13:06:15+03:00',
          },
          expression: 'INVESTMENT_DURATION(date3, date4, 0.05, 1.5, 4, "us_nasd")',
          result: 0.91667041,
        },
        {
          context: {
            date3: '2009-01-30T13:06:15+03:00',
            date4: '2023-07-30T13:06:15+03:00',
          },
          expression: 'INVESTMENT_DURATION(date3, date4, 0.05, 1.5, 4, "actual_360")',
          result: 1.12222596,
        },
      ],
    };

  }, { "../../datetime": 89, "../date/YEARFRAC": 183, "./COUPNUM": 198 }], 212: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');
    const { execute: INVESTMENT_DURATION } = require('./INVESTMENT_DURATION');

    module.exports = {
      name: 'INVESTMENT_MDURATION',
      category: 'Financial',
      title: 'Calculates the modified Macaulay duration of a security paying periodic interest',
      description: 'The INVESTMENT_MDURATION function calculates the modified Macaulay duration of a security paying periodic interest, such as a US Treasury Bond, based on expected yield.',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Settlement',
        description: 'The settlement date of the security, the date after issuance when the security is delivered to the buyer.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Maturity',
        description: 'The maturity or end date of the security, when it can be redeemed at face or par value.',
      }, {
        type: 'number',
        title: 'Rate',
        description: 'The annualized rate of interest.',
      }, {
        type: 'number',
        title: 'Yield',
        description: 'The expected annual yield of the security.',
      }, {
        type: 'number',
        title: 'Frequency',
        description: 'The number of interest or coupon payments per year (1, 2, or 4).',
        enum: [1, 2, 4],
      }, {
        type: 'string',
        title: 'Basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({
        schema: { type: 'number' },
      }),
      execute: ([settlement, maturity, rate, fYield, frequency, convention]) => {
        if (typeof settlement !== 'string' && typeof maturity !== 'string') {
          return null;
        }
        if (typeof frequency !== 'number' && !Number.isFinite(frequency)
          && typeof rate !== 'number' && !Number.isFinite(rate)
          && typeof fYield !== 'number' && !Number.isFinite(fYield)) {
          return null;
        }
        if (rate < 0 || fYield < 0) {
          return null;
        }
        const startDate = parseDateTime(settlement);
        const endDate = parseDateTime(maturity);
        if (startDate >= endDate) {
          return null;
        }
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          const valuesPeriod = [1, 2, 4];
          if (!valuesBasis.includes(basis) || !valuesPeriod.includes(frequency)) {
            return null;
          }
          let fRet = INVESTMENT_DURATION([settlement, maturity, rate, fYield, frequency, basis]);
          fRet /= 1.0 + (fYield / frequency);
          return fRet;
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '2010-01-01T13:06:15+03:00',
            date2: '2039-01-01T13:06:15+03:00',
          },
          expression: 'INVESTMENT_MDURATION(date1, date2, 0.03, 0.12, 2, "us_nasd")',
          result: 9.83962981,
        },
        {
          context: {
            date1: '2010-01-01T13:06:15+03:00',
            date2: '2039-01-01T13:06:15+03:00',
          },
          expression: 'INVESTMENT_MDURATION(date1, date2, 0.03, 0.12, 2)',
          result: 9.83962981,
        },
        {
          context: {
            date3: '2009-01-30T13:06:15+03:00',
            date4: '2023-07-30T13:06:15+03:00',
          },
          expression: 'INVESTMENT_MDURATION(date3, date4, 0.05, 1.5, 4, "us_nasd")',
          result: 0.66666939,
        },
        {
          context: {
            date3: '2009-01-30T13:06:15+03:00',
            date4: '2023-07-30T13:06:15+03:00',
          },
          expression: 'INVESTMENT_MDURATION(date3, date4, 0.05, 1.5, 4, "actual_360")',
          result: 0.81616434,
        },
      ],
    };

  }, { "../../datetime": 89, "./INVESTMENT_DURATION": 211 }], 213: [function (require, module, exports) {
    const { INVESTMENT_PDURATION } = require('../../financial');

    module.exports = {
      name: 'INVESTMENT_PDURATION',
      category: 'Financial',
      title: 'Periods for an investment',
      description: 'Returns the number of periods for an investment to reach a specific value at a given rate',
      parameters: [{
        type: 'number',
        title: 'Rate',
        description: 'The rate at which the investment grows each period',
      }, {
        type: 'number',
        title: 'Present value',
        description: 'The investment\'s current value.',
      }, {
        type: 'number',
        title: 'Future value',
        description: 'The investment\'s desired future value.',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, present, future]) => {
        if (rate == null || present == null || future == null) {
          return null;
        }
        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || typeof present !== 'number'
          || !Number.isFinite(present)
          || typeof future !== 'number'
          || !Number.isFinite(future)
        ) {
          return null;
        }
        const result = INVESTMENT_PDURATION(rate, present, future);
        return Number.isFinite(result) ? result : null;
      },
      examples: [
        { expression: 'INVESTMENT_PDURATION(0.1, 10, 20)', result: 7.272540897 },
        { expression: 'INVESTMENT_PDURATION(1, 10, 20)', result: 1 },
        { expression: 'INVESTMENT_PDURATION(1, 10, 30)', result: 1.584962501 },
        { expression: 'INVESTMENT_PDURATION(1, 20, 20)', result: 0 },
      ],
    };

  }, { "../../financial": 91 }], 214: [function (require, module, exports) {
    const { IPMT } = require('../../financial');

    module.exports = {
      name: 'IPMT',
      category: 'Financial',
      title: 'IPMT',
      description: 'Calculates the payment on interest for an investment based on constant-amount periodic payments and a constant interest rate.',
      parameters: [{
        type: 'number',
        title: 'Rate',
        description: 'The interest rate',
      }, {
        type: 'number',
        title: 'Period',
        description: 'The amortization period, in terms of number of periods. '
          + 'Period must be at least "1" and at most "Number of periods"',
      }, {
        type: 'number',
        title: 'Number of periods',
        description: 'The number of payments to be made',
      }, {
        type: 'number',
        title: 'Present value',
        description: 'The current value of the annuity.',
      }, {
        type: 'number',
        title: 'Future value',
        description: 'The future value remaining after the final payment has been made (Default is 0).',
        default: 0,
      }, {
        type: 'boolean',
        title: 'Payments at the beginning of period',
        description: 'Whether payments are due at the end (false) or beginning (true) of each period',
        default: false,
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, period, periods, present, future, isOnBeginning]) => {
        if (rate == null || period == null || periods == null || present == null) {
          return null;
        }
        const futureSaved = future || 0;
        const isOnBeginningSaved = isOnBeginning || false;

        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || typeof period !== 'number'
          || !Number.isFinite(period)
          || typeof periods !== 'number'
          || !Number.isFinite(periods)
          || typeof present !== 'number'
          || !Number.isFinite(present)
          || typeof futureSaved !== 'number'
          || !Number.isFinite(futureSaved)
          || typeof isOnBeginningSaved !== 'boolean'
        ) {
          return null;
        }
        const result = IPMT(rate, period, periods, present, futureSaved, isOnBeginningSaved);
        return Number.isFinite(result) ? Number(result.toFixed(2)) : null;
      },
      examples: [
        { expression: 'IPMT(0.1, 4, 12, 100)', result: -8.45 },
        { expression: 'IPMT(0.5, 4, 12, 100)', result: -49.08 },
        { expression: 'IPMT(0.5, 6, 12, 100)', result: -47.44 },
        { expression: 'IPMT(0.5, 6, 12, 100, 200)', result: -42.32 },
        { expression: 'IPMT(0.5, 6, 12, 100, 200, true)', result: -28.21 },
      ],
    };

  }, { "../../financial": 91 }], 215: [function (require, module, exports) {
    // Calculates the resulting amount
    const irrResult = (values, dates, rate) => {
      const r = rate + 1;
      let result = values[0];
      // eslint-disable-next-line no-plusplus
      for (let i = 1; i < values.length; i++) {
        result += values[i] / (r ** ((dates[i] - dates[0]) / 365));
      }
      return result;
    };

    // Calculates the first derivation
    const irrResultDeriv = (values, dates, rate) => {
      const r = rate + 1;
      let result = 0;
      // eslint-disable-next-line no-plusplus
      for (let i = 1; i < values.length; i++) {
        const frac = (dates[i] - dates[0]) / 365;
        result -= frac * values[i] / (r ** (frac + 1));
      }
      return result;
    };

    module.exports = {
      name: 'IRR',
      category: 'Financial',
      title: 'Calculates the internal rate of return on an investment based on a series of periodic cash flows',
      description: 'Calculates the internal rate of return on an investment based on a series of periodic cash flows.',
      parameters: [{
        type: 'array',
        title: 'Сashflow amounts',
        description: 'An array containing the income or payments associated with the investment.',
      }, {
        type: 'number',
        title: 'Rate guess',
        description: 'An estimate for what the internal rate of return will be.',
        default: 0.1,
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([array, rate]) => {
        if (!Array.isArray(array)) {
          return null;
        }
        const values = array.filter(item => (typeof item === 'number' && Number.isFinite(item)));
        if (!values.length) {
          return null;
        }
        const dates = [];
        let positive = false;
        let negative = false;

        // eslint-disable-next-line no-plusplus
        for (let i = 0; i < values.length; i++) {
          dates[i] = (i === 0) ? 0 : dates[i - 1] + 365;
          if (values[i] > 0) {
            positive = true;
          }
          if (values[i] < 0) {
            negative = true;
          }
        }

        // Return error if values does not contain at least one positive value and one negative value
        if (!positive || !negative) {
          return null;
        }

        // Initialize guess and resultRate
        const guess = (rate === undefined) ? 0.1 : rate;
        if (typeof guess !== 'number' && !Number.isFinite(guess)) {
          return null;
        }
        // Set maximum epsilon for end of iteration
        const epsMax = 1e-10;

        // Implement Newton's method
        let resultRate = 0;
        let newRate; let epsRate; let
          resultValue;
        let contLoop = true;
        do {
          resultValue = irrResult(values, dates, resultRate);
          newRate = resultRate - resultValue / irrResultDeriv(values, dates, resultRate);
          epsRate = Math.abs(newRate - resultRate);
          resultRate = newRate;
          contLoop = (epsRate > epsMax) && (Math.abs(resultValue) > epsMax);
        } while (contLoop);

        // Return internal rate of return
        return resultRate;
      },
      examples: [
        {
          expression: 'IRR(array1)',
          context: {
            array1: [-20000, -1000, 9000, 15000],
          },
          result: 0.05327951,
        },
        {
          expression: 'IRR(array1, 0.1)',
          context: {
            array1: [-20000, -1000, 9000, 15000],
          },
          result: 0.05327951,
        },
        {
          expression: 'IRR(array2, 0.8)',
          context: {
            array2: [-100000, -80000, -40000, 10000, 80000, 100000, 150000],
          },
          result: 0.10378603,
        },
      ],
    };

  }, {}], 216: [function (require, module, exports) {
    const { ISPMT } = require('../../financial');

    module.exports = {
      name: 'ISPMT',
      category: 'Financial',
      title: 'ISPMT',
      description: 'Calculates the periodic payment for an annuity investment based on constant-amount periodic payments and a constant interest rate.',
      parameters: [{
        type: 'number',
        title: 'Rate',
        description: 'The interest rate',
      }, {
        type: 'number',
        title: 'Period',
        description: 'The time frame for which you want to view the interest payment.',
      }, {
        type: 'number',
        title: 'Number of periods',
        description: 'The number of payments to be made.',
      }, {
        type: 'number',
        title: 'Present value',
        description: 'The current value of the annuity.',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, period, periods, present]) => {
        if (rate == null || period == null || periods == null || present == null) {
          return null;
        }

        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || typeof period !== 'number'
          || !Number.isFinite(period)
          || typeof periods !== 'number'
          || !Number.isFinite(periods)
          || typeof present !== 'number'
          || !Number.isFinite(present)
        ) {
          return null;
        }

        const result = ISPMT(rate, period, periods, present);
        return Number.isFinite(result) ? result : null;
      },
      examples: [
        { expression: 'ISPMT(0.1, 3, 12, 100)', result: -7.5 },
        { expression: 'ISPMT(1, 3, 12, 100)', result: -75 },
        { expression: 'ISPMT(1, 6, 12, 100)', result: -50 },
        { expression: 'ISPMT(1, 3, 12, 200)', result: -150 },
      ],
    };

  }, { "../../financial": 91 }], 217: [function (require, module, exports) {
    const { MIRR } = require('../../financial');

    module.exports = {
      name: 'MIRR',
      category: 'Financial',
      title: 'Modified internal rate',
      description: 'Calculates the modified internal rate of return on an investment based on a series of periodic cash flows '
        + 'and the difference between the interest rate paid on financing versus the return received on reinvested income. Returns result in decimal number.',
      parameters: [{
        type: 'array',
        title: 'Сashflow amounts',
        description: 'An array containing the income or payments associated with the investment.',
        items: {
          type: 'number',
        },
      }, {
        type: 'number',
        title: 'Financing rate',
        description: 'The interest rate paid on funds invested.',
      }, {
        type: 'number',
        title: 'Reinvestment return rate',
        description: 'The return (as a percentage) earned on reinvestment of income received from the investment',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([array, financingRate, reinvestmentRate]) => {
        if (!Array.isArray(array)) {
          return null;
        }

        if (typeof financingRate !== 'number'
          || !Number.isFinite(financingRate)
          || typeof reinvestmentRate !== 'number'
          || !Number.isFinite(reinvestmentRate)
        ) {
          return null;
        }

        const values = array.filter(item => (typeof item === 'number' && Number.isFinite(item)));
        if (!values.length) {
          return null;
        }

        const result = MIRR(values, financingRate, reinvestmentRate);

        return Number.isFinite(result) ? result : null;
      },
      examples: [
        { expression: 'MIRR([1, -1, 2, 3, 4, 5], 1, 1)', result: 1.709384058 },
        { expression: 'MIRR([1, -1, 2, 3, 4, 5], 1, 2.5)', result: 3.21691195 },
        { expression: 'MIRR([2, -2, 2, 3, 4, 5], 0, 1)', result: 1.208166722 },
        { expression: 'MIRR([1, -1, 2, 3, 4, 5], 2, 1)', result: 1.938250528 },
      ],
    };

  }, { "../../financial": 91 }], 218: [function (require, module, exports) {
    const { NPV } = require('../../financial');

    module.exports = {
      name: 'NET_PRESENT_VALUE',
      category: 'Financial',
      title: 'Net present value',
      description: 'Calculates the net present value of an investment based on a series of periodic cash flows and a discount rate.',
      parameters: [{
        type: 'number',
        title: 'Discount rate',
        description: 'The discount rate of the investment over one period.',
      }, {
        type: 'array',
        title: 'Cash flows',
        description: 'Future cash flows',
        items: {
          type: 'number',
        },
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, cashFlows]) => {
        if (rate == null || cashFlows == null) {
          return null;
        }
        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || !Array.isArray(cashFlows)
        ) {
          return null;
        }
        const cashFlowsValid = cashFlows.filter(item => (typeof item === 'number' && Number.isFinite(item)));
        const result = NPV(rate, cashFlowsValid);
        return Number.isFinite(result) ? Number(result.toFixed(2)) : null;
      },
      examples: [
        { expression: 'NET_PRESENT_VALUE(1, [1, 2, 3, 4, 5])', result: 1.78 },
        { expression: 'NET_PRESENT_VALUE(0.5, [1, 2, 3, 4, 5])', result: 3.89 },
        { expression: 'NET_PRESENT_VALUE(0, [1, 2, 3, 4, 5])', result: 15 },
      ],
    };

  }, { "../../financial": 91 }], 219: [function (require, module, exports) {
    const { NOMINAL } = require('../../financial');

    module.exports = {
      name: 'NOMINAL',
      category: 'Financial',
      title: 'Nominal',
      description: 'Calculates the annual nominal interest rate given the effective rate and number of compounding periods per year.',
      parameters: [{
        type: 'number',
        title: 'Effective rate',
        description: 'The effective interest rate per year.',
      }, {
        type: 'number',
        title: 'Periods per year',
        description: 'The number of compounding periods per year.',
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, periods]) => {
        if (typeof rate === 'number'
          && Number.isFinite(rate)
          && typeof periods === 'number'
          && Number.isFinite(periods)
        ) {
          const result = NOMINAL(rate, periods);
          return Number.isFinite(result) ? result : null;
        }
        return null;
      },
      examples: [
        { expression: 'NOMINAL(0.1, 5)', result: 0.09622438246 },
        { expression: 'NOMINAL(0.5, 5)', result: 0.422358856 },
        { expression: 'NOMINAL(1, 5)', result: 0.743491775 },
        { expression: 'NOMINAL(1, 6)', result: 0.7347722899 },
      ],
    };

  }, { "../../financial": 91 }], 220: [function (require, module, exports) {
    const { NPER } = require('../../financial');

    module.exports = {
      name: 'NPER',
      category: 'Financial',
      title: 'Number of payment periods',
      description: 'calculates the number of payment periods for an investment based on constant-amount periodic payments and a constant interest rate.',
      parameters: [{
        type: 'number',
        title: 'Rate',
        description: 'The interest rate',
      }, {
        type: 'number',
        title: 'Payment amount',
        description: 'The amount of each payment made',
      }, {
        type: 'number',
        title: 'Present value',
        description: 'The current value of the annuity',
      }, {
        type: 'number',
        title: 'Future value',
        description: 'The future value remaining after the final payment has been made',
        default: 0,
      }, {
        type: 'boolean',
        title: 'Payments at the beginning of period',
        description: 'Whether payments are due at the end (false) or beginning (true) of each period',
        default: false,
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, amount, present, future, isOnBeginning]) => {
        if (rate == null || amount == null || present === null) {
          return null;
        }
        const futureSaved = future || 0;
        const isOnBeginningSaved = isOnBeginning || false;

        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || typeof amount !== 'number'
          || !Number.isFinite(amount)
          || typeof present !== 'number'
          || !Number.isFinite(present)
          || typeof futureSaved !== 'number'
          || !Number.isFinite(futureSaved)
          || typeof isOnBeginningSaved !== 'boolean'
        ) {
          return null;
        }

        const result = NPER(rate, amount, present, futureSaved, isOnBeginningSaved);
        return Number.isFinite(result) ? result : null;
      },
      examples: [
        { expression: 'NPER(0.1, 4, 100)', result: -13.14406259 },
        { expression: 'NPER(0.5, 4, 100)', result: -6.419022583 },
        { expression: 'NPER(0.5, 6, 100)', result: -5.508716229 },
        { expression: 'NPER(0.5, 6, 100, -200)', result: 1.573717172 },
        { expression: 'NPER(0.5, 6, 100, -200, true)', result: 1.513842809 },
      ],
    };

  }, { "../../financial": 91 }], 221: [function (require, module, exports) {
    const { PMT } = require('../../financial');

    module.exports = {
      name: 'PMT',
      category: 'Financial',
      title: 'PMT',
      description: 'Calculates the periodic payment for an annuity investment based on constant-amount periodic payments and a constant interest rate.',
      parameters: [{
        type: 'number',
        title: 'Rate',
        description: 'The interest rate',
      }, {
        type: 'number',
        title: 'Number of periods',
        description: 'The number of payments to be made',
      }, {
        type: 'number',
        title: 'Present value',
        description: 'The current value of the annuity.',
      }, {
        type: 'number',
        title: 'Future value',
        description: 'The future value remaining after the final payment has been made (Default is 0).',
        default: 0,
      }, {
        type: 'boolean',
        title: 'Payments at the beginning of period',
        description: 'Whether payments are due at the end (false) or beginning (true) of each period',
        default: false,
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, periods, present, future, isOnBeginning]) => {
        if (rate == null || periods == null || present == null) {
          return null;
        }
        const futureSaved = future || 0;
        const isOnBeginningSaved = isOnBeginning || false;

        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || typeof periods !== 'number'
          || !Number.isFinite(periods)
          || typeof present !== 'number'
          || !Number.isFinite(present)
          || typeof future !== 'number'
          || !Number.isFinite(future)
          || typeof isOnBeginningSaved !== 'boolean'
        ) {
          return null;
        }

        const result = PMT(rate, periods, present, futureSaved, isOnBeginningSaved);

        if (Number.isFinite(result)) {
          return Number(result.toFixed(2));
        }
        return null;
      },
      examples: [
        { expression: 'PMT(0.1, 4, 100)', result: -31.55 },
        { expression: 'PMT(0.5, 4, 100)', result: -62.31 },
        { expression: 'PMT(0.5, 6, 100)', result: -54.81 },
        { expression: 'PMT(0.5, 6, 100, 200)', result: -64.44 },
        { expression: 'PMT(0.5, 6, 100, 200, true)', result: -42.96 },
      ],
    };

  }, { "../../financial": 91 }], 222: [function (require, module, exports) {
    const { PPMT } = require('../../financial');

    module.exports = {
      name: 'PPMT',
      category: 'Financial',
      title: 'PPMT',
      description: 'Calculates the payment on interest for an investment based on constant-amount periodic payments and a constant interest rate.',
      parameters: [{
        type: 'number',
        title: 'Rate',
        description: 'The interest rate',
      }, {
        type: 'number',
        title: 'Period',
        description: 'The amortization period, in terms of number of periods. '
          + 'Period must be at least "1" and at most "Number of periods"',
      }, {
        type: 'number',
        title: 'Number of periods',
        description: 'The number of payments to be made',
      }, {
        type: 'number',
        title: 'Present value',
        description: 'The current value of the annuity.',
      }, {
        type: 'number',
        title: 'Future value',
        description: 'The future value remaining after the final payment has been made (Default is 0).',
        default: 0,
      }, {
        type: 'boolean',
        title: 'Payments at the beginning of period',
        description: 'Whether payments are due at the end (false) or beginning (true) of each period',
        default: false,
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, period, periods, present, future, isOnBeginning]) => {
        if (rate == null || period == null || periods == null || present == null) {
          return null;
        }
        const futureSaved = future || 0;
        const isOnBeginningSaved = isOnBeginning || false;

        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || typeof period !== 'number'
          || !Number.isFinite(period)
          || typeof periods !== 'number'
          || !Number.isFinite(periods)
          || typeof present !== 'number'
          || !Number.isFinite(present)
          || typeof futureSaved !== 'number'
          || !Number.isFinite(futureSaved)
          || typeof isOnBeginningSaved !== 'boolean'
        ) {
          return null;
        }
        const result = PPMT(rate, period, periods, present, futureSaved, isOnBeginningSaved);
        return Number.isFinite(result) ? Number(result.toFixed(2)) : null;
      },
      examples: [
        { expression: 'PPMT(0.1, 4, 12, 100)', result: -6.22 },
        { expression: 'PPMT(0.5, 4, 12, 100)', result: -1.31 },
        { expression: 'PPMT(0.5, 6, 12, 100)', result: -2.95 },
        { expression: 'PPMT(0.5, 6, 12, 100, 200)', result: -8.85 },
        { expression: 'PPMT(0.5, 6, 12, 100, 200, true)', result: -5.9 },
      ],
    };

  }, { "../../financial": 91 }], 223: [function (require, module, exports) {
    const { PV } = require('../../financial');

    module.exports = {
      name: 'PRESENT_VALUE',
      category: 'Financial',
      title: 'Present value',
      description: 'Calculates the present value of an annuity investment based on constant-amount periodic payments and a constant interest rate.',
      parameters: [{
        type: 'number',
        title: 'Rate',
        description: 'The interest rate',
      }, {
        type: 'number',
        title: 'Periods',
        description: 'The number of payments to be made',
      }, {
        type: 'number',
        title: 'Payment amount',
        description: 'The amount per period to be paid',
      }, {
        type: 'number',
        title: 'Future value',
        description: 'The future value remaining after the final payment has been made',
        default: 0,
      }, {
        type: 'boolean',
        title: 'Payments at the beginning of period',
        description: 'Whether payments are due at the end (false) or beginning (true) of each period',
        default: false,
      }],
      analyze: () => ({ schema: { type: 'number' } }),
      execute: ([rate, periods, amount, future, isOnBeginning]) => {
        if (rate == null || periods == null || amount === null) {
          return null;
        }
        const futureSaved = future || 0;
        const isOnBeginningSaved = isOnBeginning || false;

        if (typeof rate !== 'number'
          || !Number.isFinite(rate)
          || typeof periods !== 'number'
          || !Number.isFinite(periods)
          || typeof amount !== 'number'
          || !Number.isFinite(amount)
          || typeof futureSaved !== 'number'
          || !Number.isFinite(futureSaved)
          || typeof isOnBeginningSaved !== 'boolean'
        ) {
          return null;
        }

        const result = PV(rate, periods, amount, futureSaved, isOnBeginningSaved);
        return Number.isFinite(result) ? Number(result.toFixed(2)) : null;
      },
      examples: [
        { expression: 'PRESENT_VALUE(0.1, 4, 100)', result: -316.99 },
        { expression: 'PRESENT_VALUE(0.5, 4, 100)', result: -160.49 },
        { expression: 'PRESENT_VALUE(0.5, 6, 100)', result: -182.44 },
        { expression: 'PRESENT_VALUE(0.5, 6, 100, 200)', result: -200.00 },
        { expression: 'PRESENT_VALUE(0.5, 6, 100, 200, true)', result: -291.22 },
      ],
    };

  }, { "../../financial": 91 }], 224: [function (require, module, exports) {
    const { parseDateTime } = require('../../datetime');
    const { execute: COUPNUM } = require('./COUPNUM');
    const { execute: COUPDAYS } = require('./COUPDAYS');
    const { execute: COUPDAYBS } = require('./COUPDAYBS');
    const { execute: COUPDAYSNC } = require('./COUPDAYSNC');

    module.exports = {
      name: 'PRICE',
      category: 'Financial',
      title: 'Calculates the price of a security paying periodic interest',
      description: 'The PRICE function calculates the price of a security paying periodic interest, such as a US Treasury Bond, based on expected yield',
      parameters: [{
        type: 'string',
        format: 'date-time',
        title: 'Settlement',
        description: 'The settlement date of the security, the date after issuance when the security is delivered to the buyer.',
      }, {
        type: 'string',
        format: 'date-time',
        title: 'Maturity',
        description: 'The maturity or end date of the security, when it can be redeemed at face or par value.',
      }, {
        type: 'number',
        title: 'Rate',
        description: 'The annualized rate of interest.',
      }, {
        type: 'number',
        title: 'Yield',
        description: 'The expected annual yield of the security.',
      }, {
        type: 'number',
        title: 'Redemption',
        description: 'The redemption value of the security.',
      }, {
        type: 'number',
        title: 'Frequency',
        description: 'The number of interest or coupon payments per year (1, 2, or 4).',
        enum: [1, 2, 4],
      }, {
        type: 'string',
        title: 'Basis',
        description: 'An indicator of what day count method to use:\n'
          + '\n- `us_nasd`: Default. US (NASD), i.e. 30 day month and 360 day year.'
          + '\n- `actual`: Uses the actual number of days per month and year.'
          + '\n- `actual_360`: Uses the actual number of days per month, but with a 360 day year.'
          + '\n- `actual_365`: Uses the actual number of days per month, but assumes a 365 day year, so leap years are not recognized.'
          + '\n- `european`: Similar to us_nasd, but uses European standards instead of US when the months are not 30 days long.',
        enum: ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'],
        default: 'us_nasd',
      }],
      analyze: () => ({
        schema: { type: 'number' },
      }),
      execute: ([settlement, maturity, rate, fYield, redemption, frequency, convention]) => {
        if (typeof settlement !== 'string' && typeof maturity !== 'string') {
          return null;
        }
        if (typeof frequency !== 'number' && !Number.isFinite(frequency)
          && typeof rate !== 'number' && !Number.isFinite(rate)
          && typeof fYield !== 'number' && !Number.isFinite(fYield)
          && typeof redemption !== 'number' && !Number.isFinite(redemption)) {
          return null;
        }
        if (rate < 0 || fYield < 0 || redemption <= 0) {
          return null;
        }
        const startDate = parseDateTime(settlement);
        const endDate = parseDateTime(maturity);
        if (startDate >= endDate) {
          return null;
        }
        if (startDate.isValid() && endDate.isValid()) {
          const basis = convention || 'us_nasd';
          const valuesBasis = ['us_nasd', 'actual', 'actual_360', 'actual_365', 'european'];
          const valuesPeriod = [1, 2, 4];
          if (!valuesBasis.includes(basis) || !valuesPeriod.includes(frequency)) {
            return null;
          }
          const fE = COUPDAYS([settlement, maturity, frequency, basis]);
          const fDSCE = COUPDAYSNC([settlement, maturity, frequency, basis]) / fE;
          const fN = COUPNUM([settlement, maturity, frequency, basis]);
          const fA = COUPDAYBS([settlement, maturity, frequency, basis]);
          let fRet = redemption / ((1 + fYield / frequency) ** (fN - 1 + fDSCE));
          fRet -= 100 * rate / frequency * fA / fE;
          const fT1 = 100 * rate / frequency;
          const fT2 = 1 + fYield / frequency;
          // eslint-disable-next-line no-plusplus
          for (let fK = 0; fK < fN; fK++) {
            fRet += fT1 / (fT2 ** (fK + fDSCE));
          }
          return fRet;
        }
        return null;
      },
      examples: [
        {
          context: {
            date1: '2010-01-02T13:06:15+03:00',
            date2: '2039-12-31T13:06:15+03:00',
          },
          expression: 'PRICE(date1, date2, 0.03, 1.2, 100, 2, "us_nasd")',
          result: 2.49642314,
        },
        {
          context: {
            date1: '2010-01-02T13:06:15+03:00',
            date2: '2039-12-